<?php
/**
 * WooCommerce Template Functions.
 *
 * @package airtech
 */

if ( ! function_exists( 'airtech_before_content' ) ) {
	/**
	 * Before Content
	 * Wraps all WooCommerce content in wrappers which match the theme markup
	 *
	 * @since   1.0.0
	 * @return  void
	 */
	function airtech_before_content() {
		?>
		<div id="primary" class="content-area">
			<main id="main" class="site-main">
		<?php
	}
}

if ( ! function_exists( 'airtech_after_content' ) ) {
	/**
	 * After Content
	 * Closes the wrapping divs
	 *
	 * @since   1.0.0
	 * @return  void
	 */
	function airtech_after_content() {
		?>
			</main><!-- #main -->
		</div><!-- #primary -->

		<?php
		if( is_product() ) {
			do_action( 'airtech_product_sidebar' );
		}
		else {
			do_action( 'airtech_shop_sidebar' );
		}
	}
}

if ( ! function_exists( 'airtech_cart_link_fragment' ) ) {
	/**
	 * Cart Fragments
	 * Ensure cart contents update when products are added to the cart via AJAX
	 *
	 * @param  array $fragments Fragments to refresh via AJAX.
	 * @return array            Fragments to refresh via AJAX
	 */
	function airtech_cart_link_fragment( $fragments ) {
		global $woocommerce;

		ob_start();
		airtech_cart_link();
		$fragments['a.cart-contents'] = ob_get_clean();

		return $fragments;
	}
}

if ( ! function_exists( 'airtech_cart_link' ) ) {
	/**
	 * Cart Link
	 * Displayed a link to the cart including the number of items present and the cart total
	 *
	 * @return void
	 * @since  1.0.0
	 */
	function airtech_cart_link() {
		?>
			<a class="cart-contents" href="<?php echo esc_url( wc_get_cart_url() ); ?>" title="<?php esc_attr_e( 'View your shopping cart', 'airtech' ); ?>">
				<?php /* translators: %d: number of items in cart */ ?>
				<?php echo wp_kses_post( WC()->cart->get_cart_subtotal() ); ?> <span class="count"><?php echo wp_kses_data( sprintf( _n( '%d item', '%d items', WC()->cart->get_cart_contents_count(), 'airtech' ), WC()->cart->get_cart_contents_count() ) ); ?></span>
			</a>
		<?php
	}
}

if ( ! function_exists( 'airtech_product_search' ) ) {
	/**
	 * Display Product Search
	 *
	 * @since  1.0.0
	 * @uses  airtech_is_woocommerce_activated() check if WooCommerce is activated
	 * @return void
	 */
	function airtech_product_search() {
		if ( airtech_is_woocommerce_activated() ) {
			?>
			<div class="site-search">
				<?php the_widget( 'WC_Widget_Product_Search', 'title=' ); ?>
			</div>
			<?php
		}
	}
}

if ( ! function_exists( 'airtech_header_cart' ) ) {
	/**
	 * Display Header Cart
	 *
	 * @since  1.0.0
	 * @uses  airtech_is_woocommerce_activated() check if WooCommerce is activated
	 */
	function airtech_header_cart() {
		if ( airtech_is_woocommerce_activated() ) {
			if ( is_cart() ) {
				$class = 'current-menu-item';
			} else {
				$class = '';
			}
		ob_start();
			?>
		<ul id="site-header-cart" class="site-header-cart menu">
			<li class="<?php echo esc_attr( $class ); ?>">
				<?php airtech_cart_link(); ?>
			</li>
			<li>
				<?php the_widget( 'WC_Widget_Cart', 'title=' ); ?>
			</li>
		</ul>
			<?php
		$output = ob_get_clean();
		echo wp_kses_post($output);
		}
	}
}

if ( ! function_exists( 'airtech_upsell_display' ) ) {
	/**
	 * Upsells
	 * Replace the default upsell function with our own which displays the correct number product columns
	 *
	 * @since   1.0.0
	 * @return  void
	 * @uses    woocommerce_upsell_display()
	 */
	function airtech_upsell_display() {
		$columns = apply_filters( 'airtech_upsells_columns', 3 );
		woocommerce_upsell_display( -1, $columns );
	}
}

if ( ! function_exists( 'airtech_sorting_wrapper' ) ) {
	/**
	 * Sorting wrapper
	 *
	 * @since   1.4.3
	 * @return  void
	 */
	function airtech_sorting_wrapper() {
		echo '<div class="airtech-sorting">';
	}
}

if ( ! function_exists( 'airtech_sorting_wrapper_close' ) ) {
	/**
	 * Sorting wrapper close
	 *
	 * @since   1.4.3
	 * @return  void
	 */
	function airtech_sorting_wrapper_close() {
		echo '</div>';
	}
}

if ( ! function_exists( 'airtech_product_columns_wrapper' ) ) {
	/**
	 * Product columns wrapper
	 *
	 * @since   2.2.0
	 * @return  void
	 */
	function airtech_product_columns_wrapper() {
		$columns = airtech_loop_columns();
		echo '<div class="columns-' . absint( $columns ) . '">';
	}
}

if ( ! function_exists( 'airtech_loop_columns' ) ) {
	/**
	 * Default loop columns on product archives
	 *
	 * @return integer products per row
	 * @since  1.0.0
	 */
	function airtech_loop_columns() {
		$columns = 4; // 3 products per row

		if ( function_exists( 'wc_get_default_products_per_row' ) ) {
			$columns = wc_get_default_products_per_row();
		}

		return apply_filters( 'airtech_loop_columns', $columns );
	}
}

if ( ! function_exists( 'airtech_product_columns_wrapper_close' ) ) {
	/**
	 * Product columns wrapper close
	 *
	 * @since   2.2.0
	 * @return  void
	 */
	function airtech_product_columns_wrapper_close() {
		echo '</div>';
	}
}

if ( ! function_exists( 'airtech_shop_messages' ) ) {
	/**
	 * Airtech shop messages
	 *
	 * @since   1.4.4
	 * @uses    airtech_do_shortcode
	 */
	function airtech_shop_messages() {
		if ( ! is_checkout() ) {
			echo wp_kses_post( airtech_do_shortcode( 'woocommerce_messages' ) );
		}
	}
}

if ( ! function_exists( 'airtech_woocommerce_pagination' ) ) {
	/**
	 * Airtech WooCommerce Pagination
	 * WooCommerce disables the product pagination inside the woocommerce_product_subcategories() function
	 * but since Airtech adds pagination before that function is excuted we need a separate function to
	 * determine whether or not to display the pagination.
	 *
	 * @since 1.4.4
	 */
	function airtech_woocommerce_pagination() {
		if ( woocommerce_products_will_display() ) {
			woocommerce_pagination();
		}
	}
}

if ( ! function_exists( 'airtech_recent_products' ) ) {
	/**
	 * Display Recent Products
	 * Hooked into the `homepage` action in the homepage template
	 *
	 * @since  1.0.0
	 * @param array $args the product section args.
	 * @return void
	 */
	function airtech_recent_products( $args ) {
		$args = apply_filters(
			'airtech_recent_products_args', array(
				'limit'   => 4,
				'columns' => 4,
				'orderby' => 'date',
				'order'   => 'desc',
				'title'   => __( 'New In', 'airtech' ),
			)
		);

		$shortcode_content = airtech_do_shortcode(
			'products', apply_filters(
				'airtech_recent_products_shortcode_args', array(
					'orderby'  => esc_attr( $args['orderby'] ),
					'order'    => esc_attr( $args['order'] ),
					'per_page' => intval( $args['limit'] ),
					'columns'  => intval( $args['columns'] ),
				)
			)
		);

		/**
		 * Only display the section if the shortcode returns products
		 */
		if ( false !== strpos( $shortcode_content, 'product' ) ) {
			echo '<section class="airtech-product-section airtech-recent-products" aria-label="' . esc_attr__( 'Recent Products', 'airtech' ) . '">';

			do_action( 'airtech_homepage_before_recent_products' );

			echo '<h2 class="section-title">' . wp_kses_post( $args['title'] ) . '</h2>';

			do_action( 'airtech_homepage_after_recent_products_title' );

			echo wp_kses_post($shortcode_content); // WPCS: XSS ok.

			do_action( 'airtech_homepage_after_recent_products' );

			echo '</section>';
		}
	}
}

if ( ! function_exists( 'airtech_featured_products' ) ) {
	/**
	 * Display Featured Products
	 * Hooked into the `homepage` action in the homepage template
	 *
	 * @since  1.0.0
	 * @param array $args the product section args.
	 * @return void
	 */
	function airtech_featured_products( $args ) {
		$args = apply_filters(
			'airtech_featured_products_args', array(
				'limit'      => 4,
				'columns'    => 4,
				'orderby'    => 'date',
				'order'      => 'desc',
				'visibility' => 'featured',
				'title'      => __( 'We Recommend', 'airtech' ),
			)
		);

		$shortcode_content = airtech_do_shortcode(
			'products', apply_filters(
				'airtech_featured_products_shortcode_args', array(
					'per_page'   => intval( $args['limit'] ),
					'columns'    => intval( $args['columns'] ),
					'orderby'    => esc_attr( $args['orderby'] ),
					'order'      => esc_attr( $args['order'] ),
					'visibility' => esc_attr( $args['visibility'] ),
				)
			)
		);

		/**
		 * Only display the section if the shortcode returns products
		 */
		if ( false !== strpos( $shortcode_content, 'product' ) ) {
			echo '<section class="airtech-product-section airtech-featured-products" aria-label="' . esc_attr__( 'Featured Products', 'airtech' ) . '">';

			do_action( 'airtech_homepage_before_featured_products' );

			echo '<h2 class="section-title">' . wp_kses_post( $args['title'] ) . '</h2>';

			do_action( 'airtech_homepage_after_featured_products_title' );

			echo wp_kses_post($shortcode_content); // WPCS: XSS ok.

			do_action( 'airtech_homepage_after_featured_products' );

			echo '</section>';
		}
	}
}

if ( ! function_exists( 'airtech_popular_products' ) ) {
	/**
	 * Display Popular Products
	 * Hooked into the `homepage` action in the homepage template
	 *
	 * @since  1.0.0
	 * @param array $args the product section args.
	 * @return void
	 */
	function airtech_popular_products( $args ) {
		$args = apply_filters(
			'airtech_popular_products_args', array(
				'limit'   => 4,
				'columns' => 4,
				'orderby' => 'rating',
				'order'   => 'desc',
				'title'   => __( 'Fan Favorites', 'airtech' ),
			)
		);

		$shortcode_content = airtech_do_shortcode(
			'products', apply_filters(
				'airtech_popular_products_shortcode_args', array(
					'per_page' => intval( $args['limit'] ),
					'columns'  => intval( $args['columns'] ),
					'orderby'  => esc_attr( $args['orderby'] ),
					'order'    => esc_attr( $args['order'] ),
				)
			)
		);

		/**
		 * Only display the section if the shortcode returns products
		 */
		if ( false !== strpos( $shortcode_content, 'product' ) ) {
			echo '<section class="airtech-product-section airtech-popular-products" aria-label="' . esc_attr__( 'Popular Products', 'airtech' ) . '">';

			do_action( 'airtech_homepage_before_popular_products' );

			echo '<h2 class="section-title">' . wp_kses_post( $args['title'] ) . '</h2>';

			do_action( 'airtech_homepage_after_popular_products_title' );

			echo wp_kses_post($shortcode_content); // WPCS: XSS ok.

			do_action( 'airtech_homepage_after_popular_products' );

			echo '</section>';
		}
	}
}

if ( ! function_exists( 'airtech_on_sale_products' ) ) {
	/**
	 * Display On Sale Products
	 * Hooked into the `homepage` action in the homepage template
	 *
	 * @param array $args the product section args.
	 * @since  1.0.0
	 * @return void
	 */
	function airtech_on_sale_products( $args ) {
		$args = apply_filters(
			'airtech_on_sale_products_args', array(
				'limit'   => 4,
				'columns' => 4,
				'orderby' => 'date',
				'order'   => 'desc',
				'on_sale' => 'true',
				'title'   => __( 'On Sale', 'airtech' ),
			)
		);

		$shortcode_content = airtech_do_shortcode(
			'products', apply_filters(
				'airtech_on_sale_products_shortcode_args', array(
					'per_page' => intval( $args['limit'] ),
					'columns'  => intval( $args['columns'] ),
					'orderby'  => esc_attr( $args['orderby'] ),
					'order'    => esc_attr( $args['order'] ),
					'on_sale'  => esc_attr( $args['on_sale'] ),
				)
			)
		);

		/**
		 * Only display the section if the shortcode returns products
		 */
		if ( false !== strpos( $shortcode_content, 'product' ) ) {
			echo '<section class="airtech-product-section airtech-on-sale-products" aria-label="' . esc_attr__( 'On Sale Products', 'airtech' ) . '">';

			do_action( 'airtech_homepage_before_on_sale_products' );

			echo '<h2 class="section-title">' . wp_kses_post( $args['title'] ) . '</h2>';

			do_action( 'airtech_homepage_after_on_sale_products_title' );

			echo wp_kses_post($shortcode_content); // WPCS: XSS ok.

			do_action( 'airtech_homepage_after_on_sale_products' );

			echo '</section>';
		}
	}
}

if ( ! function_exists( 'airtech_best_selling_products' ) ) {
	/**
	 * Display Best Selling Products
	 * Hooked into the `homepage` action in the homepage template
	 *
	 * @since 2.0.0
	 * @param array $args the product section args.
	 * @return void
	 */
	function airtech_best_selling_products( $args ) {
		$args = apply_filters(
			'airtech_best_selling_products_args', array(
				'limit'   => 4,
				'columns' => 4,
				'orderby' => 'popularity',
				'order'   => 'desc',
				'title'   => esc_attr__( 'Best Sellers', 'airtech' ),
			)
		);

		$shortcode_content = airtech_do_shortcode(
			'products', apply_filters(
				'airtech_best_selling_products_shortcode_args', array(
					'per_page' => intval( $args['limit'] ),
					'columns'  => intval( $args['columns'] ),
					'orderby'  => esc_attr( $args['orderby'] ),
					'order'    => esc_attr( $args['order'] ),
				)
			)
		);

		/**
		 * Only display the section if the shortcode returns products
		 */
		if ( false !== strpos( $shortcode_content, 'product' ) ) {
			echo '<section class="airtech-product-section airtech-best-selling-products" aria-label="' . esc_attr__( 'Best Selling Products', 'airtech' ) . '">';

			do_action( 'airtech_homepage_before_best_selling_products' );

			echo '<h2 class="section-title">' . wp_kses_post( $args['title'] ) . '</h2>';

			do_action( 'airtech_homepage_after_best_selling_products_title' );

			echo wp_kses_post($shortcode_content); // WPCS: XSS ok.

			do_action( 'airtech_homepage_after_best_selling_products' );

			echo '</section>';
		}
	}
}

if ( ! function_exists( 'airtech_promoted_products' ) ) {
	/**
	 * Featured and On-Sale Products
	 * Check for featured products then on-sale products and use the appropiate shortcode.
	 * If neither exist, it can fallback to show recently added products.
	 *
	 * @since  1.5.1
	 * @param integer $per_page total products to display.
	 * @param integer $columns columns to arrange products in to.
	 * @param boolean $recent_fallback Should the function display recent products as a fallback when there are no featured or on-sale products?.
	 * @uses  airtech_is_woocommerce_activated()
	 * @uses  wc_get_featured_product_ids()
	 * @uses  wc_get_product_ids_on_sale()
	 * @uses  airtech_do_shortcode()
	 * @return void
	 */
	function airtech_promoted_products( $per_page = '2', $columns = '2', $recent_fallback = true ) {
		if ( airtech_is_woocommerce_activated() ) {

			if ( wc_get_featured_product_ids() ) {

				echo '<h2>' . esc_html__( 'Featured Products', 'airtech' ) . '</h2>';

				echo airtech_do_shortcode(
					'featured_products', array(
						'per_page' => $per_page,
						'columns'  => $columns,
					)
				); // WPCS: XSS ok.
			} elseif ( wc_get_product_ids_on_sale() ) {

				echo '<h2>' . esc_html__( 'On Sale Now', 'airtech' ) . '</h2>';

				echo airtech_do_shortcode(
					'sale_products', array(
						'per_page' => $per_page,
						'columns'  => $columns,
					)
				); // WPCS: XSS ok.
			} elseif ( $recent_fallback ) {

				echo '<h2>' . esc_html__( 'New In Store', 'airtech' ) . '</h2>';

				echo airtech_do_shortcode(
					'recent_products', array(
						'per_page' => $per_page,
						'columns'  => $columns,
					)
				); // WPCS: XSS ok.
			}
		}
	}
}

if ( ! function_exists( 'airtech_single_product_pagination' ) ) {
	/**
	 * Single Product Pagination
	 *
	 * @since 2.3.0
	 */
	function airtech_single_product_pagination() {
		if ( class_exists( 'Airtech_Product_Pagination' ) || true !== get_theme_mod( 'airtech_product_pagination' ) ) {
			return;
		}

		// Show only products in the same category?
		$in_same_term   = apply_filters( 'airtech_single_product_pagination_same_category', true );
		$excluded_terms = apply_filters( 'airtech_single_product_pagination_excluded_terms', '' );
		$taxonomy       = apply_filters( 'airtech_single_product_pagination_taxonomy', 'product_cat' );

		$previous_product = airtech_get_previous_product( $in_same_term, $excluded_terms, $taxonomy );
		$next_product     = airtech_get_next_product( $in_same_term, $excluded_terms, $taxonomy );

		if ( ! $previous_product && ! $next_product ) {
			return;
		}

		?>
		<nav class="airtech-product-pagination" aria-label="<?php esc_attr_e( 'More products', 'airtech' ); ?>">
			<?php if ( $previous_product ) : ?>
				<a href="<?php echo esc_url( $previous_product->get_permalink() ); ?>" rel="prev">
					<?php echo wp_kses_post( $previous_product->get_image() ); ?>
					<span class="airtech-product-pagination__title"><?php echo wp_kses_post( $previous_product->get_name() ); ?></span>
				</a>
			<?php endif; ?>

			<?php if ( $next_product ) : ?>
				<a href="<?php echo esc_url( $next_product->get_permalink() ); ?>" rel="next">
					<?php echo wp_kses_post( $next_product->get_image() ); ?>
					<span class="airtech-product-pagination__title"><?php echo wp_kses_post( $next_product->get_name() ); ?></span>
				</a>
			<?php endif; ?>
		</nav><!-- .airtech-product-pagination -->
		<?php
	}
}

if ( ! function_exists( 'airtech_sticky_single_add_to_cart' ) ) {
	/**
	 * Sticky Add to Cart
	 *
	 * @since 2.3.0
	 */
	function airtech_sticky_single_add_to_cart() {
		global $product;

		if ( class_exists( 'Airtech_Sticky_Add_to_Cart' ) || true !== get_theme_mod( 'airtech_sticky_add_to_cart' ) ) {
			return;
		}

		if ( ! is_product() ) {
			return;
		}

		$params = apply_filters(
			'airtech_sticky_add_to_cart_params', array(
				'trigger_class' => 'entry-summary',
			)
		);

		wp_localize_script( 'airtech-sticky-add-to-cart', 'airtech_sticky_add_to_cart_params', $params );

		wp_enqueue_script( 'airtech-sticky-add-to-cart' );
		?>
			<section class="airtech-sticky-add-to-cart">
				<div class="col-full">
					<div class="airtech-sticky-add-to-cart__content">
						<?php echo wp_kses_post( woocommerce_get_product_thumbnail() ); ?>
						<div class="airtech-sticky-add-to-cart__content-product-info">
							<span class="airtech-sticky-add-to-cart__content-title"><?php esc_attr_e( 'You\'re viewing:', 'airtech' ); ?> <strong><?php the_title(); ?></strong></span>
							<span class="airtech-sticky-add-to-cart__content-price"><?php echo wp_kses_post( $product->get_price_html() ); ?></span>
							<?php echo wp_kses_post( wc_get_rating_html( $product->get_average_rating() ) ); ?>
						</div>
						<a href="<?php echo esc_url( $product->add_to_cart_url() ); ?>" class="airtech-sticky-add-to-cart__content-button button alt">
							<?php echo esc_attr( $product->add_to_cart_text() ); ?>
						</a>
					</div>
				</div>
			</section><!-- .airtech-sticky-add-to-cart -->
		<?php
	}
}

if ( ! function_exists( 'airtech_woocommerce_brands_archive' ) ) {
	/**
	 * Display brand image on brand archives
	 * Requires WooCommerce Brands.
	 *
	 * @since  2.3.0
	 * @link   https://woocommerce.com/products/brands/
	 * @uses   is_tax()
	 * @uses   wp_kses_post()
	 * @uses   get_brand_thumbnail_image()
	 * @uses   get_queried_object()
	 * @return void
	 */
	function airtech_woocommerce_brands_archive() {
		if ( is_tax( 'product_brand' ) ) {
			echo wp_kses_post( get_brand_thumbnail_image( get_queried_object() ) );
		}
	}
}

if ( ! function_exists( 'airtech_woocommerce_brands_single' ) ) {
	/**
	 * Output product brand image for use on single product pages
	 * Requires WooCommerce Brands.
	 *
	 * @since  2.3.0
	 * @link   https://woocommerce.com/products/brands/
	 * @uses   airtech_do_shortcode()
	 * @uses   wp_kses_post()
	 * @return void
	 */
	function airtech_woocommerce_brands_single() {
		$brand = airtech_do_shortcode(
			'product_brand', array(
				'class' => '',
			)
		);

		if ( empty( $brand ) ) {
			return;
		}

		?>
		<div class="airtech-wc-brands-single-product">
			<?php echo wp_kses_post( $brand ); ?>
		</div>
		<?php
	}
}
