<?php
/**
 * Airtech Class
 *
 * @since    2.0.0
 * @package  airtech
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Airtech' ) ) :

	/**
	 * The main Airtech class
	 */
	class Airtech {

		/**
		 * Setup class.
		 *
		 * @since 1.0
		 */
		public function __construct() {
			add_action( 'after_setup_theme', array( $this, 'setup' ) );
			add_action( 'widgets_init', array( $this, 'widgets_init' ) );
			add_action( 'wp_enqueue_scripts', array( $this, 'scripts' ), 10 );
			add_action( 'wp_enqueue_scripts', array( $this, 'child_scripts' ), 30 ); // After WooCommerce.
			add_action( 'enqueue_block_editor_assets', array( $this, 'block_editor_assets' ) );
			add_filter( 'body_class', array( $this, 'body_classes' ) );
			add_filter( 'wp_page_menu_args', array( $this, 'page_menu_args' ) );
			add_filter( 'navigation_markup_template', array( $this, 'navigation_markup_template' ) );
			add_action( 'enqueue_embed_scripts', array( $this, 'print_embed_styles' ) );
			add_filter( 'block_editor_settings', array( $this, 'custom_editor_settings' ), 10, 2 );
		}

		/**
		 * Sets up theme defaults and registers support for various WordPress features.
		 *
		 * Note that this function is hooked into the after_setup_theme hook, which
		 * runs before the init hook. The init hook is too late for some features, such
		 * as indicating support for post thumbnails.
		 */
		public function setup() {
			/*
			 * Load Localisation files.
			 *
			 * Note: the first-loaded translation file overrides any following ones if the same translation is present.
			 */

			// Loads wp-content/languages/themes/airtech-it_IT.mo.
			load_theme_textdomain( 'airtech', trailingslashit( WP_LANG_DIR ) . 'themes' );

			// Loads wp-content/themes/child-theme-name/languages/it_IT.mo.
			load_theme_textdomain( 'airtech', get_stylesheet_directory() . '/languages' );

			// Loads wp-content/themes/airtech/languages/it_IT.mo.
			load_theme_textdomain( 'airtech', get_template_directory() . '/languages' );

			/**
			 * Add default posts and comments RSS feed links to head.
			 */
			add_theme_support( 'automatic-feed-links' );

			/*
			 * Enable support for Post Thumbnails on posts and pages.
			 *
			 * @link https://developer.wordpress.org/reference/functions/add_theme_support/#Post_Thumbnails
			 */
			add_theme_support( 'post-thumbnails' );

			/**
			 * Register menu locations.
			 */
			register_nav_menus(
				apply_filters(
					'airtech_register_nav_menus', array(
						'primary-menu'   => __( 'Primary Menu', 'airtech' ),
					)
				)
			);

			/*
			 * Switch default core markup for search form, comment form, comments, galleries, captions and widgets
			 * to output valid HTML5.
			 */
			add_theme_support(
				'html5', apply_filters(
					'airtech_html5_args', array(
						'search-form',
						'comment-form',
						'comment-list',
						'gallery',
						'caption',
						'widgets',
					)
				)
			);

			/**
			 * Setup the WordPress core custom background feature.
			 */
			add_theme_support(
				'custom-background', apply_filters(
					'airtech_custom_background_args', array(
						'default-color' => apply_filters( 'airtech_default_background_color', 'ffffff' ),
						'default-image' => '',
					)
				)
			);

			/**
			 *  Add support for the Site Logo plugin and the site logo functionality in JetPack
			 *  https://github.com/automattic/site-logo
			 *  http://jetpack.me/
			 */
			add_theme_support(
				'site-logo', apply_filters(
					'airtech_site_logo_args', array(
						'size' => 'full',
					)
				)
			);

			/**
			 * Declare support for title theme feature.
			 */
			add_theme_support( 'title-tag' );

			/**
			 * Declare support for selective refreshing of widgets.
			 */
			add_theme_support( 'customize-selective-refresh-widgets' );

			/**
			 * Add support for Block Styles.
			 */
			add_theme_support( 'wp-block-styles' );

			/**
			 * Add support for full and wide align images.
			 */
			add_theme_support( 'align-wide' );

			/**
			 * Add support for editor styles.
			 */
			add_theme_support( 'editor-styles' );

			/**
			 * Enqueue editor styles.
			 */
			add_editor_style( array( 'assets/css/base/gutenberg-editor.css', $this->google_fonts() ) );

			/**
			 * Add support for responsive embedded content.
			 */
			add_theme_support( 'responsive-embeds' );

			/**
			 * Add support for HFE plugin.
			 */
			add_theme_support( 'header-footer-elementor' );

		}

		/**
		 * Register widget area.
		 *
		 * @link https://codex.wordpress.org/Function_Reference/register_sidebar
		 */
		public function widgets_init() {
			$sidebar_args['sidebar'] = array(
				'name'        => __( 'Sidebar', 'airtech' ),
				'id'          => 'sidebar-1',
				'description' => 'Default Sidebar, appears on Default pages and Posts.',
			);

			if( class_exists( 'WooCommerce' ) ) {

				$sidebar_args['shop'] = array(
					'name'        => __( 'Shop Sidebar', 'airtech' ),
					'id'          => 'shop-sidebar',
					'description' => 'Appears on pages powered by Woocommerce. Except product single page.',
				);

				$sidebar_args['product'] = array(
					'name'        => __( 'Product Sidebar', 'airtech' ),
					'id'          => 'product-sidebar',
					'description' => 'Appears on product single page',
				);

			}

			$sidebar_args['header'] = array(
				'name'        => __( 'Below Header', 'airtech' ),
				'id'          => 'header-1',
				'description' => __( 'Widgets added to this region will appear beneath the header and above the main content.', 'airtech' ),
			);

			$rows    = intval( apply_filters( 'airtech_footer_widget_rows', 1 ) );
			$regions = intval( apply_filters( 'airtech_footer_widget_columns', 4 ) );

			for ( $row = 1; $row <= $rows; $row++ ) {
				for ( $region = 1; $region <= $regions; $region++ ) {
					$footer_n = $region + $regions * ( $row - 1 ); // Defines footer sidebar ID.
					$footer   = sprintf( 'footer_%d', $footer_n );

					if ( 1 === $rows ) {
						/* translators: 1: column number */
						$footer_region_name = sprintf( __( 'Footer Column %1$d', 'airtech' ), $region );

						/* translators: 1: column number */
						$footer_region_description = sprintf( __( 'Widgets added here will appear in column %1$d of the footer.', 'airtech' ), $region );
					} else {
						/* translators: 1: row number, 2: column number */
						$footer_region_name = sprintf( __( 'Footer Row %1$d - Column %2$d', 'airtech' ), $row, $region );

						/* translators: 1: column number, 2: row number */
						$footer_region_description = sprintf( __( 'Widgets added here will appear in column %1$d of footer row %2$d.', 'airtech' ), $region, $row );
					}

					$sidebar_args[ $footer ] = array(
						'name'        => $footer_region_name,
						'id'          => sprintf( 'footer-%d', $footer_n ),
						'description' => $footer_region_description,
					);
				}
			}

			$sidebar_args = apply_filters( 'airtech_sidebar_args', $sidebar_args );

			foreach ( $sidebar_args as $sidebar => $args ) {
				$widget_tags = array(
					'before_widget' => '<div id="%1$s" class="widget %2$s">',
					'after_widget'  => '</div>',
					'before_title'  => '<span class="gamma widget-title">',
					'after_title'   => '</span>',
				);

				/**
				 * Dynamically generated filter hooks. Allow changing widget wrapper and title tags. See the list below.
				 *
				 * 'airtech_header_widget_tags'
				 * 'airtech_sidebar_widget_tags'
				 *
				 * 'airtech_footer_1_widget_tags'
				 * 'airtech_footer_2_widget_tags'
				 * 'airtech_footer_3_widget_tags'
				 * 'airtech_footer_4_widget_tags'
				 */
				$filter_hook = sprintf( 'airtech_%s_widget_tags', $sidebar );
				$widget_tags = apply_filters( $filter_hook, $widget_tags );

				if ( is_array( $widget_tags ) ) {
					register_sidebar( $args + $widget_tags );
				}
			}
		}

		/**
		 * Enqueue scripts and styles.
		 *
		 * @since  1.0.0
		 */
		public function scripts() {
			global $airtech_version;

			/**
			 * Styles
			 */
			wp_enqueue_style( 'airtech-gutenberg-blocks', get_template_directory_uri() . '/assets/css/base/gutenberg-blocks.css', '', $airtech_version );
			wp_style_add_data( 'airtech-gutenberg-blocks', 'rtl', 'replace' );

			wp_enqueue_style( 'airtech-icons', get_template_directory_uri() . '/assets/css/base/icons.css', '', $airtech_version );
			wp_style_add_data( 'airtech-icons', 'rtl', 'replace' );

		    wp_enqueue_style( 'bootstrap', get_template_directory_uri() . '/assets/css/bootstrap.min.css', false, null, 'all');
		    wp_enqueue_style( 'fontawesome', get_template_directory_uri() . '/assets/css/font-awesome.min.css', false, null, 'all');
		    wp_enqueue_style( 'linearicons', get_template_directory_uri() . '/assets/vendors/linearicons/style.css', false, null, 'all');
		    wp_enqueue_style( 'stroke-icon', get_template_directory_uri() . '/assets/vendors/stroke-icon/style.css', false, null, 'all');
		    wp_enqueue_style( 'owl-carousel', get_template_directory_uri() . '/assets/vendors/owl-carousel/assets/owl.carousel.min.css', false, null, 'all');
		    wp_enqueue_style( 'simpleLightbox', get_template_directory_uri() . '/assets/vendors/lightbox/simpleLightbox.css', false, null, 'all');
		    wp_enqueue_style( 'magnific-popup', get_template_directory_uri() . '/assets/vendors/popup/magnific-popup.css', false, null, 'all');
		    wp_enqueue_style( 'nice-select', get_template_directory_uri() . '/assets/vendors/nice-selector/css/nice-select.css', false, null, 'all');
		    wp_enqueue_style( 'tempusdominusbootstrap', get_template_directory_uri() . '/assets/vendors/time-date-picker/tempusdominus-bootstrap-4.min.css', false, null, 'all');
			wp_enqueue_style( 'airtech-style', get_template_directory_uri() . '/style.css', '', $airtech_version );
			wp_style_add_data( 'airtech-style', 'rtl', 'replace' );
		    wp_enqueue_style( 'airtech-styles', get_template_directory_uri() . '/assets/css/themestyles.css', false, null, 'all');
		    wp_enqueue_style( 'airtech-responsive', get_template_directory_uri() . '/assets/css/responsive.css', false, null, 'all');


			/**
			 * Fonts
			 */
			wp_enqueue_style( 'airtech-fonts', $this->google_fonts(), array(), null );

			/**
			 * Scripts
			 */
			$suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';

			wp_enqueue_script( 'airtech-skip-link-focus-fix', get_template_directory_uri() . '/assets/js/skip-link-focus-fix' . $suffix . '.js', array(), '20130115', true );


			if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
				wp_enqueue_script( 'comment-reply' );
			}

			wp_enqueue_script( 'popper', get_template_directory_uri() . '/assets/js/theme/popper.min.js', false, null, true);
			wp_enqueue_script( 'bootstrap', get_template_directory_uri() . '/assets/js/theme/bootstrap.min.js', false, null, true);
			wp_enqueue_script( 'owl-carousel', get_template_directory_uri() . '/assets/vendors/owl-carousel/owl.carousel.min.js', false, null, true);
			wp_enqueue_script( 'magnific-popup', get_template_directory_uri() . '/assets/vendors/popup/jquery.magnific-popup.min.js', false, null, true);
			wp_enqueue_script( 'momentwithlocales', get_template_directory_uri() . '/assets/vendors/time-date-picker/moment-with-locales.min.js', false, null, true);
			wp_enqueue_script( 'tempusdominusbootstrap', get_template_directory_uri() . '/assets/vendors/time-date-picker/tempusdominus-bootstrap-4.min.js', false, null, true);
			wp_enqueue_script( 'simplelightbox', get_template_directory_uri() . '/assets/vendors/lightbox/simpleLightbox.min.js', false, null, true);
			wp_enqueue_script( 'nice-select', get_template_directory_uri() . '/assets/vendors/nice-selector/js/jquery.nice-select.min.js', false, null, true);
			wp_enqueue_script( 'imagesloaded' );
			wp_enqueue_script( 'isotope', get_template_directory_uri() . '/assets/vendors/isotop/isotope.pkgd.min.js', false, null, true);

			wp_enqueue_script( 'airtech-themedist', get_template_directory_uri() . '/assets/js/theme/theme.js', false, null, true);

		}

		/**
		 * Register Google fonts.
		 *
		 * @since 2.4.0
		 * @return string Google fonts URL for the theme.
		 */
		public function google_fonts() {

		$fonts_url = '';
		$fonts     = array();
		$subsets   = 'latin,latin-ext';


		/* translators: If there are characters in your language that are not supported by Merriweather, translate this to 'off'. Do not translate into your own language. */
		if ( 'off' !== _x( 'on', 'Google font: on or off', 'airtech' ) ) {
			$fonts[] = 'Open Sans:300,300i,400,400i,600,700';
			$fonts[] = 'Playfair Display:400,700,900';
			$fonts[] = 'Poppins:300,400,400i,500,500i,600,600i,700';
		}

		if ( $fonts ) {
			$fonts_url = add_query_arg( array(
				'family' => urlencode( implode( '|', $fonts ) ),
				'subset' => urlencode( $subsets ),
			), 'https://fonts.googleapis.com/css' );
		}

		return $fonts_url;
		}

		/**
		 * Enqueue supplemental block editor assets.
		 *
		 * @since 2.4.0
		 */
		public function block_editor_assets() {
			global $airtech_version;

			// Styles.
			wp_enqueue_style( 'airtech-editor-block-styles', get_theme_file_uri( '/assets/css/base/gutenberg-blocks.css' ), false, $airtech_version, 'all' );
			wp_style_add_data( 'airtech-editor-block-styles', 'rtl', 'replace' );

			// JS.
			$suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';
			wp_enqueue_script( 'airtech-editor', get_template_directory_uri() . '/assets/js/editor' . $suffix . '.js', array( 'wp-data', 'wp-dom-ready', 'wp-edit-post' ), $airtech_version, true );
		}

		/**
		 * Enqueue child theme stylesheet.
		 * A separate function is required as the child theme css needs to be enqueued _after_ the parent theme
		 * primary css and the separate WooCommerce css.
		 *
		 * @since  1.5.3
		 */
		public function child_scripts() {
			if ( is_child_theme() ) {
				$child_theme = wp_get_theme( get_stylesheet() );
				wp_enqueue_style( 'airtech-child-style', get_stylesheet_uri(), array(), $child_theme->get( 'Version' ) );
			}
		}

		/**
		 * Get our wp_nav_menu() fallback, wp_page_menu(), to show a home link.
		 *
		 * @param array $args Configuration arguments.
		 * @return array
		 */
		public function page_menu_args( $args ) {
			$args['show_home'] = true;
			return $args;
		}

		/**
		 * Adds custom classes to the array of body classes.
		 *
		 * @param array $classes Classes for the body element.
		 * @return array
		 */
		public function body_classes( $classes ) {
			// Adds a class to blogs with more than 1 published author.
			if ( is_multi_author() ) {
				$classes[] = 'group-blog';
			}

			/**
			 * Adds a class when WooCommerce is not active.
			 *
			 * @todo Refactor child themes to remove dependency on this class.
			 */
			$classes[] = 'no-wc-breadcrumb';

			/**
			 * What is this?!
			 * Take the blue pill, close this file and forget you saw the following code.
			 * Or take the red pill, filter airtech_make_me_cute and see how deep the rabbit hole goes...
			 */
			$cute = apply_filters( 'airtech_make_me_cute', false );

			if ( true === $cute ) {
				$classes[] = 'airtech-cute';
			}

			// If our main sidebar doesn't contain widgets, adjust the layout to be full-width. only for non WC powered pages though.
			if( class_exists( 'WooCommerce' ) ) {
				if ( ! is_active_sidebar( 'sidebar-1' ) && ! airtech_is_woocommerce_page() ) {
					$classes[] = 'airtech-full-width-content';
				}

				// If our shop sidebar doesn't contain widgets, adjust the layout to be full-width. Only on WC powered pages.
				if ( ! is_active_sidebar( 'shop-sidebar' ) && airtech_is_woocommerce_page() ) {
					$classes[] = 'airtech-full-width-content';
				}

				// If our product sidebar doesn't contain widgets, adjust the layout to be full-width. Only on products page.
				if ( ! is_active_sidebar( 'product-sidebar' ) && is_product() ) {
					$classes[] = 'airtech-full-width-content';
				}
			}
			else {
				if ( ! is_active_sidebar( 'sidebar-1' ) ) {
					$classes[] = 'airtech-full-width-content';
				}
			}

			// Add class when using homepage template + featured image.
			if ( is_page_template( 'template-homepage.php' ) && has_post_thumbnail() ) {
				$classes[] = 'has-post-thumbnail';
			}

			// Add class when Secondary Navigation is in use.
			if ( has_nav_menu( 'secondary' ) ) {
				$classes[] = 'airtech-secondary-navigation';
			}

			// Add class if align-wide is supported.
			if ( current_theme_supports( 'align-wide' ) ) {
				$classes[] = 'airtech-align-wide';
			}

			// Add class if align-wide is supported.
			if ( ! defined( 'TT_FW_ROOT' ) ) {
				$classes[] = 'no-blvplug';
			}

			// Add class to trigger full width for service.
			if ( is_singular('tt_project') ) {
				$classes[] = 'blv-fullwidth';
			}

			// Add class for sticky menu.
			if ( ! get_theme_mod('hdr_sticky', '0') ) {
				$classes[] = 'no-sticky';
			}

			// Add class for sticky menu.
			if ( ! get_theme_mod('hdr_sticky_mob', '0') ) {
				$classes[] = 'no-sticky-m';
			}
			return $classes;
		}


		/**
		 * Adds a custom parameter to the editor settings that is used
		 * to track whether the main sidebar has widgets.
		 *
		 * @since 2.4.3
		 * @param array   $settings Default editor settings.
		 * @param WP_Post $post Post being edited.
		 *
		 * @return array Filtered block editor settings.
		 */
		public function custom_editor_settings( $settings, $post ) {
			$settings['mainSidebarActive'] = false;

			if ( is_active_sidebar( 'sidebar-1' ) ) {
				$settings['mainSidebarActive'] = true;
			}

			return $settings;
		}

		/**
		 * Custom navigation markup template hooked into `navigation_markup_template` filter hook.
		 */
		public function navigation_markup_template() {
			$template  = '<nav id="post-navigation" class="navigation %1$s" role="navigation" aria-label="' . esc_html__( 'Post Navigation', 'airtech' ) . '">';
			$template .= '<h2 class="screen-reader-text">%2$s</h2>';
			$template .= '<div class="nav-links">%3$s</div>';
			$template .= '</nav>';

			return apply_filters( 'airtech_navigation_markup_template', $template );
		}

		/**
		 * Add styles for embeds
		 */
		public function print_embed_styles() {
			wp_enqueue_style( 'source-sans-pro', '//fonts.googleapis.com/css?family=Source+Sans+Pro:400,300,300italic,400italic,700,900' );
			$accent_color     = get_theme_mod( 'airtech_accent_color' );
			$background_color = airtech_get_content_background_color();
			?>
			<style type="text/css">
				.wp-embed {
					padding: 2.618em !important;
					border: 0 !important;
					border-radius: 3px !important;
					font-family: "Source Sans Pro", "Open Sans", sans-serif !important;
					background-color: <?php echo esc_html( airtech_adjust_color_brightness( $background_color, -7 ) ); ?> !important;
				}

				.wp-embed .wp-embed-featured-image {
					margin-bottom: 2.618em;
				}

				.wp-embed .wp-embed-featured-image img,
				.wp-embed .wp-embed-featured-image.square {
					min-width: 100%;
					margin-bottom: .618em;
				}

				a.wc-embed-button {
					padding: .857em 1.387em !important;
					font-weight: 600;
					background-color: <?php echo esc_attr( $accent_color ); ?>;
					color: #fff !important;
					border: 0 !important;
					line-height: 1;
					border-radius: 0 !important;
					box-shadow:
						inset 0 -1px 0 rgba(#000,.3);
				}

				a.wc-embed-button + a.wc-embed-button {
					background-color: #60646c;
				}
			</style>
			<?php
		}
	}
endif;

return new Airtech();


