<?php

namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Group_Control_Typography;



if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Master_Header_1 extends Widget_Base
{

    use TP_Style_Trait, TP_Icon_Trait, TP_Offcanvas_Trait, TP_Menu_Trait, TP_Column_Trait, TP_Query_Trait;

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tp-master-header-1';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Master Header 1', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }


    /**
     * Menu index.
     *
     * @access protected
     * @var $nav_menu_index
     */
    protected $nav_menu_index = 1;

    /**
     * Retrieve the menu index.
     *
     * Used to get index of nav menu.
     *
     * @since 1.3.0
     * @access protected
     *
     * @return string nav index.
     */
    protected function get_nav_menu_index()
    {
        return $this->nav_menu_index++;
    }

    /**
     * Retrieve the list of available menus.
     *
     * Used to get the list of available menus.
     *
     * @since 1.3.0
     * @access private
     *
     * @return array get WordPress menus list.
     */
    private function get_available_menus()
    {

        $menus = wp_get_nav_menus();

        $options = [];

        foreach ($menus as $menu) {
            $options[$menu->slug] = $menu->name;
        }

        return $options;
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */

    protected function register_controls()
    {
        $this->register_controls_section();
        $this->style_tab_content();
    }

    protected function register_controls_section()
    {

        // layout Panel
        $this->start_controls_section(
            'tp_header_top',
            [
                'label' => esc_html__('Header Info', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_header_sticky',
            [
                'label' => esc_html__('Enable Sticky', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'tp_header_right_switch',
            [
                'label' => esc_html__('Header Right On/Off', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

        if (function_exists('masterstudy_lms_pre_get_document_title')) {

            // category section
            $this->start_controls_section(
                'tp_header_category_sec',
                [
                    'label' => esc_html__('Category', 'tpcore'),
                    'tab' => Controls_Manager::TAB_CONTENT,
                ]
            );

            // header_category_switch
            $this->add_control(
                'tp_header_category_switch',
                [
                    'label' => esc_html__('Enable Category?', 'tpcore'),
                    'type' => Controls_Manager::SWITCHER,
                    'label_on' => esc_html__('Show', 'tpcore'),
                    'label_off' => esc_html__('Hide', 'tpcore'),
                    'return_value' => 'yes',
                    'default' => 'yes',
                    'separator' => 'before',
                ]
            );

            $this->add_control(
                'tp_header_category_text',
                [
                    'label' => esc_html__('Category Text', 'tpcore'),
                    'type' => Controls_Manager::TEXT,
                    'default' => esc_html__('Category', 'tpcore'),
                    'placeholder' => esc_html__('Your Text', 'tpcore'),
                    'label_block' => true,
                    'condition' => [
                        'tp_header_category_switch' => 'yes'
                    ]
                ]
            );

            $this->add_control(
                'tp_header_category_list',
                [
                    'label' => esc_html__('Select Cagetories', 'tpcore'),
                    'type' => Controls_Manager::SELECT2,
                    'multiple' => true,
                    'options' => tp_get_categories('stm_lms_course_taxonomy'),
                    'label_block' => true,
                    'condition' => [
                        'tp_header_category_switch' => 'yes'
                    ]
                ]
            );

            $this->end_controls_section();


            // layout Panel
            $this->start_controls_section(
                'profile_general_section_logged',
                array(
                    'label' => esc_html__('Login', 'tpcore'),
                )
            );

            // header_Login_switch
            $this->add_control(
                'tp_header_login_switch',
                [
                    'label' => esc_html__('Enable Login?', 'tpcore'),
                    'type' => Controls_Manager::SWITCHER,
                    'label_on' => esc_html__('Show', 'tpcore'),
                    'label_off' => esc_html__('Hide', 'tpcore'),
                    'return_value' => 'yes',
                    'default' => 'yes',
                    'separator' => 'before',
                ]
            );

            $this->add_control(
                'profile_general_icon',
                array(
                    'name' => 'profile_general_icon_selected',
                    'label' => esc_html__('Icon', 'tpcore'),
                    'type' => Controls_Manager::ICONS,
                    'default' => array(
                        'value' => 'fal fa-user',
                        'library' => 'fa-solid',
                    ),
                )
            );

            $this->add_control(
                'auth_links_btn_link',
                array(
                    'label' => esc_html__('Button Link', 'tpcore'),
                    'type' => Controls_Manager::URL,
                    'dynamic' => array(
                        'active' => true,
                    ),
                    'placeholder' => site_url() . '/user-account',
                    'default' => array(
                        'url' => site_url() . '/user-account',
                    ),
                    'condition' => array(
                        'tp_header_login_switch' => 'yes',
                    ),
                )
            );

            $this->add_control(
                'auth_links_btn_text',
                array(
                    'label' => esc_html__('Button Text', 'tpcore'),
                    'type' => Controls_Manager::TEXT,
                    'dynamic' => array(
                        'active' => true,
                    ),
                    'default' => esc_html__('Login/Sign Up', 'tpcore'),
                    'condition' => array(
                        'tp_header_login_switch' => 'yes',
                    ),
                )
            );

            $this->end_controls_section();
        }

        // Header Right
        $this->start_controls_section(
            'tp_header_right',
            [
                'label' => esc_html__('Header Right', 'tpcore'),
            ]
        );

        // header_search_switch
        $this->add_control(
            'tp_header_search_switch',
            [
                'label' => esc_html__('Header Search Switch', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 0,
            ]
        );
        $this->tp_single_icon_control('theme_btn', 'tp_header_search_switch', 'yes');

        // header_cart_switch
        $this->add_control(
            'tp_header_cart_switch',
            [
                'label' => esc_html__('Enable Cart?', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        // cart icon controls
        $this->tp_single_icon_control('header_cart', 'tp_header_cart_switch', 'yes');

        $this->end_controls_section();

        // header button controls
        $this->tp_link_render_controls('header_btn', 'Button');


        // _tp_image
        $this->start_controls_section(
            '_tp_image',
            [
                'label' => esc_html__('Site Logo', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_logo_black',
            [
                'label' => esc_html__('Choose Logo Black', 'tpcore'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'tp_logo_width',
            [
                'type' => \Elementor\Controls_Manager::NUMBER,
                'label' => esc_html__('Logo Width', 'tpcore'),
                'description' => esc_html__('This number will count by "PX" and maximum number is 1000', 'tpcore'),
                'placeholder' => '0',
                'min' => 10,
                'max' => 1000,
                'step' => 1,
                'default' => 125,
                'selectors' => [
                    '{{WRAPPER}} .tp-el-logo img' => 'width: {{VALUE}}px;',
                ],
            ]
        );

        $this->end_controls_section();

        // menu
        $this->start_controls_section(
            'section_menu',
            [
                'label' => __('Menu', 'tpcore'),
            ]
        );

        $menus = $this->get_available_menus();

        if (!empty($menus)) {
            $this->add_control(
                'menu',
                [
                    'label' => __('Menu', 'tpcore'),
                    'type' => Controls_Manager::SELECT,
                    'options' => $menus,
                    'default' => array_keys($menus)[0],
                    'save_default' => true,
                    /* translators: %s Nav menu URL */
                    'description' => sprintf(__('Go to the <a href="%s" target="_blank">Menus screen</a> to manage your menus.', 'tpcore'), admin_url('nav-menus.php')),
                ]
            );
        } else {
            $this->add_control(
                'menu',
                [
                    'type' => Controls_Manager::RAW_HTML,
                    /* translators: %s Nav menu URL */
                    'raw' => sprintf(__('<strong>There are no menus in your site.</strong><br>Go to the <a href="%s" target="_blank">Menus screen</a> to create one.', 'tpcore'), admin_url('nav-menus.php?action=edit&menu=0')),
                    'content_classes' => 'elementor-panel-alert elementor-panel-alert-info',
                ]
            );
        }

        $this->add_control(
            'menu_last_item',
            [
                'label' => __('Last Menu Item', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'none' => __('Default', 'tpcore'),
                    'cta' => __('Button', 'tpcore'),
                ],
                'default' => 'none',
                'condition' => [
                    'layout!' => 'expandible',
                ],
            ]
        );

        $this->end_controls_section();


        // offcanvas
        $this->tp_offcanvas_controls('offcanvas', 'Offcanvas');
    }

    // style_tab_content
    protected function style_tab_content()
    {

        // menu
        $this->start_controls_section(
            'section_header_style',
            [
                'label' => __('Header Style', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'background',
                'types' => ['classic', 'gradient', 'video'],
                'exclude' => ['video'],
                'selector' => '{{WRAPPER}} .tp_el_header',
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'tp_box_shadow',
                'selector' => '{{WRAPPER}} .tp_el_header',
            ]
        );
        $this->add_control(
            'tp_header_padding',
            [
                'label' => esc_html__('Padding', 'tpcore'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .tp_el_header' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        // header sticky
        $this->add_control(
            '_content_header_sticky',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __('Header Sticky', 'tpcore'),
                'separator' => 'before'
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'tp_sticky_background',
                'types' => ['classic', 'gradient', 'video'],
                'exclude' => ['video'],
                'selector' => '{{WRAPPER}} .tp_el_header.tp-header-sticky',
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'tp_sticky_box_shadow',
                'selector' => '{{WRAPPER}} .tp_el_header.tp-header-sticky',
            ]
        );

        $this->end_controls_section();

        $this->tp_link_controls_style('', 'btn2_style', 'Button', '.tp-el-btn2');


        $this->start_controls_section(
            'tp_menu_style_section',
            [
                'label' => esc_html__('Mega Menu', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
                'separator' => 'before',
            ]
        );


        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_menuss_title_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .tp-megamenu-home-item a',
            ]
        );

        $this->start_controls_tabs(
            'style_tabs'
        );

        $this->start_controls_tab(
            'style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_menu_title_color',
            [
                'label' => esc_html__('Menu Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-home-item a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_menu_title_hvr_color',
            [
                'label' => esc_html__('Menu Hover Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-home-item:hover a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();


        // main Menu style
        $this->start_controls_section(
            'tp_main_menu_style_section',
            [
                'label' => esc_html__('Main Menu', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_main_menu_title_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .main-menu > nav > ul > li > a',
            ]
        );

        $this->start_controls_tabs(
            'main_style_tabs'
        );

        $this->start_controls_tab(
            'main_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_main_menu_title_color',
            [
                'label' => esc_html__('Menu Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .main-menu > nav > ul > li > a,
                    {{WRAPPER}} .main-menu > nav > ul > li.has-dropdown > a::after' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'main_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_main_menu_title_hvr_color',
            [
                'label' => esc_html__('Menu Hover Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .main-menu > nav > ul > li:hover > a,
                    {{WRAPPER}} .main-menu > nav > ul > li:hover > a:after' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();


        // Link Menu style
        $this->start_controls_section(
            'tp_mobile_link_menu_style_section',
            [
                'label' => esc_html__('Link Menu', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_mobile_link_menu_title_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .tp-megamenu-small-content .tp-megamenu-list a,
                {{WRAPPER}} .main-menu > nav > ul > li > .tp-submenu li > a,
                {{WRAPPER}} .tp-megamenu-fullwidth-list ul li a',
            ]
        );

        $this->start_controls_tabs(
            'mobile_link_style_tabs'
        );

        $this->start_controls_tab(
            'mobile_link_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_mobile_link_menu_title_color',
            [
                'label' => esc_html__('Menu Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-small-content .tp-megamenu-list a,
                {{WRAPPER}} .main-menu > nav > ul > li > .tp-submenu li > a,
                {{WRAPPER}} .tp-megamenu-fullwidth-list ul li a' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_mobile_link_menu_toggle_color',
            [
                'label' => esc_html__('Menu Bar Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-small-content .tp-megamenu-list a::before,
                    {{WRAPPER}} .main-menu > nav > ul > li > .tp-submenu li > a::before,
                    {{WRAPPER}} .tp-megamenu-fullwidth-list ul li a:before' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'mobile_link_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_mobile_link_menu_title_hvr_color',
            [
                'label' => esc_html__('Menu Hover Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-small-content .tp-megamenu-list a:hover,
                {{WRAPPER}} .main-menu > nav > ul > li > .tp-submenu li > a:hover,
                {{WRAPPER}} .tp-megamenu-fullwidth-list ul li a:hover' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        // Mobile Menu style
        $this->start_controls_section(
            'tp_mobile_main_menu_style_section',
            [
                'label' => esc_html__('Mobile Menu', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_mobile_main_menu_title_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .tp-main-menu-mobile ul li > a',
            ]
        );

        $this->start_controls_tabs(
            'mobile_main_style_tabs'
        );

        $this->start_controls_tab(
            'mobile_main_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_mobile_main_menu_title_color',
            [
                'label' => esc_html__('Menu Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-main-menu-mobile ul li > a' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_mobile_main_menu_toggle_color',
            [
                'label' => esc_html__('Menu Toggle Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-main-menu-mobile ul li.has-dropdown > a .dropdown-toggle-btn::after' => 'background-color: {{VALUE}} !important',
                    '{{WRAPPER}} .tp-main-menu-mobile ul li.has-dropdown > a .dropdown-toggle-btn::before' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'mobile_main_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_mobile_main_menu_title_hvr_color',
            [
                'label' => esc_html__('Menu Hover Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-main-menu-mobile ul li > a.expanded, 
                    {{WRAPPER}} .tp-main-menu-mobile ul li:hover > a' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_mobile_main_menu_toggle_hvr_color',
            [
                'label' => esc_html__('Menu Toggle Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-main-menu-mobile ul li.has-dropdown > a.expanded .dropdown-toggle-btn::after,
                    {{WRAPPER}} .offcanvas__2 .tp-main-menu-mobile ul li:hover > a .dropdown-toggle-btn::before,
                    {{WRAPPER}} .offcanvas__2 .tp-main-menu-mobile ul li:hover > a .dropdown-toggle-btn::after' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        // Close Btn style
        $this->start_controls_section(
            'tp_mobile_close_btn_style_section',
            [
                'label' => esc_html__('Close Button', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_mobile_close_btn_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .offcanvas__close-btn',
            ]
        );

        $this->start_controls_tabs(
            'close_button_style_tabs'
        );

        $this->start_controls_tab(
            'close_button_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_close_button_menu_title_color',
            [
                'label' => esc_html__('Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .offcanvas__close-btn' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_close_button_menu_toggle_color',
            [
                'label' => esc_html__('Bg Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .offcanvas__close-btn' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'close_button_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_close_button_menu_title_hvr_color',
            [
                'label' => esc_html__('Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .offcanvas__close-btn:hover' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_close_button_menu_title_hvrbg_color',
            [
                'label' => esc_html__('Bg Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .offcanvas__close-btn:hover' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();


    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $control_id = 'header_btn';

        $btn_id = 'theme_btn';

        if (!empty($settings['menu'])):
            $menu = $settings['menu'];
        else:
            $menu = '';
        endif;

        $menus = $this->get_available_menus();
        require_once get_parent_theme_file_path() . '/inc/class-navwalker.php';

        $args = [
            'echo' => false,
            'menu' => $menu,
            'menu_class' => 'tp-nav-menu ',
            'menu_id' => 'menu-' . $this->get_nav_menu_index() . '-' . $this->get_id(),
            'fallback_cb' => 'Acadia_Navwalker_Class::fallback',
            'container' => '',
            'walker' => new Acadia_Navwalker_Class,
        ];

        $menu_html = wp_nav_menu($args);

        $offcanvas_image_size = tp_get_img_size($settings, 'tp_offcanvas_logo_size');

        $img = tp_get_img($settings, 'tp_logo_black', 'full', false);

        $is_sticky = $settings['tp_header_sticky'] == 'yes' ? 'header-sticky' : 'no-sticky';

        $edit_class = tp_is_elementor_edit_mode() ? 'acadia-elementor-header-edit-mode' : '';

        $this->tp_link_attributes_render('header_btn', 'tp-btn-inner tp-el-btn2', $this->get_settings());

        ?>

        <!-- header-area-start -->
        <header class="header-area p-relative">
            <div id="<?php echo esc_attr($is_sticky); ?>"
                class="tp_el_header tp-header-masterstudy tp-header-2 <?php echo esc_attr($edit_class); ?>">
                <div class="container custom-container-larg">
                    <div class="row align-items-center">
                        <div class="col-xxl-3 col-xl-4 col-lg-6 col-6">
                            <div class="tp-header-2-right d-flex align-items-center">
                                <?php if (!empty($img['tp_logo_black'])): ?>
                                    <div class="tp-header-inner-logo tp-el-logo">
                                        <a href="<?php print esc_url(home_url('/')); ?>">
                                            <img src="<?php echo esc_url($img['tp_logo_black']) ?>"
                                                alt="<?php echo esc_attr($img['tp_logo_black_alt']) ?>">
                                        </a>
                                    </div>
                                <?php endif; ?>

                                <?php if (!empty($settings['tp_header_category_switch'])): ?>
                                    <div class="tp-header-2-category tp-header-inner-category home-2 d-none d-md-block">
                                        <div class="tp-header-2-category-icon">
                                            <p>
                                                <span>
                                                    <svg width="14" height="14" viewBox="0 0 14 14" fill="none"
                                                        xmlns="http://www.w3.org/2000/svg">
                                                        <path class="color"
                                                            d="M13.9998 4.564V1.386C13.9998 0.399 13.5518 0 12.4388 0H9.6108C8.4978 0 8.0498 0.399 8.0498 1.386V4.557C8.0498 5.551 8.4978 5.943 9.6108 5.943H12.4388C13.5518 5.95 13.9998 5.551 13.9998 4.564Z"
                                                            fill="currentColor" />
                                                        <path opacity="0.4"
                                                            d="M13.9998 12.439V9.611C13.9998 8.498 13.5518 8.05 12.4388 8.05H9.6108C8.4978 8.05 8.0498 8.498 8.0498 9.611V12.439C8.0498 13.552 8.4978 14 9.6108 14H12.4388C13.5518 14 13.9998 13.552 13.9998 12.439Z"
                                                            fill="#1A73E8" />
                                                        <path opacity="0.4"
                                                            d="M5.95 4.564V1.386C5.95 0.399 5.502 0 4.389 0H1.561C0.448 0 0 0.399 0 1.386V4.557C0 5.551 0.448 5.943 1.561 5.943H4.389C5.502 5.95 5.95 5.551 5.95 4.564Z"
                                                            fill="#1A73E8" />
                                                        <path class="color"
                                                            d="M5.95 12.439V9.611C5.95 8.498 5.502 8.05 4.389 8.05H1.561C0.448 8.05 0 8.498 0 9.611V12.439C0 13.552 0.448 14 1.561 14H4.389C5.502 14 5.95 13.552 5.95 12.439Z"
                                                            fill="currentColor" />
                                                    </svg>
                                                </span>
                                                <?php echo tp_kses($settings['tp_header_category_text']); ?>
                                            </p>

                                            <?php if (!empty($settings['tp_header_category_list']) && function_exists('masterstudy_lms_pre_get_document_title')): ?>
                                                <div class="tp-header-2-category-list">
                                                    <ul>
                                                        <?php foreach ($settings['tp_header_category_list'] as $key => $value):
                                                            $term = get_term_by('slug', $value, 'stm_lms_course_taxonomy');
                                                            ?>
                                                            <!-- slug valu check -->
                                                            <?php if (!empty($term)): ?>
                                                                <li>
                                                                    <a
                                                                        href="<?php echo get_term_link($term); ?>"><?php echo tp_kses($term->name); ?></a>
                                                                </li>
                                                            <?php endif; ?>
                                                        <?php endforeach; ?>
                                                    </ul>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="col-xxl-6 col-xl-6 col-lg-6 d-none d-xl-block">
                            <div class="main-menu text-xl-center d-none d-xl-block">
                                <nav class="tp-main-menu-content">
                                    <?php echo $menu_html; ?>
                                </nav>
                            </div>
                        </div>

                        <?php if (!empty($settings['tp_header_right_switch'])): ?>
                            <div class="col-xxl-3 col-xl-2 col-lg-6 col-6">
                                <div class="tp-header-2-contact d-flex align-items-center justify-content-end">

                                    <?php if (!empty($settings['tp_header_search_switch'])): ?>
                                        <div class="tp-header-inner-search">
                                            <button class="tp-search-open-btn">
                                                <?php tp_render_signle_icon_html($settings, 'theme_btn', ''); ?>
                                            </button>
                                        </div>
                                    <?php endif; ?>

                                    <?php if ($settings['tp_header_cart_switch'] == 'yes' && class_exists('WooCommerce')): ?>
                                        <div class="tp-header-2-cart home-2 d-none d-xxl-block">
                                            <button class="cartmini-open-btn">
                                                <span class="d-none">
                                                    <?php echo esc_html__('Cart', 'tpcore'); ?>
                                                </span>
                                                <?php tp_render_signle_icon_html($settings, 'header_cart'); ?>
                                                <?php
                                                $cart_count = !is_null(WC()->cart) ? WC()->cart->get_cart_contents_count() : 0;
                                                ?>
                                                <i><?php echo esc_html($cart_count); ?></i>
                                            </button>
                                        </div>
                                    <?php endif; ?>

                                    <?php if (!empty($settings['tp_' . $control_id . '_text']) && $settings['tp_' . $control_id . '_button_show'] == 'yes'): ?>
                                        <div class="tp-header-inner-btn home-2 d-none d-xxl-block">
                                            <a <?php echo $this->get_render_attribute_string('tp-button-arg' . $control_id . ''); ?>>
                                                <?php echo $settings['tp_' . $control_id . '_text']; ?>
                                            </a>
                                        </div>
                                    <?php endif; ?>

                                    <?php if (!empty($settings['tp_header_login_switch']) && function_exists('masterstudy_lms_pre_get_document_title')): ?>
                                        <div class="tp-master-login">
                                            <?php if (!is_user_logged_in()): ?>
                                                <a href="<?php echo esc_url($settings['auth_links_btn_link']['url']); ?>"
                                                    class="ms-lms-authorization">
                                                    <span class="ms-lms-authorization-icon">
                                                        <i class="<?php echo esc_attr(!empty($settings['profile_lms_icon']['value']) ? $settings['profile_lms_icon']['value'] : 'fas fa-user'); ?>"
                                                            aria-hidden="true"></i>
                                                    </span>
                                                    <span class="ms-lms-authorization-title">
                                                        <?php echo esc_html($settings['auth_links_btn_text']); ?>
                                                    </span>
                                                </a>
                                            <?php else: ?>
                                                <?php
                                                $icon = $settings['profile_general_icon']['value'];
                                                $user = \STM_LMS_User::get_current_user();
                                                ?>

                                                <a href="<?php echo esc_url(home_url('/user-account')); ?>">
                                                    <i class="<?php echo esc_attr($icon); ?>"></i>
                                                    <?php echo esc_html($user['login']); ?>
                                                </a>

                                            <?php endif; ?>
                                        </div>

                                    <?php endif; ?>

                                    <div class="offcanvas-btn d-xl-none ml-30">
                                        <button class="offcanvas-open-btn">
                                            <i class="fa-solid fa-bars"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </header>
        <!-- header-area-end -->

        <?php if (class_exists('WooCommerce') && $settings['tp_header_cart_switch'] == 'yes'): ?>
            <?php print acadia_shopping_cart(); ?>
            <div class="body-overlay"></div>
        <?php endif; ?>

        <?php include(TPCORE_ELEMENTS_PATH . '/header/header-offcanvas.php'); ?>
        <?php include(TPCORE_ELEMENTS_PATH . '/header-side/header-side-master.php'); ?>

        <?php
    }
}

$widgets_manager->register(new TP_Master_Header_1());
