<?php
namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;
use \Elementor\Control_Media;
use \Elementor\Group_Control_Image_Size;

use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Text_Shadow;
use \Elementor\Group_Control_Typography;
use \Elementor\Core\Schemes\Typography;
use \Elementor\Group_Control_Background;
use \Elementor\Repeater;

if (!defined('ABSPATH'))
	exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Banner extends Widget_Base
{

	use TP_Style_Trait, TP_Animation_Trait;

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'tp-banner';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return __('Banner', 'tpcore');
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'tp-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['tpcore'];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends()
	{
		return ['tpcore'];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */

	protected function register_controls()
	{
		$this->register_controls_section();
		$this->style_tab_content();
	}

	protected function register_controls_section()
	{

		// layout Panel
		$this->tp_design_layout('Layout Style', 1);

		$this->start_controls_section(
			'tp_banner_section',
			[
				'label' => esc_html__('Banner Controls', 'tpcore'),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'tp_banner_subtitle',
			[
				'label' => esc_html__('Subtitle', 'tpcore'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__('NOVEL', 'tpcore'),
				'placeholder' => esc_html__('Your Text', 'tpcore'),
				'label_block' => true,
			]
		);
		$this->add_control(
			'tp_banner_title',
			[
				'label' => esc_html__('Title', 'tpcore'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__('Huge Sale! Dont miss Out', 'tpcore'),
				'placeholder' => esc_html__('Your Text', 'tpcore'),
				'label_block' => true,
			]
		);

		$this->add_control(
			'tp_banner_title_tag',
			[
				'label' => esc_html__('Title HTML Tag', 'tpcore'),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'h1' => [
						'title' => esc_html__('H1', 'tpcore'),
						'icon' => 'eicon-editor-h1'
					],
					'h2' => [
						'title' => esc_html__('H2', 'tpcore'),
						'icon' => 'eicon-editor-h2'
					],
					'h3' => [
						'title' => esc_html__('H3', 'tpcore'),
						'icon' => 'eicon-editor-h3'
					],
					'h4' => [
						'title' => esc_html__('H4', 'tpcore'),
						'icon' => 'eicon-editor-h4'
					],
					'h5' => [
						'title' => esc_html__('H5', 'tpcore'),
						'icon' => 'eicon-editor-h5'
					],
					'h6' => [
						'title' => esc_html__('H6', 'tpcore'),
						'icon' => 'eicon-editor-h6'
					]
				],
				'default' => 'h4',
				'toggle' => false,
			]
		);

		$this->add_control(
			'tp_banner_image',
			[
				'label' => esc_html__('Thumbnail', 'tpcore'),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->add_control(
			'tp_banner_link_text',
			[
				'label' => esc_html__('Button Text', 'tpcore'),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__('Shop Now', 'tpcore'),
				'placeholder' => esc_html__('Your Text', 'tpcore'),
				'label_block' => true,
			]
		);

		tp_render_links_controls($this, 'banner_link');

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name' => 'tp_image_size',
				'default' => 'full',
				'exclude' => [
					'custom'
				]
			]
		);

		$this->end_controls_section();

	}

	protected function style_tab_content()
	{
		$this->tp_section_style_controls('section', 'Section - Style', '.tp-el-section');
		$this->tp_basic_style_controls('b_heading_subtitle', 'Section - Subtitle', '.tp-el-subtitle');
		$this->tp_basic_style_controls('b_heading_title', 'Section - Title', '.tp-el-title');
		$this->tp_link_controls_style('', 'b_btn1_style', 'Button', '.tp-el-btn');

		$this->start_controls_section(
			'tp_banner_style_section',
			[
				'label' => esc_html__('Image Style', 'tpcore'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'tp_banner_border-radius',
			[
				'label' => esc_html__('Border Radius', 'tpcore'),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors' => [
					'{{WRAPPER}} .tp_el_item_image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'tp_banner_width',
			[
				'label' => esc_html__('Width', 'tpcore'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 5,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .tp_el_item_image' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'tp_banner_height',
			[
				'label' => esc_html__('Height', 'tpcore'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 5,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .tp_el_item_image' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		?>

		<?php if ($settings['tp_design_style'] == 'layout-2'): ?>

		<?php else:

			$img = tp_get_img($settings, 'tp_banner_image', 'tp_image_size');

			$attrs = tp_get_repeater_links_attr($settings, 'banner_link');
			extract($attrs);

			$links_attrs = [
				'href' => $link,
				'target' => $target,
				'rel' => $rel,
			];

			$this->add_render_attribute('title_args', 'class', 'tp-featured-shop-banner-title tp-el-title');
			?>


			<div class="tp-shop-feature-thumb p-relative">
				<img class="tp_el_item_image" src="<?php echo esc_url($img['tp_banner_image']); ?>"
					alt="<?php echo esc_attr($img['tp_banner_image_alt']); ?>">

				<div class="tp-shop-feature-content">
					<?php if (!empty($settings['tp_banner_subtitle'])): ?>
						<span class="tp-el-subtitle"><?php echo tp_kses($settings['tp_banner_subtitle']); ?></span>
					<?php endif; ?>


					<?php if (!empty($settings['tp_banner_title'])):
						printf(
							'<%1$s %2$s>%3$s</%1$s>',
							tag_escape($settings['tp_banner_title_tag']),
							$this->get_render_attribute_string('title_args'),
							tp_kses($settings['tp_banner_title']),
						);
					endif;
					?>
				</div>

				<?php if (!empty($settings['tp_banner_link_text'])): ?>
					<div class="tp-shop-feature-btn">
						<a class="tp-el-btn" <?php echo tp_implode_html_attributes($links_attrs) ?>>
							<?php echo tp_kses($settings['tp_banner_link_text']); ?>
							<span>
								<svg width="6" height="10" viewBox="0 0 6 10" fill="none" xmlns="http://www.w3.org/2000/svg">
									<path d="M1 9L5 5L1 1" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"
										stroke-linejoin="round"></path>
								</svg>
							</span>
						</a>
					</div>
				<?php endif; ?>
			</div>

		<?php endif;
	}
}

$widgets_manager->register(new TP_Banner());