---
layout: default
title: postMessage module
subtitle: Live-Update the preview pane without a refresh
mainMaxWidth: 55rem;
bodyClasses: page module
---

WordPress allow you to live-update the preview pane by writing some custom JS for your controls (see the [Customizer Javascript API post](https://developer.wordpress.org/themes/customize-api/the-customizer-javascript-api/#preview-js-and-controls-js)) for more details.

Kirki however allows you to skip that and automatically generates the custom scripts for you - provided your fields have the necessary arguments.

### Using the `output` argument

If your field already has an `output` argument defined, you can use that to auto-generate the necessary scripts. All you have to do is add `'transport' => 'auto'` to your field.

**Example:**

```php
Kirki::add_field( 'kirki_demo', array(
	'type'        => 'color',
	'settings'    => 'color_setting_hex',
	'label'       => __( 'Color Control (hex-only)', 'kirki' ),
	'description' => esc_attr__( 'This is a color control', 'kirki' ),
	'section'     => 'color_section',
	'default'     => '#0088CC',
	'transport'   => 'auto',
	'output' => array(
		array(
			'element'  => 'body',
			'property' => 'color',
		),
		array(
			'element'  => '.my-super-cool-css-class',
			'property' => 'background-color',
		),
	),
) );
```

### Writing a custom script

If you use `postMessage` in the `transport` argument then you can write your own script. For example to achieve the same result as with the above example, we would have this:

##### PHP
```php
Kirki::add_field( 'kirki_demo', array(
	'type'        => 'color',
	'settings'    => 'color_setting_hex',
	'label'       => __( 'Color Control (hex-only)', 'kirki' ),
	'description' => esc_attr__( 'This is a color control', 'kirki' ),
	'section'     => 'color_section',
	'default'     => '#0088CC',
	'transport'   => 'postMessage',
) );
```

##### JS

```js
wp.customize( 'color_setting_hex', function( value ) {

	// When the value changes.
	value.bind( function( newval ) {

		// Add CSS to elements.
		jQuery( 'body' ).css( 'color', newval );
		jQuery( '.my-super-cool-css-class' ).css( 'background-color', newval );
	} );
} );
```

For reference the JS that Kirki automatically generates follows a different approach.
Instead of adding the styles inline in the elements like the above example code, Kirki creates a `<style>` in the `<head>` of your document for each control and then adds the CSS there.
The end result is similar, the only difference is that inline styles override rules set from stylesheets.
Since Kirki has to work with a variety of themes and coding styles the extra `<style>` method ensures that what you see in the preview will be the same as the end result on your frontend. Undoubtedly the code above is shorter and better for a well-written theme, but unfortunately we can't assume that every theme that uses Kirki will be well-written.

The JS generated by Kirki when using `'transport' => 'auto'` is this:

```js
wp.customize('color_setting_hex',function(value){value.bind(function(newval){if(null===document.getElementById('kirki-postmessage-color_setting_hex')||'undefined'===typeof document.getElementById('kirki-postmessage-color_setting_hex')){jQuery('head').append('<style id="kirki-postmessage-color_setting_hex"></style>');}newval0=newval;newval1=newval;var cssContent='body{color:'+newval0+';}.my-super-cool-css-class{background-color:'+newval1+';}';jQuery('#kirki-postmessage-color_setting_hex').text(cssContent);jQuery('#kirki-postmessage-color_setting_hex').appendTo('head');});});
```

Unminimized/beautified this is what it would look like:

```js
wp.customize( 'color_setting_hex', function( value ) {

	// When the value changes.
	value.bind( function( newval ) {

		// Generate the CSS.
		var cssContent = 'body{color:' + newval + ';}.my-super-cool-css-class{background-color:' + newval + ';}';

		// Check if we already have a <style> in the <head> referencing this control.
		if (
			null === document.getElementById( 'kirki-postmessage-color_setting_hex' ) ||
			'undefined' === typeof document.getElementById( 'kirki-postmessage-color_setting_hex' )
		) {

			// Append the <style> to the <head>.
			jQuery( 'head' ).append( '<style id="kirki-postmessage-color_setting_hex"></style>' );
		}

		// Add the CSS to the <style> and append.
		jQuery( '#kirki-postmessage-color_setting_hex' ).text( cssContent );
		jQuery( '#kirki-postmessage-color_setting_hex' ).appendTo( 'head' );
	} );
} );
```
