<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       http://www.europadns.com/
 * @since      1.0.0
 *
 * @package    Portfolio_Milo
 * @subpackage Portfolio_Milo/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Portfolio_Milo
 * @subpackage Portfolio_Milo/includes
 * @author     Europadns <nicolae@europadns.com>
 */
class Portfolio_Milo {

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      Portfolio_Milo_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $portfolio_milo;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {

        $this->portfolio_milo = 'portfolio-milo';
        $this->version = '1.0.0';

        $this->load_dependencies();
        $this->set_locale();
        $this->milo_portfolio_initiate_custom_post();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - Portfolio_Milo_Loader. Orchestrates the hooks of the plugin.
     * - Portfolio_Milo_i18n. Defines internationalization functionality.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies() {

        /**
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-portfolio-milo-loader.php';

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-portfolio-milo-i18n.php';

        $this->loader = new Portfolio_Milo_Loader();
    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the Portfolio_Milo_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale() {

        $plugin_i18n = new Portfolio_Milo_i18n();
        $plugin_i18n->set_domain($this->get_portfolio_milo());

        $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_portfolio_milo() {
        return $this->portfolio_milo;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    Portfolio_Milo_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }

    public function milo_portfolio_initiate_custom_post() {
        // Default custom portfolio
        add_action('init', 'milo_portfolio_create_custom_post');

        function milo_portfolio_create_custom_post() {

            global $milo_theme_options;
            $milo_flush = 0;
            if (isset($milo_theme_options['milo-portfolio-slug'])) {
                if (trim($milo_theme_options['milo-portfolio-slug']) != '') {

                    $milo_theme_options['milo-portfolio-slug'] = sanitize_title_with_dashes($milo_theme_options['milo-portfolio-slug']);

                    $option_value = get_option('milo-portfolio-check');

                    if ($option_value == false || $option_value != $milo_theme_options['milo-portfolio-slug']) {
                        update_option('milo-portfolio-check', $milo_theme_options['milo-portfolio-slug']);
                        $milo_flush = 1;
                    }

                    define('MILO_SLUG', $milo_theme_options['milo-portfolio-slug']);
                } else if (get_option('milo-portfolio-check') != null) {
                    update_option('milo-portfolio-check', null);
                    $milo_flush = 1;
                }
            }

            if (!defined('MILO_SLUG')) {
                define('MILO_SLUG', 'milo-portfolio');
            }

            register_post_type('milo-portfolio', array(
                'labels' => array(
                    'name' => __('Portfolio', 'portfolio-milo'),
                    'singular_name' => __('Portfolio', 'portfolio-milo')
                ),
                'public' => true,
                'has_archive' => true,
                'rewrite' => array('slug' => sanitize_title_with_dashes(MILO_SLUG)),
                'show_in_nav_menus' => true,
                'supports' => array(
                    'title', 'editor', 'excerpt',
                    'thumbnail'),
                    )
            );

            if ($milo_flush == 1) {
                flush_rewrite_rules();
            }
        }

        // Registers the portfolio category
        function milo_portfolio_create_taxonomies() {
            $labels = array(
                'name' => __('Categories', 'portfolio-milo'),
                'singular_name' => __('Category', 'portfolio-milo'),
                'search_items' => __('Search Categories', 'portfolio-milo'),
                'all_items' => __('All Categories', 'portfolio-milo'),
                'parent_item' => __('Parent Category', 'portfolio-milo'),
                'parent_item_colon' => __('Parent Category:', 'portfolio-milo'),
                'edit_item' => __('Edit Category', 'portfolio-milo'),
                'update_item' => __('Update Category', 'portfolio-milo'),
                'add_new_item' => __('Add New Category', 'portfolio-milo'),
                'new_item_name' => __('New Category Name', 'portfolio-milo'),
                'menu_name' => __('Categories', 'portfolio-milo'),
            );

            $args = array(
                'hierarchical' => true,
                'labels' => $labels,
                'show_ui' => true,
                'show_admin_column' => true,
                'query_var' => true,
                'rewrite' => array('slug' => 'categories'),
            );

            register_taxonomy('portfolio_categories', array('milo-portfolio'), $args);
        }

        add_action('init', 'milo_portfolio_create_taxonomies', 0);
    }

}
