<?php

namespace Workdo\MobileServiceManagement\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

use Workdo\MobileServiceManagement\DataTables\MobileServiceContractDataTable;

use Workdo\MobileServiceManagement\Entities\MobileServiceContract;

use Workdo\MobileServiceManagement\Events\CreateMobileServiceContract;
use Workdo\MobileServiceManagement\Events\UpdateMobileServiceContract;
use Workdo\MobileServiceManagement\Events\DestroyMobileServiceContract;

class MobileServiceContractController extends Controller
{
    public function index(MobileServiceContractDataTable $datatable)
    {
        if (Auth::user()->isAbleTo('mobileservice_contracts manage')) {
            return $datatable->render('mobile-service-management::mobileservice-contracts.index');
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function create()
    {
        if (Auth::user()->isAbleTo('mobileservice_contracts create')) {
            return view('mobile-service-management::mobileservice-contracts.create');
        } else {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function store(Request $request)
    {
        if (Auth::user()->isAbleTo('mobileservice_contracts create')) {
            $validator = Validator::make($request->all(), [
                'customer_name'    => 'required|string',
                'contract_type'    => 'required|string',
                'start_date'       => 'required|date',
                'end_date'         => 'required|date|after_or_equal:start_date',
                'contract_details' => 'required|string',
            ]);

            if ($validator->fails()) {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $mobileservice_contract                   = new MobileServiceContract();
            $mobileservice_contract->customer_name    = $request->customer_name;
            $mobileservice_contract->contract_type    = $request->contract_type;
            $mobileservice_contract->start_date       = $request->start_date;
            $mobileservice_contract->end_date         = $request->end_date;
            $mobileservice_contract->contract_details = $request->contract_details;
            $mobileservice_contract->workspace        = getActiveWorkSpace();
            $mobileservice_contract->created_by       = creatorId();
            $mobileservice_contract->save();

            event(new CreateMobileServiceContract($request, $mobileservice_contract));

            return redirect()->route('mobileservice-contract.index')->with('success', __('The contract has been created successfully'));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function edit($id)
    {
        if (Auth::user()->isAbleTo('mobileservice_contracts edit')) {
            $mobileservice_contract = MobileServiceContract::where('id', $id)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($mobileservice_contract) {
                return view('mobile-service-management::mobileservice-contracts.edit', compact('mobileservice_contract'));
            } else {
                return response()->json(['error' => __('Contract not found.')], 404);
            }
        } else {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function update(Request $request, $id)
    {
        if (Auth::user()->isAbleTo('mobileservice_contracts edit')) {
            $validator = Validator::make($request->all(), [
                'customer_name'    => 'required|string',
                'contract_type'    => 'required|string',
                'start_date'       => 'required|date',
                'end_date'         => 'required|date|after_or_equal:start_date',
                'contract_details' => 'required|string',
            ]);

            if ($validator->fails()) {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $mobileservice_contract = MobileServiceContract::where('id', $id)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($mobileservice_contract) {
                $mobileservice_contract->customer_name    = $request->customer_name;
                $mobileservice_contract->contract_type    = $request->contract_type;
                $mobileservice_contract->start_date       = $request->start_date;
                $mobileservice_contract->end_date         = $request->end_date;
                $mobileservice_contract->contract_details = $request->contract_details;
                $mobileservice_contract->save();

                event(new UpdateMobileServiceContract($request, $mobileservice_contract));

                return redirect()->route('mobileservice-contract.index')->with('success', __('The contract details are updated successfully'));
            } else {
                return redirect()->back()->with('error', __('Contract not found.'));
            }
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function destroy($id)
    {
        if (Auth::user()->isAbleTo('mobileservice_contracts delete')) {
            $mobileservice_contract = MobileServiceContract::where('id', $id)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($mobileservice_contract) {
                event(new DestroyMobileServiceContract($mobileservice_contract));
                $mobileservice_contract->delete();

                return redirect()->back()->with('success', __('The contract has been deleted'));
            } else {
                return redirect()->back()->with('error', __('Contract not found.'));
            }
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function viewcontract($id)
    {
        if (Auth::user()->isAbleTo('mobileservice_contracts manage')) {
            $mobileservice_contract = MobileServiceContract::where('id', $id)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($mobileservice_contract) {
                return view('mobile-service-management::mobileservice-contracts.description', compact('mobileservice_contract'));
            } else {
                return response()->json(['error' => __('Contract not found.')], 404);
            }
        } else {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }
}
