<?php

namespace Workdo\MobileServiceManagement\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

use Workdo\MobileServiceManagement\Entities\MobileServiceBrand;

use Workdo\MobileServiceManagement\Events\CreateMobileServiceBrand;
use Workdo\MobileServiceManagement\Events\UpdateMobileServiceBrand;
use Workdo\MobileServiceManagement\Events\DestroyMobileServiceBrand;

class MobileServiceBrandController extends Controller
{

    public function index(Request $request)
    {
        if (Auth::user()->isAbleTo('mobileservice_devicebrand manage')) {
            $brands = MobileServiceBrand::where('created_by', creatorId())->where('workspace', getActiveWorkSpace())->get();
            return view('mobile-service-management::service-setup.device-brands.index', compact('brands'));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function create()
    {
        if (Auth::user()->isAbleTo('mobileservice_devicebrand create')) {
            return view('mobile-service-management::service-setup.device-brands.create');
        } else {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function store(Request $request)
    {
        if (Auth::user()->isAbleTo('mobileservice_devicebrand create')) {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string'
            ]);

            if ($validator->fails()) {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $brand             = new MobileServiceBrand();
            $brand->name       = $request->name;
            $brand->workspace  = getActiveWorkSpace();
            $brand->created_by = creatorId();
            $brand->save();

            event(new CreateMobileServiceBrand($request, $brand));

            return redirect()->back()->with('success', __('The device brand has been created successfully'));
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function edit(Request $request, $id)
    {
        if (Auth::user()->isAbleTo('mobileservice_devicebrand edit')) {
            $brand = MobileServiceBrand::where('id', $id)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($brand) {
                return view('mobile-service-management::service-setup.device-brands.edit', compact('brand'));
            } else {
                return response()->json(['error' => __('Device brand not found.')], 404);
            }
        } else {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function update(Request $request, $id)
    {
        if (Auth::user()->isAbleTo('mobileservice_devicebrand edit')) {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string'
            ]);

            if ($validator->fails()) {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $brand = MobileServiceBrand::where('id', $id)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($brand) {
                $brand->name = $request->name;
                $brand->save();

                event(new UpdateMobileServiceBrand($request, $brand));

                return redirect()->back()->with('success', __('The device brand details are updated successfully'));
            } else {
                return redirect()->back()->with('error', __('Device brand not found.'));
            }
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function destroy($id)
    {
        if (Auth::user()->isAbleTo('mobileservice_devicebrand delete')) {
            $brand = MobileServiceBrand::where('id', $id)
                ->where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->first();

            if ($brand) {
                event(new DestroyMobileServiceBrand($brand));
                $brand->delete();

                return redirect()->back()->with('success', __('The device brand has been deleted'));
            } else {
                return redirect()->back()->with('error', __('Device brand not found.'));
            }
        } else {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
}
