<?php

namespace Workdo\MobileServiceManagement\DataTables;

use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

use Workdo\MobileServiceManagement\Entities\MobileServiceReview;

class MobileServiceReviewDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        $rowColumn = ['service_id', 'mobile_service_id', 'rating', 'review', 'created_at'];

        $dataTable = (new EloquentDataTable($query))
            ->addIndexColumn()
            ->editColumn('service_id', function (MobileServiceReview $data) {
                $href = \Laratrust::hasPermission('mobileservice_servicerequest show') ? route('mobileservice.service-request.show', ['serviceId' => encrypt($data->service_id)]) : "#";
                return '<a class="btn btn-outline-primary"  href="' . $href . '" >' . $data->service_id . '</a>';
            })
            ->editColumn('mobile_service_id', function (MobileServiceReview $data) {
                return $data->service ? $data->service->name : __('Other service');
            })
            ->filterColumn('mobile_service_id', function ($query, $keyword) {
                if (stripos(__('Other service'), $keyword) !== false) {
                    $query->where('mobile_service_id', 0);
                } else {
                    $query->whereHas('service', function ($q) use ($keyword) {
                        $q->where('name', 'LIKE', "%$keyword%");
                    });
                }
            })
            ->editColumn('rating', function (MobileServiceReview $data) {
                $stars = '';
                for ($i = 1; $i <= 5; $i++) {
                    $stars .= $i <= $data->rating ? '<i class="fas fa-star text-warning"></i>' : '<i class="far fa-star text-muted"></i>';
                }
                return $stars;
            })
            ->editColumn('review', function (MobileServiceReview $data) {
                $url  = route('mobileservice.reviews.review', $data->id);
                $html = '<a class="action-item" data-url="' . $url . '" data-ajax-popup="true" data-bs-toggle="tooltip" title="' . __('Review') . '" data-title="' . __('Review') . '"><i class="fa fa-comment"></i></a>';
                return $html;
            })
            ->editColumn('created_at', function (MobileServiceReview $data) {
                return company_date_formate($data->created_at);
            })
            ->filterColumn('created_at', function ($query, $keyword) {
                try {
                    if (\Carbon\Carbon::hasFormat($keyword, 'd-m-Y')) {
                        $date = \Carbon\Carbon::createFromFormat('d-m-Y', $keyword)->format('Y-m-d');
                        return $query->where('created_at', 'LIKE', "%$date%");
                    } elseif (\Carbon\Carbon::hasFormat($keyword, 'm-Y')) {
                        $date = \Carbon\Carbon::createFromFormat('m-Y', $keyword)->format('Y-m');
                        return $query->where('created_at', 'LIKE', "%$date%");
                    } elseif (\Carbon\Carbon::hasFormat($keyword, 'd-m')) {
                        $date = \Carbon\Carbon::createFromFormat('d-m', $keyword)->format('m-d');
                        return $query->where('created_at', 'LIKE', "%$date%");
                    } else {
                        $hasDay   = false;
                        $hasMonth = false;
                        $hasYear = false;
                        if (\Carbon\Carbon::hasFormat($keyword, 'd') && strlen($keyword) <= 2) {
                            $day = \Carbon\Carbon::createFromFormat('d', $keyword)->format('d');
                            $query->whereRaw('DAY(created_at) = ?', [$day]);
                            $hasDay = true;
                        }
                        if (\Carbon\Carbon::hasFormat($keyword, 'm') && strlen($keyword) <= 2) {
                            $month = \Carbon\Carbon::createFromFormat('m', $keyword)->format('m');
                            $query->orWhereRaw('MONTH(created_at) = ?', [$month]);
                            $hasMonth = true;
                        }
                        if (preg_match('/^\d{4}$/', $keyword)) {
                            $year = \Carbon\Carbon::createFromFormat('Y', $keyword)->format('Y');
                            $query->orWhereRaw('YEAR(created_at) = ?', [$year]);
                            $hasYear = true;
                        }

                        if ($hasDay || $hasMonth || $hasYear) {
                            return $query;
                        }
                    }
                } catch (\Exception $e) {
                    \Log::error('Invalid date format: ' . $keyword);
                }
            });

        if (\Laratrust::hasPermission('mobileservice_review delete')) {
            $dataTable->addColumn('action', function (MobileServiceReview $data) {
                return view('mobile-service-management::reviews.action', compact('data'));
            });
            $rowColumn[] = 'action';
        }

        return $dataTable->rawColumns($rowColumn);
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(MobileServiceReview $model): QueryBuilder
    {
        return $model->with('service')
            ->where('mobile_service_reviews.created_by', creatorId())
            ->where('mobile_service_reviews.workspace', getActiveWorkSpace());
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        $dataTable = $this->builder()
            ->setTableId('mobile-service-reviews')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(0)
            ->language([
                "paginate"          => [
                    "next"     => '<i class="ti ti-chevron-right"></i>',
                    "previous" => '<i class="ti ti-chevron-left"></i>'
                ],
                'lengthMenu'        => "_MENU_" . __('Entries Per Page'),
                "searchPlaceholder" => __('Search...'),
                "search"            => "",
                "info"              => __('Showing _START_ to _END_ of _TOTAL_ entries')
            ])
            ->initComplete('function() {
                var table = this;
                var searchInput = $(\'#\'+table.api().table().container().id+\' label input[type="search"]\');
                searchInput.removeClass(\'form-control form-control-sm\');
                searchInput.addClass(\'dataTable-input\');
                var select = $(table.api().table().container()).find(".dataTables_length select").removeClass(\'custom-select custom-select-sm form-control form-control-sm\').addClass(\'dataTable-selector\');
            }');

        $exportButtonConfig = [
            'extend'    => 'collection',
            'className' => 'btn btn-light-secondary dropdown-toggle',
            'text'      => '<i class="ti ti-download me-2" data-bs-toggle="tooltip" data-bs-placement="top" data-bs-original-title="Export"></i>',
            'buttons'   => [
                [
                    'extend'        => 'print',
                    'text'          => '<i class="fas fa-print me-2"></i> ' . __('Print'),
                    'className'     => 'btn btn-light text-primary dropdown-item',
                    'exportOptions' => ['columns' => [0, 1, 3]],
                ],
                [
                    'extend'        => 'csv',
                    'text'          => '<i class="fas fa-file-csv me-2"></i> ' . __('CSV'),
                    'className'     => 'btn btn-light text-primary dropdown-item',
                    'exportOptions' => ['columns' => [0, 1, 3]],
                ],
                [
                    'extend'        => 'excel',
                    'text'          => '<i class="fas fa-file-excel me-2"></i> ' . __('Excel'),
                    'className'     => 'btn btn-light text-primary dropdown-item',
                    'exportOptions' => ['columns' => [0, 1, 3]],
                ],
            ],
        ];

        $buttonsConfig = array_merge([
            $exportButtonConfig,
            [
                'extend'    => 'reset',
                'className' => 'btn btn-light-danger',
            ],
            [
                'extend'    => 'reload',
                'className' => 'btn btn-light-warning',
            ],
        ]);

        $dataTable->parameters([
            "dom"          => "
        <'dataTable-top'<'dataTable-dropdown page-dropdown'l><'dataTable-botton table-btn dataTable-search tb-search  d-flex justify-content-end gap-2'Bf>>
        <'dataTable-container'<'col-sm-12'tr>>
        <'dataTable-bottom row'<'col-5'i><'col-7'p>>",
            'buttons'      => $buttonsConfig,
            "drawCallback" => 'function( settings ) {
                var tooltipTriggerList = [].slice.call(
                    document.querySelectorAll("[data-bs-toggle=tooltip]")
                  );
                  var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                    return new bootstrap.Tooltip(tooltipTriggerEl);
                  });
                  var popoverTriggerList = [].slice.call(
                    document.querySelectorAll("[data-bs-toggle=popover]")
                  );
                  var popoverList = popoverTriggerList.map(function (popoverTriggerEl) {
                    return new bootstrap.Popover(popoverTriggerEl);
                  });
                  var toastElList = [].slice.call(document.querySelectorAll(".toast"));
                  var toastList = toastElList.map(function (toastEl) {
                    return new bootstrap.Toast(toastEl);
                  });
            }'
        ]);

        $dataTable->language([
            'buttons' => [
                'create' => __('Create'),
                'export' => __('Export'),
                'print'  => __('Print'),
                'reset'  => __('Reset'),
                'reload' => __('Reload'),
                'excel'  => __('Excel'),
                'csv'    => __('CSV'),
            ]
        ]);

        return $dataTable;
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        $column = [
            Column::make('id')->searchable(false)->visible(false)->exportable(false)->printable(false),
            Column::make('No')->title(__('No'))->data('DT_RowIndex')->name('DT_RowIndex')->searchable(false)->orderable(false),
            Column::make('service_id')->title(__('Service ID')),
            Column::make('name')->title(__('Name')),
            Column::make('email')->title(__('Email')),
            Column::make('mobile_service_id')->title(__('Service')),
            Column::make('rating')->title(__('Rating')),
            Column::make('review')->title(__('Review'))->searchable(false)->orderable(false),
            Column::make('created_at')->title(__('Created')),
        ];

        if (\Laratrust::hasPermission('mobileservice_review delete')) {
            $action = [
                Column::computed('action')
                    ->title(__('Action'))
                    ->exportable(false)
                    ->printable(false)
                    ->addClass('text-end')
                    ->width(60),
            ];
            $column = array_merge($column, $action);
        }

        return $column;
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Reviews_' . date('YmdHis');
    }
}
