<?php
defined('ABSPATH') or exit;

class Zynith_SEO_Sitemap {

	private $max_urls = 25000;
	private $base_path;

	public static function init() {
		return new self();
	}

	public function __construct() {
		$this->base_path = Zynith_SEO_Settings::get_abspath();
		register_activation_hook(ZYNITH_SEO_FILE, [$this, 'add_event']);
		register_deactivation_hook(ZYNITH_SEO_FILE, [$this, 'remove_event']);
		add_filter('wp_sitemaps_enabled', '__return_false');
		add_action('template_redirect', [$this, 'redirect_default_sitemap']);
		add_filter('cron_schedules', [$this, 'add_schedules']);
		add_action('zynith_seo_generate_sitemap', [$this, 'generate_sitemap']);

		// Hook into post save actions
		add_action('save_post', [$this, 'generate_sitemap_on_update']);
		add_action('delete_post', [$this, 'generate_sitemap_on_update']);
		add_action('wp_trash_post', [$this, 'generate_sitemap_on_update']);
	}

	public function add_event() {
		if (!wp_next_scheduled('zynith_seo_generate_sitemap')) {
			wp_schedule_event(time(), 'every_six_hours', 'zynith_seo_generate_sitemap');
		}
		flush_rewrite_rules();
	}

	public function remove_event() {
		wp_clear_scheduled_hook('zynith_seo_generate_sitemap');
		$this->delete_files();
	}

	public function delete_files() {
		$sitemap_name = Zynith_SEO_Settings::get_sitemap_name();
		$sitemap_base_name = preg_replace('/\.xml$/', '', $sitemap_name);
		$paths = [
			Zynith_SEO_Settings::get_sitemap_path(),
			$this->base_path . 'sitemap.xsl',
			$this->base_path . 'sitemap-index.xsl'
		];
		if ($handle = opendir($this->base_path)) {
			while (false !== ($file = readdir($handle))) {
				if ($file != "." && $file != ".." && preg_match('/^[a-zA-Z0-9_-]+-' . preg_quote($sitemap_base_name) . '-\d+\.xml$/', $file)) {
					$paths[] = $this->base_path . $file;
				}
			}
			closedir($handle);
		}
		foreach ($paths as $path) {
			if (file_exists($path)) {
				@unlink($path);
			}
		}
	}

	public function redirect_default_sitemap() {
		$sitemap_filename = Zynith_SEO_Settings::get_sitemap_name();
		if ($sitemap_filename === 'wp-sitemap.xml') {
			return;
		}
		$current_url = $_SERVER['REQUEST_URI'];
		if (strpos($current_url, 'wp-sitemap.xml') !== false) {
			wp_redirect(home_url($sitemap_filename), 301);
			exit;
		}
	}

	public function add_schedules($schedules) {
		$custom_schedules = ['every_six_hours' => [
			'interval' => HOUR_IN_SECONDS * 6,
			'display' => __('Every Six Hours', ZYNITH_SEO_TEXT_DOMAIN)
		]];
		return array_merge($schedules, $custom_schedules);
	}

	public function generate_sitemap() {
		$this->delete_files();
		$urls = $this->generate_sitemap_urls();
		$total_urls = array_sum(array_map('count', $urls));
		$multi_sitemap = $total_urls > $this->max_urls;
		$this->update_sitemap_stylesheet($total_urls, $multi_sitemap);
		if (!$multi_sitemap) {
			$this->write_to_path(Zynith_SEO_Settings::get_sitemap_path(), $this->get_sitemap_xml($urls));
		} else {
			$sitemap_name = Zynith_SEO_Settings::get_sitemap_name();
			$sitemap_base_name = preg_replace('/\.xml$/', '', $sitemap_name);
			$sitemap_index = [];
			foreach ($urls as $type => $type_urls) {
				$chunks = array_chunk($type_urls, $this->max_urls);
				foreach ($chunks as $index => $chunk) {
					$filename = $type . '-' . $sitemap_base_name . '-' . ($index + 1) . '.xml';
					$path = $this->base_path . $filename;
					$this->write_to_path($path, $this->get_sitemap_xml([$type => $chunk]));
					$sitemap_index[] = $filename;
				}
			}
			$this->save_sitemap_index($sitemap_index);
		}
		flush_rewrite_rules();
	}

	// New method to generate sitemap on post update
	public function generate_sitemap_on_update($post_id) {
		// Verify if this is an auto-save routine.
		if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
			return;
		}

		// Check permissions and ensure it's a valid post type
		if (!current_user_can('edit_post', $post_id) || !in_array(get_post_type($post_id), array_keys(Zynith_SEO_Post_Types::get_post_types()))) {
			return;
		}

		// Generate the sitemap
		$this->generate_sitemap();
	}

	public function generate_sitemap_urls() {
		global $wpdb;
		$urls = [];
		$authors = [];
		$post_types = array_keys(Zynith_SEO_Post_Types::get_post_types());
		$noindex_post_types = get_option('zynith_noindex_post_types');
		$batch_size = 100;
		foreach ($post_types as $post_type) {
			if (!empty($noindex_post_types) && in_array($post_type, array_keys($noindex_post_types))) {
				continue;
			}
			$offset = 0;
			do {
				$posts = $wpdb->get_results($wpdb->prepare(
					"SELECT ID, post_author, post_modified_gmt FROM {$wpdb->prefix}posts 
					WHERE post_type = %s AND post_status = 'publish' 
					ORDER BY post_modified_gmt DESC LIMIT %d OFFSET %d", $post_type, $batch_size, $offset));
				if (empty($posts)) {
					break;
				}
				foreach ($posts as $post) {
					if ('yes' === get_post_meta($post->ID, '_custom_noindex', true)) {
						continue;
					}
					$authors[$post->post_author] = $post->post_author;
					$urls[$post_type][] = [
						'loc' => htmlspecialchars(get_permalink($post->ID)),
						'lastmod' => get_the_modified_date('c', $post->ID)
					];
				}
				$offset += $batch_size;
			} while (!empty($posts));
		}
		$taxonomies = array_keys(Zynith_SEO_Taxonomies::get_taxonomies());
		$noindex_taxonomies = get_option('zynith_noindex_taxonomies');
		foreach ($taxonomies as $taxonomy) {
			if ($taxonomy === 'product_shipping_class' || (!empty($noindex_taxonomies) && in_array($taxonomy, array_keys($noindex_taxonomies)))) {
				continue;
			}
			$terms = get_terms(['taxonomy' => $taxonomy, 'hide_empty' => false]);
			foreach ($terms as $term) {
				if ('yes' === get_term_meta($term->term_id, '_custom_noindex', true)) {
					continue;
				}
				$urls[$taxonomy][] = [
					'loc' => htmlspecialchars(get_term_link($term->term_id, $taxonomy)),
					'lastmod' => ''
				];
			}
		}
		$noindex_archives = get_option('zynith_noindex_archives');
		if (!empty($noindex_archives) && !in_array('author', array_keys($noindex_archives))) {
			foreach ($authors as $author_id) {
				if ('yes' === get_user_meta($author_id, '_custom_noindex', true)) {
					continue;
				}
				$urls[$taxonomy][] = [
					'loc' => htmlspecialchars(get_author_posts_url($author_id)),
					'lastmod' => ''
				];
			}
		}

		// Filter out Divi-related URLs
		$divi_patterns = ['/layout_type/', '/module_width/', '/scope/'];
		foreach ($urls as $type => $url_entries) {
			$urls[$type] = array_filter($url_entries, function ($url) use ($divi_patterns) {
				foreach ($divi_patterns as $pattern) {
					if (strpos($url['loc'], $pattern) !== false) {
						return false;
					}
				}
				return true;
			});
		}
		return $urls;
	}

	private function update_sitemap_stylesheet($total_urls, $multi_sitemap) {
		$stylesheet_path = $this->base_path . 'sitemap.xsl';
		$index_stylesheet_path = $this->base_path . 'sitemap-index.xsl';
		$main_updated = $this->write_to_path($stylesheet_path, $this->get_sitemap_stylesheet($total_urls, $multi_sitemap));
		$index_updated = false;
		if ($multi_sitemap) {
			$index_updated = $this->write_to_path($index_stylesheet_path, $this->get_sitemap_index_stylesheet());
		}
		return $multi_sitemap ? $main_updated && $index_updated : $main_updated;
	}

	public function get_sitemap_stylesheet($total_urls, $multi_sitemap) {
		return '<?xml version="1.0" encoding="utf-8"?>
			<xsl:stylesheet version="2.0" xmlns:sitemap="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
			<xsl:output method="html" version="1.0" encoding="UTF-8" indent="yes"/>
			<xsl:template match="/">
			<html xmlns="http://www.w3.org/1999/xhtml" lang="' . substr(get_locale(), 0, 2) . '">
            <head>
				<meta charset="utf-8"/>
				<meta name="viewport" content="width=device-width, initial-scale=1"/>
				<title>Sitemap &#x2d; ' . get_bloginfo('name') . '</title>
				<style style="text/css">
					body { font-family: Verdana,Geneva,sans-serif; margin: 0; padding: 22px; }
					h1 { font-size: 26px; font-weight: 400; margin: 11px 0; color: #851CE2; }
					a { color: #0060D8; text-decoration: none; overflow-wrap: anywhere; }
					a:hover { color: #465D91; border-bottom: 1px solid #465D91; }
					#content { max-width: 768px; margin: 0 auto; }
					table { margin: 11px 0; font-size: 14px; border: 1px solid #c8c6c6; border-spacing: 0; table-layout: fixed; }
					th, td { padding: 10px; font-size: 13px; text-align: left; border-bottom: 1px solid #c8c6c6; overflow-wrap: anywhere; }
					tbody > tr:nth-of-type(odd) { background-color: #E5E2E1; }
					#footer { padding: 11px 0; font-size: 13px; color: #474747; }
					#footer a { color: inherit; }
				</style>
			</head>
            <body>
				<div id="content">
					<h1>' . get_bloginfo('name') . ' — Sitemap</h1>
					<a href="' . home_url() . '">' . home_url() . '</a>
					<table>
						<thead>
							<tr>
								<th>URL</th>
								<th>Date Updated</th>
							</tr>
						</thead>
						<tbody>
							<xsl:for-each select="sitemap:urlset/sitemap:url">
							<xsl:variable name="itemURL" select="sitemap:loc"/>
							<xsl:variable name="lastmod" select="sitemap:lastmod"/>
							<tr>
								<td>
									<a href="{$itemURL}">
										<xsl:choose>
											<xsl:when test="string-length($itemURL)&gt;95">
												<xsl:value-of select="substring($itemURL,0,93)"/>...</xsl:when>
												<xsl:otherwise>
													<xsl:value-of select="$itemURL"/>
												</xsl:otherwise>
											</xsl:choose>
										</a>
									</td>
									<td>
										<xsl:value-of select="$lastmod"/>
									</td>
								</tr>
								</xsl:for-each>
							</tbody>
						</table>
						<div id="footer">Sitemap generated by <a href="https://zynith.app/" target="_blank" rel="nofollow noopener">Zynith SEO</a> on ' . date('c') . (!$multi_sitemap ? ' and generated a total of ' . $total_urls . ' URLs.' : '.') . '</div>
					</div>
				</body>
				</html>
				</xsl:template>
				</xsl:stylesheet>';
	}

	public function get_sitemap_index_stylesheet() {
		return '<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet version="2.0" xmlns:sitemap="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
    <xsl:output method="html" encoding="utf-8" indent="yes"/>
    <xsl:template match="/">
        <html xmlns="http://www.w3.org/1999/xhtml" lang="' . substr(get_locale(), 0, 2) . '">
            <head>
                <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
                <meta name="viewport" content="width=device-width, initial-scale=1"/>
                <title>Sitemap Index for {{ site_name }}</title>
                <style>
                    body { font-family: Verdana,Geneva,sans-serif; margin: 0; padding: 22px; }
                    h1 { font-size: 26px; font-weight: 400; margin: 11px 0; color: #851CE2; }
					a { color: #0060D8; text-decoration: none; overflow-wrap: anywhere; }
					a:hover { color: #465D91; border-bottom: 1px solid #465D91; }
					#content { max-width: 768px; margin: 0 auto; }
					ol { margin: 11px 0; font-size: 14px; border: 1px solid #c8c6c6; }
					li { padding: 11px; color: #0060D8; }
					#footer { padding: 11px 0; font-size: 13px; color: #474747; }
					#footer a { color: inherit; }
                </style>
            </head>
            <body>
                <div id="content">
					<h1>' . get_bloginfo('name') . ' — Sitemap Index</h1>
					<a href="' . home_url() . '">' . home_url() . '</a>
					<ol>
						<xsl:for-each select="sitemap:sitemapindex/sitemap:sitemap">
							<li>
								<a href="{sitemap:loc}">
										<xsl:value-of select="sitemap:loc"/>
								</a>
							</li>
						</xsl:for-each>
					</ol>
					<div id="footer">Sitemap index generated by <a href="https://zynith.app/" target="_blank" rel="nofollow noopener">Zynith SEO</a> on ' . date('c') . '.</div>
				</div>
            </body>
        </html>
    </xsl:template>
</xsl:stylesheet>';
	}

	private function write_to_path($path, $content) {
		if (!file_exists(dirname($path))) {
			mkdir(dirname($path), 0755, true);
		}
		return file_put_contents($path, $content) !== false;
	}

	public function get_sitemap_xml($urls_by_type) {
		$stylesheet_path = $this->base_path . 'sitemap.xsl';
		$xml = '<?xml version="1.0" encoding="utf-8"?>';
		$xml .= file_exists($stylesheet_path) ? "<?xml-stylesheet type=\"text/xsl\" href=\"" . home_url('sitemap.xsl') . "\"?>" : "";
		$xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"><!-- Sitemap was generated on ' . date('c') . ' -->';
		if (!empty($urls_by_type)) {
			foreach ($urls_by_type as $post_type => $urls) {
				$post_type_obj = get_post_type_object($post_type);
				$post_type_name = $post_type_obj ? $post_type_obj->labels->singular_name : $post_type;
				$xml .= '<!-- ' . $post_type_name . ' URLs -->';
				foreach ($urls as $url) {
					$xml .= '<url>';
					$xml .= "<loc>{$url['loc']}</loc>";
					if (!empty($url['lastmod'])) {
						$xml .= "<lastmod>{$url['lastmod']}</lastmod>";
					}
					$xml .= "</url>";
				}
			}
		}
		$xml .= '</urlset>';
		return $xml;
	}

	private function save_sitemap_index($sitemaps) {
		$stylesheet_path = $this->base_path . 'sitemap-index.xsl';
		$index_content = '<?xml version="1.0" encoding="utf-8"?>';
		$index_content .= file_exists($stylesheet_path) ? "<?xml-stylesheet type=\"text/xsl\" href=\"" . home_url('sitemap-index.xsl') . "\"?>" : "";
		$index_content .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"><!-- Sitemap was generated on ' . date('c') . ' -->' . "\n";
		foreach ($sitemaps as $sitemap) {
			$index_content .= '<sitemap>';
			$index_content .= '<loc>' . esc_url(home_url($sitemap)) . '</loc>';
			$index_content .= "</sitemap>";
		}
		$index_content .= '</sitemapindex>';
		file_put_contents(Zynith_SEO_Settings::get_sitemap_path(), $index_content);
	}
}