<?php
defined('ABSPATH') or exit;

class Zynith_SEO_Settings {
	
	public $settings = [];
	
	private static $instance;
	private static $settings_home_title = 'Dynamic Homepage Settings';
	private static $plugin_pages = ['zynith-seo-settings', 'schema-templates', 'zynith-script-manager', '404-monitor-settings'];
	
	public static function init() {
		if (null === self::$instance) self::$instance = new self();
		return self::$instance;
	}

	public function __construct() {
		add_action('init', [$this, 'initialize_settings']);
		add_action('admin_menu', [$this, 'add_settings_menus']);
		add_action('admin_init', [$this, 'register_settings']);
		add_action('admin_notices', [$this, 'show_admin_notices']);
		add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
		add_filter('robots_txt', [$this, 'filter_robots_txt'], 999, 2);
	}
	
	public function initialize_settings() {
        $this->settings = $this->get_settings();
    }
	
	public function get_settings() {
		$sitemap_name = self::get_sitemap_name();
		$settings = [
    		'zynith-seo-settings' => [
				'full_title' => __(ZYNITH_SEO_NAME . ' Settings', ZYNITH_SEO_TEXT_DOMAIN),
				'menu_title' => __(ZYNITH_SEO_NAME, ZYNITH_SEO_TEXT_DOMAIN),
				'groups' => [
					'business' => [
						'title' => __('Business Details', ZYNITH_SEO_TEXT_DOMAIN),
						'description' => __("Enter your business details here to dynamically insert them into meta titles and descriptions, and to enhance schema markup across your website.", ZYNITH_SEO_TEXT_DOMAIN),
						'fields' => [
							[
								'id' => 'business_name',
								'type' => 'text',
								'label' => __('Business Name:', ZYNITH_SEO_TEXT_DOMAIN),
								'placeholder' => 'Business Name',
								'description' => sprintf(__("Use %s%%%%business_name%%%%%s as a placeholder.", ZYNITH_SEO_TEXT_DOMAIN), '<code>', '</code>')
							],
							[
								'id' => 'business_email',
								'type' => 'email',
								'label' => __('Business Email:', ZYNITH_SEO_TEXT_DOMAIN),
								'placeholder' => 'Business Email',
								'description' => sprintf(__("Use %s%%%%business_email%%%%%s as a placeholder.", ZYNITH_SEO_TEXT_DOMAIN), '<code>', '</code>')
							],
							[
								'id' => 'phone_number',
								'type' => 'tel',
								'label' => __('Phone Number:', ZYNITH_SEO_TEXT_DOMAIN),
								'placeholder' => 'Phone Number',
								'description' => sprintf(__("Use %s%%%%business_phone%%%%%s as a placeholder.", ZYNITH_SEO_TEXT_DOMAIN), '<code>', '</code>')
							],
							[
								'id' => 'street_address',
								'type' => 'text',
								'label' => __('Street Address:', ZYNITH_SEO_TEXT_DOMAIN),
								'placeholder' => 'Street Address',
								'description' => sprintf(__("Use %s%%%%street_address%%%%%s as a placeholder.", ZYNITH_SEO_TEXT_DOMAIN), '<code>', '</code>')
							],
							[
								'id' => 'address_locality',
								'type' => 'text',
								'label' => __('Locality:', ZYNITH_SEO_TEXT_DOMAIN),
								'placeholder' => 'Enter city or suburb',
								'description' => sprintf(__("Use %s%%%%locality%%%%%s as a placeholder.", ZYNITH_SEO_TEXT_DOMAIN), '<code>', '</code>')
							],
							[
								'id' => 'address_region',
								'type' => 'text',
								'label' => __('Region:', ZYNITH_SEO_TEXT_DOMAIN),
								'placeholder' => 'State or Province',
								'description' => sprintf(__("Use %s%%%%region%%%%%s as a placeholder.", ZYNITH_SEO_TEXT_DOMAIN), '<code>', '</code>')
							],
							[
								'id' => 'postal_code',
								'type' => 'text',
								'label' => __('Postal Code:', ZYNITH_SEO_TEXT_DOMAIN),
								'placeholder' => 'Postal Code',
								'description' => sprintf(__("Use %s%%%%postal_code%%%%%s as a placeholder.", ZYNITH_SEO_TEXT_DOMAIN), '<code>', '</code>')
							],
							[
								'id' => 'address_country',
								'type' => 'text',
								'label' => __('Country Name:', ZYNITH_SEO_TEXT_DOMAIN),
								'placeholder' => 'Country Name',
								'description' => sprintf(__("Use %s%%%%country_name%%%%%s as a placeholder.", ZYNITH_SEO_TEXT_DOMAIN), '<code>', '</code>')
							],
							[
								'id' => 'address_country_code',
								'type' => 'text',
								'label' => __('Country Code:', ZYNITH_SEO_TEXT_DOMAIN),
								'maxlength' => 2,
								'placeholder' => __("Enter the two-letter ISO country code, e.g., US, AU"),
								'description' => sprintf(__("Use %s%%%%country_code%%%%%s as a placeholder.", ZYNITH_SEO_TEXT_DOMAIN), '<code>', '</code>')
							],
							[
								'id' => 'logo',
								'type' => 'image',
								'label' => __('Logo:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => sprintf(__("Use %s%%%%logo_url%%%%%s as a placeholder.", ZYNITH_SEO_TEXT_DOMAIN), '<code>', '</code>')
							],
							[
								'id' => 'default_og_image',
								'type' => 'image',
								'label' => __('Default Open Graph Image:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => __('The Open Graph image is displayed when sharing on social media, such as Facebook.', ZYNITH_SEO_TEXT_DOMAIN)
							]
						]
					],
					'homepage' => [
						'title' => __(self::$settings_home_title, ZYNITH_SEO_TEXT_DOMAIN),
						'description' => sprintf(__('%sThis section is only for a non-static homepage or posts page. You can view and edit the homepage and posts page settings %shere%s.%s', ZYNITH_SEO_TEXT_DOMAIN ), '<p>',
												 '<a href="' . admin_url('options-reading.php') . '">', '</a>', '</p>'),
						'fields' => [
							[
								'id' => 'homepage_meta_title',
								'type' => 'text',
								'label' => __('Meta Title:', ZYNITH_SEO_TEXT_DOMAIN),
								'placeholder' => __('Enter a SEO-friendly title for your homepage', ZYNITH_SEO_TEXT_DOMAIN)
							],[
								'id' => 'homepage_meta_description',
								'type' => 'textarea',
								'label' => __('Meta Description:', ZYNITH_SEO_TEXT_DOMAIN),
								'placeholder' => __('Provide a brief description of your homepage', ZYNITH_SEO_TEXT_DOMAIN)
							],[
								'id' => 'homepage_schema_markup',
								'type' => 'code',
								'label' => __('Homepage Schema Markup:', ZYNITH_SEO_TEXT_DOMAIN),
								'placeholder' => __('Enter your JSON-LD structured data for the homepage here, surrounded in braces { ... }', ZYNITH_SEO_TEXT_DOMAIN)
							]
						]
					],
					'global_robots' => [
						'title' => __('Global Robots Settings', ZYNITH_SEO_TEXT_DOMAIN),
						'description' => __("This section allows you to configure and customize how search engines crawl and index various parts of your website, including the generation of dynamic Robots.txt to manage site access."),
						'fields' => [
							[
								'id' => 'custom_sitemap_filename',
								'type' => 'text',
								'label' => __('Sitemap Filename:', ZYNITH_SEO_TEXT_DOMAIN),
								'required' => true,
								'default' => $sitemap_name
							],
							[
								'id' => 'sitemap_url',
								'type' => 'message',
								'label' => __('View Sitemap:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => '<a href="' . esc_url(home_url($sitemap_name)) . '" target="_blank">' . esc_html(home_url($sitemap_name)) . '</a>'
							],
							[
								'id' => 'remove_canonical',
								'type' => 'checkbox',
								'label' => __('Remove Canonical Tag:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => __('Enable this option if you want another plugin to manage the canonical tags for your pages, potentially to avoid conflicts or to use specialized functionality.', ZYNITH_SEO_TEXT_DOMAIN)
							],
							[
								'id' => 'global_noindex',
								'type' => 'message',
								'label' => __("Global 'noindex':", ZYNITH_SEO_TEXT_DOMAIN),
								'description' => sprintf(__('Apply %snoindex%s globally to the following page types listed below.', ZYNITH_SEO_TEXT_DOMAIN), '<strong>', '</strong>')
							],
							[
								'id' => 'noindex_post_types',
								'type' => 'checkbox_group',
								'label' => '',
								'options' => Zynith_SEO_Post_Types::get_post_types()
							],
							[
								'id' => 'noindex_taxonomies',
								'type' => 'checkbox_group',
								'label' => '',
								'options' => Zynith_SEO_Taxonomies::get_taxonomies(),
								'default' => ['product_shipping_class']
							],
							[
								'id' => 'noindex_archives',
								'type' => 'checkbox_group',
								'label' => '',
								'options' => [
									'author' => ['label' => __('Author', ZYNITH_SEO_TEXT_DOMAIN )],
									'date' => ['label' => __('Date', ZYNITH_SEO_TEXT_DOMAIN )],
									'search' => ['label' => __('Search', ZYNITH_SEO_TEXT_DOMAIN)]
								],
								'default' => ['search']
							],
							[
								'id' => 'global_nofollow',
								'type' => 'message',
								'label' => __("Global 'nofollow':", ZYNITH_SEO_TEXT_DOMAIN),
								'description' => sprintf(__('Apply %snofollow%s globally to the following page types listed below.', ZYNITH_SEO_TEXT_DOMAIN), '<strong>', '</strong>')
							],
							[
								'id' => 'nofollow_post_types',
								'type' => 'checkbox_group',
								'label' => '',
								'options' => Zynith_SEO_Post_Types::get_post_types()
							],
							[
								'id' => 'nofollow_taxonomies',
								'type' => 'checkbox_group',
								'label' => '',
								'options' => Zynith_SEO_Taxonomies::get_taxonomies()
							],
							[
								'id' => 'nofollow_archives',
								'type' => 'checkbox_group',
								'label' => '',
								'options' => [
									'author' => ['label' => __('Author', ZYNITH_SEO_TEXT_DOMAIN)],
									'date' => ['label' => __('Date', ZYNITH_SEO_TEXT_DOMAIN)],
									'search' => ['label' => __('Search', ZYNITH_SEO_TEXT_DOMAIN)],
								]
							],
							[
								'id' => 'dynamic_robots_txt',
								'type' => 'checkbox',
								'label' => __("Use Dynamic 'robots.txt':", ZYNITH_SEO_TEXT_DOMAIN),
								'description' => __('Disabling this uses the robots.txt content entered below.', ZYNITH_SEO_TEXT_DOMAIN),
								'default' => 0
							]
						]
					],
					'wp_settings' => [
						'title' => __('WordPress Settings', ZYNITH_SEO_TEXT_DOMAIN),
						'fields' => [							
							[
								'id' => 'remove_category_base',
								'type' => 'checkbox',
								'label' => __('Category Base:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => sprintf(__('This option removes the category base from your URLs, changing "%1$s" to "%2$s". While this can simplify your URLs, using %3$sWordPress’s native Permalinks settings%4$s is generally safer and more reliable for such changes.', ZYNITH_SEO_TEXT_DOMAIN), '<code>' . esc_url(home_url('/category/uncategorized/')) . '</code>', '<code>' . esc_url(home_url('/uncategorized/')) . '</code>', '<a href="' . admin_url('options-permalink.php') . '">', '</a>')
							],		
							[
								'id' => 'remove_rss_feeds',
								'type' => 'checkbox',
								'label' => __('RSS Feeds:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => __("Enable this option to remove all default RSS feed links from your website's head section, which can help simplify your site's HTML output if you do not use RSS.", ZYNITH_SEO_TEXT_DOMAIN)
							],[
								'id' => 'remove_generator_tags',
								'type' => 'checkbox',
								'label' => __('Generator Tags:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => __("Activating this will strip WordPress version numbers and other generator tags from your site's HTML. This can be a good practice to obscure your site's WordPress version for security reasons.", ZYNITH_SEO_TEXT_DOMAIN)
							],[
								'id' => 'disable_media_endpoint',
								'type' => 'checkbox',
								'label' => __('Disable Media REST API Endpoint:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => __('Turn this on to disable the REST API endpoint for media items (/wp/v2/media). Useful for enhancing security or performance if your site does not need API access to media resources.', ZYNITH_SEO_TEXT_DOMAIN)
							],[
								'id' => 'disable_gutenberg_editor',
								'type' => 'checkbox',
								'label' => __('Disable Gutenberg Editor:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => __('Select this option to revert the default editor from Gutenberg to the Classic Editor. This is useful if you prefer the traditional editing experience or have compatibility issues with Gutenberg.', ZYNITH_SEO_TEXT_DOMAIN)
							]
						]
					],
					'zynith_seo_settings' => [
						'title' => __(ZYNITH_SEO_NAME . ' Settings', ZYNITH_SEO_TEXT_DOMAIN),
						'fields' => [
							[
								'id' => 'style_select',
								'type' => 'select',
								'label' => __('Select Zynith Style Mode:', ZYNITH_SEO_TEXT_DOMAIN),
								'options' => [
									'default' => __('Light Mode', ZYNITH_SEO_TEXT_DOMAIN),
									'dark' => __('Dark Mode', ZYNITH_SEO_TEXT_DOMAIN)
								],
								'default' => 'default'
							],[
								'id' => 'enable_404_monitor',
								'type' => 'checkbox',
								'label' => __('Enable 404 Monitor:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => __('Enable this option to monitor and log 404 (Not Found) errors on your website.', ZYNITH_SEO_TEXT_DOMAIN),
								'default' => 1
							],[
								'id' => 'license_key',
								'type' => 'password',
								'label' => __('Zynith SEO License Key:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => __('The license key allows you to verify your purchase and receive updates for the plugin.', ZYNITH_SEO_TEXT_DOMAIN),
								'purge_button' => true,
								'purge_button_text' => __('Remove Key', ZYNITH_SEO_TEXT_DOMAIN),
								'confirm_dialog' => __('Are you sure you want to remove the license key? You will no longer receive plugin updates.', ZYNITH_SEO_TEXT_DOMAIN),
							]
						]
						],
					'toc_settings' => [
						'title' => __('ToC Settings', ZYNITH_SEO_TEXT_DOMAIN),
						'fields' => [
							[
								'id' => 'toc_implementation',
								'type' => 'message',
								'label' => __('Instructions:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => sprintf(__('You can add a Table of Contents to any content by using the shortcode: %sTo implement the Table of Contents in your templates with PHP, use: %s', ZYNITH_SEO_TEXT_DOMAIN),
														 '<code>[zynith-toc]</code><br />', '<code>&lt;?php zynith_toc(); ?&gt;</code>')
							],[
								'id' => 'toc_heading_text',
								'type' => 'text',
								'label' => __('ToC Heading Text:', ZYNITH_SEO_TEXT_DOMAIN),
								'default' => 'Table of Contents'
							],[
								'id' => 'toc_heading_1_enabled',
								'type' => 'checkbox',
								'label' => __('Show Heading 1:', ZYNITH_SEO_TEXT_DOMAIN)
							],[
								'id' => 'toc_heading_2_enabled',
								'type' => 'checkbox',
								'label' => __('Show Heading 2:', ZYNITH_SEO_TEXT_DOMAIN),
								'default' => 1
							],[
								'id' => 'toc_heading_3_enabled',
								'type' => 'checkbox',
								'label' => __('Show Heading 3:', ZYNITH_SEO_TEXT_DOMAIN)
							],[
								'id' => 'toc_heading_4_enabled',
								'type' => 'checkbox',
								'label' => __('Show Heading 4:', ZYNITH_SEO_TEXT_DOMAIN)
							],[
								'id' => 'toc_heading_5_enabled',
								'type' => 'checkbox',
								'label' => __('Show Heading 5:', ZYNITH_SEO_TEXT_DOMAIN)
							],[
								'id' => 'toc_heading_6_enabled',
								'type' => 'checkbox',
								'label' => __('Show Heading 6:', ZYNITH_SEO_TEXT_DOMAIN)
							],[
								'id' => 'toc_css',
								'type' => 'code',
								'label' => __('Customize CSS:', ZYNITH_SEO_TEXT_DOMAIN),
								'default' => self::get_default_toc_css()
							],[
								'id' => 'toc_css_restore',
								'type' => 'message',
								'label' => __('Restore Default CSS:', ZYNITH_SEO_TEXT_DOMAIN),
								'default' => self::get_default_toc_css(),
								'description' => sprintf(__("To revert back to the Zynith default ToC CSS, click %sRestore Default CSS%s then make sure to click 'Save Changes' below.", ZYNITH_SEO_TEXT_DOMAIN),
														 '<a href="#zynith_toc_css" class="button-small" onClick="restore_toc_css();" style="font-weight: 500; cursor: pointer;">', '</a>')
							]
						]
					],
					'breadcrumbs' => [
						'title' => __('Breadcrumb Settings', ZYNITH_SEO_TEXT_DOMAIN),
						'fields' => [
							[
								'id' => 'bc_implementation',
								'type' => 'message',
								'label' => __('Instructions:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => sprintf(__('You can add breadcrumbs to any content by using the shortcode: %sTo implement breadcrumbs in your templates with PHP, use: %s', ZYNITH_SEO_TEXT_DOMAIN ),
														 '<code>[zynith-breadcrumbs]</code><br />', '<code>&lt;?php zynith_seo_breadcrumbs(); ?&gt;</code>')
							],
							[
								'id' => 'add_breadcrumbs_schema',
								'type' => 'checkbox',
								'label' => __('Add breadcrumbs schema:', ZYNITH_SEO_TEXT_DOMAIN),
								'description' => __('Automatically add breadcrumbs schema when using breadcrumbs.', ZYNITH_SEO_TEXT_DOMAIN)
							]
						]
					],
					'import_export' => [
						'title' => __('Import/Export', ZYNITH_SEO_TEXT_DOMAIN),
						'fields' => []
					]
				]
			]
		];
		
		$use_dynamic_robots_txt = (bool) get_option(ZYNITH_SEO_TEXT_PREFIX . 'dynamic_robots_txt', false);
		if ($use_dynamic_robots_txt) {
			$lastIndex = count($settings['zynith-seo-settings']['groups']['global_robots']['fields']) - 1;
			$settings['zynith-seo-settings']['groups']['global_robots']['fields'][$lastIndex]['description'] = __('We will automatically add your sitemap and a few other settings using dynamic robots.txt.', ZYNITH_SEO_TEXT_DOMAIN);
			$settings['zynith-seo-settings']['groups']['global_robots']['fields'][] = [
				'id' => 'robots_txt_remove_wp_default',
				'type' => 'checkbox',
				'label' => __('Default robots.txt Output:', ZYNITH_SEO_TEXT_DOMAIN),
				'description' => __('Remove the default robots.txt output added by WordPress in the dynamic robots.txt', ZYNITH_SEO_TEXT_DOMAIN),
				'default' => 1
			];
			$settings['zynith-seo-settings']['groups']['global_robots']['fields'][] = [
				'id' => 'robots_txt_allow_pdf_index',
				'type' => 'checkbox',
				'label' => __('PDF Indexing:', ZYNITH_SEO_TEXT_DOMAIN),
				'description' => __('Allow indexing of your PDF files', ZYNITH_SEO_TEXT_DOMAIN),
				'default' => 1
			];
		}
		else {
			$settings['zynith-seo-settings']['groups']['global_robots']['fields'][] = [
				'id' => 'robots_txt_content',
				'type' => 'code',
				'label' => __('robots.txt Content:', ZYNITH_SEO_TEXT_DOMAIN),
				'placeholder' => __('Specify custom rules for robots.txt.', ZYNITH_SEO_TEXT_DOMAIN),				
				'description' => sprintf(__('%sRecommended syntax for robots.txt is the following:%s', ZYNITH_SEO_TEXT_DOMAIN ), '<p>',
											 '</p><p style="padding-left: 10px;border-left: 2px solid #2d80ea;text-wrap: nowrap;">User-agent: *<br />Disallow: <br />Sitemap: ' . home_url(self::get_sitemap_name()) . '</p>')
			];
		}
		
		return $settings;
	}
	
	public static function get_default_toc_css() {
        return '
details.zynith-toc > * {
	transition: max-height 0.5s ease-in-out;
    overflow: auto;
}
details.zynith-toc[open] > * {
    max-height: 500px;
}
details.zynith-toc:not([open]) > * {
    max-height: 44px;
}
details.zynith-toc {
	max-width: 100%;
	width: fit-content;
	border: 1px solid gray;
    padding: 20px;
}
details.zynith-toc > summary {
	font-size: 1.3em;
	cursor: pointer;
}
.zynith-toc ul {
	margin: 5px 0;
	padding-left: 0;
	list-style-type: none;
}
.zynith-toc li { margin-bottom: 0.1em; }
.zynith-toc .toc-h1 a { font-weight: bold; }
.zynith-toc .toc-h2 { padding-left: 20px; }
.zynith-toc .toc-h3 { padding-left: 40px; }
.zynith-toc .toc-h4 { padding-left: 60px; }
.zynith-toc .toc-h5 { padding-left: 80px; }
.zynith-toc .toc-h6 { padding-left: 100px; }
        ';
    }
	
	public static function get_sitemap_name() {
		$default_filename = 'sitemap.xml';
		$filename = get_option(ZYNITH_SEO_TEXT_PREFIX . 'custom_sitemap_filename', $default_filename);
		if (substr($filename, -4) !== '.xml') $filename .= '.xml';
		return sanitize_file_name($filename);
	}
		
	public function add_settings_menus() {
		$main_menu_slug = ZYNITH_SEO_TEXT_DOMAIN . '-settings';
		add_menu_page(
			__(ZYNITH_SEO_NAME . 'Settings', ZYNITH_SEO_TEXT_DOMAIN), // The title of the main page
			__(ZYNITH_SEO_NAME, ZYNITH_SEO_TEXT_DOMAIN), // The menu title
			'manage_options', // Required capability for seeing this menu
			$main_menu_slug, // The slug name to refer to this menu
			[$this, 'render_zynith_seo_settings_page'], // The function that displays the page content
			ZYNITH_SEO_ICON, // The icon for the menu
			90 // Position where the menu should appear
		);
		add_submenu_page(
			$main_menu_slug,
			__('Settings', ZYNITH_SEO_TEXT_DOMAIN), // Page title
			__('Settings', ZYNITH_SEO_TEXT_DOMAIN), // Menu title
			'manage_options',
			$main_menu_slug,
			[$this, 'render_zynith_seo_settings_page']
		);
		add_action('admin_menu', function() use ($main_menu_slug) {
			$this->rearrange_admin_menu($main_menu_slug);
		}, 100);
	}
	
	private function rearrange_admin_menu($main_menu_slug) {
		if (empty($GLOBALS['submenu'][$main_menu_slug])) return;
		$positions = [$main_menu_slug => 0];
		$enable_404_monitor = get_option(ZYNITH_SEO_TEXT_PREFIX . 'enable_404_monitor', true);
		foreach (self::$plugin_pages as $index => $page_slug) {
			if ($page_slug === '404-monitor-settings' && !$enable_404_monitor) continue;
			$positions[$page_slug] = $index + 1;
		}
		$new_structure = [];
		$fallback_counter = count($positions);
		foreach ($GLOBALS['submenu'][$main_menu_slug] as $item) {
			$position = $positions[$item[2]] ?? $fallback_counter++;
			$new_structure[$position] = $item;
		}
		ksort($new_structure);
		$GLOBALS['submenu'][$main_menu_slug] = $new_structure;
	}
	
	public function render_zynith_seo_settings_page() {
		$current_page = htmlspecialchars(filter_input(INPUT_GET, 'page') ?? ZYNITH_SEO_TEXT_DOMAIN . '-settings');
		$current_tab = filter_input(INPUT_GET, 'tab') ?? 'business';
		$option_group = $current_page . '_' . $current_tab;
		$is_static_homepage = (get_option('show_on_front') === 'page');
		$tabs = [];
		if (isset($this->settings[$current_page]['groups'])) foreach ($this->settings[$current_page]['groups'] as $key => $group) if (!$is_static_homepage || $key !== 'homepage') $tabs[$key] = $group['title'] ?? '';
		echo '<div class="wrap zynith-wrap">';
		echo '<header>';
		echo '<img src="' . ZYNITH_SEO_ICON . '" alt="Zynith SEO" height="30" width="30">';
		echo '<h1>' . esc_html($this->settings[$current_page]['full_title']) . '</h1>';
		echo '</header>';
		echo '<div class="nav-tab-wrapper">';
		foreach ($tabs as $tab_key => $tab_label) {
			$active = ($tab_key == $current_tab) ? 'nav-tab-active' : '';
			echo '<a href="?page=' . esc_attr($current_page) . '&tab=' . esc_attr($tab_key) . '" class="nav-tab ' . $active . '">' . esc_html($tab_label) . '</a>';
		}
		echo '</div>';
		if ($current_tab === 'import_export') {
			$this->plugin_importer();
		}
		else {
			echo '<div class="postbox"><form id="zynith-seo-form-options-' . $current_tab . '" method="post" action="options.php">';
			settings_fields($option_group);
			do_settings_sections($current_page . '_' . $current_tab);
			submit_button();
			echo '</form></div>';
			if ($current_tab === 'zynith_seo_settings') $this->add_purge_option();
		}
		echo '</div>';
	}
	
	public function plugin_importer() {
		?>
		<!-- Import Form -->
		<div class="postbox">
		<form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" enctype="multipart/form-data">
			<?php wp_nonce_field(ZYNITH_SEO_TEXT_PREFIX . 'import_export_action', ZYNITH_SEO_TEXT_PREFIX . 'import_nonce'); ?>
			<h2>Import SEO Data</h2>
			<table class="form-table">
				<tbody>
					<tr>
						<th scope="row">Select SEO Plugin for Import:</th>
						<td>
							<select name="seo_plugin" id="seo_plugin_import" class="regular-text styled-select">
								<option value="zynith">Zynith SEO</option>
								<option value="yoast">Yoast SEO</option>
								<option value="rankmath">Rank Math</option>
							</select>
						</td>
					</tr>
					<tr>
						<th scope="row">Upload CSV File:</th>
						<td>
							<label class="label-file" for="csv_file">
								<input type="file" name="csv_file" id="csv_file" accept=".csv" />Choose File
							</label>
						</td>
					</tr>
					<tr>
						<th scope="row"></th>
						<td>
							<input type="submit" name="import_seo" class="button-primary" value="Import" />
						</td>
					</tr>
				</tbody>
			</table>
		</form>
		</div>
		
		<!-- Export Form -->
		<div class="postbox">
		<form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
			<?php wp_nonce_field(ZYNITH_SEO_TEXT_PREFIX . 'import_export_action', ZYNITH_SEO_TEXT_PREFIX . 'export_nonce'); ?>
			<h2>Export SEO Data</h2>
			<table class="form-table">
				<tbody>
					<tr>
						<th scope="row">Select SEO Plugin for Export:</th>
						<td>
							<select name="seo_plugin_export" id="seo_plugin_export" class="regular-text styled-select">
								<option value="zynith">Zynith SEO</option>
								<option value="yoast">Yoast SEO</option>
								<option value="rankmath">Rank Math</option>
							</select>
						</td>
					</tr>
					<tr>
						<th scope="row"></th>
						<td>
							<input type="submit" name="export_seo" class="button-primary" value="Export" />
						</td>
					</tr>
				</tbody>
			</table>
		</form>
		</div>
		<?php
	}
	
	public function add_purge_option() {
		$page = filter_input(INPUT_GET, 'page');
		$action_url = admin_url('admin.php?page=' . $page);
		if (!empty($page)) {
			$action_url .= '&action=purge-zynith';
			$action_url = add_query_arg('_wpnonce', wp_create_nonce('purge-zynith'), $action_url);
		}
		echo sprintf(
			'<div class="postbox-warning">
				<h2>%s</h2>
				<p><strong>%s</strong></p>
				<p>%s</p>
				<p>%s</p>
				<a href="%s" id="purge_zynith" class="button-delete" onclick="return confirm(\'%s\')">%s</a>
			</div>',
			__('Purge Zynith SEO Data', ZYNITH_SEO_TEXT_DOMAIN),
			__('Warning: This action is irreversible.', ZYNITH_SEO_TEXT_DOMAIN),
			__('Pressing the button below will permanently delete all data and database tables created by the Zynith SEO plugin.', ZYNITH_SEO_TEXT_DOMAIN),
			__('Please ensure you have backed up your data if necessary before proceeding.', ZYNITH_SEO_TEXT_DOMAIN),
			esc_url($action_url),
			__('Are you sure you want to permanently delete all Zynith SEO data? This action cannot be undone.', ZYNITH_SEO_TEXT_DOMAIN),
			__('Purge Zynith SEO Data', ZYNITH_SEO_TEXT_DOMAIN)
		);
	}

	public function register_settings() {
		$this->register_settings_fields($this->settings);
		$this->export_seo_data();
		$this->import_seo_data();
		$this->sitemap_xml_actions();
		$this->purge_option();
		$this->purge_zynith_meta();
	}
	
	public function register_settings_fields($settings) {
		if (isset($_POST['purge_action'])) {
			$field_id = $_POST['purge_action'];
			update_option($field_id, '');
			add_settings_error($field_id, 'password_purged', 'Password has been removed.', 'updated');
    	}
		$show_on_front = get_option('show_on_front');
		$homepage_id = get_option('page_on_front');
		$blogpage_id = get_option('page_for_posts');
		foreach ($settings as $page => $data) {
			foreach ($data['groups'] as $tab_key => $group) {
				if ($show_on_front === 'page' && !empty($homepage_id) && !empty($blogpage_id) && $group['title'] === __(self::$settings_home_title, ZYNITH_SEO_TEXT_DOMAIN)) continue;
				$section_id = $page . '_' . $tab_key;
				$option_group = $page . '_' . $tab_key;
				$description_callback = function() use ($group) {
					if (isset($group['description'])) echo '<p>' . wp_kses_post($group['description']) . '</p>';
				};
				add_settings_section(
					$section_id,
					$group['title'] ?? '',
					$description_callback,
					$section_id
				);
				foreach ($group['fields'] as $field) {
					register_setting(
						$option_group,
						ZYNITH_SEO_TEXT_PREFIX . $field['id'],
						isset($field['sanitize_callback']) ? $field['sanitize_callback'] : null
					);
					add_settings_field(
						ZYNITH_SEO_TEXT_PREFIX . $field['id'],
						$field['label'] ?? '',
						[$this, 'render_settings_field'],
						$section_id,
						$section_id,
						$field
					);
					if ($field['type'] === 'password') {
						$field_id = ZYNITH_SEO_TEXT_PREFIX . $field['id'];
						$new_password = $_POST[$field_id] ?? '';
						if (!empty($new_password)) update_option($field_id, $new_password);
					}
				}
			}
		}
	}
	
	public function render_settings_field($field) {
		if ($field['id'] === 'custom_sitemap_filename') {
			$regenerate_url = add_query_arg(['page' => 'zynith-seo-settings', 'action' => 'regenerate-sitemap-xml'], admin_url('admin.php'));
			$regenerate_url = wp_nonce_url($regenerate_url, 'regenerate-sitemap-xml');
			$field['description'] = sprintf(__("%sRegenerate Sitemap%s to force it to update now.", ZYNITH_SEO_TEXT_DOMAIN), '<a href="' . $regenerate_url . '" class="button-small"">', '</a>');
		}
		$field_id = ZYNITH_SEO_TEXT_PREFIX . $field['id'];
		$type = $field['type'];
		if ($type === 'checkbox') {
			echo '<label for="' . esc_attr($field_id) . '" class="check-switch"><input name="' . esc_attr($field_id) . '" id="' . esc_attr($field_id) . '" type="checkbox" value="1"' . (!empty(get_option($field_id, $field['default'] ?? '')) ? ' checked' : '') . (($field['disabled'] ?? false) ? ' disabled' : '') . ' /><span class="slider"></span></label>';
		}
		elseif ($type === 'checkbox_group') {
			echo '<fieldset>';
			if (!empty($field['options'])) {
				$default = get_option($field_id, $field['default'] ?? '');
				$default = is_array($default) ? $default : [$default];
				$i = 0;
				foreach ($field['options'] as $key => $option) {
					$checked = in_array($key, $default, true) || (!empty($default[$key]) && $default[$key] === '1');
					echo '<div class="row"><label>' . esc_html($option['label']) . ':</label>
							<label for="' . esc_attr($field_id) . '-' . $i . '" class="check-switch">
								<input name="' . esc_attr($field_id) . '[' . esc_attr($key) . ']" id="' . esc_attr($field_id) . '-' . $i . '" type="checkbox" value="1"' . ($checked ? ' checked' : '') . ' />
								<span class="slider"></span>
							</label></div>';
					$i++;
				}
			}
			echo '</fieldset>';
		}
		elseif ($type === 'email' || $type === 'tel' || $type === 'text') {
			$placeholder = isset($field['placeholder']) ? esc_attr($field['placeholder']) : '';
			$maxlength = isset($field['maxlength']) ? esc_attr($field['maxlength']) : '';
			$required = !empty($field['required']) ? 'required' : '';
			echo '<input name="' . esc_attr($field_id) . '" id="' . esc_attr($field_id) . '" type="' . esc_attr($type) . '" value="' . esc_attr(get_option($field_id, $field['default'] ?? '')) . '" ' . ($placeholder !== '' ? 'placeholder="' . $placeholder . '"' : '') . ' ' . ($maxlength !== '' ? 'maxlength="' . $maxlength . '"' : '') . ' class="regular-text" ' . $required . ' />';
		}
		elseif ($type === 'image') {
			$image_id = get_option($field_id);
			$image_object = '';
			if (!empty($image_id)) $image_object = wp_get_attachment_image($image_id, 'medium');
			echo '<input name="' . esc_attr($field_id) . '" id="' . esc_attr($field_id) . '" type="hidden" value="' . esc_attr($image_id) . '" />
				<style>#' . esc_attr($field_id) . '-preview .image img { margin-bottom: 10px; }</style>
				<div id="' . esc_attr($field_id) . '-preview"><div class="image">' . $image_object . '</div></div>
				<input class="upload-image" type="button" data-id="' . esc_attr($field_id) . '" value="' . esc_attr__('Select Image', ZYNITH_SEO_TEXT_DOMAIN) . '" />
				<input class="remove-image" type="button" data-id="' . esc_attr($field_id) . '" value="' . esc_attr__('Remove Image', ZYNITH_SEO_TEXT_DOMAIN) . '"' . (empty($image_id) ? ' style="display: none;"' : '') . ' />';
		}
		elseif ($type === 'password') {
			$stored_password = get_option($field_id);
			$placeholder = $field['placeholder'] ?? '';
			$purge_button_text = $field['purge_button_text'] ?? __('Purge', ZYNITH_SEO_TEXT_DOMAIN);
			$confirm_dialog = $field['confirm_dialog'] ?? __('Are you sure?', ZYNITH_SEO_TEXT_DOMAIN);
			if (!empty($stored_password)) {
				echo '<p>' . __('A key has been set. If you need to change it, please ', ZYNITH_SEO_TEXT_DOMAIN);
				echo '<button type="button" class="button-small purge-button" onclick="purgePassword();">' . esc_html($purge_button_text) . '</button> first and then set a new one.</p>';
				echo '<input type="hidden" name="' . esc_attr($field_id) . '" id="' . esc_attr($field_id) . '" value="' . esc_attr($stored_password) . '">';
				$field['description'] = '';
				echo '<script>
						function purgePassword() {
							let confirmAction = confirm("' . esc_js($confirm_dialog) . '");
							if (confirmAction) {
								document.getElementById("' . esc_attr($field_id) . '").value = "";
								document.getElementById("submit").click();
							}
						}
					</script>';
			}
			else {
				echo '<input type="password" name="' . esc_attr($field_id) . '" id="' . esc_attr($field_id) . '" value="" placeholder="' . esc_attr($placeholder) . '" class="regular-text">';
			}
		}
		elseif ($type === 'select') {
			echo '<select name="' . esc_attr($field_id) . '" id="' . esc_attr($field_id) . '" class="regular-text styled-select">';
			if (!empty($field['options'])) {
				$default = get_option($field_id, $field['default'] ?? '');
				foreach ($field['options'] as $key => $value) {
					$selected = (string) $key === (string) $default ? ' selected' : '';
					echo '<option value="' . esc_attr($key) . '"' . $selected . '>' . esc_html($value) . '</option>';
				}
			}
			echo '</select>';
		}
		elseif ($type === 'textarea' || $type === 'code') {
			$placeholder = isset($field['placeholder']) ? esc_attr($field['placeholder']) : '';
			$rows = ($type === 'textarea') ? '2' : '16';
			$class = ($type === 'textarea') ? 'small-text' : 'large-text code';
			echo '<textarea name="' . esc_attr($field_id) . '" id="' . esc_attr($field_id) . '"' . ($placeholder !== '' ? ' placeholder="' . $placeholder . '" ' : '') . 'rows="' . $rows . '" class="' . $class . '">' . esc_textarea(get_option($field_id, $field['default'] ?? '')) . '</textarea>';
		}
		if (!empty($field['description'])) echo '<div class="description" id="' . esc_attr($field_id) . '-description">' . $field['description'] . '</div>';
	}
	
	public function export_seo_data() {
		if (isset($_POST['export_seo'], $_POST[ZYNITH_SEO_TEXT_PREFIX . 'export_nonce']) && wp_verify_nonce($_POST[ZYNITH_SEO_TEXT_PREFIX . 'export_nonce'], ZYNITH_SEO_TEXT_PREFIX . 'import_export_action')) {
			$seo_plugin = sanitize_text_field($_POST['seo_plugin_export']);
			$meta_keys = [];
			switch ($seo_plugin) {
				case 'rankmath':
					$meta_keys = ['rank_math_title', 'rank_math_description', 'rank_math_schema', 'rank_math_noindex', 'rank_math_nofollow', 'rank_math_focus_keyword'];
					break;
				case 'yoast':
					$meta_keys = ['_yoast_wpseo_title', '_yoast_wpseo_metadesc', '_yoast_wpseo_schema', '_yoast_wpseo_noindex', '_yoast_wpseo_nofollow', '_yoast_wpseo_focuskw'];
					break;
				case 'zynith':
					$meta_keys = ['_custom_meta_title', '_custom_meta_description', '_custom_schema', '_custom_noindex', '_custom_nofollow', '_custom_target_keyword'];
					break;
			}
			$filename = $seo_plugin . '-seo-data.csv';
			if (ob_get_length()) ob_end_clean();
			header('Content-Type: text/csv');
			header('Content-Disposition: attachment; filename="' . $filename . '"');
        	header('Pragma: no-cache');
        	header('Expires: 0');
			$output = fopen('php://output', 'w');
			fputcsv($output, array_merge(['Post/Page ID'], $meta_keys));
			$args = ['post_type' => ['post', 'page'], 'posts_per_page' => -1, 'post_status' => 'publish'];
			$seo_query = new WP_Query($args);
			if ($seo_query->have_posts()) {
				while ($seo_query->have_posts()) {
					$seo_query->the_post();
					$post_id = get_the_ID();
					$row = array($post_id);
					foreach ($meta_keys as $key) {
						$meta_value = get_post_meta($post_id, $key, true);
						$row[] = $meta_value;
					}
					fputcsv($output, $row);
				}
				wp_reset_postdata();
			}
			fclose($output);
			exit;
			wp_redirect(add_query_arg([
				'page' => 'zynith-seo-settings',
				'tab' => 'import_export',
				'export_status' => 'success'
			], admin_url('admin.php')));
		}
	}
	
	public function import_seo_data() {
		if (isset($_POST['import_seo'], $_POST[ZYNITH_SEO_TEXT_PREFIX . 'import_nonce']) && wp_verify_nonce($_POST[ZYNITH_SEO_TEXT_PREFIX . 'import_nonce'], ZYNITH_SEO_TEXT_PREFIX . 'import_export_action')) {
			$csv_file = $_FILES['csv_file'];
			if ($csv_file['error'] === UPLOAD_ERR_OK && $csv_file['type'] === 'text/csv') {
				$csv_data = array_map('str_getcsv', file($csv_file['tmp_name']));
				array_shift($csv_data);
				foreach ($csv_data as $row) {
					$post_id = (int) $row[0];
					update_post_meta($post_id, '_custom_meta_title', sanitize_text_field($row[1] ?? ''));
					update_post_meta($post_id, '_custom_meta_description', sanitize_text_field($row[2] ?? ''));
					update_post_meta($post_id, '_custom_schema', wp_kses_post($row[3] ?? ''));
					update_post_meta($post_id, '_custom_noindex', sanitize_text_field($row[4] ?? ''));
					update_post_meta($post_id, '_custom_nofollow', sanitize_text_field($row[5] ?? ''));
					update_post_meta($post_id, '_custom_target_keyword', sanitize_text_field($row[6] ?? ''));
				}
			}
			wp_redirect(add_query_arg([
				'page' => 'zynith-seo-settings',
				'tab' => 'import_export',
				'import_status' => 'success'
			], admin_url('admin.php')));
			exit;
		}
	}
		
	public function sitemap_xml_actions() {
		$action = filter_input(INPUT_GET, 'action');
		if ($action === 'regenerate-sitemap-xml') {
			$nonce = filter_input(INPUT_GET, '_wpnonce');
			if (!wp_verify_nonce($nonce, 'regenerate-sitemap-xml')) wp_die(__('Verification failed.', ZYNITH_SEO_TEXT_DOMAIN));
			(new Zynith_SEO_Sitemap())->generate_sitemap();
			$current_page = filter_input(INPUT_GET, 'page');
			$current_tab = filter_input(INPUT_GET, 'tab') ?: 'global_robots';
			$redirect_url = admin_url('admin.php?page=' . urlencode($current_page) . '&tab=' . urlencode($current_tab));
			$redirect_url = add_query_arg('sitemap-generated', 'true', $redirect_url);
			wp_redirect($redirect_url);
			exit;
		}        
	}
	
	public function purge_option() {
		$action = filter_input(INPUT_POST, 'action');
		if ($action !== 'purge-option') return;
		$option = htmlspecialchars(filter_input(INPUT_POST, 'option'));
		$deleted = (bool) delete_option($option);
		$message = !$deleted ? __('We were unable to remove the value.', ZYNITH_SEO_TEXT_DOMAIN) : '';
		wp_send_json(
			[
				'success' => $deleted,
				'message' => $message
			]
		);
	}
	
	public function purge_zynith_meta() {
		global $wpdb;
		$action = filter_input(INPUT_GET, 'action');
		if ($action !== 'purge-zynith') return;
		$nonce = filter_input(INPUT_GET, '_wpnonce');
		if (!wp_verify_nonce($nonce, 'purge-zynith')) wp_die(__('Verification failed.', ZYNITH_SEO_TEXT_DOMAIN));
		$meta_keys = [
			'_custom_meta_title',
			'_custom_meta_description',
			'_custom_noindex',
			'_custom_nofollow',
			'_custom_target_keyword',
			'_custom_schema'
		];
		foreach ($meta_keys as $meta_key) $wpdb->delete($wpdb->postmeta, ['meta_key' => $meta_key]);
		$option_like = $wpdb->esc_like(ZYNITH_SEO_TEXT_PREFIX) . '%';
    	$wpdb->query($wpdb->prepare("DELETE FROM $wpdb->options WHERE option_name LIKE %s", $option_like));
		$page = filter_input(INPUT_GET, 'page');
		$tab = filter_input(INPUT_GET, 'tab');
		$redirect_url = admin_url('admin.php?page=' . $page . '&tab=' . $tab);
		wp_redirect(add_query_arg('zynith-purged', 'true', $redirect_url));
		exit;
	}
	
	public function show_admin_notices() {
		if (!is_admin() || filter_input(INPUT_GET, 'page') !== 'zynith-seo-settings') return;
		$notice = '';
		if (filter_input(INPUT_GET, 'settings-updated') == 'true') {
			$tab = filter_input(INPUT_GET, 'tab') ?: 'business';
			$tab_title = isset($this->settings['zynith-seo-settings']['groups'][$tab]['title']) ? $this->settings['zynith-seo-settings']['groups'][$tab]['title'] : ucfirst($tab);
			$notice = (filter_input(INPUT_GET, 'robots_txt_content')) ? __("robots.txt file updated successfully.", ZYNITH_SEO_TEXT_DOMAIN) : $tab_title . __(' Saved', ZYNITH_SEO_TEXT_DOMAIN);
		}
		elseif (filter_input(INPUT_GET, 'sitemap-generated')) {
			$notice = __("Sitemap was regenerated.", ZYNITH_SEO_TEXT_DOMAIN);
		}
		elseif (filter_input(INPUT_GET, 'zynith-purged')) {
			$notice = __("Zynith SEO meta data purged", ZYNITH_SEO_TEXT_DOMAIN);
		}
		if (!empty($notice)) echo '<div class="notice notice-success"><p>' . esc_html($notice) . '</p></div>';		
		if (filter_input(INPUT_GET, 'tab') == 'global_robots' && !get_option('blog_public')) echo '<div class="notice notice-error"><p>' . sprintf(__('Your website is currently set to discourage search engines from indexing it. To use the robots settings effectively, please %schange the search engine visibility settings%s in the WordPress Reading settings.', ZYNITH_SEO_TEXT_DOMAIN), '<a href="' . esc_url(admin_url('options-reading.php')) . '">', '</a>') . '</p></div>';
		if ($export_status = filter_input(INPUT_GET, 'export_status')) {
			$message = $export_status === 'success' ? 'CSV file exported successfully to downloads folder.' : 'Failed to export CSV file.';
			echo '<div class="notice notice-success is-dismissible"><p>' . esc_html($message) . '</p></div>';
		}
		if ($import_status = filter_input(INPUT_GET, 'import_status')) {
			$message = $import_status === 'success' ? 'CSV file imported successfully.' : 'Failed to import file.';
			echo '<div class="notice notice-success is-dismissible"><p>' . esc_html($message) . '</p></div>';
		}
	}
	
	public static function get_sitemap_path() {
		return self::get_abspath() . self::get_sitemap_name();
	}
	
	public static function get_abspath() {
		$doc_root = filter_input(INPUT_SERVER, 'DOCUMENT_ROOT');
		if (empty($doc_root)) $doc_root = wp_is_writable(ABSPATH) ? ABSPATH : dirname(WP_CONTENT_DIR);
		return htmlspecialchars(rtrim($doc_root, '/')) . '/';
	}
	
	public function enqueue_scripts() {
		$current_version = ZYNITH_SEO_IN_DEVELOPMENT ? time() : ZYNITH_SEO_VERSION;
		$selected_style = get_option(ZYNITH_SEO_TEXT_PREFIX . 'style_select', 'default');
		$plugin_url = plugin_dir_url(ZYNITH_SEO_FILE) . 'assets/' . ZYNITH_SEO_TEXT_DOMAIN;
		if (isset($_GET['page']) && in_array($_GET['page'], self::$plugin_pages)) {
			wp_enqueue_style(ZYNITH_SEO_TEXT_DOMAIN . '-styles', $plugin_url . '-styles.css', [], $current_version);
			wp_enqueue_style(ZYNITH_SEO_TEXT_DOMAIN . '-admin-layout', $plugin_url . '-admin.css', [], $current_version);
			wp_enqueue_style(ZYNITH_SEO_TEXT_DOMAIN . '-admin-appearance', $plugin_url . '-admin-' . $selected_style . '.css', [], $current_version);
			if ($_GET['page'] !== 'schema-templates') wp_register_script(ZYNITH_SEO_TEXT_DOMAIN . '-js', $plugin_url . '-admin.js', ['jquery'], $current_version, true);
		}
		global $pagenow;
		$wp_pages = ['post.php', 'post-new.php', 'edit.php', 'edit-tags.php', 'user-edit.php', 'profile.php', 'term.php'];
		if (in_array($pagenow, $wp_pages)) {
			wp_enqueue_style(ZYNITH_SEO_TEXT_DOMAIN . '-styles', $plugin_url . '-styles.css', [], $current_version);
			wp_enqueue_style(ZYNITH_SEO_TEXT_DOMAIN . '-metabox-layout', $plugin_url . '-metabox.css', [], $current_version);
			wp_enqueue_style(ZYNITH_SEO_TEXT_DOMAIN . '-metabox-appearance', $plugin_url . '-metabox-' . $selected_style . '.css', [], $current_version);
			wp_register_script(ZYNITH_SEO_TEXT_DOMAIN . '-js', $plugin_url . '-metabox.js', [], $current_version, true);
		}
		wp_enqueue_script(ZYNITH_SEO_TEXT_DOMAIN . '-js');
		wp_localize_script(ZYNITH_SEO_TEXT_DOMAIN . '-js', 'custom_image', [
			'title' => __('Select Image', ZYNITH_SEO_TEXT_DOMAIN),
			'button' => __('Select Image', ZYNITH_SEO_TEXT_DOMAIN),
			'og_title' => __('Select Open Graph Image', ZYNITH_SEO_TEXT_DOMAIN),
			'og_button' => __('Select Image', ZYNITH_SEO_TEXT_DOMAIN),
			'logo_title' => __('Select Logo Image', ZYNITH_SEO_TEXT_DOMAIN),
			'logo_button' => __('Select Image', ZYNITH_SEO_TEXT_DOMAIN)
		]);
		wp_localize_script('jquery', 'customAdminScript', ['nonce' => wp_create_nonce('save_meta_data_nonce')]);
		wp_enqueue_media();
	}
	
	public function filter_robots_txt($output, $public) {
		if (!$public) return $output;
		$use_dynamic_robots_txt = (bool) get_option(ZYNITH_SEO_TEXT_PREFIX . 'dynamic_robots_txt', false);
		$custom_content = get_option(ZYNITH_SEO_TEXT_PREFIX . 'robots_txt_content', '');
		if ($use_dynamic_robots_txt != 1 && !empty($custom_content)) return $custom_content;
		return $output;
	}
}