<?php
defined('ABSPATH') or exit;

class Zynith_SEO_Script_Manager {
	
	public static $allowed_html = [
    	'a' => ['href' => true, 'title' => true, 'target' => true, 'rel' => true],
    	'abbr' => ['title' => true, 'class' => true, 'id' => true, 'style' => true],
    	'address' => ['class' => true, 'id' => true, 'style' => true],
    	'bdi' => ['class' => true, 'id' => true],
    	'br' => [],
    	'button' => ['class' => true, 'id' => true, 'style' => true, 'type' => true, 'onclick' => true],
    	'div' => ['class' => true, 'id' => true, 'style' => true],
		'iframe' => ['src' => true, 'height' => true, 'width' => true, 'style' => true, 'allow' => true, 'allowfullscreen' => true],
   		'img' => ['src' => true, 'alt' => true, 'width' => true, 'height' => true, 'class' => true, 'id' => true, 'style' => true],
    	'li' => ['class' => true, 'id' => true, 'style' => true],
    	'meta' => ['name' => true, 'content' => true],
		'noscript' => [],
    	'ol' => ['class' => true, 'id' => true, 'style' => true],
    	'p' => ['class' => true, 'id' => true, 'style' => true],
    	'script' => ['src' => true, 'type' => true, 'async' => true, 'defer' => true],
    	'span' => ['class' => true, 'id' => true, 'style' => true],
    	'style' => ['src' => true, 'type' => true],
    	'ul' => ['class' => true, 'id' => true, 'style' => true],
		'chat-widget' => ['location-id' => true, 'style' => true, 'sub-heading' => true, 'prompt-avatar' => true, 'agency-name' => true, 'agency-website' => true, 'locale' => true, 'send-label' => true, 'primary-color' => true]
	];
	
	public static function init() {
		return new self();
	}
	
	public function __construct() {
		add_action('admin_menu', [$this, 'add_script_manager_page']);
		add_action('admin_init', [$this, 'handle_form_submission']);
		add_action('admin_notices', [$this, 'display_admin_notices']);
		add_action('wp_head', [$this, 'insert_snippets']);
		add_action('wp_footer', [$this, 'insert_snippets_footer']);
		add_action('wp_body_open', [$this, 'insert_snippets_body']);
	}
	
	public function add_script_manager_page() {
        add_submenu_page(
            'zynith-seo-settings',
            __('Zynith SEO Script Manager', ZYNITH_SEO_TEXT_DOMAIN),
            __('Script Manager', ZYNITH_SEO_TEXT_DOMAIN),
            'manage_options',
            'zynith-script-manager',
            [$this, 'display_script_manager_page']
        );
    }
	
	public function display_script_manager_page() {
		global $wpdb;
		$snippet_to_edit = null;
		if (isset($_GET['edit_snippet'])) {
			$snippet_id = intval($_GET['edit_snippet']);
			$table_name = $wpdb->prefix . 'zynith_snippets';
			$snippet_to_edit = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $snippet_id));
			if (!$snippet_to_edit) {
				echo '<div class="error"><p>' . __('Snippet not found.', ZYNITH_SEO_TEXT_DOMAIN) . '</p></div>';
				return;
			}
		}
		$applyToAll = !empty($snippet_to_edit) && ($snippet_to_edit->pages_posts === 'all' || $snippet_to_edit->pages_posts === '');
		?>
		<div class="wrap zynith-wrap">
			<header>
				<img src="<?php echo ZYNITH_SEO_ICON; ?>" alt="Zynith SEO" height="30" width="30">
				<h1><?php echo isset($snippet_to_edit) ? __('Zynith SEO Script Manager: Edit Snippet', ZYNITH_SEO_TEXT_DOMAIN) : __('Zynith SEO Script Manager', ZYNITH_SEO_TEXT_DOMAIN); ?></h1>
			</header>
			<p><?php _e('Manage JavaScript, HTML, and CSS code snippets that can be inserted into specific parts of your site. Useful for adding analytics, ads, or custom CSS/JS.', ZYNITH_SEO_TEXT_DOMAIN); ?></p>
			<div class="postbox">
				<form method="post" action="">
					<?php wp_nonce_field('zynith_snippet_save', 'zynith_snippet_nonce'); ?>
					<input type="hidden" name="zynith_snippet_action" value="<?php echo isset($snippet_to_edit) && $snippet_to_edit ? 'update_snippet' : 'save_snippet'; ?>">
					<?php if (isset($snippet_to_edit) && $snippet_to_edit): ?>
					<input type="hidden" name="snippet_id" value="<?php echo esc_attr($snippet_to_edit->id); ?>">
					<?php endif; ?>
					<div class="row-top">
						<div class="col-small">
							<label for="snippet_name"><?php _e('Snippet Name:', ZYNITH_SEO_TEXT_DOMAIN); ?></label>
							<input type="text" class="regular-text" name="snippet_name" id="snippet_name" value="<?php echo isset($snippet_to_edit) && $snippet_to_edit ? esc_attr($snippet_to_edit->name) : ''; ?>" required>
							<div>
								<label for="snippet_pages_posts"><?php _e('Apply to Pages/Posts:', ZYNITH_SEO_TEXT_DOMAIN); ?><br />
									<span class="description"><?php _e('Hold down ctrl to select multiple.', ZYNITH_SEO_TEXT_DOMAIN); ?></span>
								</label>
								<div>
									<?php
										$selected_pages_posts = isset($snippet_to_edit) ? explode(',', $snippet_to_edit->pages_posts) : [];
										$post_types = get_post_types(['public' => true], 'objects');
									?>
									<select name="snippet_pages_posts[]" id="snippet_pages_posts" multiple style="width: 100%; height: 150px;">
										<?php
											foreach ($post_types as $post_type) {
												$args = [
													'post_type' => $post_type->name,
													'posts_per_page' => -1,
													'post_status' => 'publish',
												];
												$posts = get_posts($args);
												if ($posts) {
													echo '<optgroup label="' . esc_attr($post_type->label) . '">';
													foreach ($posts as $post) {
														$selected = in_array(intval($post->ID), $selected_pages_posts) ? ' selected="selected"' : '';
														echo '<option value="' . esc_attr($post->ID) . '"' . $selected . '>';
														echo esc_html($post->post_title);
														echo '</option>';
													}
													echo '</optgroup>';
												}
											}
										?>
									</select>
								</div>
								<label for="apply_to_all"><?php _e('Apply to All Pages:', ZYNITH_SEO_TEXT_DOMAIN); ?> <input type="checkbox" name="apply_to_all" id="apply_to_all" <?php echo $applyToAll ? 'checked' : ''; ?>/></label>
							</div>
							<label for="snippet_placement"><?php _e('Placement:', ZYNITH_SEO_TEXT_DOMAIN); ?></label>
							<select name="snippet_placement" id="snippet_placement">
								<?php
								$placements = ['head' => 'Head', 'body' => 'Body', 'footer' => 'Footer'];
								foreach ($placements as $value => $label):
									$selected = isset($snippet_to_edit) && $snippet_to_edit && $snippet_to_edit->placement == $value ? ' selected' : '';
								?>
								<option value="<?php echo esc_attr($value); ?>"<?php echo $selected; ?>><?php echo esc_html($label); ?></option>
									<?php endforeach; ?>
							</select>
						</div>
						<div class="col-large">
							<label for="snippet_code">Snippet Code:</label>
							<textarea class="code" id="snippet_code" name="snippet_code" rows="6" required><?php echo isset($snippet_to_edit) && $snippet_to_edit ? stripslashes(esc_textarea($snippet_to_edit->code)) : ''; ?></textarea>
							<div class="inside">
								<p>Allowed Tags and Properties:</p>
								<?php $this->display_allowed_html_tags(); ?>
							</div>
						</div>
					</div>
					<button class="button button-primary" name="submit" id="submit" type="submit"><?php echo isset($snippet_to_edit) && $snippet_to_edit ? 'Update Snippet' : 'Save Snippet'; ?></button>
				</form>
			</div>
		<?php $this->display_snippets(); ?>
		</div>
		<script>
			document.addEventListener('DOMContentLoaded', function () {
					let applyToAllCheckbox = document.getElementById('apply_to_all');
					let selectElement = document.getElementById('snippet_pages_posts');
					let selectedValues = <?php echo json_encode(explode(',', $snippet_to_edit->pages_posts ?? '')); ?>;
					
					function toggleSelectDisabledState(disabled) {
						selectElement.disabled = disabled;
						if (disabled) {
							Array.from(selectElement.options).forEach(function(option) {
								option.selected = true;
							});
						}
						else {
							Array.from(selectElement.options).forEach(function(option) { option.selected = false; });
							selectElement.disabled = false;
						}
					}
					toggleSelectDisabledState(<?php echo $applyToAll ? 'true' : 'false'; ?>);
					
					if (selectElement) {
						selectedValues = selectedValues.map(Number);
						
						Array.from(selectElement.options).forEach(function(option) {
							if (selectedValues.includes(Number(option.value))) {
								option.selected = true;
							}
						});
						
					}
					applyToAllCheckbox.addEventListener("change", function() { toggleSelectDisabledState(this.checked); });
				});
		</script>
		<?php
	}
	
	public function display_allowed_html_tags() {
		$html = '<ul class="quicktags">';
		foreach (self::$allowed_html as $tag => $attributes) {
			$html .= '<li><code>' . $tag . '</code>';
			if (!empty($attributes)) {
				$attr_list = [];
				foreach ($attributes as $attr => $enabled) if ($enabled) $attr_list[] = $attr;
				$html .= ' (' . implode(', ', $attr_list) . ')';
			}
			$html .= '</li>';
		}
		$html .= '</ul>';
		echo $html;
	}
	
	public function display_snippets() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'zynith_snippets';
		$snippets = $wpdb->get_results("SELECT * FROM $table_name ORDER BY name ASC");

		echo '<div class="postbox">';
		echo '<h2>' . __('Existing Snippets', ZYNITH_SEO_TEXT_DOMAIN) . '</h2>';
		if ($snippets) {
			echo '<table class="wp-list-table widefat striped">';
			echo '<thead>';
			echo '<tr>';
			echo '<th class="sortable">' . esc_html__('Date Created', ZYNITH_SEO_TEXT_DOMAIN) . '</th>';
			echo '<th class="sortable">' . esc_html__('Name', ZYNITH_SEO_TEXT_DOMAIN) . '</th>';
			echo '<th class="sortable">' . esc_html__('Placement', ZYNITH_SEO_TEXT_DOMAIN) . '</th>';
			echo '<th class="sortable">' . esc_html__('Code Preview', ZYNITH_SEO_TEXT_DOMAIN) . '</th>';
			echo '<th class="sortable">' . esc_html__('Pages/Posts', ZYNITH_SEO_TEXT_DOMAIN) . '</th>';
			echo '<th>' . esc_html__('Actions', ZYNITH_SEO_TEXT_DOMAIN) . '</th>';
			echo '</tr>';
			echo '</thead>';
			echo '<tbody>';
			foreach ($snippets as $snippet) {
				echo '<tr>';
				echo '<td>' . date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($snippet->added)) . '</td>';
				echo '<td>' . esc_html($snippet->name) . '</td>';
				echo '<td>' . esc_html($snippet->placement) . '</td>';
				$code_preview = substr(stripslashes($snippet->code), 0, 92);
				$code_preview = htmlspecialchars($code_preview);
				if (strlen(stripslashes($snippet->code)) > 92) $code_preview .= '...';
				echo '<td><code title="' . esc_attr(stripslashes($snippet->code)) . '" style="cursor: pointer;">' . $code_preview . '</code></td>';
			 	if (empty($snippet->pages_posts) || $snippet->pages_posts === 'all') {
					$pages_posts_list = __('All Pages', ZYNITH_SEO_TEXT_DOMAIN);
				}
				else {
					$pages_posts_ids = explode(',', $snippet->pages_posts);
					$pages_posts_titles = [];
					foreach ($pages_posts_ids as $id) {
						$post = get_post($id);
						if ($post) $pages_posts_titles[] = $post->post_title . ' (' . $post->post_type . ')';
					}
					$pages_posts_list = implode(', ', $pages_posts_titles);
				}
				echo '<td>' . esc_html($pages_posts_list) . '</td>';
				echo '<td style="display: flex;">';
				echo '<a href="?page=zynith-script-manager&edit_snippet=' . esc_attr($snippet->id) . '" class="button-secondary">' . esc_html__('Edit', ZYNITH_SEO_TEXT_DOMAIN) . '</a>';
				$toggle_action = $snippet->enabled ? 'disable_snippet' : 'enable_snippet';
				$toggle_label = $snippet->enabled ? __('Disable', ZYNITH_SEO_TEXT_DOMAIN) : __('Enable', ZYNITH_SEO_TEXT_DOMAIN);
				echo '<form method="post" action="">';
				echo wp_nonce_field('zynith_snippet_action', 'zynith_snippet_nonce');
				echo '<input type="hidden" name="zynith_snippet_action" value="' . esc_attr($toggle_action) . '">';
				echo '<input type="hidden" name="snippet_id" value="' . esc_attr($snippet->id) . '">';
				echo '<button type="submit" class="button-secondary">' . esc_html($toggle_label) . '</button>';
				echo '</form>';
				echo '<form method="post" action="">';
				echo wp_nonce_field('zynith_snippet_delete', 'zynith_snippet_nonce');
				echo '<input type="hidden" name="zynith_snippet_action" value="delete_snippet">';
				echo '<input type="hidden" name="snippet_id" value="' . esc_attr($snippet->id) . '">';
				echo '<button type="submit" class="button-secondary" onclick="return confirm(\'Are you sure you want to delete this snippet?\');">Delete</button>';
				echo '</form>';
				echo '</td>';
				echo '</tr>';
			}
			echo '</tbody>';
			echo '</table>';
		}
		else {
			echo '<p>No snippets found.</p>';
		}
		echo '</div>';
	}
	
	public function handle_form_submission() {
		if (!current_user_can('manage_options')) return;
		if (isset($_POST['zynith_snippet_action'])) {
			$action = $_POST['zynith_snippet_action'];
			if (($action === 'save_snippet' || $action === 'update_snippet') && (!isset($_POST['zynith_snippet_nonce']) || !wp_verify_nonce($_POST['zynith_snippet_nonce'], 'zynith_snippet_save'))) {
				return;
			}
			elseif (($action === 'enable_snippet' || $action === 'disable_snippet') && (!isset($_POST['zynith_snippet_nonce']) || !wp_verify_nonce($_POST['zynith_snippet_nonce'], 'zynith_snippet_action'))) {
				return;
			}
			elseif ($action === 'delete_snippet' && (!isset($_POST['zynith_snippet_nonce']) || !wp_verify_nonce($_POST['zynith_snippet_nonce'], 'zynith_snippet_delete'))) {
				return;
			}
		}
		$action = isset($_POST['zynith_snippet_action']) ? sanitize_text_field($_POST['zynith_snippet_action']) : '';
		if (!empty($action)) {
			global $wpdb;
			$table_name = $wpdb->prefix . 'zynith_snippets';
			$redirect_url = admin_url('admin.php?page=zynith-script-manager');
			$allowed_html = self::$allowed_html;
			$result = false;
			$snippet_id = isset($_POST['snippet_id']) ? intval($_POST['snippet_id']) : 0;
			$snippet_name = isset($_POST['snippet_name']) ? sanitize_text_field(wp_unslash($_POST['snippet_name'])) : 'Snippet';
			switch ($_POST['zynith_snippet_action']) {
				case 'save_snippet':
				case 'update_snippet':
					$apply_to_all = isset($_POST['apply_to_all']) ? 'all' : '';
					$pages_posts = !$apply_to_all && isset($_POST['snippet_pages_posts']) ? implode(',', array_map('intval', $_POST['snippet_pages_posts'])) : $apply_to_all;
					$placement = sanitize_text_field(wp_unslash($_POST['snippet_placement']));
					$code = wp_kses(wp_unslash($_POST['snippet_code']), $allowed_html);
					$data = [
						'name' => $snippet_name,
						'pages_posts' => $pages_posts,
						'placement' => $placement,
						'code' => $code,
						'enabled' => 1
					];
					
					if ($this->is_duplicate_snippet($data, $snippet_id)) {
						$message = urlencode("A similar snippet already exists and was not saved to prevent duplicates.");
						wp_redirect($redirect_url . "&notice=error&message=" . $message);
						exit;
					}
					
					if ($_POST['zynith_snippet_action'] === 'save_snippet') {
						$result = $wpdb->insert($table_name, $data);
						$action = 'saved';
					}
					else {
						$result = $wpdb->update($table_name, $data, ['id' => $snippet_id]);
						$action = 'updated';
					}
					break;
				case 'delete_snippet':
					$result = $wpdb->delete($table_name, ['id' => $snippet_id]);
					$action = 'deleted';
					break;
				case 'enable_snippet':
				case 'disable_snippet':
					$new_status = ($action === 'enable_snippet') ? 1 : 0;
					$result = $wpdb->update($table_name, ['enabled' => $new_status], ['id' => $snippet_id]);
					$action = $new_status ? 'enabled' : 'disabled';
					break;
			}
			$status = ($result !== false) ? 'success' : 'error';
			$message = ($result !== false) ? stripslashes("{$snippet_name} {$action}.") : stripslashes("Failed to {$action}.");
			wp_redirect($redirect_url . "&notice=$status&message=" . urlencode($message));
			exit;
		}
	}
	
	private function is_duplicate_snippet($data, $snippet_id = 0) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'zynith_snippets';
		$query = $wpdb->prepare(
			"SELECT COUNT(*) FROM $table_name WHERE name = %s AND pages_posts = %s AND placement = %s AND code = %s AND id != %d",
			$data['name'], $data['pages_posts'], $data['placement'], $data['code'], $snippet_id
		);
		return $wpdb->get_var($query) > 0;
	}
	
	public function display_admin_notices() {
		if (isset($_GET['notice']) && isset($_GET['message'])) {
			$class = ($_GET['notice'] === 'success') ? 'notice-success' : 'notice-error';
			$message = urldecode($_GET['message']);
			echo "<div class='notice {$class}'><p>{$message}</p></div>";
		}
	}
		
	public function insert_snippets() {
		global $wpdb, $post;
		$table_name = $wpdb->prefix . 'zynith_snippets';
		$current_post_id = isset($post->ID) ? $post->ID : null;
		$snippets = $wpdb->get_results("SELECT * FROM $table_name WHERE enabled = 1");
		foreach ($snippets as $snippet) {
			if ($snippet->pages_posts === 'all' || in_array($current_post_id, explode(',', $snippet->pages_posts)) || empty($snippet->pages_posts)) {
				if ($snippet->placement === 'head') {
					echo "<!-- Snippet Name: " . esc_html($snippet->name) . " -->\n";
					echo stripslashes($snippet->code);
				}
			}
		}
	}
	
	public function insert_snippets_footer() {
		global $wpdb, $post;
		$table_name = $wpdb->prefix . 'zynith_snippets';
		$current_post_id = isset($post->ID) ? $post->ID : null;
		$snippets = $wpdb->get_results("SELECT * FROM $table_name WHERE enabled = 1");
		foreach ($snippets as $snippet) {
			if ($snippet->pages_posts === 'all' || in_array($current_post_id, explode(',', $snippet->pages_posts)) || empty($snippet->pages_posts)) {
				if ($snippet->placement === 'footer' || $snippet->placement == 'both') {
					echo "<!-- Zynith Snippet: " . esc_html($snippet->name) . " -->\n";
					echo stripslashes($snippet->code);
				}
			}
		}
	}
	
	public function insert_snippets_body() {
		global $wpdb, $post;
		$table_name = $wpdb->prefix . 'zynith_snippets';
		$current_post_id = isset($post->ID) ? $post->ID : null;
		$snippets = $wpdb->get_results("SELECT * FROM $table_name WHERE enabled = 1");
		foreach ($snippets as $snippet) {
			if ($snippet->pages_posts === 'all' || in_array($current_post_id, explode(',', $snippet->pages_posts)) || empty($snippet->pages_posts)) {
				if ($snippet->placement === 'body') {
					echo "<!-- Zynith Snippet: " . esc_html($snippet->name) . " -->\n";
					echo stripslashes($snippet->code);
				}
			}
		}
	}
}