<?php
defined('ABSPATH') or exit;

class Zynith_SEO_Import {

    protected $active_plugins;

    public function __construct() {
        $this->set_active_plugins();
        register_activation_hook(ZYNITH_SEO_FILE, [ $this, 'update_meta_data' ]);
    }

    public static function init() {
        return new self();
    }

    private function set_active_plugins() {
        if (!file_exists(ABSPATH . 'wp-admin/includes/plugin.php')) return [];
		
        include_once(ABSPATH . 'wp-admin/includes/plugin.php');

        $this->active_plugins = [
            'yoast' => is_plugin_active('wordpress-seo/wp-seo.php'),
            'rank_math' => is_plugin_active('seo-by-rank-math/rank-math.php'),
            'aioseo' => is_plugin_active('all-in-one-seo-pack/all_in_one_seo_pack.php'),
            'slim_seo' => is_plugin_active('slim-seo/slim-seo.php'),
            'seopress' => is_plugin_active('wp-seopress/seopress.php'),
            'squirrly' => is_plugin_active('squirrly-seo/squirrly.php'),
            'seo_framework' => is_plugin_active('autodescription/autodescription.php'),
        ];
    }

    private function has_old_seo_plugin() {
        if (!empty($this->active_plugins)) foreach ($this->active_plugins as $plugin => $active) if ($active) return true;
        return false;
    }

    private function get_squirrly_meta() {
        $meta_array = [];

        if (!empty($this->active_plugins['squirrly'])) {
            global $wpdb;

            $squirrly_meta = $wpdb->get_results("SELECT `post`, `seo` FROM `{$wpdb->prefix}qss`");

            if (!empty($squirrly_meta)) {
                foreach ($squirrly_meta as $meta_row) {
                    $squirrly_post = unserialize($meta_row->post);
                    $squirrly_seo = unserialize($meta_row->seo);

                    if (empty($squirrly_post['ID']) || empty($squirrly_seo['doseo'])) continue;
                    
                    $meta_array[$squirrly_post['ID']] = [
                        'title' => $squirrly_seo['title'] ?? '',
                        'description' => $squirrly_seo['description'] ?? '',
                        'schema' => $squirrly_seo['jsonld'] ?? '',
                        'keyword' => $squirrly_seo['keywords'] ?? '',
                        'canonical' => $squirrly_seo['canonical'] ?? '',
                        'noindex' => (bool) ($squirrly_seo['noindex'] ?? false),
                        'nofollow' => (bool) ($squirrly_seo['nofollow'] ?? false),
                    ];
                }
            }
        }

        return $meta_array;
    }

    public function update_meta_data() {
        if (!$this->has_old_seo_plugin()) return;

        $posts = get_posts([
            'post_type' => array_keys(Zynith_SEO_Post_Types::get_post_types()),
            'post_status' => 'publish',
            'numberposts' => -1,
        ]);

        if (empty($posts)) return;

        $squirrly_meta = $this->get_squirrly_meta();
        
        foreach ($posts as $post) $this->update_post($post->ID, $squirrly_meta);
	}

    private function update_post($post_id, $squirrly_meta) {
        // Initialize metadata variables
        $meta_title = '';
        $meta_description = '';
        $schema = '';
        $target_keyword = '';
        $noindex = '';
        $nofollow = '';

        // Get the SEO metadata values based on which plugin is active
        if ($this->active_plugins['yoast']) {
            $meta_title = get_post_meta($post_id, '_yoast_wpseo_title', true);
            $meta_description = get_post_meta($post_id, '_yoast_wpseo_metadesc', true);
            $target_keyword = get_post_meta($post_id, '_yoast_wpseo_focuskw', true);
            $noindex = get_post_meta($post_id, '_yoast_wpseo_meta-robots-noindex', true);
            $nofollow = get_post_meta($post_id, '_yoast_wpseo_meta-robots-nofollow', true);
        }
		elseif ($this->active_plugins['rank_math']) {
            $meta_title = get_post_meta($post_id, 'rank_math_title', true);
            $meta_description = get_post_meta($post_id, 'rank_math_description', true);
            $target_keyword = get_post_meta($post_id, 'rank_math_focus_keyword', true);
            $noindex = get_post_meta($post_id, 'rank_math_robots', true) && in_array('noindex', get_post_meta($post_id, 'rank_math_robots', true)) ? 'yes' : 'no';
            $nofollow = get_post_meta($post_id, 'rank_math_robots', true) && in_array('nofollow', get_post_meta($post_id, 'rank_math_robots', true)) ? 'yes' : 'no';
        }
		elseif ($this->active_plugins['aioseo']) {
            $meta_title = get_post_meta($post_id, '_aioseo_title', true);
            $meta_description = get_post_meta($post_id, '_aioseo_description', true);
            $target_keyword = get_post_meta($post_id, '_aioseo_keywords', true);
            $noindex = get_post_meta($post_id, '_aioseo_robots_noindex', true);
            $nofollow = get_post_meta($post_id, '_aioseo_robots_nofollow', true);
        }
		elseif ($this->active_plugins['slim_seo']) {
            $slim_seo_meta = get_post_meta($post_id, 'slim_seo', true);
            
            if (!empty($slim_seo_meta)) {
                // Slim SEO adds "facebook_image", "twitter_image" and "canonical" to array as well
                $meta_title = $slim_seo_meta['title'] ?? '';
                $meta_description = $slim_seo_meta['description'] ?? '';
                $target_keyword = '';
                $noindex = $slim_seo_meta['noindex'] ? 'yes' : 'no';
                $nofollow = 'no';
            }
        }
		elseif ($this->active_plugins['seopress']) {
            $meta_title = get_post_meta($post_id, '_seopress_titles_title', true);
            $meta_description = get_post_meta($post_id, '_seopress_titles_desc', true);
            $target_keyword = get_post_meta($post_id, '_seopress_analysis_target_kw', true);
            $noindex = get_post_meta($post_id, '_seopress_robots_index', true) == "yes" ? 'no' : 'yes'; // In SEOPress, 'yes' indicates to not allow indexing
            $nofollow = get_post_meta($post_id, '_seopress_robots_follow', true) == "yes" ? 'no' : 'yes'; // In SEOPress, 'yes' indicates to not allow following
        }
		elseif ($this->active_plugins['squirrly']) {
            $post_meta = $squirrly_meta[ $post_id ] ?? null;

            if (!empty($post_meta)) {
                $meta_title = $post_meta['title'];
                $meta_description = $post_meta['description'];
                $schema = $post_meta['schema'];
                $target_keyword = $post_meta['keyword'];
                $canonical = $post_meta['canonical'];
                $noindex = $post_meta['noindex'] ? 'yes' : 'no';
                $nofollow = $post_meta['nofollow'] ? 'yes' : 'no';
            }
        }
		elseif ($this->active_plugins['seo_framework']) {
            // _genesis_canonical_uri, _social_image_url
            $meta_title = get_post_meta($post_id, '_genesis_title', true);
            $meta_description = get_post_meta($post_id, '_genesis_description', true);
            $noindex = get_post_meta($post_id, '_genesis_noindex', true) ? 'yes' : 'no';
            $nofollow = get_post_meta($post_id, '_genesis_nofollow', true) ? 'yes' : 'no';
        }

        // Sanitize the custom fields
        $meta_title = sanitize_text_field($meta_title);
        $meta_description = sanitize_textarea_field($meta_description);
        $schema = sanitize_textarea_field($schema);
        $noindex = ($noindex === true || $noindex === 'yes') ? 'yes' : 'no';
        $nofollow = ($nofollow === true || $nofollow === 'yes') ? 'yes' : 'no';
        $target_keyword = sanitize_text_field($target_keyword);

        // Check if the custom fields exist before updating
        if (empty(get_post_meta($post_id, '_custom_meta_title', true))) update_post_meta($post_id, '_custom_meta_title', $meta_title);
        if (empty(get_post_meta($post_id, '_custom_meta_description', true))) update_post_meta($post_id, '_custom_meta_description', $meta_description);
        if (empty(get_post_meta($post_id, '_custom_schema', true))) update_post_meta($post_id, '_custom_schema', $schema);
        if (empty(get_post_meta($post_id, '_custom_noindex', true))) update_post_meta($post_id, '_custom_noindex', $noindex);
        if (empty(get_post_meta($post_id, '_custom_nofollow', true))) update_post_meta($post_id, '_custom_nofollow', $nofollow);
        if (empty(get_post_meta($post_id, '_custom_target_keyword', true))) update_post_meta($post_id, '_custom_target_keyword', $target_keyword);
    }
}