<?php
defined('ABSPATH') or exit;

class Zynith_SEO_Content {
	
	public $post_types;
    public $taxonomies;
	
	public static function init() {
		return new self();
	}
    
    public function __construct() {
		add_action('template_redirect', [$this, 'set_post_types_and_taxonomies'], 1);
		add_filter('pre_get_document_title', [$this, 'modify_document_title'], 100);
		add_action('wp_head', [$this, 'add_last_modified_meta'], 2);
		add_action('wp_head', [$this, 'output_meta_data'], 100);
		add_action('wp_head', [$this, 'output_canonical_url'], 5);
		add_action('wp_head', [$this, 'output_social_meta'], 6);
		add_action('wp_head', [$this, 'scripts_insert_head'], 7);
		add_action('wp_footer', [$this, 'scripts_insert_body'], 10);
		add_filter('wp_robots', [$this, 'modify_robots_post_meta'], 4);
		add_filter('robots_txt', [$this, 'modify_robots_txt'], 20);
	}
	
    public function set_post_types_and_taxonomies() {
        $this->post_types = array_keys(Zynith_SEO_Post_Types::get_post_types());
        $this->taxonomies = array_keys(Zynith_SEO_Taxonomies::get_taxonomies());
    }
	
	public function modify_document_title($title) {
		$metadata = '';
		$type = '';
		$post = null;
		
		if (is_singular($this->post_types)) {
			$post = get_post();
			$type = 'post';
		}
		elseif ($term = get_queried_object()) {
			if ($this->is_supported_taxonomy($term)) {
				$post = $term;
				$type = 'taxonomy';
			}
		}
		elseif (is_author()) {
			$post = get_userdata(get_query_var('author'));
			$type = 'author';
		}
		elseif (is_home()) {
			$front_page_id = get_option('page_on_front');
        	$posts_page_id = get_option('page_for_posts');
			
			if (is_front_page() && !empty($front_page_id)) {
				$post = get_post($front_page_id);
				$type = 'post';
			}
			elseif (!is_front_page() && !empty($posts_page_id)) {
				$post = get_post($posts_page_id);
				$type = 'post';
			}
			
			if (is_null($post)) {
				$meta_title = get_option('zynith_homepage_meta_title');
				if (!empty($meta_title)) { $title = esc_html($meta_title); }
			}

		}
		
		if (!is_null($post)) {
			$metadata = Zynith_SEO_Meta_Boxes::setObjectMetadata($type, $post);
			$meta_title = $metadata['meta_title'] ?? get_post_meta($post->ID, '_custom_meta_title', true);
			
			if (!empty($meta_title)) {
				$title = Zynith_SEO_Meta_Boxes::replace_seo_variables($meta_title, $metadata);
				$title = esc_html($title);
			}
			else { $title = get_the_title($post->ID) . ' | ' . get_bloginfo('name'); }
		}
        return $title;
    }
		
    private function is_supported_taxonomy($term) {
		if (is_a($term, 'WP_Term') && !empty($term->taxonomy)) return (bool) in_array($term->taxonomy, $this->taxonomies);
		return false;
	}
    
	public function add_last_modified_meta() {
		if (is_single() || is_page()) {
			global $post;
			$modified_time = get_the_modified_time('c', $post->ID);
			echo "<meta property=\"article:modified_time\" content=\"" . esc_attr($modified_time) . "\" />\n";
		}
	}
	
	public function output_meta_data() {
		$post = null;
		$type = ''; 
		$meta_description = '';
		$schema = '';
		
		if (is_singular($this->post_types)) {
			$post = get_post();
			$type = 'post';
			$schema = get_post_meta($post->ID, '_custom_schema', true);
		}
		elseif (is_author()) {
			$post = get_userdata(get_the_author_meta('ID'));
			$type = 'author';
			$schema = get_user_meta($post->ID, '_custom_schema', true);
		}
		elseif ($term = get_queried_object()) {
			if ($this->is_supported_taxonomy($term)) {
				$post = $term;
				$type = 'taxonomy';
				$schema = get_term_meta($post->ID, '_custom_schema', true);
			}
		}
		elseif (is_home()) {
			$type = 'home';
			$post = (is_front_page()) ? null : get_post((int)get_option('page_for_posts'));
			$schema = get_option('zynith_homepage_schema_markup');
		}
		
		$metadata = Zynith_SEO_Meta_Boxes::setObjectMetadata($type, $post);
		$meta_description = $metadata['meta_description'] ?? '';
		
		if (empty($meta_description) && !is_null($post)) $meta_description = html_entity_decode(get_the_excerpt($post->ID), ENT_QUOTES, 'UTF-8');
		if (!empty($meta_description)) $meta_description = Zynith_SEO_Meta_Boxes::replace_seo_variables($meta_description, $metadata);
		echo '<meta name="description" content="' . esc_attr($meta_description) . '">' . "\n";
		if (!empty($schema)) {
			$schema = Zynith_SEO_Meta_Boxes::replace_seo_variables($schema, $metadata);
			if (!empty($schema)) echo '<script type="application/ld+json">' . $schema . '</script>' . "\n";
		}
    }
	
    public function get_canonical_url() {
		$canonical_url = null;

		if (is_front_page()) {
			$canonical_url = home_url('/');
		}
		elseif (is_home() && !is_front_page()) {
			$blog_home_id = get_option('page_for_posts');
			if (!empty($blog_home_id)) $canonical_url = get_permalink($blog_home_id);
		}
		elseif (is_singular($this->post_types)) {
			$post = get_post();
			$canonical_url = get_permalink($post->ID);
		}
		elseif (is_category() || is_tag() || is_tax()) {
			$term = get_queried_object();
			if (!empty($term->taxonomy)) $canonical_url = get_term_link($term->term_id, $term->taxonomy);
		}
		elseif (is_archive()) {
			if (is_date()) {
				if (is_day()) {
					$canonical_url = get_day_link(get_query_var('year'), get_query_var('monthnum'), get_query_var('day'));
				}
				elseif (is_month()) {
					$canonical_url = get_month_link(get_query_var('year'), get_query_var('monthnum'));
				}
				elseif (is_year()) {
					$canonical_url = get_year_link(get_query_var('year'));
				}
			}
			elseif (is_author()) {
				$canonical_url = get_author_posts_url(get_query_var('author'), get_query_var('author_name'));
			}
			elseif (is_post_type_archive()) {
				$canonical_url = get_post_type_archive_link(get_query_var('post_type'));
			}
		}
		return trailingslashit($canonical_url);
	}
	
   	public function output_canonical_url() {
		$remove_canonical_tag = get_option('zynith_remove_canonical', false);
		if (!$remove_canonical_tag) {
			$canonical_url = $this->get_canonical_url();
			if (!empty($canonical_url)) echo '<link rel="canonical" href="' . esc_url($canonical_url) . '">' . "\n";
		}
	}
	
    public function scripts_insert_head() {
		echo get_option('zynith_gtm_header_script');
	}
	
    public function scripts_insert_body() {
		echo get_option('zynith_gtm_body_script');
	}
	
    public function output_social_meta() {
		$default_image = get_option('zynith_default_og_image', null);
		$canonical_url = $this->get_canonical_url();
		$og_type = 'website';
		$meta_title = null;
		$meta_description = null;
		$image_url = null;
		$type = '';
		$post = null;
		$title = '';
		$post_excerpt = '';
		$metadata = [];
		
		if (is_singular($this->post_types)) {
			$post = get_post();
			$type = 'post';
			$image_url = get_post_meta($post->ID, '_custom_meta_og_image', true);
			$og_type = 'article';
			
			if (empty($image_url)) {
				$featured_image_id = get_post_thumbnail_id($post->ID);
				if ($featured_image_id) $image_url = wp_get_attachment_image_url($featured_image_id, 'full');
			}

			// If still no image, try to find the first image in content
			if (empty($image_url)) {
				$content = $post->post_content;
				preg_match('/<img.+src=[\'"]([^\'"]+)[\'"].*>/i', $content, $matches);
				if (!empty($matches[1])) $image_url = $matches[1];
			}
		}
		elseif ($term = get_queried_object()) {
			if ($this->is_supported_taxonomy($term)) {
				$post = $term;
				$type = 'taxonomy';	
			}			
		}
		elseif (is_author()) {
			$post = get_userdata(get_query_var('author'));
			$type = 'author';
			$user_id = get_the_author_meta('ID');
			$image_url = get_user_meta($user_id, '_custom_meta_og_image', true);
			$meta_title = get_user_meta($user_id, '_custom_meta_title', true);
			$meta_description = get_user_meta($user_id, '_custom_meta_description', true);
			$title = $post->display_name;
		}
		elseif (is_home()) {
			if (is_front_page()) {
				// Dynamic home page: Latest Posts
				$type = 'home';
				$post = (is_front_page()) ? null : get_post((int)get_option('page_for_posts'));
				$meta_title = get_option('zynith_homepage_meta_title');
				$meta_description = get_option('zynith_homepage_meta_description');
				$title = get_the_title();
			}
			else {
				// Static home page
				$home_page_id = (int)get_option('page_for_posts');
				$post = get_post($home_page_id);
				$type = 'post';
				$image_url = get_post_meta($home_page_id, '_custom_meta_og_image', true);
				$meta_title = get_post_meta($home_page_id, '_custom_meta_title', true);
				$meta_description = get_post_meta($home_page_id, '_custom_meta_description', true);
				$title = get_the_title($post->ID);
				if (empty($meta_title)) $meta_title = get_option('zynith_homepage_meta_title');
				if (empty($meta_description)) $meta_description = get_option('zynith_homepage_meta_description');
			}
		}
		
		if (!is_null($post)) $metadata = Zynith_SEO_Meta_Boxes::setObjectMetadata($type, $post);
		
		if (empty($$meta_title)) $meta_title = $metadata['meta_title'] ?? '';
		if (empty($meta_description)) $meta_description = $metadata['meta_description'] ?? '';
		
		if (!empty($meta_title)) {
			$meta_title = Zynith_SEO_Meta_Boxes::replace_seo_variables($meta_title, $metadata);
			echo '<meta property="og:title" content="' . esc_attr($meta_title) . '">' . "\n";
			echo '<meta property="twitter:title" content="' . esc_attr($meta_title) . '">' . "\n";
		}
		if (!empty($meta_description)) {
			$meta_description = Zynith_SEO_Meta_Boxes::replace_seo_variables($meta_description, $metadata);
			echo '<meta property="og:description" content="' . esc_attr($meta_description) . '">' . "\n";
			echo '<meta property="twitter:description" content="' . esc_attr($meta_description) . '">' . "\n";
		}
		
		echo '<meta property="og:type" content="' . esc_attr($og_type) . '" />';
		
		if (empty($image_url) && !empty($default_image)) {
			if (is_numeric($default_image)) {
				$image_url = wp_get_attachment_image_url($default_image, 'full');
			}
			elseif (is_string($default_image)) {
				$image_url = $default_image;
			}
		}
		else {
			$image_url = $metadata['meta_og_image'] ?? '';
		}
		
		if (!empty($image_url)) {
			$image_info = pathinfo($image_url);
			$image_title = preg_replace('%\s*[-_\s]+\s*%', ' ', $image_info['filename']);
			$image_alt_text = ucwords(mb_strtolower(trim($image_title)));
			echo '<meta property="og:image" content="' . esc_url($image_url) . '">' . "\n";
			echo '<meta property="og:image:alt" content="' . esc_attr($image_alt_text) . '">' . "\n";
			echo '<meta property="twitter:card" content="summary_large_image">' . "\n";
			echo '<meta property="twitter:image" content="' . esc_url($image_url) . '">' . "\n";
			echo '<meta property="twitter:image:alt" content="' . esc_attr($image_alt_text) . '">' . "\n";
		}
		
		if (!empty($canonical_url)) echo '<meta property="og:url" content="' . esc_url($canonical_url) . '">' . "\n";
		echo '<meta property="og:locale" content="' . get_locale() . '">' . "\n";
		echo '<meta property="og:site_name" content="' . get_bloginfo('name') . '">' . "\n";
	}
	
	public function modify_robots_post_meta($directives) {
        $blog_public = (bool) get_option('blog_public');
        if (!$blog_public) return $directives;

		if (is_singular($this->post_types)) {
			$post = get_post();
            if (empty($post)) return $directives;
        
            $noindex = get_post_meta($post->ID, '_custom_noindex', true);
            $nofollow = get_post_meta($post->ID, '_custom_nofollow', true);
            $noindex_post_types = get_option('zynith_noindex_post_types');
            $nofollow_post_types = get_option('zynith_nofollow_post_types');
        
            // Now, we established that we're in a public post, and want to do our own robots meta. Unset whatever parameters you are modifying
            unset($directives['noindex'], $directives['nofollow']);

            // If noindex is set, add it to the directives array
            if (!empty($noindex_post_types) && in_array($post->post_type, array_keys($noindex_post_types))) $directives['noindex'] = true;
            if ($noindex === 'yes') $directives['noindex'] = true;

            // If nofollow is set, add it to the directives array
            if (!empty($nofollow_post_types) && in_array($post->post_type, array_keys($nofollow_post_types))) $directives['nofollow'] = true;
            if ($nofollow === 'yes') $directives['nofollow'] = true;
            if (ZYNITH_SEO_IS_WOOCOMMERCE) {
				if (is_cart() || is_checkout() || is_account_page() || is_view_order_page() || is_checkout_pay_page() || is_edit_account_page() || is_order_received_page() || is_add_payment_method_page() || is_lost_password_page()) {
                	$directives['noindex'] = true;
            	}
			}
        }
		elseif (($term = get_queried_object()) || is_author() || is_date() || is_search()) {
			if ($this->is_supported_taxonomy($term)) {
				$noindex = get_term_meta($term->term_id, '_custom_noindex', true);
                $nofollow = get_term_meta($term->term_id, '_custom_nofollow', true);
                $noindex_taxonomies = get_option('zynith_noindex_taxonomies');
                $nofollow_taxonomies = get_option('zynith_nofollow_taxonomies');
                unset($directives['noindex'], $directives['nofollow']);
                if ($noindex === 'yes' || (!empty($noindex_taxonomies) && in_array($term->taxonomy, array_keys($noindex_taxonomies)))) $directives['noindex'] = true;
                if ($nofollow === 'yes' || (!empty($nofollow_taxonomies) && in_array($term->taxonomy, array_keys($nofollow_taxonomies)))) $directives['nofollow'] = true;
            }
			elseif (is_author() == 1 || is_date() || is_search()) {
				$noindex = false;
				$nofollow = false;
				$type = '';
				if (is_author()) {
					$user_id = get_the_author_meta('ID');
					$noindex = get_user_meta($user_id, '_custom_noindex', true);
					$nofollow = get_user_meta($user_id, '_custom_nofollow', true);
					$type = 'author';
				}
				if (is_date()) $type = 'date';
				if (is_search()) $type = 'search';
				$noindex_archives = get_option('zynith_noindex_archives');
				$nofollow_archives = get_option('zynith_nofollow_archives');
				if ($noindex === 'yes' || (!empty($noindex_archives) && in_array($type, array_keys($noindex_archives)))) {
					$directives['noindex'] = true;
					unset($directives['index']);
				}
				else {
    				unset($directives['noindex']);
				}
				if ($nofollow === 'yes' || (!empty($nofollow_archives) && in_array($type, array_keys($nofollow_archives)))) {
					$directives['nofollow'] = true;
					unset($directives['follow']);
				}
				else {
    				unset($directives['nofollow']);
				}
			}
        }
        if (!empty($directives['noindex']) || !empty($directives['nofollow'])) unset($directives['max-image-preview']);
        if (empty($directives['noindex']) || empty($directives['nofollow'])) {
            $additional_directives = [];
            if (empty($directives['noindex'])) $additional_directives['index'] = true;
            if (empty($directives['nofollow'])) $additional_directives['follow'] = true;
            $directives = array_merge($additional_directives, $directives);
        }
        return $directives;
    }
	
	public function modify_robots_txt($output) {
        if (!get_option('blog_public')) return "User-agent: *\nDisallow: /";
		
        $use_dynamic_robots_txt = (bool) get_option('zynith_dynamic_robots_txt', false);
        if (!$use_dynamic_robots_txt) return $output;
        
        $sitemap_url = home_url(Zynith_SEO_Settings::get_sitemap_name());
        $allow_pdf_index = (bool) get_option('zynith_robots_txt_allow_pdf_index', true);
        $remove_wp_default = (bool) get_option('zynith_robots_txt_remove_wp_default', true);
        $output .= "\n";

        // If WP default is disabled, start with a blank output
        if ($remove_wp_default) $output = "User-agent: *\nDisallow: /wp-json/\n\n";
        $output .= "User-agent: *\n";
        if (!$allow_pdf_index) $output .= "Disallow: *.pdf\n";

        // Cloudflare email protection
		$output .= "Disallow: /cdn-cgi/l/email-protection/\n";
        
        // Cloudflare Challenges
		$output .= "Disallow: /cdn-cgi/bm/cv/\n";
		$output .= "Disallow: /cdn-cgi/challenge-platform/\n\n";

        // Sitemap XML
		$output .= "Sitemap: {$sitemap_url}\n\n";

        // Block harmful bots
        $output .= "User-agent: Nuclei\n";
        $output .= "User-agent: WikiDo\n";
        $output .= "User-agent: Riddler\n";
        $output .= "User-agent: PetalBot\n";
        $output .= "User-agent: Zoominfobot\n";
        $output .= "User-agent: Go-http-client\n";
        $output .= "User-agent: Node/simplecrawler\n";
        $output .= "User-agent: CazoodleBot\n";
        $output .= "User-agent: dotbot/1.0\n";
        $output .= "User-agent: Gigabot\n";
        $output .= "User-agent: Barkrowler\n";
        $output .= "User-agent: BLEXBot\n";
        $output .= "User-agent: magpie-crawler\n";
        $output .= "Disallow: /";

        return $output;
    }
}