<?php
defined('ABSPATH') or exit;

class Zynith_SEO_404_Monitor {
	
	public static function init() {
		return new self();
	}
	
	public function __construct() {
        register_deactivation_hook(__FILE__, [$this, 'delete_404_log_table']);
		if (get_option(ZYNITH_SEO_TEXT_PREFIX . 'enable_404_monitor', true)) {
            add_action('admin_menu', [$this, 'add_404_monitor_page']);
            add_action('template_redirect', [$this, 'log_404_errors']);
		    add_action('admin_init', [$this, 'handle_delete_404_logs']);
        }
	}
		
    public function delete_404_log_table() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'zynith_404_log';
        $wpdb->query("DROP TABLE IF EXISTS $table_name");
    }

    public function add_404_monitor_page() {
		add_submenu_page(
			'zynith-seo-settings',
			__('404 Monitor', ZYNITH_SEO_TEXT_DOMAIN),
			__('404 Monitor', ZYNITH_SEO_TEXT_DOMAIN),
			'manage_options',
			'404-monitor-settings',
			[$this, 'display_404_monitor_settings']
		);
    }

    public function display_404_monitor_settings() {
    	if (!current_user_can('manage_options')) wp_die('You do not have permission to access this page.');
        global $wpdb;
        $table_name = $wpdb->prefix . 'zynith_404_log';
		if (isset($_POST['delete_all_404_logs'])) {
			$wpdb->query("TRUNCATE TABLE $table_name");
			echo '<div class="notice notice-success"><p>All 404 logs have been deleted.</p></div>';
		}
        $per_page = 50;
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $offset = ($current_page - 1) * $per_page;
        $total_logs = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
        $total_pages = ceil($total_logs / $per_page);
        $logs = $wpdb->get_results("SELECT * FROM $table_name ORDER BY timestamp DESC LIMIT $per_page OFFSET $offset", ARRAY_A);
        echo '<div class="wrap zynith-wrap">
			<header>
				<img src="' . ZYNITH_SEO_ICON . '" alt="Zynith SEO" height="30" width="30">
				<h1>' . __('404 Monitor', ZYNITH_SEO_TEXT_DOMAIN) . '</h1>
			</header>';
        if (empty($logs)) {
			echo '<p>' . __('No 404 errors have been logged yet.', ZYNITH_SEO_TEXT_DOMAIN) . '</p>';
		}
		else {
			echo '<p>' . __('Monitor 404 hits to identify and address potentially broken links, enhancing navigation and maintaining a robust site structure.', ZYNITH_SEO_TEXT_DOMAIN) . '</p>';
			echo '<div class="postbox">';
			echo '<form id="zynith-form-404" method="post" action="">';
			echo '<table class="wp-list-table widefat striped" id="zynith-table-404">';
			echo '<thead><tr><th><input type="checkbox" id="select-all" /><label for="select-all">' . __('Select All', ZYNITH_SEO_TEXT_DOMAIN) . '</label></th>
			<th class="sortable">' . __('404 URL', ZYNITH_SEO_TEXT_DOMAIN) . '</th><th class="sortable">' . __('User Agent', ZYNITH_SEO_TEXT_DOMAIN) . '</th>
			<th class="sortable">' . __('Timestamp', ZYNITH_SEO_TEXT_DOMAIN) . '</th></tr></thead>';
			echo '<tbody>';
			foreach ($logs as $log) {
				echo '<tr>';
				echo '<td><input type="checkbox" class="log-checkbox" name="logs[]" value="' . $log['id'] . '"></td>';
				echo '<td>' . esc_html($log['url']) . '</td>';
				echo '<td>' . esc_html($log['user_agent']) . '</td>';
				echo '<td>' . esc_html($log['timestamp']) . '</td>';
				echo '</tr>';
			}
			echo '</tbody>';
			echo '</table>';
            $page_links = paginate_links([
                'base' => add_query_arg('paged', '%#%'),
                'format' => '',
                'prev_text' => '&laquo; Previous',
                'next_text' => 'Next &raquo;',
                'total' => $total_pages,
                'current' => $current_page,
            ]);
            if ($page_links) echo '<div class="tablenav"><div class="tablenav-pages">' . $page_links . '</div></div>';
            echo '<p><input type="submit" class="button button-primary" name="delete_404_logs" value="Delete Selected 404 Logs" style="margin: 0 11px 0 0;">';
			echo '<input type="submit" class="button button-primary" name="delete_all_404_logs" value="Delete All 404 Logs"></p>';
            echo '</form></div>';
        }
        echo '</div>';
    }

    public function log_404_errors() {
        if (is_404()) {
            global $wpdb;
            $table_name = $wpdb->prefix . 'zynith_404_log';
            $log = [
                'url' => esc_url($_SERVER['REQUEST_URI']),
                'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? esc_html($_SERVER['HTTP_USER_AGENT']) : 'Unknown User Agent'
            ];
            $wpdb->insert($table_name, $log);
        }
    }
	
    public function handle_delete_404_logs() {
        if (isset($_POST['delete_404_logs'])) {
            global $wpdb;
            $table_name = $wpdb->prefix . 'zynith_404_log';
            $logs_to_delete = isset($_POST['logs']) ? $_POST['logs'] : [];
            if (!empty($logs_to_delete)) {
                $ids = implode(',', array_map('intval', $logs_to_delete));
                $wpdb->query("DELETE FROM $table_name WHERE id IN ($ids)");
            }
        }
    }
}