<?php
/**
 * Zynith SEO - 404 Monitor
 *
 * Logs 404 requests into a custom table and provides an admin UI to review/manage logs.
 * - Uses tolerant insert (checks columns before including them) to avoid "Unknown column" errors.
 * - Uses dbDelta() on every ensure to upgrade schema safely (adds missing columns like `referrer`).
 * - Positive enable flag: zynith_seo_404_monitor_enabled (1 = on, 0 = off).
 */

defined('ABSPATH') || exit;

//
// -----------------------------------------------------------------------------
// Constants & Helpers
// -----------------------------------------------------------------------------
/**
 * Return the fully-qualified table name.
 *
 * @global wpdb $wpdb
 * @return string
 */
function zynith_seo_404_table() {
    global $wpdb;
    return $wpdb->prefix . 'zynith_404_log';
}

/**
 * Desired schema for dbDelta (id, url, referrer, user_agent, logged_at)
 * NOTE: Using "logged_at" instead of "timestamp" to avoid keyword confusion.
 *
 * @return string SQL
 */
function zynith_seo_404_schema_sql() {
    $table           = zynith_seo_404_table();
    $charset_collate = $GLOBALS['wpdb']->get_charset_collate();

    // dbDelta requires exact formatting rules (PRIMARY KEY on its own line, etc.)
    return "CREATE TABLE {$table} (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        url text NOT NULL,
        referrer text NULL,
        user_agent text NOT NULL,
        logged_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY  (id)
    ) {$charset_collate};";
}

/**
 * Ensure table exists and is up to date. Safe to call repeatedly.
 */
function zynith_seo_ensure_404_table() {
    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta( zynith_seo_404_schema_sql() );
}

/**
 * Get column list for the 404 table (used for tolerant inserts).
 *
 * @return string[] Column names, lowercase
 */
function zynith_seo_404_columns() {
    global $wpdb;
    $table = zynith_seo_404_table();

    // Quick existence check first (no errors in logs if missing).
    $exists = $wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table));
    if ($exists !== $table) return [];
    
    // Suppress errors briefly to avoid log noise on edge cases.
    $old = $wpdb->suppress_errors(true);
    $cols = $wpdb->get_col("DESC {$table}", 0);
    $wpdb->suppress_errors($old);
    if (!is_array($cols)) return [];
    
    return array_map('strtolower', $cols);
}

/**
 * Is monitor enabled?
 * Default ON (1).
 *
 * @return bool
 */
function zynith_seo_404_monitor_enabled() {
    $val = get_option( 'zynith_seo_404_monitor_enabled', '1' );
    return (string) $val === '1';
}

//
// -----------------------------------------------------------------------------
// Activation / Early ensure
// -----------------------------------------------------------------------------
/**
 * Ensure schema on plugin load (keeps old installs upgraded without manual click).
 */
add_action( 'plugins_loaded', 'zynith_seo_ensure_404_table' );

//
// -----------------------------------------------------------------------------
// Admin Menu & Page
// -----------------------------------------------------------------------------
add_action( 'admin_menu', 'zynith_seo_404_monitor_menu' );
function zynith_seo_404_monitor_menu() {
    add_submenu_page(
        'zynith_seo_dashboard',
        __( '404 Monitor', 'zynith-seo' ),
        __( '404 Monitor', 'zynith-seo' ),
        'manage_options',
        'zynith-seo-404-monitor',
        'zynith_seo_render_404_monitor_page'
    );
}

/**
 * Admin page renderer.
 */
function zynith_seo_render_404_monitor_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    // Ensure table exists/upgrades (in case admin lands here before plugins_loaded).
    zynith_seo_ensure_404_table();

    global $wpdb;
    $table = zynith_seo_404_table();

    // Handle actions.
    if ( isset( $_POST['zynith_seo_404_action'] ) ) {
        check_admin_referer( 'zynith_seo_404_actions' );

        $action = sanitize_text_field( $_POST['zynith_seo_404_action'] );

        if ( 'delete_row' === $action && isset( $_POST['row_id'] ) ) {
            $rid = absint( $_POST['row_id'] );
            $wpdb->delete( $table, array( 'id' => $rid ), array( '%d' ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        }

        if ( 'toggle_enable' === $action && isset( $_POST['enabled'] ) ) {
            update_option( 'zynith_seo_404_monitor_enabled', $_POST['enabled'] === '1' ? '1' : '0' );
        }

        if ( 'purge_days' === $action && isset( $_POST['days'] ) ) {
            $days = max( 1, absint( $_POST['days'] ) );
            // Prepared delete: compare datetime.
            $cutoff = gmdate( 'Y-m-d H:i:s', time() - ( DAY_IN_SECONDS * $days ) );
            $wpdb->query( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.NotPrepared
                $wpdb->prepare( "DELETE FROM {$table} WHERE logged_at < %s", $cutoff )
            );
        }

        if ( 'truncate' === $action ) {
            // TRUNCATE can't be prepared with identifiers; use safe table function & guard with nonce/caps.
            $wpdb->query( "TRUNCATE TABLE {$table}" ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
        }
    }

    // Fetch latest 100 rows.
    $limit   = 100;
    $results = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->prepare( "SELECT id, url, referrer, user_agent, logged_at FROM {$table} ORDER BY logged_at DESC LIMIT %d", $limit )
    );

    $enabled = zynith_seo_404_monitor_enabled();
    ?>
    <div class="wrap">
        <h1><?php esc_html_e( 'Zynith SEO 404 Monitor Logs', 'zynith-seo' ); ?></h1>

        <form method="post" style="margin:12px 0;">
            <?php wp_nonce_field( 'zynith_seo_404_actions' ); ?>
            <input type="hidden" name="zynith_seo_404_action" value="toggle_enable" />
            <label>
                <input type="checkbox" name="enabled" value="1" <?php checked( $enabled ); ?> />
                <?php esc_html_e( 'Enable 404 Monitor', 'zynith-seo' ); ?>
            </label>
            <?php submit_button( __( 'Save', 'zynith-seo' ), 'primary small', '', false ); ?>
        </form>

        <form method="post" style="margin:12px 0;display:flex;gap:8px;align-items:center;">
            <?php wp_nonce_field( 'zynith_seo_404_actions' ); ?>
            <input type="hidden" name="zynith_seo_404_action" value="purge_days" />
            <label>
                <?php esc_html_e( 'Purge entries older than', 'zynith-seo' ); ?>
                <input type="number" name="days" min="1" value="30" style="width:80px;" />
                <?php esc_html_e( 'days', 'zynith-seo' ); ?>
            </label>
            <?php submit_button( __( 'Purge', 'zynith-seo' ), 'secondary small', '', false ); ?>
        </form>

        <form method="post" onsubmit="return confirm('Are you sure you want to clear ALL 404 logs?');" style="margin:12px 0;">
            <?php wp_nonce_field( 'zynith_seo_404_actions' ); ?>
            <input type="hidden" name="zynith_seo_404_action" value="truncate" />
            <?php submit_button( __( 'Clear All Logs', 'zynith-seo' ), 'delete small', '', false ); ?>
        </form>

        <table class="widefat fixed striped">
            <thead>
            <tr>
                <th><?php esc_html_e( 'URL', 'zynith-seo' ); ?></th>
                <th><?php esc_html_e( 'Referrer', 'zynith-seo' ); ?></th>
                <th><?php esc_html_e( 'User Agent', 'zynith-seo' ); ?></th>
                <th><?php esc_html_e( 'Logged At', 'zynith-seo' ); ?></th>
                <th><?php esc_html_e( 'Actions', 'zynith-seo' ); ?></th>
            </tr>
            </thead>
            <tbody>
            <?php if ( $results ) : ?>
                <?php foreach ( $results as $row ) : ?>
                    <tr>
                        <td style="word-break:break-all;"><?php echo esc_url( $row->url ); ?></td>
                        <td style="word-break:break-all;"><?php echo $row->referrer ? esc_url( $row->referrer ) : esc_html__( 'Direct', 'zynith-seo' ); ?></td>
                        <td style="word-break:break-all;">
                            <?php
                            $ref = (string) $row->referrer;
                            if ($ref === '' || strtolower($ref) === 'direct') {
                                echo esc_html__('Direct', 'zynith-seo');
                            }
                            elseif (filter_var($ref, FILTER_VALIDATE_URL)) {
                                echo esc_url( $ref );
                            }
                            else {
                                echo esc_html( $ref );
                            }
                            ?>
                        </td>
                        <td style="word-break:break-all;"><?php echo esc_html( $row->user_agent ); ?></td>
                        <td><?php echo esc_html( $row->logged_at ); ?></td>
                        <td>
                            <form method="post" style="display:inline;">
                                <?php wp_nonce_field( 'zynith_seo_404_actions' ); ?>
                                <input type="hidden" name="zynith_seo_404_action" value="delete_row" />
                                <input type="hidden" name="row_id" value="<?php echo esc_attr( $row->id ); ?>" />
                                <?php submit_button( __( 'Delete', 'zynith-seo' ), 'secondary small', '', false ); ?>
                            </form>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php else : ?>
                <tr><td colspan="5"><?php esc_html_e( 'No 404 errors logged yet.', 'zynith-seo' ); ?></td></tr>
            <?php endif; ?>
            </tbody>
        </table>
    </div>
    <?php
}

//
// -----------------------------------------------------------------------------
// Logging (tolerant insert)
// -----------------------------------------------------------------------------
/**
 * Logs only when:
 *  - monitor enabled, and
 *  - current request resolves to a 404.
 */
add_action( 'template_redirect', 'zynith_seo_log_404_errors', 20 );
function zynith_seo_log_404_errors() {
    if (!zynith_seo_404_monitor_enabled() || !is_404()) return;
    
    // Ensure table/scheme exists (first request after update is safe).
    zynith_seo_ensure_404_table();

    // Reduce volume from bots & assets
    $path = isset($_SERVER['REQUEST_URI']) ? parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH) : '';
    $ext  = strtolower(pathinfo($path ?? '', PATHINFO_EXTENSION));
    $skip_ext = array('jpg','jpeg','png','webp','gif','svg','ico','css','js','woff','woff2','ttf','eot','map');
    $skip_prefixes = array('/.well-known/', '/ads.txt', '/app-ads.txt', '/sellers.json');
    foreach ($skip_prefixes as $p) if (strpos($path, $p) === 0) return;
    if ($ext && in_array($ext, $skip_ext, true)) return;
    
    global $wpdb;
    $table = zynith_seo_404_table();

    // Gather fields safely.
    $url        = isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
    $referrer   = ! empty( $_SERVER['HTTP_REFERER'] ) ? esc_url_raw( wp_unslash( $_SERVER['HTTP_REFERER'] ) ) : 'Direct';
    $user_agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : 'Unknown';
    $logged_at  = current_time( 'mysql' ); // WP-local time string.

    // Tolerant insert: only include columns that actually exist.
    $cols = zynith_seo_404_columns();

    $data = array();
    $fmt  = array();

    if ( in_array( 'url', $cols, true ) ) {
        $data['url'] = $url;
        $fmt[]       = '%s';
    }
    if ( in_array( 'referrer', $cols, true ) ) {
        $data['referrer'] = $referrer;
        $fmt[]            = '%s';
    }
    if ( in_array( 'user_agent', $cols, true ) ) {
        $data['user_agent'] = $user_agent;
        $fmt[]              = '%s';
    }
    // Support either 'logged_at' (preferred) or legacy 'timestamp'.
    if ( in_array( 'logged_at', $cols, true ) ) {
        $data['logged_at'] = $logged_at;
        $fmt[]             = '%s';
    } elseif ( in_array( 'timestamp', $cols, true ) ) { // legacy column name
        $data['timestamp'] = $logged_at;
        $fmt[]             = '%s';
    }

    // If table is missing expected columns, skip silently to avoid fatal thrash.
    if ( empty( $data ) ) {
        return;
    }

    $wpdb->insert( $table, $data, $fmt ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
}

//
// -----------------------------------------------------------------------------
// Optional: Activation hook to default-enable monitor & ensure schema
// -----------------------------------------------------------------------------
register_activation_hook( __FILE__, function () {
    if ( get_option( 'zynith_seo_404_monitor_enabled', null ) === null ) {
        update_option( 'zynith_seo_404_monitor_enabled', '1' );
    }
    zynith_seo_ensure_404_table();
} );
