<?php
/**
 * This file belongs to the YIT Plugin Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly

/**
 * Main class
 *
 * @class   YITH_WC_Share_For_Discounts
 * @package Yithemes
 * @since   1.0.0
 * @author  Your Inspiration Themes
 */

if ( ! class_exists( 'YITH_WC_Share_For_Discounts' ) ) {

	class YITH_WC_Share_For_Discounts {

		/**
		 * Single instance of the class
		 *
		 * @var \YITH_WC_Share_For_Discounts
		 * @since 1.0.0
		 */
		protected static $instance;

		/**
		 * Panel object
		 *
		 * @var     /Yit_Plugin_Panel object
		 * @since   1.0.0
		 * @see     plugin-fw/lib/yit-plugin-panel.php
		 */
		protected $_panel = null;

		/**
		 * @var string Premium version landing link
		 */
		protected $_premium_landing = 'https://yithemes.com/themes/plugins/yith-woocommerce-share-for-discounts/';

		/**
		 * @var string Plugin official documentation
		 */
		protected $_official_documentation = 'https://docs.yithemes.com/yith-woocommerce-share-for-discounts/';

		/**
		 * @var string Yith WooCommerce Share For Discounts panel page
		 */
		protected $_panel_page = 'yith-wc-share-for-discounts';

		/**
		 * Returns single instance of the class
		 *
		 * @return \YITH_WC_Share_For_Discounts
		 * @since 1.0.0
		 */
		public static function get_instance() {

			if ( is_null( self::$instance ) ) {

				self::$instance = new self;

			}

			return self::$instance;

		}

		/**
		 * Constructor
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function __construct() {

			if ( ! function_exists( 'WC' ) ) {
				return;
			}

			//Load plugin framework
			add_action( 'plugins_loaded', array( $this, 'plugin_fw_loader' ), 12 );
			add_action( 'plugins_loaded', array( $this, 'include_privacy_text' ), 20 );
			// register plugin to licence/update system
			add_action( 'wp_loaded', array( $this, 'register_plugin_for_activation' ), 99 );
			add_action( 'admin_init', array( $this, 'register_plugin_for_updates' ) );
			add_filter( 'plugin_action_links_' . plugin_basename( YWSFD_DIR . '/' . basename( YWSFD_FILE ) ), array( $this, 'action_links' ) );
			add_filter( 'yith_show_plugin_row_meta', array( $this, 'plugin_row_meta' ), 10, 5 );
			add_action( 'admin_menu', array( $this, 'add_menu_page' ), 5 );

			$this->includes();

			add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
			add_action( 'woocommerce_admin_field_ywsfd-coupon', 'YWSFD_Custom_Coupon::output' );

			if ( get_option( 'ywsfd_enable_plugin' ) == 'yes' ) {


				add_action( 'init', array( $this, 'initialize_styles' ) );


				add_action( 'wp_login', array( $this, 'switch_to_logged_user' ), 10, 2 );
				add_filter( 'woocommerce_email_classes', array( $this, 'add_ywsfd_custom_email' ) );
				add_filter( 'ywsfd_can_get_coupon', array( $this, 'can_get_coupon' ), 10, 2 );
				add_filter( 'ywsfd_onsale_variations', array( $this, 'get_onsale_variations' ), 10, 2 );

				if ( get_option( 'ywsfd_coupon_purge' ) == 'yes' ) {
					add_action( 'ywsfd_trash_coupon_cron', array( $this, 'trash_expired_coupons' ) );
				}

				if ( is_admin() ) {
					add_action( 'admin_notices', array( $this, 'check_active_options' ), 10 );
				} else {
					add_action( 'wp_head', array( $this, 'add_opengraph_meta' ), 5 );
					add_action( 'wp_head', array( $this, 'theme_compatibility' ) );
					add_filter( 'language_attributes', array( $this, 'add_opengraph_doctype' ) );
					add_action( 'wp_enqueue_scripts', array( $this, 'frontend_scripts' ) );
					add_action( 'woocommerce_before_single_product', array( $this, 'show_ywsfd_product_page' ), 5 );
					add_action( 'woocommerce_before_checkout_form', array( $this, 'show_ywsfd_checkout_page' ) );
					add_action( 'woocommerce_before_cart', array( $this, 'show_ywsfd_cart_page' ) );
					add_action( 'woocommerce_add_to_cart', array( $this, 'check_coupon' ), 10, 2 );
					add_action( 'woocommerce_before_checkout_process', array( $this, 'check_coupon_checkout' ) );
					add_action( 'woocommerce_check_cart_items', array( $this, 'coupon_validation' ) );
				}

			}

		}

		/**
		 * Files inclusion
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		private function includes() {

			include_once( 'includes/functions-ywsfd.php' );
			include_once( 'includes/class-ywsfd-ajax.php' );
			include_once( 'includes/class-ywsfd-shortcode.php' );

			if ( get_option( 'ywsfd_enable_twitter' ) == 'yes' ) {
				require_once( 'includes/twitteroauth/autoload.php' );
			}

			if ( get_option( 'ywsfd_enable_linkedin' ) == 'yes' ) {
				require_once( 'includes/linkedinauth/linkedin.php' );
			}

			if ( is_admin() ) {
				include_once( 'templates/admin/custom-coupon.php' );
				include_once( 'templates/admin/class-ywsfd-custom-coupon-purge.php' );
			}

		}

		/**
		 * Add styles for major themes compatibility
		 *
		 * @since   1.2.1
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function theme_compatibility() {

			//AVADA
			if ( class_exists( 'Avada' ) ) {

				?>

                <style type="text/css">

                    .cart-collaterals .ywsfd-wrapper {
                        float: left;
                        border: 1px solid #e0dede;
                        width: 48%;
                        padding: 30px;
                        margin: 0 0 10px 0;
                    }

                    .cart-collaterals .ywsfd-wrapper .ywsfd-social {

                        padding: 0;
                        margin: 0;
                        border: 0 none;
                    }

                </style>

				<?php

			}

			//NEIGHBORHOOD
			if ( function_exists( 'sf_is_neighborhood' ) ) {

				?>

                <style type="text/css">

                    .checkout .ywsfd-wrapper {
                        width: 380px;
                        float: left;
                        min-height: 1px;
                        margin-left: 20px;
                    }

                    @media (min-width: 1200px) {

                        .checkout .ywsfd-wrapper {
                            width: 470px;
                            margin-left: 30px;
                        }

                    }

                    @media (max-width: 979px) and (min-width: 768px) {

                        .checkout .ywsfd-wrapper {
                            width: 290px;
                            margin-left: 20px;
                        }

                    }

                    @media (max-width: 767px) {

                        .checkout .ywsfd-wrapper {
                            display: block;
                            float: none;
                            width: 100%;
                            max-width: 100%;
                            margin-left: 0;
                            -webkit-box-sizing: border-box;
                            -moz-box-sizing: border-box;
                            box-sizing: border-box;
                        }

                    }

                </style>

				<?php

			}

		}

		/**
		 * ADMIN FUNCTIONS
		 */

		/**
		 * Add a panel under YITH Plugins tab
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Alberto Ruggiero
		 * @use     /Yit_Plugin_Panel class
		 * @see     plugin-fw/lib/yit-plugin-panel.php
		 */
		public function add_menu_page() {

			if ( ! empty( $this->_panel ) ) {
				return;
			}

			$admin_tabs = array(
				'general' => __( 'General Settings', 'yith-woocommerce-share-for-discounts' ),
				'coupon'  => __( 'Coupon Settings', 'yith-woocommerce-share-for-discounts' ),
				'share'   => __( 'Sharing Settings', 'yith-woocommerce-share-for-discounts' ),
			);

			$args = array(
				'create_menu_page' => true,
				'parent_slug'      => '',
				'page_title'       => __( 'Share For Discounts', 'yith-woocommerce-share-for-discounts' ),
				'menu_title'       => 'Share For Discounts',
				'capability'       => 'manage_options',
				'parent'           => '',
				'parent_page'      => 'yith_plugin_panel',
				'page'             => $this->_panel_page,
				'admin-tabs'       => $admin_tabs,
				'options-path'     => YWSFD_DIR . 'plugin-options'
			);

			$this->_panel = new YIT_Plugin_Panel_WooCommerce( $args );

		}

		/**
		 * Check if active options have at least a social network selected
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function check_active_options() {

			$errors = array();

			if ( isset( $_POST['ywsfd_enable_facebook'] ) && '1' == $_POST['ywsfd_enable_facebook'] && $_POST['ywsfd_appid_facebook'] == '' ) {

				$errors[] = __( 'You need to add a Facebook App ID', 'yith-woocommerce-share-for-discounts' );
			}

			if ( isset( $_POST['ywsfd_enable_twitter'] ) && '1' == $_POST['ywsfd_enable_twitter'] && $_POST['ywsfd_twitter_app_id'] == '' ) {

				$errors[] = __( 'You need to add a Twitter Consumer Key (API Key)', 'yith-woocommerce-share-for-discounts' );

			}

			if ( isset( $_POST['ywsfd_enable_twitter'] ) && '1' == $_POST['ywsfd_enable_twitter'] && $_POST['ywsfd_twitter_app_secret'] == '' ) {

				$errors[] = __( 'You need to add a Twitter Consumer Secret (API Secret)', 'yith-woocommerce-share-for-discounts' );

			}

			if ( isset( $_POST['ywsfd_enable_linkedin'] ) && '1' == $_POST['ywsfd_enable_linkedin'] && $_POST['ywsfd_linkedin_app_id'] == '' ) {

				$errors[] = __( 'You need to add a Linkedin Client ID', 'yith-woocommerce-share-for-discounts' );

			}

			if ( isset( $_POST['ywsfd_enable_linkedin'] ) && '1' == $_POST['ywsfd_enable_linkedin'] && $_POST['ywsfd_linkedin_app_secret'] == '' ) {

				$errors[] = __( 'You need to add a Linkedin Client Secret', 'yith-woocommerce-share-for-discounts' );

			}

			$errors = apply_filters( 'ywsfd_additional_notices', $errors );

			if ( ! empty( $errors ) ) {

				?>
                <div class="error">

                    <p>
                        <b><?php _e( 'YITH WooCommerce Share For Discounts', 'yith-woocommerce-share-for-discounts' ) ?></b>
                    </p>

                    <ul>

						<?php foreach ( $errors as $error ): ?>

                            <li> <?php echo $error; ?></li>

						<?php endforeach; ?>
                    </ul>

                </div>

				<?php

			}

		}

		/**
		 * Trash expired coupons
		 *
		 * @since   1.0.8
		 *
		 * @param   $return
		 *
		 * @return  mixed
		 * @author  Alberto Ruggiero
		 */
		public function trash_expired_coupons( $return = false ) {

			$args = array(
				'post_type'      => 'shop_coupon',
				'post_status'    => 'publish',
				'posts_per_page' => - 1,
				'meta_query'     => array(
					'relation' => 'AND',
					array(
						'key'   => 'generated_by',
						'value' => 'ywsfd',
					),
					array(
						'relation' => 'OR',
						array(
							'key'     => 'expiry_date',
							'value'   => date( 'Y-m-d', strtotime( "today" ) ),
							'compare' => '<',
							'type'    => 'DATE'
						),
						array(
							'key'     => 'usage_count',
							'value'   => 1,
							'compare' => '>='
						)
					)
				)
			);

			$query = new WP_Query( $args );
			$count = $query->post_count;

			if ( $query->have_posts() ) {

				while ( $query->have_posts() ) {

					$query->the_post();

					wp_trash_post( $query->post->ID );

				}

			}

			wp_reset_query();
			wp_reset_postdata();

			if ( $return ) {

				return $count;

			}

			return null;

		}

		/**
		 * Initializes CSS and javascript
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function admin_scripts() {

			if ( isset( $_GET['page'] ) && $_GET['page'] == 'yith-wc-share-for-discounts' ) {

				wp_enqueue_style( 'ywsfd-admin', YWSFD_ASSETS_URL . '/css/ywsfd-admin' . ywsfd_get_minified() . '.css' );

				wp_enqueue_script( 'ywsfd-admin', YWSFD_ASSETS_URL . '/js/ywsfd-admin' . ywsfd_get_minified() . '.js', array( 'jquery' ) );

				$params = array(
					'ajax_url' => admin_url( 'admin-ajax.php' ),
				);

				wp_localize_script( 'ywsfd-admin-premium', 'ywsfd_admin', $params );

			}

		}

		/**
		 * Register privacy text
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function include_privacy_text() {
			include_once( 'includes/class-ywsfd-privacy.php' );
		}

		/**
		 * Add the YWSFD_Share_Mail class to WooCommerce mail classes
		 *
		 * @since   1.0.0
		 *
		 * @param   $email_classes
		 *
		 * @return  array
		 * @author  Alberto Ruggiero
		 */
		public function add_ywsfd_custom_email( $email_classes ) {

			$email_classes['YWSFD_Share_Mail'] = include( 'includes/class-ywsfd-share-email.php' );

			return $email_classes;
		}

		/**
		 * FRONTEND FUNCTIONS
		 */

		/**
		 * Get onsale variations
		 *
		 * @since   1.3.1
		 *
		 * @param   $onsale_variations
		 * @param   $product_id
		 *
		 * @return  array
		 * @author  Alberto Ruggiero
		 */
		public function get_onsale_variations( $onsale_variations, $product_id ) {

			if ( ! $product_id ) {
				return $onsale_variations;
			}

			if ( get_option( 'ywsfd_share_product_onsale_enable', 'yes' ) == 'no' ) {

				$product = wc_get_product( $product_id );

				if ( ! $product ) {
					return $onsale_variations;
				}

				if ( $product->is_type( 'variable' ) ) {

					$variations = array_filter( $product->get_available_variations() );

					if ( count( $variations ) > 0 ) {

						foreach ( $variations as $variation ) {

							$product_variation = wc_get_product( $variation['variation_id'] );

							if ( $product_variation->get_sale_price() ) {

								$onsale_variations[] = $variation['variation_id'];

							}

						}

					}

				}

			}

			return $onsale_variations;

		}

		/**
		 * Initializes CSS
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function initialize_styles() {

			$deps = array();

			if ( apply_filters( 'ywsfd_load_fontawesome', true ) ) {

				wp_register_style( 'font-awesome', "https://maxcdn.bootstrapcdn.com/font-awesome/4.6.3/css/font-awesome.min.css", array(), '4.6.3' );
				$deps = array( 'font-awesome' );
			}

			wp_register_style( 'ywsfd-frontend', YWSFD_ASSETS_URL . '/css/ywsfd-frontend' . ywsfd_get_minified() . '.css', $deps );

		}

		/**
		 * Initializes CSS and javascript
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function frontend_scripts() {

			global $post;

			$post_url       = $post_title = $post_id = '';
			$custom_url     = get_option( 'ywsfd_custom_url' );
			$custom_message = get_option( 'ywsfd_custom_message' );

			if ( $post ) {
				$post_id    = $post->ID;
				$post_url   = get_permalink( $post->ID );
				$post_title = get_the_title( $post->ID );
			}

			wp_enqueue_style( 'ywsfd-frontend' );

			if ( ! apply_filters( 'ywsfd_can_get_coupon', true, $post_id ) || ( is_shop() && apply_filters( 'ywsfd_show_on_shop', false ) == false ) || ( is_account_page() && apply_filters( 'ywsfd_show_on_myaccount', false ) == false ) ) {
				return;
			}

			$params = array(
				'ajax_social_url'    => add_query_arg( 'action', 'ywsfd_get_coupon', str_replace( array( 'https:', 'http:' ), '', admin_url( 'admin-ajax.php' ) ) ),
				'ajax_load_url'      => add_query_arg( 'action', 'ywsfd_load', str_replace( array( 'https:', 'http:' ), '', admin_url( 'admin-ajax.php' ) ) ),
				'apply_coupon_nonce' => wp_create_nonce( 'apply-coupon' ),
				'post_id'            => $post_id,
				'sharing'            => array(
					'url'              => apply_filters( 'ywsfd_post_url', ( $custom_url != '' ? $custom_url : $post_url ) ),
					'message'          => apply_filters( 'ywsfd_post_message', ( $custom_message != '' ? $custom_message : $post_title ) ),
					'twitter_username' => ( get_option( 'ywsfd_user_twitter' ) != '' ? ( ' - ' . __( 'via', 'yith-woocommerce-share-for-discounts' ) . ' @' . get_option( 'ywsfd_user_twitter' ) ) : '' ),
				),
				'locale'             => get_locale(),
				'facebook'           => 'no',
				'twitter'            => 'no',
				'email'              => 'no',
				'linkedin'           => 'no',
				'custom_url'         => ( $custom_url != '' ? 'yes' : 'no' ),
				'onsale_variations'  => apply_filters( 'ywsfd_onsale_variations', array(), $post_id )
			);

			if ( get_option( 'ywsfd_enable_facebook' ) == 'yes' && get_option( 'ywsfd_appid_facebook' ) != '' ) {

				$params['facebook']  = 'yes';
				$params['fb_app_id'] = get_option( 'ywsfd_appid_facebook' );

			}

			if ( get_option( 'ywsfd_enable_twitter' ) == 'yes' ) {

				$params['twitter']           = 'yes';
				$params['twitter_login']     = YWSFD_URL . 'templates/frontend/twitter-login.php';
				$params['twitter_close']     = __( 'You closed the parent window. The authorization process has been suspended.', 'yith-woocommerce-share-for-discounts' );
				$params['twitter_fail']      = __( 'The authorization process has failed.', 'yith-woocommerce-share-for-discounts' );
				$params['twitter_auth_ajax'] = add_query_arg( 'action', 'ywsfd_get_twitter_url', str_replace( array( 'https:', 'http:' ), '', admin_url( 'admin-ajax.php' ) ) );
				$params['twitter_send_ajax'] = add_query_arg( 'action', 'ywsfd_send_tweet', str_replace( array( 'https:', 'http:' ), '', admin_url( 'admin-ajax.php' ) ) );

			}

			if ( get_option( 'ywsfd_enable_linkedin' ) == 'yes' ) {

				$params['linkedin']           = 'yes';
				$params['linkedin_login']     = YWSFD_URL . 'templates/frontend/linkedin-login.php';
				$params['linkedin_close']     = __( 'You closed the parent window. The authorization process has been suspended.', 'yith-woocommerce-share-for-discounts' );
				$params['linkedin_fail']      = __( 'The authorization process has failed.', 'yith-woocommerce-share-for-discounts' );
				$params['linkedin_auth_ajax'] = add_query_arg( 'action', 'ywsfd_get_linkedin_url', str_replace( array( 'https:', 'http:' ), '', admin_url( 'admin-ajax.php' ) ) );
				$params['linkedin_send_ajax'] = add_query_arg( 'action', 'ywsfd_send_linkedin', str_replace( array( 'https:', 'http:' ), '', admin_url( 'admin-ajax.php' ) ) );

			}

			if ( get_option( 'ywsfd_enable_email' ) == 'yes' ) {

				$params['email']          = 'yes';
				$params['ajax_email_url'] = add_query_arg( 'action', 'ywsfd_send_friend_mail', str_replace( array( 'https:', 'http:' ), '', admin_url( 'admin-ajax.php' ) ) );

			}

			wp_enqueue_script( 'ywsfd-frontend', YWSFD_ASSETS_URL . '/js/ywsfd-frontend.js', array( 'jquery' ) );

			wp_localize_script( 'ywsfd-frontend', 'ywsfd', apply_filters( 'ywsfd_scripts_filter', $params ) );

		}

		/**
		 * Get the position and show YWSFD in product page
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function show_ywsfd_product_page() {

			if ( get_option( 'ywsfd_share_product_enable' ) != 'no' ) {

				global $post;

				if ( ! apply_filters( 'ywsfd_can_get_coupon', true, ( isset( $post->ID ) ? $post->ID : '' ) ) ) {

					return;

				}

				$position = get_option( 'ywsfd_share_product_position' );

				switch ( $position ) {

					case '1':
						$args = array(
							'hook'     => 'single_product',
							'priority' => 15
						);
						break;

					case '2':
						$args = array(
							'hook'     => 'single_product',
							'priority' => 25
						);
						break;

					case '3':
						$args = array(
							'hook'     => 'after_single_product',
							'priority' => 5
						);
						break;

					case '4':
						$args = array(
							'hook'     => 'after_single_product',
							'priority' => 15
						);
						break;

					case '5':
						$args = array(
							'hook'     => 'after_single_product',
							'priority' => 25
						);
						break;

					default:
						$args = array(
							'hook'     => 'before_single_product',
							'priority' => 5
						);

				}

				$args   = apply_filters( 'ywsfd_share_position', $args );
				$action = apply_filters( 'ywsfd_override_standard_position', 'woocommerce_' . $args['hook'] . '_summary' );

				add_action( $action, array( $this, 'add_ywsfd_template' ), $args['priority'] );

			}

		}

		/**
		 * Get the position and show YWSFD in checkout page
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function show_ywsfd_checkout_page() {

			if ( get_option( 'ywsfd_share_checkout_enable' ) == 'yes' ) {

				$position = get_option( 'ywsfd_share_checkout_position' );

				switch ( $position ) {

					case '1':
						$args = array(
							'hook'     => 'after',
							'priority' => 10
						);
						break;

					default:
						$args = array(
							'hook'     => 'before',
							'priority' => 10
						);

				}

				$args = apply_filters( 'ywsfd_share_position_checkout', $args );


				add_action( 'woocommerce_checkout_' . $args['hook'] . '_customer_details', array( $this, 'add_ywsfd_template' ), $args['priority'] );

			}

		}

		/**
		 * Get the position and show YWSFD in cart page
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function show_ywsfd_cart_page() {

			if ( get_option( 'ywsfd_share_cart_enable' ) == 'yes' ) {

				$position = get_option( 'ywsfd_share_cart_position' );

				switch ( $position ) {

					case '1':
						$args = array(
							'hook'     => 'cart_collaterals',
							'priority' => 10
						);
						break;

					case '2':
						$args = array(
							'hook'     => 'after_cart',
							'priority' => 10
						);
						break;

					default:
						$args = array(
							'hook'     => 'before_cart',
							'priority' => 11
						);

				}

				$args = apply_filters( 'ywsfd_share_position_cart', $args );

				add_action( 'woocommerce_' . $args['hook'], array( $this, 'add_ywsfd_template' ), $args['priority'] );

			}

		}

		/**
		 * Add YWSFD to product page
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function add_ywsfd_template() {

			if ( $this->check_social_active() ) : ?>

                <div id="YWSFD_wrapper" class="woocommerce ywsfd-wrapper">

					<?php

					global $post;

					$post_id = isset( $post ) ? $post->ID : '';

					if ( ! YITH_WSFD()->coupon_already_assigned( $post_id ) ) {
						$this->print_share_bar( $post_id );
					} else {
						$this->print_after_share();
					}

					?>

                </div>

			<?php endif;

		}

		/**
		 * Print share buttons
		 *
		 * @since   1.4.9
		 *
		 * @param   $post_id
		 *
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function print_share_bar( $post_id ) {

			$title_container = apply_filters( 'ywsfd_share_title_container', '<h2>%s</h2>' );
			$custom_url      = get_option( 'ywsfd_custom_url' );
			$custom_message  = get_option( 'ywsfd_custom_message' );
			$post_url        = get_permalink( $post_id );
			$post_title      = get_the_title( $post_id );
			$product         = wc_get_product( $post_id );
			$social_params   = apply_filters( 'ywsfd_social_params', array(
				'sharing'       => array(
					'url'              => apply_filters( 'ywsfd_post_url', ( $custom_url != '' ? $custom_url : $post_url ) ),
					'message'          => apply_filters( 'ywsfd_post_message', ( $custom_message != '' ? $custom_message : $post_title ) ),
					'twitter_username' => ( get_option( 'ywsfd_user_twitter' ) != '' ? ( ' - ' . __( 'via', 'yith-woocommerce-share-for-discounts' ) . ' @' . get_option( 'ywsfd_user_twitter' ) ) : '' ),
					'form_action'      => $post_url,
					'post_id'          => $post_id,
				),
				'facebook'      => get_option( 'ywsfd_enable_facebook' ),
				'facebook_type' => get_option( 'ywsfd_button_type_facebook' ),
				'twitter'       => get_option( 'ywsfd_enable_twitter' ),
				'linkedin'      => get_option( 'ywsfd_enable_linkedin' ),
				'email'         => get_option( 'ywsfd_enable_email' ),
			) );

			?>

			<?php if ( $product ) {
				$title = apply_filters( 'ywsfd_share_title', get_option( 'ywsfd_share_title' ) );
			} else {
				$title = apply_filters( 'ywsfd_share_title_page', get_option( 'ywsfd_share_title_page' ) );
			} ?>

			<?php echo sprintf( $title_container, $title ) ?>

            <div class="ywsfd-social">
                <div class="ywsfd-errors"></div>

				<?php
				/**
				 * FACEBOOK
				 */
				if ( $social_params['facebook'] == 'yes' ) {

					?>
                    <div id="fb-root"></div>
					<?php

					if ( $social_params['facebook_type'] != 'share' ) {
						wc_get_template( '/frontend/fb-like-btn.php', array( 'social_params' => $social_params ), YWSFD_TEMPLATE_PATH, YWSFD_TEMPLATE_PATH );
					}

					if ( $social_params['facebook_type'] != 'like' ) {
						wc_get_template( '/frontend/fb-share-btn.php', array( 'social_params' => $social_params ), YWSFD_TEMPLATE_PATH, YWSFD_TEMPLATE_PATH );
					}

				}

				/**
				 * TWITTER
				 */
				if ( $social_params['twitter'] == 'yes' ) {
					wc_get_template( '/frontend/tw-tweet-btn.php', array( 'social_params' => $social_params ), YWSFD_TEMPLATE_PATH, YWSFD_TEMPLATE_PATH );
				}

				/**
				 * LINKEDIN
				 */
				if ( $social_params['linkedin'] == 'yes' ) {
					wc_get_template( '/frontend/lnk-share-btn.php', array( 'social_params' => $social_params ), YWSFD_TEMPLATE_PATH, YWSFD_TEMPLATE_PATH );
				}

				/**
				 * EMAIL
				 */
				if ( $social_params['email'] == 'yes' ) {
					wc_get_template( '/frontend/mail-btn.php', array( 'social_params' => $social_params ), YWSFD_TEMPLATE_PATH, YWSFD_TEMPLATE_PATH );
				}

				do_action( 'ywsfd_additional_social', $social_params );

				/**
				 * TWITTER FORM
				 */
				if ( $social_params['twitter'] == 'yes' ) {
					wc_get_template( '/frontend/tw-tweet-form.php', array( 'social_params' => $social_params ), YWSFD_TEMPLATE_PATH, YWSFD_TEMPLATE_PATH );
				}

				/**
				 * LINKEDIN FORM
				 */
				if ( $social_params['linkedin'] == 'yes' ) {
					wc_get_template( '/frontend/lnk-share-form.php', array( 'social_params' => $social_params ), YWSFD_TEMPLATE_PATH, YWSFD_TEMPLATE_PATH );
				}

				/**
				 * EMAIL FORM
				 */
				if ( $social_params['email'] == 'yes' ) {
					wc_get_template( '/frontend/mail-form.php', array( 'social_params' => $social_params ), YWSFD_TEMPLATE_PATH, YWSFD_TEMPLATE_PATH );
				}

				do_action( 'ywsfd_additional_forms', $social_params );

				?>

            </div>

			<?php apply_filters( 'ywsfd_after_social', '' );

		}

		/**
		 * Print thank-you message
		 *
		 * @since   1.4.9
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function print_after_share() {
			$title_container = apply_filters( 'ywsfd_share_title_container', '<h2>%s</h2>' ); ?>
			<?php echo sprintf( $title_container, apply_filters( 'ywsfd_share_title_after', get_option( 'ywsfd_share_title_after' ) ) ) ?>

            <div class="ywsfd-social">
            <div class="ywsfd-after-share">
				<?php echo apply_filters( 'ywsfd_share_message', get_option( 'ywsfd_share_text_after' ) ); ?>
            </div>
            </div><?php
		}


		/**
		 * Check if at least a social network is active
		 *
		 * @since   1.0.0
		 * @return  boolean
		 * @author  Alberto Ruggiero
		 */
		public function check_social_active() {

			$socials = apply_filters( 'ywsfd_available_socials', array(
				'facebook',
				'twitter',
				'linkedin',
				'email'
			) );

			$active = false;

			foreach ( $socials as $social ) {

				if ( get_option( 'ywsfd_enable_' . $social ) == 'yes' ) {

					$active = true;

				}

			}

			return $active;

		}

		/**
		 * Add opengraph doctype
		 *
		 * @since   1.0.0
		 *
		 * @param   $output
		 *
		 * @return  string
		 * @author  Alberto Ruggiero
		 */
		public function add_opengraph_doctype( $output ) {

			$enabled = get_option( 'ywsfd_fbmeta_enable' ) == 'yes';

			if ( ! apply_filters( 'ywsfd_hide_og_meta', $enabled ) ) {
				return $output;
			}

			return $output . ' xmlns:og="http://opengraphprotocol.org/schema/" xmlns:fb="http://www.facebook.com/2008/fbml"';

		}

		/**
		 * Add opengraph meta
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function add_opengraph_meta() {

			$enabled = get_option( 'ywsfd_fbmeta_enable' ) == 'yes';

			if ( ! apply_filters( 'ywsfd_hide_og_meta', $enabled ) ) {
				return;
			}

			global $post, $wp;

			$default_title       = get_option( 'ywsfd_fbmeta_title' );
			$default_description = get_option( 'ywsfd_fbmeta_description' );
			$default_image       = get_option( 'ywsfd_fbmeta_image' );
			$default_permalink   = home_url( add_query_arg( array(), $wp->request ) ) . '/';

			$title = $description = $permalink = $image = '';

			if ( $post && ! is_product_category() ) {

				if ( has_post_thumbnail( $post->ID ) ) {

					$image = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'large' );
					$image = $image[0];

				}

				$title       = get_the_title();
				$ogcontent = $post->post_excerpt ? $post->post_excerpt : $post->post_content;
				$description = strip_tags( $ogcontent );
				$permalink   = get_permalink();

			}

			$title       = ( $title == '' ? ( $default_title != '' ? $default_title : get_option( 'blogname' ) ) : $title );
			$description = ( $description == '' ? ( $default_description != '' ? $default_description : get_option( 'blogdescription' ) ) : $description );
			$image       = ( $image == '' ? $default_image : $image );
			$permalink   = ( $permalink == '' ? $default_permalink : $permalink );
			?>

            <meta property="fb:app_id" content="<?php echo get_option( 'ywsfd_appid_facebook' ) ?>" />
            <meta property="og:type" content="website" />
            <meta property="og:site_name" content="<?php echo get_option( 'blogname' ) ?>" />
            <meta property="og:url" content="<?php echo $permalink ?>" />
            <meta property="og:title" content="<?php echo $title ?>" />
			<?php if ( $image ): ?>
                <meta property="og:image" content="<?php echo esc_attr( $image ) ?>" />
			<?php endif; ?>
            <meta property="og:description" content="<?php echo $description ?>" />

			<?php

		}

		/**
		 * Check if current product was shared from user and get coupon code
		 *
		 * @since   1.0.0
		 *
		 * @param   $product_id
		 *
		 * @return  boolean|string
		 * @author  Alberto Ruggiero
		 */
		public function coupon_already_assigned( $product_id = false ) {

			if ( ! $product_id ) {

				global $post;

				if ( ! isset( $post->ID ) ) {
					return false;
				}

				$product_id = $post->ID;

			}

			$result    = false;
			$user_data = $this->get_user_data();

			if ( isset( $user_data['email'] ) && $user_data['email'] != '' ) {

				$customer_ref = array(
					'key'     => 'customer_email',
					'value'   => $user_data['email'],
					'compare' => '=',
				);

			} else {

				$customer_ref = array(
					'key'     => 'customer_guest_id',
					'value'   => $user_data['guest_id'],
					'compare' => '=',
				);

			}

			$coupon_options = get_option( 'ywsfd_coupon' );

			switch ( $coupon_options['discount_type'] ) {
				case 'fixed_cart':
				case 'percent':

					$args = array(
						'post_type'   => 'shop_coupon',
						'post_status' => 'publish',
						'meta_query'  => array(
							$customer_ref,
						),
						'date_query'  => array(
							array(
								'year'  => date( 'Y' ),
								'month' => date( 'm' ),
								'day'   => date( 'd' ),
							),
						),
					);
					break;

				default:

					if ( get_post_type( $product_id ) == 'product' ) {

						$args = array(
							'post_type'   => 'shop_coupon',
							'post_status' => 'publish',
							'meta_query'  => array(
								'relation' => 'AND',
								array(
									'relation' => 'OR',
									array(
										'key'     => 'product_ids',
										'value'   => $product_id,
										'compare' => '=',
									),
									array(
										'key'     => 'product_ids',
										'value'   => $product_id . ',',
										'compare' => 'LIKE',
									),
									array(
										'key'     => 'product_ids',
										'value'   => ',' . $product_id . ',',
										'compare' => 'LIKE',
									),
									array(
										'key'     => 'product_ids',
										'value'   => ',' . $product_id,
										'compare' => 'LIKE',
									),

								),
								$customer_ref,
							),
							'date_query'  => array(
								array(
									'year'  => date( 'Y' ),
									'month' => date( 'm' ),
									'day'   => date( 'd' ),
								),
							),
						);

					} else {

						$args = array(
							'post_type'   => 'shop_coupon',
							'post_status' => 'publish',
							'meta_query'  => array(
								'relation' => 'AND',
								array(
									'key'     => 'other_page_id',
									'value'   => '',
									'compare' => '!=',
								),
								$customer_ref,
							),
							'date_query'  => array(
								array(
									'year'  => date( 'Y' ),
									'month' => date( 'm' ),
									'day'   => date( 'd' ),
								),
							),
						);

					}

			}

			add_filter( 'posts_where', array( $this, 'modify_coupon_where' ) );
			$query = get_posts( $args );
			remove_filter( 'posts_where', array( $this, 'modify_coupon_where' ) );

			if ( ! empty( $query ) ) {

				$result = $query[0]->post_title;

			}

			return $result;

		}

		public function modify_coupon_where( $where ) {

			$where = preg_replace( '/(\'%)(\d+)/', '\'${2}', $where );
			$where = preg_replace( '/(\d+)(%\')/', '${1}\'', $where );

			return $where;

		}

		/**
		 * Get current user data
		 *
		 * @since   1.0.0
		 * @return  array
		 * @author  Alberto Ruggiero
		 */
		public function get_user_data() {

			$user_data = array(
				'nickname' => '',
				'email'    => '',
				'guest_id' => ''
			);

			if ( is_user_logged_in() ) {

				$current_user = wp_get_current_user();

				$user_data['nickname'] = get_user_meta( $current_user->ID, 'nickname', true );
				$user_data['email']    = get_user_meta( $current_user->ID, 'billing_email', true );


			} else {

				if ( ! WC()->session->has_session() ) {
					WC()->session->set_customer_session_cookie( true );
				}

				$guest_id = WC()->session->get( 'ywsfd_guest_id' );

				if ( empty( $guest_id ) ) {

					$guest_id = uniqid( rand(), false );
					WC()->session->set( 'ywsfd_guest_id', $guest_id );

				}

				$user_data['nickname'] = __( 'Guest', 'yith-woocommerce-share-for-discounts' );
				$user_data['guest_id'] = $guest_id;

			}

			return $user_data;

		}

		/**
		 * Re-assign coupon when user is logged
		 *
		 * @since   1.0.0
		 *
		 * @param   $user_login
		 * @param   $user
		 *
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function switch_to_logged_user( $user_login, $user ) {

			if ( ! WC()->session->has_session() ) {
				WC()->session->set_customer_session_cookie( true );
			}
			$guest_id = WC()->session->get( 'ywsfd_guest_id' );

			if ( ! empty( $guest_id ) ) {

				$this->assign_guest_coupon( $guest_id, $user->user_email );

				WC()->session->set( 'ywsfd_guest_id', false );
			}

		}

		/**
		 * Re-assign coupon on checkout
		 *
		 * @since   1.0.7
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function check_coupon_checkout() {

			if ( ! WC()->session->has_session() ) {
				WC()->session->set_customer_session_cookie( true );
			}

			$guest_id = WC()->session->get( 'ywsfd_guest_id' );

			if ( ! empty( $guest_id ) ) {

				$this->assign_guest_coupon( $guest_id, $_POST['billing_email'] );

			}

		}

		/**
		 * Re-assign coupon when to provided email address
		 *
		 * @since   1.0.7
		 *
		 * @param   $guest_id
		 * @param   $user_email
		 *
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function assign_guest_coupon( $guest_id, $user_email ) {

			$found_ids = array();
			$args      = array(
				'post_type'   => 'shop_coupon',
				'post_status' => 'publish',
				'meta_query'  => array(
					array(
						'key'     => 'customer_guest_id',
						'value'   => $guest_id,
						'compare' => '=',
					),
				),
				'date_query'  => array(
					array(
						'year'  => date( 'Y' ),
						'month' => date( 'm' ),
						'day'   => date( 'd' ),
					),
				),
			);

			$query = new WP_Query( $args );

			if ( $query->have_posts() ) {

				while ( $query->have_posts() ) {

					$query->the_post();
					$found_ids[] = $query->post->ID;

				}

			}

			wp_reset_query();
			wp_reset_postdata();


			if ( ! empty( $found_ids ) ) {

				foreach ( $found_ids as $coupon_id ) {

					update_post_meta( $coupon_id, 'customer_email', $user_email );
					update_post_meta( $coupon_id, 'customer_guest_id', '' );

				}

			}

		}

		/**
		 * Creates a coupon with specific settings
		 *
		 * @since   1.0.0
		 *
		 * @param   $user_data
		 * @param   $product_id
		 *
		 * @return  string
		 * @author  Alberto Ruggiero
		 */
		public function create_coupon( $user_data, $product_id ) {

			$coupon_first_part  = apply_filters( 'ywsfd_coupon_code_first_part', $user_data['nickname'] );
			$coupon_separator   = apply_filters( 'ywsfd_coupon_code_separator', '-' );
			$coupon_second_part = apply_filters( 'ywsfd_coupon_code_second_part', current_time( 'YmdHis' ) );

			$coupon_code = $coupon_first_part . $coupon_separator . $coupon_second_part;

			$coupon_option = apply_filters( 'ywsfd_coupon_options', get_option( 'ywsfd_coupon' ) );

			$coupon_data = array(
				'post_title'   => $coupon_code,
				'post_excerpt' => $coupon_option['description'],
				'post_content' => '',
				'post_status'  => 'publish',
				'post_author'  => 1,
				'post_type'    => 'shop_coupon'
			);

			$coupon_id     = wp_insert_post( $coupon_data );
			$ve            = get_option( 'gmt_offset' ) > 0 ? '+' : '-';
			$expiry_date   = ( $coupon_option['expiry_days'] != '' ) ? date( 'Y-m-d', strtotime( '+' . $coupon_option['expiry_days'] . ' days' . $ve . get_option( 'gmt_offset' ) . ' HOURS' ) ) : '';
			$discount_type = $coupon_option['discount_type'];

			switch ( $discount_type ) {
				case 'fixed_cart':
				case 'percent':
					$product_ids = '';
					break;
				default:

					if ( ! empty( $product_id ) ) {

						if ( get_post_type( $product_id ) == 'product' ) {

							$product_ids = $product_id;

							if ( $discount_type == 'percent_product' && version_compare( WC()->version, '2.7.0', '>=' ) ) {
								$discount_type = 'percent';
							}

						} else {

							$product_ids = '';
							update_post_meta( $coupon_id, 'other_page_id', $product_id );

						}

					} else {
						$product_ids = '';
					}

			}

			$variations = apply_filters( 'ywsfd_onsale_variations', array(), $product_ids );

			update_post_meta( $coupon_id, 'discount_type', $discount_type );
			update_post_meta( $coupon_id, 'coupon_amount', $coupon_option['coupon_amount'] );
			update_post_meta( $coupon_id, 'free_shipping', ( isset( $coupon_option['free_shipping'] ) && $coupon_option['free_shipping'] != '' ? 'yes' : 'no' ) );
			update_post_meta( $coupon_id, 'product_ids', $product_ids );
			update_post_meta( $coupon_id, 'customer_email', $user_data['email'] );
			update_post_meta( $coupon_id, 'customer_guest_id', $user_data['guest_id'] );
			update_post_meta( $coupon_id, 'generated_by', 'ywsfd' );
			update_post_meta( $coupon_id, 'exclude_product_ids', implode( ',', $variations ) );

			if ( apply_filters( 'ywsfd_eternal_coupon', false ) == false ) {

				update_post_meta( $coupon_id, 'expiry_date', $expiry_date );
				update_post_meta( $coupon_id, 'usage_limit', '1' );
				update_post_meta( $coupon_id, 'usage_limit_per_user', '1' );

			}

			if ( apply_filters( 'ywsfd_individual_use', false ) ) {
				update_post_meta( $coupon_id, 'individual_use', 'yes' );
			}

			do_action( 'ywsfd_additional_coupon_features', $coupon_id, $coupon_option );

			return $coupon_code;

		}

		/**
		 * Check if the coupon for current product needs to be added after adding product to cart
		 *
		 * @since   1.0.0
		 *
		 * @param $cart_item_key
		 * @param $product_id
		 *
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function check_coupon( $cart_item_key, $product_id ) {

			$coupon_id = $this->coupon_already_assigned( $product_id );

			if ( $coupon_id && ! in_array( strtolower( $coupon_id ), WC()->cart->applied_coupons ) ) {

				WC()->cart->add_discount( $coupon_id );

			}

		}

		/**
		 * Prevent multiple discount on the same article.
		 *
		 * @since   1.1.3
		 * @return  void
		 * @author  Alberto Ruggiero
		 */
		public function coupon_validation() {

			if ( ! WC()->session->has_session() ) {
				WC()->session->set_customer_session_cookie( true );
			}

			$today    = getdate();
			$guest_id = WC()->session->get( 'ywsfd_guest_id' );

			foreach ( WC()->cart->get_coupons() as $coupon ) {

				if ( yit_get_prop( $coupon, 'generated_by', true ) == 'ywsfd' && get_post_status( yit_get_prop( $coupon, 'id' ) ) == 'publish' ) {

					$product_ids       = yit_get_prop( $coupon, 'product_ids' );
					$customer_email    = yit_get_prop( $coupon, 'customer_email' );
					$customer_guest_id = yit_get_prop( $coupon, 'customer_guest_id' );

					if ( is_user_logged_in() && $customer_guest_id != '' ) {
						WC()->cart->remove_coupon( yit_get_prop( $coupon, 'code' ) );
						continue;
					}

					if ( $customer_email != '' ) {

						$customer_condition = array(
							'key'     => 'customer_email',
							'value'   => ( is_array( $customer_email ) ? implode( ',', $customer_email ) : $customer_email ),
							'compare' => '='
						);

					} else {

						$customer_condition = array(
							'key'     => 'customer_guest_id',
							'value'   => ( is_array( $customer_guest_id ) ? implode( ',', $customer_guest_id ) : $customer_guest_id ),
							'compare' => '='
						);

					}

					$args = array(
						'post_type'      => 'shop_coupon',
						'post_status'    => 'publish',
						'posts_per_page' => - 1,
						'post__not_in'   => array( yit_get_prop( $coupon, 'id' ) ),
						'date_query'     => array(
							array(
								'column' => 'post_date_gmt',
								'year'   => $today['year'],
								'month'  => $today['mon'],
								'day'    => $today['mday'],
							),
						),
						'meta_query'     => array(
							'relation' => 'AND',
							array(
								'key'   => 'generated_by',
								'value' => 'ywsfd',
							),
							array(
								'relation' => 'AND',
								array(
									'key'     => 'product_ids',
									'value'   => ( is_array( $product_ids ) ? implode( ',', $product_ids ) : $product_ids ),
									'compare' => '=',
								),
								$customer_condition
							)
						)
					);

					$query = new WP_Query( $args );

					if ( $query->have_posts() ) {

						while ( $query->have_posts() ) {

							$query->the_post();

							wp_trash_post( $query->post->ID );

							WC()->cart->remove_coupon( $query->post->post_title );

						}

					}

					if ( ! empty( $guest_id ) && ! empty( $customer_guest_id ) && ( is_array( $customer_guest_id ) ? implode( ',', $customer_guest_id ) : $customer_guest_id ) != $guest_id ) {

						WC()->cart->remove_coupon( yit_get_prop( $coupon, 'code' ) );

					}

					wp_reset_query();
					wp_reset_postdata();

				}

			}

		}

		/**
		 * Current product can be shared?
		 *
		 * @since   1.3.1
		 *
		 * @param   $value
		 * @param   $product_id
		 *
		 * @return  bool
		 * @author  Alberto Ruggiero
		 */
		public function can_get_coupon( $value, $product_id ) {

			$product = wc_get_product( $product_id );

			if ( ! $product ) {
				return $value;
			}

			if ( get_option( 'ywsfd_share_product_onsale_enable', 'yes' ) == 'no' ) {

				if ( $product->is_type( 'variable' ) ) {

					$variations        = array_filter( $product->get_available_variations() );
					$onsale_variations = $this->get_onsale_variations( array(), $product_id );

					if ( count( $variations ) == count( $onsale_variations ) ) {

						$value = false;

					}

				} else {

					if ( $product->get_sale_price() ) {

						$value = false;

					}

				}

			}

			return $value;

		}

		/**
		 * YITH FRAMEWORK
		 */

		/**
		 * Load plugin framework
		 *
		 * @since   1.0.0
		 * @return  void
		 * @author  Andrea Grillo <andrea.grillo@yithemes.com>
		 */
		public function plugin_fw_loader() {
			if ( ! defined( 'YIT_CORE_PLUGIN' ) ) {
				global $plugin_fw_data;
				if ( ! empty( $plugin_fw_data ) ) {
					$plugin_fw_file = array_shift( $plugin_fw_data );
					require_once( $plugin_fw_file );
				}
			}
		}

		/**
		 * Register plugins for activation tab
		 *
		 * @since   2.0.0
		 * @return  void
		 * @author  Andrea Grillo <andrea.grillo@yithemes.com>
		 */
		public function register_plugin_for_activation() {
			if ( ! class_exists( 'YIT_Plugin_Licence' ) ) {
				require_once 'plugin-fw/licence/lib/yit-licence.php';
				require_once 'plugin-fw/licence/lib/yit-plugin-licence.php';
			}
			YIT_Plugin_Licence()->register( YWSFD_INIT, YWSFD_SECRET_KEY, YWSFD_SLUG );
		}

		/**
		 * Register plugins for update tab
		 *
		 * @since   2.0.0
		 * @return  void
		 * @author  Andrea Grillo <andrea.grillo@yithemes.com>
		 */
		public function register_plugin_for_updates() {
			if ( ! class_exists( 'YIT_Upgrade' ) ) {
				require_once( 'plugin-fw/lib/yit-upgrade.php' );
			}
			YIT_Upgrade()->register( YWSFD_SLUG, YWSFD_INIT );
		}

		/**
		 * Action Links
		 *
		 * add the action links to plugin admin page
		 *
		 * @since   1.0.0
		 *
		 * @param   $links | links plugin array
		 *
		 * @return  array
		 * @author  Alberto Ruggiero
		 * @use     plugin_action_links_{$plugin_file_name}
		 */
		public function action_links( $links ) {

			$links = yith_add_action_links( $links, $this->_panel_page, true );

			return $links;

		}

		/**
		 * Plugin row meta
		 *
		 * add the action links to plugin admin page
		 *
		 * @since   1.0.0
		 *
		 * @param   $new_row_meta_args
		 * @param   $plugin_meta
		 * @param   $plugin_file
		 * @param   $plugin_data
		 * @param   $status
		 * @param   $init_file
		 *
		 * @return  array
		 * @author  Andrea Grillo <andrea.grillo@yithemes.com>
		 * @use     plugin_row_meta
		 */
		public function plugin_row_meta( $new_row_meta_args, $plugin_meta, $plugin_file, $plugin_data, $status, $init_file = 'YWSFD_INIT' ) {

			if ( defined( $init_file ) && constant( $init_file ) == $plugin_file ) {
				$new_row_meta_args['slug']       = YWSFD_SLUG;
				$new_row_meta_args['is_premium'] = true;
			}

			return $new_row_meta_args;

		}

	}

}

