<?php
use angelleye\PayPal\PayPal;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
} // Exit if accessed directly

if( ! class_exists( 'YITH_Vendors_Gateway_Paypal_Masspay' ) ) {
    /**
     * YITH Gateway Paypal
     *
     * Define methods and properties for class that manages payments via paypal
     *
     * @package   YITH_Marketplace
     * @author    Your Inspiration <info@yourinspiration.it>
     * @license   GPL-2.0+
     * @link      http://yourinspirationstore.it
     * @copyright 2014 Your Inspiration
     */
    class YITH_Vendors_Gateway_Paypal_Masspay extends YITH_Vendors_Gateway {
	    /**
	     * @var string gateway slug
	     */
        protected $_id = 'paypal-masspay';

	    /**
	     * @var string gateway name
	     */
	    protected $_method_title = 'PayPal MassPay';

	    /**
	     * Status for payments correctly sent
	     *
	     * @cont string Status for payments correctly sent
	     *
	     * @since 1.0
	     */
	    const PAYMENT_STATUS_OK = 'Success';

	    /**
	     * Status for payments failed
	     *
	     * @cont string Status for payments failed
	     *
	     * @since 1.0
	     */
	    const PAYMENT_STATUS_FAIL = 'Failure';

	    /**
	     * YITH_Vendors_Gateway_Paypal_Masspay constructor.
	     *
	     * @param $gateway
	     */
	    public function __construct( $gateway ) {
		    if( apply_filters( 'yith_deprecated_paypal_service_support', false ) ) {
		    	parent::__construct( $gateway );

		    	//Change default value for PayPal gateway option
		    	add_filter( 'yith_wcmv_is_enable_gateway_default_value', '__yith_wcmv_return_yes' );

			    /* === Admin Panel === */
			    add_filter( 'yith_wcmv_panel_gateways_options', 'YITH_Vendors_Gateway_Paypal_Masspay::add_paypal_section_options' );
			    add_filter( 'yith_wcmv_panel_sections', 'YITH_Vendors_Gateway_Paypal_Masspay::add_paypal_section' );
			    add_filter( 'yith_wcmv_panel_gateways_options', 'YITH_Vendors_Gateway_Paypal_Masspay::remove_vendor_payment_choosing' );

			    add_action( 'init', 'YITH_Vendors_Gateway_Paypal_Masspay::restore_paypal_options', 99 );

			    if( $this->is_enabled() ){
				    /* === Commissions Table === */
				    // Bulk Actions
				    add_filter( 'yith_wcmv_commissions_bulk_actions', 'YITH_Vendors_Gateway_Paypal_Masspay::commissions_bulk_actions' );
				    //Add MassPay Button
				    add_action( 'yith_wcmv_before_user_actions', 'YITH_Vendors_Gateway_Paypal_Masspay::add_masspay_pay_button', 10, 1 );

				    /* === Vendor's Panel === */
				    //Payments tab
				    add_action( 'yith_wcmv_vendor_panel_payments', 'YITH_Vendors_Gateway_Paypal_Masspay::add_vendor_panel_payments_options', 10, 1 );

				    // hook the IPNListener
				    add_action( 'init', array( $this, 'handle_notification' ), 30 );
			    }
		    }

		    else {
			    add_filter( 'yith_wcmv_show_enabled_gateways_table', array( $this, 'disable_all' ) );
            }
	    }

	    /**
	     * Retreive the paypal options array from this plugin.
	     *
	     * @author Andrea Grillo <andrea.grillo@yithemes.com>
	     *
	     * @return array opaypla option array
	     */
	    public static function get_paypal_options_array() {
		    return apply_filters( 'yith_wcmv_paypal_gateways_options', array(
				    'paypal-masspay' => array(

					    'payments_options_start' => array(
						    'type' => 'sectionstart',
					    ),

					    'payments_options_title' => array(
						    'title' => __( 'PayPal settings', 'yith-woocommerce-product-vendors' ),
						    'type'  => 'title',
						    'desc'  => __( 'Please note!
							From December 2017, PayPal deprecated MassPay and Adaptive Paymentsmethods.
							This implies it is no longer possible to request for the activation of these services for new accounts. 
							However, those who have the services already active will still be able to use them.',
							    'yith-woocommerce-product-vendors'
						    ),
					    ),

					    'payments_enable_service'              => array(
						    'id'      => 'yith_wcmv_enable_paypal-masspay_gateway',
						    'type'    => 'checkbox',
						    'title'   => __( 'Enable/Disable', 'yith-woocommerce-product-vendors' ),
						    'desc'    => __( 'Enable PayPal gateway', 'yith-woocommerce-product-vendors' ),
						    'default' => 'yes'
					    ),

					    'payment_gateway' => array(
						    'id'      => 'payment_gateway',
						    'type'    => 'select',
						    'title'   => __( 'PayPal Service', 'yith-woocommerce-product-vendors' ),
						    'desc'    => __( 'Choose the PayPal service to pay the commissions to vendors (the only option currently available is MassPay).', 'yith-woocommerce-product-vendors' ),
						    'options' => apply_filters( 'yith_wcmv_payments_gateway', array(
								    'masspay' => __( 'MassPay', 'yith-woocommerce-product-vendors' ),
							    )
						    ),
						    'default' => 'masspay'
					    ),

					    'payment_method' => array(
						    'id'      => 'payment_method',
						    'type'    => 'select',
						    'title'   => __( 'Payment Method', 'yith-woocommerce-product-vendors' ),
						    'desc'    => __( 'Choose how to pay the commissions to vendors', 'yith-woocommerce-product-vendors' ),
						    'options' => array(
							    'manual' => __( 'Pay manually', 'yith-woocommerce-product-vendors' ),
							    'choose' => __( 'Let vendors decide', 'yith-woocommerce-product-vendors' ),
						    ),
						    'default' => 'choose',
					    ),

					    'payment_minimum withdrawals' => array(
						    'id'                => 'payment_minimum_withdrawals',
						    'type'              => 'number',
						    'title'             => __( 'Minimum Withdrawal', 'yith-woocommerce-product-vendors' ) . ' ' . get_woocommerce_currency_symbol(),
						    'desc'              => __( "Set the minimum value for commission withdrawals. This setting will update all vendors' accounts that still have a threshold lower than the one set.", 'yith-woocommerce-product-vendors' ),
						    'custom_attributes' => array(
							    'min' => 1
						    ),
						    'default'           => 1
					    ),

					    'paypal_sandbox'              => array(
						    'id'      => 'paypal_sandbox',
						    'type'    => 'checkbox',
						    'title'   => __( 'Sandbox environment', 'yith-woocommerce-product-vendors' ),
						    'desc'    => __( 'Set environment as sandbox, for test purpose', 'yith-woocommerce-product-vendors' ),
						    'default' => 'yes'
					    ),

					    'paypal_api_username'         => array(
						    'id'    => 'paypal_api_username',
						    'type'  => 'text',
						    'title' => __( 'API Username', 'yith-woocommerce-product-vendors' ),
						    'desc'  => sprintf( __( 'API username of PayPal administration account (if empty, settings of PayPal in <a href="%s">WooCommmerce Settings page</a> apply).', 'yith-woocommerce-product-vendors' ), admin_url( 'admin.php?page=wc-settings&tab=checkout&section=wc_gateway_paypal' ) )
					    ),
					    'paypal_api_password'         => array(
						    'id'    => 'paypal_api_password',
						    'type'  => 'text',
						    'title' => __( 'API Password', 'yith-woocommerce-product-vendors' ),
						    'desc'  => sprintf( __( 'API password of PayPal administration account (if empty, settings of PayPal in <a href="%s">WooCommmerce Settings page</a> apply).', 'yith-woocommerce-product-vendors' ), admin_url( 'admin.php?page=wc-settings&tab=checkout&section=wc_gateway_paypal' ) )
					    ),
					    'paypal_api_signature'        => array(
						    'id'    => 'paypal_api_signature',
						    'type'  => 'text',
						    'title' => __( 'API Signature', 'yith-woocommerce-product-vendors' ),
						    'desc'  => sprintf( __( 'API signature of PayPal administration account (if empty, settings of PayPal in <a href="%s">WooCommmerce Settings page</a> apply).', 'yith-woocommerce-product-vendors' ), admin_url( 'admin.php?page=wc-settings&tab=checkout&section=wc_gateway_paypal' ) )
					    ),
					    'paypal_payment_mail_subject' => array(
						    'id'    => 'paypal_payment_mail_subject',
						    'type'  => 'text',
						    'title' => __( 'Payment Email Subject', 'yith-woocommerce-product-vendors' ),
						    'desc'  => __( 'Subject for email sent by PayPal to customers when payment request is registered', 'yith-woocommerce-product-vendors' )
					    ),
					    'paypal_ipn_notification_url' => array(
						    'id'                => 'paypal_ipn_notification_url',
						    'type'              => 'text',
						    'title'             => __( 'Notification URL', 'yith-woocommerce-product-vendors' ),
						    'desc'              => __( 'Copy this URL and set it into PayPal admin panel, to receive IPN from their server', 'yith-woocommerce-product-vendors' ),
						    'default'           => site_url() . '/?paypal_ipn_response=true',
						    'css'               => 'width: 400px;',
						    'custom_attributes' => array(
							    'readonly' => 'readonly'
						    )
					    ),

					    'vendors_options_end' => array(
						    'type' => 'sectionend',
					    ),
				    )
			    )
		    );
	    }

	    /**
	     * Add PayPal Section
	     *
	     * @author Andrea Grillo <andrea.grillo@yithemes.com>
	     *
	     * @return array paypal option array
	     */
	    public static function add_paypal_section( $sections ){
		    $sections['gateways']['paypal-masspay'] = _x( 'PayPal (Deprecated Services)', '[Admin]: Option description', 'yith-woocommerce-product-vendors' );
		    return $sections;
	    }

	    /**
	     * Add  paypal options array from this plugin.
	     *
	     * @author Andrea Grillo <andrea.grillo@yithemes.com>
	     *
	     * @return array paypal option array
	     */
	    public static function add_paypal_section_options( $options ){
		    return array_merge( $options, self::get_paypal_options_array() );
	    }

	    /**
	     * Add Pay Bulk Actions
	     *
	     * @param $actions array Bulk actions for commissions table
	     *
	     * @return array allowed bulk actions
	     * @author Andrea Grillo <andrea.grillo@yithemes.com>
	     */
	    public static function commissions_bulk_actions( $actions ) {
		    if ( 'masspay' == get_option( 'payment_gateway', 'masspay' ) ) {
			    $actions = array_merge( array( 'pay' => __( 'Pay with MassPay', 'yith-woocommerce-product-vendors' ) ), $actions );
		    }

		    return $actions;
	    }

	    /**
	     * Show Pay Button for MassPay service
	     *
	     * @param $commission YITH_Commission the commission to pay
	     *
	     * @return void
	     * @author Andrea Grillo <andrea.grillo@yithemes.com>
	     */
	    public static function add_masspay_pay_button( $commission ) {
		    if ( $commission->has_status( 'unpaid' ) && 'masspay' == get_option( 'payment_gateway', 'masspay' ) && $commission instanceof YITH_Commission ) {
			    $masspay_uri = esc_url( wp_nonce_url( add_query_arg( array( 'action'        => 'pay_commission',
			                                                                'commission_id' => $commission->id
			    ), admin_url( 'admin.php' ) ), 'yith-vendors-pay-commission' ) );
			    $masspay_uri = apply_filters( 'yith_wcmv_commissions_list_table_masspay_pay_button_url', $masspay_uri, $commission );
			    printf( '<a class="button tips pay" href="%1$s" data-tip="%2$s">%2$s</a>', $masspay_uri, __( 'Pay', 'yith-woocommerce-product-vendors' ) );
		    }
	    }

	    /**
	     * Add Payments option to Payment tab
	     *
	     * @param $args array template args
	     *
	     * @return void
	     * @author Andrea Grillo <andrea.grillo@yithemes.com>
	     */
	    public static function add_vendor_panel_payments_options( $args = array() ) {
		    $currency_symbol = get_woocommerce_currency_symbol();
		    $payments_type   = array(
			    'instant'   => __( 'Instant Payment', 'yith-woocommerce-product-vendors' ),
			    'threshold' => __( 'Payment threshold', 'yith-woocommerce-product-vendors' )
		    );
		    $step            = 'any';
		    $min             = get_option( 'payment_minimum_withdrawals', 1 );
		    $vendor          = ! empty( $args['vendor'] ) && $args['vendor'] instanceof YITH_Vendor ? $args['vendor'] : yith_get_vendor( 'current', 'user' );

		    if ( 'choose' == get_option( 'payment_method', false ) ) : ?>
			    <?php ob_start(); ?>
			    <div class="form-field">
				    <label for="vendor_payment_type"><?php _e( 'Payment type:', 'yith-woocommerce-product-vendors' ) ?></label>
				    <select name="yith_vendor_data[payment_type]" id="vendor_payment_type" class="vendor_payment_type">
					    <?php foreach ( $payments_type as $value => $label ) : ?>
						    <option <?php selected( $vendor->payment_type, $value ) ?>
							    value="<?php echo $value ?>"><?php echo $label ?></option>
					    <?php endforeach; ?>
				    </select>
				    <br/>
				    <span
					    class="description"><?php _e( 'Choose payment method for crediting commissions', 'yith-woocommerce-product-vendors' ); ?></span>
			    </div>


			    <div class="form-field">
				    <label class="yith_vendor_payment_threshold"
				           for="yith_vendor_payment_threshold"><?php _e( 'Threshold', 'yith-woocommerce-product-vendors' ); ?></label>
				    <input type="number" class="payment-threshold-field" name="yith_vendor_data[threshold]"
				           id="yith_vendor_payment_threshold" value="<?php echo $vendor->threshold ?>" min="<?php echo $min ?>"
				           step="<?php echo $step ?>"/>
				    <?php echo $currency_symbol ?>
				    <br/>
				    <span class="description"><?php printf( '%s (%s: <strong>%s</strong>).',
						    __( "Minimum vendor's earning before a vendor's commissions can be paid", 'yith-woocommerce-product-vendors' ),
						    __( 'Minimum threshold allowed by site administrator is', 'yith-woocommerce-product-vendors' ),
						    wc_price( get_option( 'payment_minimum_withdrawals' ) )
					    ); ?></span>
			    </div>
			    <?php echo ob_get_clean(); ?>
		    <?php endif;
	    }

	    /**
	     * Restore PayPal Options
	     *
	     * @author Andrea Grillo <andrea.grillo@yithemes.com>
	     * @return void
	     * @ since 2.5.0
	     */
	    public static function restore_paypal_options(){
		    $is_restored = get_option( 'yith_wcmv_deprecated_paypal_options_restored', false );
		    if( ! $is_restored ){
			    $to_restore = get_option( 'yith_wcmv_deprecated_paypal_options', array() );
			    foreach( $to_restore as $option => $value ){
				    update_option( $option, $value );
			    }

			    add_option( 'yith_wcmv_deprecated_paypal_options_restored', true );
		    }
	    }

        /* === PAYMENT METHODS === */

	    /**
	     * Get the data for pay() method
	     *
	     * @args Array argument to retreive payment data
	     * @return array
	     */
	    public function get_pay_data( $args = array() ) {
	        $data = array();

		    if( ! empty( $args['type'] ) && 'massive_payment' == $args['type'] ){
		        if(
                    ! empty($args['vendor']) && $args['vendor'] instanceof YITH_Vendor
                    &&
                    ! empty( $args['amount'] )
                    &&
                    ! empty( $args['commission_ids'] )
                   ){
		            $vendor = $args['vendor'];

		            if ( empty( $vendor->paypal_email ) ) {
				        return array ();
			        }

			        $commission_ids = $args['commission_ids'];

			        //Insert Payment Field
			        $payment_id = YITH_Commissions()->register_massive_payment( $vendor->id, $vendor->get_owner(), $_GET['amount'], $commission_ids, 'processing' );

			        $data[] = array(
				        'paypal_email' => $vendor->paypal_email,
				        'amount'       => round( $args['amount'], 2 ),
				        'request_id'   => 'massive_payment_' . $payment_id
			        );

			        foreach( $commission_ids as $commission_id ){
			            $commission = YITH_Commission( $commission_id );
			            if( ! empty( $commission ) && $commission instanceof YITH_Commission ){
				            // save the commissions with other to set 'paid' after success payment
				            $commissions[] = $commission;
                        }
                    }

			        $data['commissions'] = $commissions;
                }
            }

            else {
	            $order_id = 0;

	            if( empty( $args['order_id'] )){
		            $order_id = $args['order_id'];
	            }

	            $order = wc_get_order( $order_id );
	            $commissions = $vendors_to_pay = $data = array();

	            if( ! empty( $order_id ) ){
		            // retrieve the vendors to pay
		            foreach ( $order->get_items() as $item_id => $item ) {
			            if ( ! isset( $item['commission_id'] ) ) {
				            continue;
			            }

			            $commission = YITH_Commission( $item['commission_id'] );
			            $vendor = $commission->get_vendor();

			            if ( empty( $vendor->paypal_email ) ) {
				            return array ();
			            }

			            if ( 'manual' == $vendor->payment_method ) {
				            continue;
			            }

			            if( $vendor instanceof YITH_Vendor && $vendor->is_valid() ){
				            $vendors_to_pay[ $vendor->id ] = array(
					            'vendor'      => $vendor,
					            'commissions' => $vendor->commissions_to_pay( $vendor->payment_type )
				            );
			            }
		            }
	            }

	            if( empty( $vendors_to_pay ) && ! empty( $args['commission'] ) ){
		            /** @var YITH_Commission $commission */
		            $commission = $args['commission'];
		            $vendor = $commission->get_vendor();

		            if ( empty( $vendor->paypal_email ) ) {
			            return array ();
		            }

		            $vendors_to_pay[ $vendor->id ] = $vendor;

		            $vendors_to_pay[ $vendor->id ] = array(
			            'vendor'      => $vendor,
			            'commissions' => array( $commission )
		            );
	            }

	            // get the unpaid commissions for each vendor e get the amount to pay
	            foreach ( $vendors_to_pay as $vendor_id => $args ) {
		            // save the amount to pay for each commission of vendor
		            $vendor      = $args['vendor'];
		            $commissions = $args['commissions'];
		            foreach ( $commissions as $commission ) {
			            if( ! $commission instanceof  YITH_Commission ){
				            $commission_id = $commission;
				            $commission = YITH_Commission( $commission_id );
			            }

			            $data[] = array(
				            'paypal_email' => $vendor->paypal_email,
				            'amount' => round( $commission->get_amount(), 2 ),
				            'request_id' => $commission->id
			            );

			            // save the commissions with other to set 'paid' after success payment
			            $commissions[] = $commission;
		            }
	            }

	            $data['commissions'] = $commissions;
            }

		    return $data;
	    }

        /**
         * Pay method, used to process payment requests
         *
         * @param $payment_detail  array  Array of parameters for the single requests
         * Excepts at least the following parameters for each payment to process
         * [
         *     paypal_email => string (Paypal email of the receiver)
         *     amount => float (Amount to pay to user)
         *     request_id => int (Unique id of the request paid)
         * ]
         *
         * @return array An array holding the status of the operation; it should have at least a boolean status, a verbose status and an array of messages
         * [
         *     status => bool (status of the operation)
         *     verbose_status => string (one between PAYMENT_STATUS_OK and PAYMENT_STATUS_FAIL)
         *     messages => string|array (one or more message describing operation status)
         * ]
         * @since 1.0
         * @author Antonio La Rocca <antonio.larocca@yithemes.it>
         */
        public function pay( $payment_detail ){
            // include required libraries
            require_once( dirname( dirname(__FILE__) ) . '/third-party/PayPal/PayPal.php' );

            // retrieve saved options from panel
            $stored_options = $this->get_gateway_options();
	        $currency = get_woocommerce_currency();

            if ( empty( $stored_options[ 'api_username' ] ) || empty( $stored_options[ 'api_password' ] ) || empty( $stored_options[ 'api_signature' ] ) ){
                return array(
                    'status' => false,
                    'verbose_status' => self::PAYMENT_STATUS_FAIL,
                    'messages' => __( 'Missing required parameters for PayPal configuration', 'yith-woocommerce-product-vendors' )
                );
            }

            $PayPalConfig = array(
                'Sandbox' => ! ( $stored_options[ 'sandbox' ] == 'no' ),
                'APIUsername' => $stored_options['api_username'],
                'APIPassword' => $stored_options['api_password'],
                'APISignature' => $stored_options['api_signature'],
                'PrintHeaders' => true,
                'LogResults' => false,
            );

            $PayPal = new PayPal($PayPalConfig);

            // Prepare request arrays
            $MPFields = array(
                'emailsubject' => $stored_options['payment_mail_subject'], // The subject line of the email that PayPal sends when the transaction is completed.  Same for all recipients.  255 char max.
                'currencycode' => $currency,                               // Three-letter currency code.
                'receivertype' => 'EmailAddress'                           // Indicates how you identify the recipients of payments in this call to MassPay.  Must be EmailAddress or UserID
            );

            $MPItems = array();

            foreach( $payment_detail as $payment ){
                $MPItems[] = array(
                    'l_email' => $payment['paypal_email'],  // Required.  Email address of recipient.  You must specify either L_EMAIL or L_RECEIVERID but you must not mix the two.
                    'l_amt' => $payment['amount'],         // Required.  Payment amount.
                    'l_uniqueid' => $payment['request_id'] // Transaction-specific ID number for tracking in an accounting system.
                );
            }

            $PayPalRequestData = array('MPFields'=>$MPFields, 'MPItems' => $MPItems);
            $PayPalResult = $PayPal->MassPay($PayPalRequestData);

            $errors = array();
            if( $PayPalResult['ACK'] == self::PAYMENT_STATUS_FAIL ){
                foreach( $PayPalResult['ERRORS'] as $error ){
                    $errors[] = $error['L_LONGMESSAGE'];
                }
            }

            return array(
                'status' => $PayPalResult['ACK'] == self::PAYMENT_STATUS_OK,
                'verbose_status' => $PayPalResult['ACK'],
                'messages' => ( $PayPalResult['ACK'] == self::PAYMENT_STATUS_FAIL ) ? implode( "\n", $errors ) : __( 'Payment sent', 'yith-woocommerce-product-vendors' )
            );
        }

        /**
         * Method used to handle notification from paypal server
         *
         * @return void
         * @since 1.0
         * @author Antonio La Rocca <antonio.larocca@yithemes.it>
         */
        public function handle_notification(){

            if( ! $this->is_enabled() ){
                return;
            }

            if ( empty( $_GET[ 'paypal_ipn_response' ] ) ) {
	            return;
            }

	        // include required libraries
	        require( dirname( dirname(__FILE__) ) . '/third-party/IPNListener/ipnlistener.php' );

	        // retrieve saved options from panel
	        $stored_options = $this->get_gateway_options();

	        $listener = new IpnListener();
	        $listener->use_sandbox = ! ( $stored_options[ 'sandbox' ] == 'no' );

	        try {
		        // process IPN request, require validation to PayPal server
		        $listener->requirePostMethod();
		        $verified = $listener->processIpn();

	        } catch (Exception $e) {
		        // fatal error trying to process IPN.
		        die();
	        }

	        // if PayPal says IPN is valid, process content
	        if ( $verified ) {
		        $request_data = $_POST;

		        if( ! isset( $request_data['payment_status'] ) ){
			        die();
		        }

		        // format payment data
		        $payment_data = array();
		        for( $i = 1; array_key_exists( 'status_' . $i, $request_data ); $i++  ){
			        $data_index = array_keys( $request_data );

			        foreach( $data_index as $index ){
				        if( strpos( $index, '_' . $i ) !== false ){
					        $payment_data[ $i ][ str_replace( '_' . $i, '', $index ) ] = $request_data[ $index ];
					        unset( $request_data[ $index ] );
				        }
			        }
		        }

		        $request_data[ 'payment_data' ] = $payment_data;

		        if( ! empty( $payment_data ) ){
			        foreach( $payment_data as $payment ){
				        if( ! isset( $payment['unique_id'] ) ){
					        continue;
				        }

				        $args = array();
				        $args['unique_id'] = $payment['unique_id'];
				        $args['gross'] = $payment['mc_gross'];
				        $args['status'] = $payment['status'];
				        $args['receiver_email'] = $payment['receiver_email'];
				        $args['currency'] = $payment['mc_currency'];
				        $args['txn_id'] = $payment['masspay_txn_id'];

				        // call action to update request status
				        do_action( 'yith_vendors_gateway_notification', $args );
			        }
		        }

	        }

	        die();
        }

	    /**
	     * Get the gateway options
	     *
	     * @return array
	     */
	    public function get_gateway_options() {

		    $api_username  = get_option( $this->gateway . '_api_username' );
			$api_password  = get_option( $this->gateway . '_api_password' );
			$api_signature = get_option( $this->gateway . '_api_signature' );

		    // If empty, get from woocommerce settings
		    if ( empty( $api_username ) && empty( $api_password ) && empty( $api_signature ) ) {
				$gateways = WC()->payment_gateways()->get_available_payment_gateways();
			    if ( isset( $gateways['paypal'] ) ) {
				    /** @var WC_Gateway_Paypal $paypal */
				    $paypal = $gateways['paypal'];

				    $api_username  = $paypal->get_option( 'api_username' );
				    $api_password  = $paypal->get_option( 'api_password' );
				    $api_signature = $paypal->get_option( 'api_signature' );
			    }
		    }

		    $args = array(
			    'sandbox'              => get_option( $this->gateway . '_sandbox' ),
			    'api_username'         => $api_username,
			    'api_password'         => $api_password,
			    'api_signature'        => $api_signature,
			    'payment_mail_subject' => get_option( $this->gateway . '_payment_mail_subject' ),
			    'ipn_notification_url' => site_url() . '/?paypal_ipn_response=true',
		    );

		    $args = wp_parse_args( $args, array() );

		    return $args;
	    }

	    /**
	     * Remove the option where give the ability to vendor to choose the payment method
	     *
	     * @param array $fields
	     *
	     * @return array
	     */
	    public static function remove_vendor_payment_choosing( $fields ) {
	        $payment_method = get_option( 'payment_method', 'choose' );
		    if ( 'choose' != $payment_method ) {
			    unset( $fields['payment_method'] );
		    }

		    return $fields;
	    }

	    /**
	     * Check if the current gateway is enabled or not
	     *
	     * @return bool TRUE if enabled, FALSE otherwise
	     */
	    public function is_enabled(){
		    $enabled = $is_masspay_enabled = false;
		    $gateway_slug = $this->get_id();

		    if( ! empty( $gateway_slug ) ){
			    $default = apply_filters( 'yith_wcmv_is_enable_gateway_default_value', 'no' );
			    $enabled = 'yes' == get_option( "yith_wcmv_enable_{$gateway_slug}_gateway", $default );
			    $is_masspay_enabled = 'masspay' == get_option( 'payment_gateway', 'masspay' );
		    }

		    return $enabled && $is_masspay_enabled;
	    }

	    /**
         * Disable All Gateway features if the integration plugin is disabled
         * @author Andrea Grillo <andrea.grillo@yithemes.com>
         *
	     * @param $available_gateways
	     *
	     * @return mixed
	     */
	    public function disable_all( $available_gateways ){
	        $gateway_id = $this->get_id();

		    if( isset( $available_gateways[ $gateway_id ] ) ){
			    unset( $available_gateways[ $gateway_id ] );
		    }

	        return $available_gateways;
        }
    }
}