<?php
/**
 * Payment Handler Premium class
 *
 * @author  Your Inspiration Themes
 * @package YITH WooCommerce Affiliates
 * @version 1.0.0
 */

/*
 * This file belongs to the YIT Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 */

if ( ! defined( 'YITH_WCAF' ) ) {
	exit;
} // Exit if accessed directly

if ( ! class_exists( 'YITH_WCAF_Payment_Handler_Premium' ) ) {
	/**
	 * WooCommerce Payment Handler Premium
	 *
	 * @since 1.0.0
	 */
	class YITH_WCAF_Payment_Handler_Premium extends YITH_WCAF_Payment_Handler {
		/**
		 * Single instance of the class for each token
		 *
		 * @var \YITH_WCAF_Payment_Handler_Premium
		 * @since 1.0.0
		 */
		protected static $instance = null;

		/**
		 * Payment schedule event
		 *
		 * @var string
		 * @since 1.0.0
		 */
		protected $_payment_type = 'manually';

		/**
		 * Gateway for automatic payments
		 *
		 * @var string
		 * @since 1.0.0
		 */
		protected $_payment_default_gateway = null;

		/**
		 * Date for day-based automatic payments
		 *
		 * @var int
		 * @since 1.0.0
		 */
		protected $_payment_date = 15;

		/**
		 * Whether to pay only commissions older than a certan number of days
		 *
		 * @var bool
		 * @since 1.2.2
		 */
		protected $_pay_only_old_commissions = false;

		/**
		 * Number of days before a brand new commission should be paid
		 *
		 * @var int
		 * @since 1.2.2
		 */
		protected $_payment_commission_age = 15;

		/**
		 * Threshold for earnings-based automatic payments
		 *
		 * @var float
		 * @since 1.0.0
		 */
		protected $_payment_threshold = 50;

		/**
		 * Registered gateways
		 *
		 * @var mixed
		 * @since 1.0.0
		 */
		private $_available_gateways = array();

		/**
		 * Constructor method
		 *
		 * @return \YITH_WCAF_Payment_Handler_Premium
		 * @since 1.0.0
		 */
		public function __construct() {
			// init class
			$this->_retrieve_options();
			parent::__construct();
			$this->init_available_gateways();

			// handle gateway notification
			add_action( 'yith_wcaf_ipn_received', array( $this, 'handle_notification' ), 10, 1 );

			// setup schedules
			add_action( 'wp', array( $this, 'pay_affiliates_setup_schedule' ) );
			add_action( 'update_option_yith_wcaf_payment_type', array( $this, 'pay_commissions_delete_schedule' ) );

			// automatic payment actions
			add_action( 'yith_wcaf_commission_status_pending', array( $this, 'pay_on_threshold_reached' ), 15, 1 );
			add_action( 'pay_commissions_action_schedule', array( $this, 'pay_commissions' ) );

			// add commissions panel handling
			add_action( 'admin_action_yith_wcaf_complete_payment', array( $this, 'handle_payments_panel_actions' ) );

			add_filter( 'yith_wcaf_general_settings', array( $this, 'filter_general_settings' ) );
		}

		/* === INIT METHODS === */

		/**
		 * Retrieve options for payment from db
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function _retrieve_options() {
			$this->_payment_type            = get_option( 'yith_wcaf_payment_type', $this->_payment_type );
			$this->_payment_default_gateway = get_option( 'yith_wcaf_payment_default_gateway', $this->_payment_default_gateway );
			$this->_payment_date            = get_option( 'yith_wcaf_payment_date', $this->_payment_date );
			$this->_pay_only_old_commissions  = get_option( 'yith_wcaf_pay_only_old_commissions', $this->_pay_only_old_commissions );
			$this->_payment_commission_age  = get_option( 'yith_wcaf_payment_commission_age', $this->_payment_commission_age );
			$this->_payment_threshold       = get_option( 'yith_wcaf_payment_threshold', $this->_payment_threshold );
		}

		/**
		 * Filter general settings, to add notification settings
		 *
		 * @param $settings mixed Original settings array
		 * @return mixed Filtered settings array
		 * @since 1.0.0
		 */
		public function filter_general_settings( $settings ) {
			$settings_options = $settings['settings'];
			$before_index = 'commission-options-end';
			$before_index_position = array_search( $before_index, array_keys( $settings_options ) );

			$settings_options_chunk_1 = array_slice( $settings_options, 0, $before_index_position + 1 );
			$settings_options_chunk_2 = array_slice( $settings_options, $before_index_position + 1, count( $settings_options ) );

			$gateways = $this->get_available_gateways();
			$gateway_options = array();
			$gateway_options['none'] = __( 'None', 'yith-woocommerce-affiliates' );

			if( ! empty( $gateways ) ){
				foreach( $gateways as $id=> $gateway ){
					$gateway_options[ $id ] = $gateway['label'];
				}
			}

			$payment_settings = array(
				'payment-options' => array(
					'title' => __( 'Payment', 'yith-woocommerce-affiliates' ),
					'type' => 'title',
					'desc' => '',
					'id' => 'yith_wcaf_payment_options'
				),

				'payment-type' => array(
					'title' => __( 'Payment type', 'yith-woocommerce-affiliates' ),
					'type' => 'select',
					'desc' => __( 'Choose payment mode to pay commissions to your affiliates', 'yith-woocommerce-affiliates' ),
					'id' => 'yith_wcaf_payment_type',
					'options' => array(
						'manually' => __( 'Manually', 'yith-woocommerce-affiliates' ),
						'automatically_on_threshold' => __( 'Automatically when reaching a threshold', 'yith-woocommerce-affiliates' ),
						'automatically_on_date' => __( 'Automatically on a specific month date', 'yith-woocommerce-affiliates' ),
						'automatically_on_both' => __( 'Automatically on a specific month date, if a specific threshold is reached', 'yith-woocommerce-affiliates' ),
						'automatically_every_day' => __( 'Automatically every day', 'yith-woocommerce-affiliates' ) // @since 1.2.2
					),
					'default' => 'manually',
					'desc_tip' => true
				),

				'payment-default-gateway' => array(
					'title' => __( 'Default gateway', 'yith-woocommerce-affiliates' ),
					'type' => 'select',
					'desc' => __( 'Choose a gateway to execute automatic payments; if you select "none" payments will only be registered but no payment requests will be issued to the gateway', 'yith-woocommerce-affiliates' ),
					'id' => 'yith_wcaf_payment_default_gateway',
					'options' => $gateway_options,
					'default' => 'none',
					'desc_tip' => true,
					'css' => 'min-width: 300px'
				),

				'payment-date' => array(
					'title' => __( 'Payment date', 'yith-woocommerce-affiliates' ),
					'type' => 'number',
					'desc' => __( 'Choose a day of the month for commission payments', 'yith-woocommerce-affiliates' ),
					'id' => 'yith_wcaf_payment_date',
					'css' => 'max-width: 50px;',
					'default' => 15,
					'custom_attributes' => array(
						'min' => 1,
						'max' => 28,
						'step' => 1
					),
					'desc_tip' => true
				),

				'payment-threshold' => array(
					'title' => __( 'Payment threshold', 'yith-woocommerce-affiliates' ),
					'type' => 'number',
					'desc' => __( 'Choose a minimum amount that an affiliate must earn before a payment is issued', 'yith-woocommerce-affiliates' ),
					'id' => 'yith_wcaf_payment_threshold',
					'css' => 'min-width: 50px;',
					'custom_attributes' => array(
						'min' => 0,
						'max' => 500,
						'step' => 'any'
					),
					'default' => '50',
					'desc_tip' => true
				),

				'payment-pay-only-old-commissions' => array(
					'title' => __( 'Pay only commissions older than', 'yith-woocommerce-affiliates' ), // @since 1.2.2
					'type' => 'checkbox',
					'desc' => __( 'Choose whether plugin should automatically pay only commissions older than a certain number of days', 'yith-woocommerce-affiliates' ), // @since 1.2.2
					'id' => 'yith_wcaf_payment_pay_only_old_commissions',
					'default' => 'no',
				),

				'payment-commission-age' => array(
					'title' => __( 'Commission age', 'yith-woocommerce-affiliates' ), // @since 1.2.2
					'type' => 'number',
					'desc' => __( 'Choose the minimum amount of days that should pass before automatically pay a commission', 'yith-woocommerce-affiliates' ), // @since 1.2.2
					'id' => 'yith_wcaf_payment_commission_age',
					'css' => 'max-width: 50px;',
					'default' => 15,
					'custom_attributes' => array(
						'min' => 1,
						'step' => 1
					),
					'desc_tip' => true
				),

				'payment-pending-notify-admin' => array(
					'title' => __( 'Notify admin', 'yith-woocommerce-affiliates' ),
					'type' => 'checkbox',
					'desc' => sprintf( '%s <a href="%s">%s</a>', __( 'Notify admin when a new payment is issued; customize email on', 'yith-woocommerce-affiliates' ), esc_url( add_query_arg( array( 'page' => 'wc-settings', 'tab' => 'email', 'section' => 'yith_wcaf_admin_paid_commission_email' ), admin_url( 'admin.php' ) ) ), __( 'WooCommerce Settings Page', 'yith-woocommerce-affiliates' ) ),
					'id' => 'yith_wcaf_payment_pending_notify_admin',
					'default' => 'yes'
				),

				'payment-options-end' => array(
					'type'  => 'sectionend',
					'id'    => 'yith_wcaf_payment_options'
				),
			);

			$settings['settings'] = array_merge(
				$settings_options_chunk_1,
				$payment_settings,
				apply_filters( 'yith_wcaf_gateway_options', array() ),
				$settings_options_chunk_2
			);

			return $settings;
		}

		/* === HELPER METHODS === */

		/**
		 * Register payments for a bunch of commissions; will create different mass pay foreach affiliate referred by commissions
		 *
		 * @param $commissions_id       array|int Array of commissions to pay IDs, or single commission id
		 * @param $proceed_with_payment bool Whether to call gateways to pay, or just register payments
		 * @param $gateway              bool|string Gateway to use for payments; default value (false) force to use default gateway
		 *
		 * @return mixed Array with payment status, when \$proceed_with_payment is enabled; false otherwise
		 */
		public function register_payment( $commissions_id, $proceed_with_payment = true, $gateway = false ) {
			// if no commission passed, return
			if ( empty( $commissions_id ) ) {
				return array(
					'status'         => false,
					'messages'       => __( 'You have to select at least one commission', 'yith-woocommerce-affiliates' ),
					'can_be_paid'    => array(),
					'cannot_be_paid' => array()
				);
			}

			// if single commission id provided, convert it to array
			if ( ! is_array( $commissions_id ) ) {
				$commissions_id = (array) $commissions_id;
			}

			$payments       = array();
			$to_pay         = array();
			$can_be_paid    = array();
			$cannot_be_paid = array();

			foreach ( $commissions_id as $id ) {
				$commission = YITH_WCAF_Commission_Handler()->get_commission( $id );

				// if can't find commission, continue
				if ( ! $commission ) {
					continue;
				}

				$affiliate_id = $commission['affiliate_id'];
				$affiliate    = YITH_WCAF_Affiliate_Handler()->get_affiliate_by_id( $affiliate_id, true );

				// if can't find affiliate, continue
				if ( ! $affiliate ) {
					continue;
				}

				// if current status doesn't allow payments, continue
				$available_status_change = YITH_WCAF_Commission_Handler()->get_available_status_change( $id );
				if( ! in_array( 'pending-payment', $available_status_change )  && ! in_array( $commission['status'], YITH_WCAF_Commission_Handler_Premium()->payment_status ) ){
					continue;
				}

				$payment_email = $affiliate['payment_email'];

				// commission can be paid; switch status to pending-payment
				$commission_status = $this->get_commission_status_after_payment( $gateway );
				YITH_WCAF_Commission_Handler_Premium()->change_commission_status( $id, $commission_status['status'], $commission_status['message'] );

				// register commissions sent to payment processor for payment
				$can_be_paid[] = $id;

				// if there is no payment registered for the affiliate, set one
				if ( ! isset( $payments[ $affiliate_id ] ) ) {
					$payments[ $affiliate_id ]                  = array();
					$payments[ $affiliate_id ]['affiliate_id']  = $affiliate_id;
					$payments[ $affiliate_id ]['payment_email'] = $payment_email;
					$payments[ $affiliate_id ]['gateway']       = $gateway ? $gateway : '';
					$payments[ $affiliate_id ]['amount']        = 0;
					$payments[ $affiliate_id ]['commissions']   = array();
				}

				$payments[ $affiliate_id ]['commissions'][] = $commission;
				$payments[ $affiliate_id ]['amount'] += doubleval( $commission['amount'] );
			}

			// register payments
			if ( ! empty( $payments ) ) {
				foreach ( $payments as $payment ) {
					$commissions  = $payment['commissions'];
					$payment_args = $payment;
					unset( $payment_args['commissions'] );

					$payment_id = $this->add( $payment_args, $commissions );

					$proceed_with_payment = apply_filters( 'yith_wcaf_proceed_with_payment', $proceed_with_payment, $payment_id, $payment );

					if ( $payment_id && $proceed_with_payment ) {
						$to_pay[] = $payment_id;
					}
				}
			}

			// register commissions that cannot be sent to payment processor for payment
			$cannot_be_paid = array_diff( $commissions_id, $can_be_paid );

			// proceed with payments
			if ( ! empty( $to_pay ) ) {
				$res = $this->pay( $gateway, $to_pay );
				return array(
					'status'         => $res['status'],
					'message'        => $res['message'],
					'can_be_paid'    => $can_be_paid,
					'cannot_be_paid' => $cannot_be_paid
				);
			}

			return array(
				'status'         => true,
				'messages'       => __( 'Payment correctly registered', 'yith-woocommerce-affiliates' ),
				'can_be_paid'    => $can_be_paid,
				'cannot_be_paid' => $cannot_be_paid
			);
		}

		/**
		 * Register payment for all pending commission of an affiliate; will create different mass pay foreach affiliate referred by commissions
		 *
		 * @param $affiliate_id         int Affiliate id
		 * @param $proceed_with_payment bool Whether to call gateways to pay, or just register payments
		 * @param $gateway              bool|string Gateway to use for payments; default value (false) force to use default gateway
		 * @param $apply_filters        bool Whether to apply any kind of filters to the query that retrieves commissions to pay (affiliate_id and commission status will be filtered anyway)
		 *
		 * @return mixed Array with payment status, when \$proceed_with_payment is enabled; false otherwise
		 */
		public function pay_all_affiliate_commissions( $affiliate_id, $proceed_with_payment = true, $gateway = false, $apply_filters = false ) {
			$args = array(
				'affiliate_id' => $affiliate_id,
				'status'       => 'pending'
			);

			if( $apply_filters ){
				if( $this->_pay_only_old_commissions && ! empty( $this->_payment_commission_age ) ){
					$current_time = time();
					$threshold_time = $current_time - ( $this->_payment_commission_age * DAY_IN_SECONDS );

					$args['interval'] = array(
						'end_date' => date( 'Y-m-d H:i:s', $threshold_time ),
					);
				}

				$args = apply_filters( 'yith_wcaf_pay_all_affiliates_commissions_artgs', $args );
			}

			$commissions = YITH_WCAF_Commission_Handler()->get_commissions( $args );

			if ( empty( $commissions ) ) {
				return array(
					'status' => false,
					'message' => __( 'Affiliate needs to have at least one unpaid commission', 'yith-woocommerce-affiliates' ),
					'can_be_paid'    => array(),
					'cannot_be_paid' => array()
				);
			}

			$commissions_ids = wp_list_pluck( $commissions, 'ID' );
			return $this->register_payment( $commissions_ids, $proceed_with_payment, $gateway );
		}

		/**
		 * Register payments for all commissions older then a specific timestamp; will create different mass pay foreach affiliate referred by commissions
		 *
		 * @param $threshold_timestamp  int Timestamp that should be used as threshold; older pending commissions will be paid
		 * @param $proceed_with_payment bool Whether to call gateways to pay, or just register payments
		 * @param $gateway              bool|string Gateway to use for payments; default value (false) force to use default gateway
		 *
		 * @return mixed Array with payment status, when \$proceed_with_payment is enabled; false otherwise
		 */
		public function pay_all_commissions_older_than( $threshold_timestamp, $proceed_with_payment = true, $gateway = false ) {
			$commissions = YITH_WCAF_Commission_Handler()->get_commissions( array(
				'status'       => 'pending',
				'interval'     => array(
					'end_date' => date( 'Y-m-d H:i:s', $threshold_timestamp ),
				),
			) );

			if ( empty( $commissions ) ) {
				return array(
					'status' => false,
					'message' => __( 'Affiliate needs to have at least one unpaid commission', 'yith-woocommerce-affiliates' ),
					'can_be_paid'    => array(),
					'cannot_be_paid' => array()
				);
			}

			$commissions_ids = wp_list_pluck( $commissions, 'ID' );
			return $this->register_payment( $commissions_ids, $proceed_with_payment, $gateway );
		}

		/* === GATEWAYS HANDLING METHODS === */

		/**
		 * Init available gateways
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function init_available_gateways() {
			$this->_available_gateways = apply_filters(
				'yith_wcaf_available_gateways',
				array_merge(
					$this->_available_gateways,
					array(
						'paypal' => array(
							'path'     => YITH_WCAF_INC . 'gateways/class.yith-wcaf-paypal-gateway.php',
							'label'    => __( 'PayPal', 'yith-woocommerce-affiliates' ),
							'class'    => 'YITH_WCAF_Paypal_Gateway',
							'mass_pay' => true
						)
					),
					! class_exists( 'YITH_Funds' ) ? array() : array(
						'funds' => array(
							'path'     => YITH_WCAF_INC . 'gateways/class.yith-wcaf-yith-ywf-gateway.php',
							'label'    => __( 'Account Funds', 'yith-woocommerce-affiliates' ),
							'class'    => 'YITH_WCAF_YITH_YWF',
							'mass_pay' => true
						)
					)
				)
			);

			// include gateways files
			if ( ! empty( $this->_available_gateways ) ) {
				foreach ( $this->_available_gateways as $gateway_id => $gateway_info ) {
					require_once( apply_filters( 'yith_wcaf_gateway_inclusion_path', $gateway_info['path'] ) );

					if ( function_exists( $gateway_info['class'] ) ) {
						$gateway_class = $gateway_info['class'];
						$gateway_class();
					}
				}
			}
		}

		/**
		 * Pay a payment instance previously created
		 *
		 * @param $gateway           string A valid gateway slug
		 * @param $payment_instances int|array Payment id(s)
		 *
		 * @return bool|mixed Payment status; false on failure
		 */
		public function pay( $gateway, $payment_instances ) {
			// if not gateway is sent, mark as completed
			if( ! $gateway ){
				if( ! empty( $payment_instances ) ){
					foreach( $payment_instances as $instance ){
						$this->change_payment_status( $instance, 'completed' );
						do_action( 'yith_wcaf_payment_sent', $instance );
					}
					do_action( 'yith_wcaf_payments_sent', (array) $payment_instances );
				}

				return array(
					'status'   => true,
					'messages' => __( 'Payments registered correctly', 'yith-woocommerce-affiliates' )
				);
			}

			// if not a registered gateway, return false
			if ( ! in_array( $gateway, array_keys( $this->_available_gateways ) ) ) {
				return array(
					'status'   => false,
					'messages' => __( 'No gateway found with the specified ID', 'yith-woocommerce-affiliates' )
				);
			}

			$gateway_class = $this->_available_gateways[ $gateway ]['class'];

			// if does not exist a singleton instance of the gateway, return false
			if ( ! function_exists( $gateway_class ) ) {
				return array(
					'status'   => false,
					'messages' => __( 'Gateway class doesn\'t match required structure; missing singleton access function', 'yith-woocommerce-affiliates' )
				);
			}

			// if does not exists method pay on gateway object, return false
			if ( ! method_exists( $gateway_class(), 'pay' ) ) {
				return array(
					'status'   => false,
					'messages' => __( 'Gateway class doesn\'t match required structure; missing pay() method', 'yith-woocommerce-affiliates' )
				);
			}

			// return payment status
			$res = $gateway_class()->pay( $payment_instances );

			if ( $res['status'] ) {
				do_action( 'yith_wcaf_payments_sent', (array) $payment_instances );
			}

			return $res;
		}

		/**
		 * Handle IPN notification (gateway should call specific action to trigger this method)
		 *
		 * @param $payment_detail mixed Payment details received by Gateway
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function handle_notification( $payment_detail ) {
			$request_id = $payment_detail['unique_id'];

			$payment = $this->get_payment( $request_id );

			if ( ! $payment ) {
				return;
			}

			$status = $payment['status'];

			if ( $status != 'pending' ) {
				return;
			}

			if ( $payment_detail['status'] == 'Completed' ) {
				$new_status = 'completed';
			} elseif ( in_array( $payment_detail['status'], array( 'Failed', 'Returned', 'Reversed', 'Blocked' ) ) ) {
				$new_status = 'cancelled';
			} else {
				$new_status = 'pending';
			}

			if ( $new_status != 'pending' ) {
				$this->change_payment_status( $request_id, $new_status );

				if ( $new_status == 'completed' ) {
					$this->update( $request_id, array(
						'transaction_key' => $payment_detail['txn_id']
					) );
				}
			}
		}

		/**
		 * Returns a list of available gateways
		 *
		 * @return mixed List of available gateways
		 * @since 1.0.0
		 */
		public function get_available_gateways() {
			return $this->_available_gateways;
		}

		/**
		 * Returns readable version of gateway name
		 *
		 * @param $gateway string Gateway unique ID
		 *
		 * @return string Human friendly version of gateway name
		 * @since 1.0.0
		 */
		public function get_readable_gateway( $gateway ) {
			$label = '';
			if ( isset( $this->_available_gateways[ $gateway ] ) ) {
				$label = $this->_available_gateways[ $gateway ]['label'];
			}

			return apply_filters( "yith_wcaf_{$gateway}_payment_gateway_name", $label );
		}

		/* === SCHEDULED ACTIONS METHODS === */

		/**
		 * Setup scheduled events for affiliates payments
		 *
		 * From 1.2.2 it schedules a single daily event (pay_commissions_action_schedule), that handles all kind of automatic payments
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function pay_affiliates_setup_schedule() {
			if ( in_array( $this->_payment_type, array( 'automatically_on_date', 'automatically_on_both', 'automatically_every_day' ) ) && ! wp_next_scheduled( 'pay_commissions_action_schedule' ) ) {
				wp_schedule_event( time(), 'daily', 'pay_commissions_action_schedule' );
			}
		}

		/**
		 * Delete schedule when date option is changed
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function pay_commissions_delete_schedule() {
			wp_clear_scheduled_hook( 'pay_commissions_action_schedule' );
		}

		/**
		 * When a commission switch to pending, check if threshold reached, and eventually pay affiliate commissions
		 *
		 * @param $commission_id int Commission id of the last commission
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function pay_on_threshold_reached( $commission_id ) {
			if ( $this->_payment_type != 'automatically_on_threshold' || empty( $this->_payment_threshold ) || empty( $this->_payment_default_gateway ) ) {
				return;
			}

			$pay     = $this->_payment_default_gateway != 'none';
			$gateway = $this->_payment_default_gateway != 'none' ? $this->_payment_default_gateway : false;

			$commission = YITH_WCAF_Commission_Handler()->get_commission( $commission_id );

			if ( ! $commission ) {
				return;
			}

			$affiliate = YITH_WCAF_Affiliate_Handler()->get_affiliate_by_id( $commission['affiliate_id'] );

			if ( ! $affiliate ) {
				return;
			}

			if ( $affiliate['balance'] > $this->_payment_threshold ) {
				$this->pay_all_affiliate_commissions( $affiliate['ID'], $pay, $gateway, true );
			}
		}

		/**
		 * Pay all pending commission for all affiliates (should be executed once a month; if also threshold is enabled, check it before pay affilaite)
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function pay_on_date_reached() {
			if ( ( $this->_payment_type != 'automatically_on_date' && $this->_payment_type != 'automatically_on_both' ) || empty( $this->_payment_date ) || empty( $this->_payment_default_gateway ) ) {
				return;
			}

			$pay     = $this->_payment_default_gateway != 'none';
			$gateway = $this->_payment_default_gateway != 'none' ? $this->_payment_default_gateway : false;

			$affiliates = YITH_WCAF_Affiliate_Handler()->get_affiliates();

			if ( ! $affiliates ) {
				return;
			}

			foreach ( $affiliates as $affiliate ) {
				if ( $this->_payment_type == 'automatically_on_both' && ( empty( $this->_payment_threshold ) || $affiliate['earnings'] <= $this->_payment_threshold ) ) {
					continue;
				}

				$this->pay_all_affiliate_commissions( $affiliate['ID'], $pay, $gateway, true );
			}
		}

		/**
		 * Execute necessary operations for affiliate payment, during scheduled action
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function pay_commissions() {
			// check whether there is a default gateway
			if( empty( $this->_payment_default_gateway ) ){
				return;
			}

			// check whether _payment_type is a valid schedule
			if( ! in_array( $this->_payment_type, array( 'automatically_on_date', 'automatically_on_both', 'automatically_every_day' ) ) ){
				return;
			}

			// check whether user supplied a valid _payment_date, if _payment_type occurs on a specific date
			if ( ( $this->_payment_type == 'automatically_on_date' || $this->_payment_type == 'automatically_on_both' ) && empty( $this->_payment_date ) ) {
				return;
			}

			// check whether we're in the correct _payment_date, if _payment_type occurs on a specific date
			if( ( $this->_payment_type == 'automatically_on_date' || $this->_payment_type == 'automatically_on_both' ) ){
				$current_day = date( 'j' );

				if( $current_day != $this->_payment_date ){
					return;
				}
			}

			$pay     = $this->_payment_default_gateway != 'none';
			$gateway = $this->_payment_default_gateway != 'none' ? $this->_payment_default_gateway : false;

			$affiliates = YITH_WCAF_Affiliate_Handler()->get_affiliates();

			if ( ! $affiliates ) {
				return;
			}

			foreach ( $affiliates as $affiliate ) {
				if ( $this->_payment_type == 'automatically_on_both' && ( empty( $this->_payment_threshold ) || $affiliate['earnings'] <= $this->_payment_threshold ) ) {
					continue;
				}

				$this->pay_all_affiliate_commissions( $affiliate['ID'], $pay, $gateway, true );
			}
		}

		/* === PANEL PAYMENTS METHODS === */

		/**
		 * Print payment panel
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function print_payment_panel() {
			// define variables to use in template
			$payment_id = isset( $_REQUEST['payment_id'] ) ? $_REQUEST['payment_id'] : false;

			if ( ! empty( $payment_id ) && $payment = $this->get_payment( $payment_id ) ) {
				// update payment email, if required
				if ( isset( $_POST['_payment_email'] ) && is_email( $_POST['_payment_email'] ) ) {
					$payment_email = trim( $_POST['_payment_email'] );

					$this->update( $payment_id, array( 'payment_email' => $payment_email ) );
					$payment['payment_email'] = $payment_email;
				}

				// retrieve affiliate user
				$user_info = get_userdata( $payment['user_id'] );
				$user      = get_user_by( 'id', $payment['user_id'] );

				$user_name = '';
				if ( $user_info->first_name || $user_info->last_name ) {
					$user_name .= esc_html( ucfirst( $user_info->first_name ) . ' ' . ucfirst( $user_info->last_name ) );
				} else {
					$user_name .= esc_html( ucfirst( $user_info->display_name ) );
				}

				$user_email = $user_info->user_email;

				// retrieve gateway specifications and commissions
				$payment_label     = $this->get_readable_gateway( $payment['gateway'] );
				$payment_trans_key = isset( $payment['transaction_key'] ) ? $payment['transaction_key'] : __( 'N/A', 'yith-woocommerce-affiliates' );
				$commissions       = $this->get_payment_commissions( $payment_id );

				// retrieve notes
				$payment_notes = $this->get_commission_notes( $payment_id );

				// retrieve available payment actions
				$available_payment_actions = array();
				$gateways = $this->get_available_gateways();

				if( $payment['status'] == 'on-hold' && ! empty( $gateways ) ) {
					foreach( $gateways as $id => $gateway ){
						$available_payment_actions[ 'pay_via_' . $id ] =  sprintf( __( 'Pay via %s', 'yith-woocommerce-affiliates' ), $gateway['label'] );
					}
				}

				// require rate panel template
				include( YITH_WCAF_DIR . 'templates/admin/payment-panel-detail.php' );
			} else {
				// prepare user rates table items
				$payments_table = new YITH_WCAF_Payments_Table_Premium();
				$payments_table->prepare_items();

				// require rate panel template
				include( YITH_WCAF_DIR . 'templates/admin/payment-panel-table.php' );
			}
		}

		/**
		 * Process bulk action for current view
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function process_bulk_actions() {
			if ( ! empty( $_REQUEST['payments'] ) ) {
				$current_action = isset( $_REQUEST['action'] ) ? $_REQUEST['action'] : '';
				$current_action = ( empty( $current_action ) && isset( $_REQUEST['action2'] ) ) ? $_REQUEST['action2'] : $current_action;
				$redirect       = esc_url_raw( add_query_arg( array(
					'page' => 'yith_wcaf_panel',
					'tab'  => 'payments'
				), admin_url( 'admin.php' ) ) );

				if( empty( $current_action ) ){
					return;
				}

				switch( $current_action ){
					case 'switch-to-completed':
						foreach( $_REQUEST['payments'] as $payment_id ){
							$res = $this->change_payment_status( $payment_id, 'completed' );
							$redirect = esc_url_raw( add_query_arg( 'commission_status_change', $res, $redirect ) );
						}
						break;
					case 'switch-to-on-hold':
						foreach( $_REQUEST['payments'] as $payment_id ){
							$res = $this->change_payment_status( $payment_id, 'on-hold' );
							$redirect = esc_url_raw( add_query_arg( 'commission_status_change', $res, $redirect ) );
						}
						break;
					case 'delete':
						foreach( $_REQUEST['payments'] as $payment_id ){
							$res = $this->delete( $payment_id );
							$redirect = esc_url_raw( add_query_arg( 'commission_deleted', $res, $redirect ) );
						}
						break;
					default:
						// handles payment actions
						$matches = array();

						if ( preg_match( '^pay_via_([a-zA-Z_-]*)$^', $current_action, $matches ) ) {
							// group payments by gateway
							$to_pay  = array();
							$gateway = $matches[1];

							if ( in_array( $gateway, array_keys( $this->_available_gateways ) ) ) {
								foreach ( $_REQUEST['payments'] as $payment_id ) {
									$payment = $this->get_payment( $payment_id );

									if ( ! $payment || $payment['status'] != 'on-hold' ) {
										continue;
									}

									if ( ! isset( $to_pay[ $gateway ] ) ) {
										$to_pay[ $gateway ] = array();
									}

									$to_pay[ $gateway ][] = $payment_id;
								}
							}

							if ( ! empty( $to_pay ) ) {
								foreach ( $to_pay as $gateway => $payment_instances ) {
									$payments = implode( '_', $payment_instances );
									$res      = $this->pay( $gateway, $payment_instances );

									if ( ! $res['status'] ) {
										$errors   = is_array( $res['messages'] ) ? implode( ',', $res['messages'] ) : $res['messages'];
										$redirect = add_query_arg( array(
											'payment_failed'            => true,
											"payment_error_{$payments}" => urlencode( $errors )
										), $redirect );
									} else {
										$redirect = add_query_arg( array(
											'payment_success'             => true,
											"payment_success_{$payments}" => true
										), $redirect );
									}
								}
							}
						}
						break;
				}

				if( isset( $_GET['payment_id'] ) ){
					$redirect = add_query_arg( 'payment_id', intval( $_GET['payment_id'] ), $redirect );
				}

				wp_redirect( esc_url_raw( $redirect ) );
				die();
			}
		}

		/**
		 * Process action to complete payments on payment panel
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function handle_payments_panel_actions() {
			$payment_id = isset( $_REQUEST['payment_id'] ) ? $_REQUEST['payment_id'] : 0;
			$gateway_id = isset( $_REQUEST['gateway'] ) ? $_REQUEST['gateway'] : '';
			$redirect   = esc_url_raw( add_query_arg( array(
				'page' => 'yith_wcaf_panel',
				'tab'  => 'payments'
			), admin_url( 'admin.php' ) ) );

			if ( ! $payment_id || ! $gateway_id || ! in_array( $gateway_id, array_keys( $this->_available_gateways ) ) ) {
				wp_redirect( $redirect );
				die();
			}

			$payment = $this->get_payment( $payment_id );

			if ( ! $payment ) {
				wp_redirect( $redirect );
				die();
			}

			$res = $this->pay( $gateway_id, $payment_id );

			if ( ! $res['status'] ) {
				$errors   = is_array( $res['messages'] ) ? implode( ',', $res['messages'] ) : $res['messages'];
				$redirect = esc_url_raw( add_query_arg( array(
					'payment_failed'              => true,
					"payment_error_{$payment_id}" => urlencode( $errors )
				), $redirect ) );
			} else {
				$redirect = esc_url_raw( add_query_arg( array(
					'payment_success'               => true,
					"payment_success_{$payment_id}" => true
				), $redirect ) );
			}

			wp_redirect( $redirect );
			die();
		}

		/**
		 * Returns single instance of the class
		 *
		 * @return \YITH_WCAF_Payment_Handler_Premium
		 * @since 1.0.0
		 */
		public static function get_instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self;
			}

			return self::$instance;
		}
	}
}

/**
 * Unique access to instance of YITH_WCAF_Payment_Handler_Premium class
 *
 * @return \YITH_WCAF_Payment_Handler_Premium
 * @since 1.0.0
 */
function YITH_WCAF_Payment_Handler_Premium() {
	return YITH_WCAF_Payment_Handler_Premium::get_instance();
}