<?php
/**
 * Affiliate Handler Premium class
 *
 * @author Your Inspiration Themes
 * @package YITH WooCommerce Affiliates
 * @version 1.0.0
 */

/*
 * This file belongs to the YIT Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 */

if ( ! defined( 'YITH_WCAF' ) ) {
	exit;
} // Exit if accessed directly

if ( ! class_exists( 'YITH_WCAF_Affiliate_Handler_Premium' ) ) {
	/**
	 * WooCommerce Affiliate Handler Premium
	 *
	 * @since 1.0.0
	 */
	class YITH_WCAF_Affiliate_Handler_Premium extends YITH_WCAF_Affiliate_Handler {

		/**
		 * Single instance of the class for each token
		 *
		 * @var \YITH_WCAF_Affiliate_Handler_Premium
		 * @since 1.0.0
		 */
		protected static $instance = null;

		/**
		 * Constructor method
		 *
		 * @return \YITH_WCAF_Affiliate_Handler_Premium
		 * @since 1.0.0
		 */
		public function __construct() {
			parent::__construct();

			add_action( 'admin_action_yith_wcaf_pay_commissions', array( $this, 'handle_pay_commissions_panel_actions' ) );
			add_filter( 'yith_wcaf_general_settings', array( $this, 'filter_general_settings' ) );
		}

		/**
		 * Filter general settings, to add notification settings
		 *
		 * @param $settings mixed Original settings array
		 * @return mixed Filtered settings array
		 * @since 1.0.0
		 */
		public function filter_general_settings( $settings ) {
			$settings_options = $settings['settings'];
			$before_index = 'referral-registration-form';
			$before_index_position = array_search( $before_index, array_keys( $settings_options ) );

			$settings_options_chunk_1 = array_slice( $settings_options, 0, $before_index_position + 1 );
			$settings_options_chunk_2 = array_slice( $settings_options, $before_index_position + 1, count( $settings_options ) );

			$affiliate_settings = array(
				'referral-registration-auto-enable' => array(
					'title' => __( 'Auto enable affiliates', 'yith-woocommerce-affiliates' ),
					'type' => 'checkbox',
					'desc' => __( 'Auto enable affiliates on registration', 'yith-woocommerce-affiliates' ),
					'id' => 'yith_wcaf_referral_registration_auto_enable',
					'default' => 'yes'
				),
			);

			$settings['settings'] = array_merge(
				$settings_options_chunk_1,
				$affiliate_settings,
				$settings_options_chunk_2
			);

			$settings_options = $settings['settings'];

			$before_index = 'referral-registration-show-surname-field';
			$before_index_position = array_search( $before_index, array_keys( $settings_options ) );

			$settings_options_chunk_1 = array_slice( $settings_options, 0, $before_index_position + 1 );
			$settings_options_chunk_2 = array_slice( $settings_options, $before_index_position + 1, count( $settings_options ) );

			$affiliate_settings = array(
				'referral-registration-notify-admin' => array(
					'title' => __( 'Notify admin', 'yith-woocommerce-affiliates' ),
					'type' => 'checkbox',
					'desc' => sprintf( '%s <a href="%s">%s</a>', __( 'Notify admin of a new affiliate registration; customize email on', 'yith-woocommerce-affiliates' ), esc_url( add_query_arg( array( 'page' => 'wc-settings', 'tab' => 'email', 'section' => 'yith_wcaf_admin_new_affiliate_email' ), admin_url( 'admin.php' ) ) ), __( 'WooCommerce Settings Page', 'yith-woocommerce-affiliates' ) ),
					'id' => 'yith_wcaf_referral_registration_notify_admin',
					'default' => 'yes'
				),
			);

			$settings['settings'] = array_merge(
				$settings_options_chunk_1,
				$affiliate_settings,
				$settings_options_chunk_2
			);

			return $settings;
		}

		/* === FORM HANDLER METHODS === */

		/**
		 * Flag a registered user as an affiliates
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function become_an_affiliate() {
			if( isset( $_REQUEST['become_an_affiliate'] ) && $_REQUEST['become_an_affiliate'] == 1 ){
				if( is_user_logged_in() ){
					$auto_enable = 'yes' == get_option( 'yith_wcaf_referral_registration_auto_enable' );
					$customer_id = get_current_user_id();
					$affiliates = $this->get_affiliates( array( 'user_id' => $customer_id ) );
					$affiliate = isset( $affiliates[0] ) ? $affiliates[0] : false;

					if( ! $affiliate ){
						$validation_error = new WP_Error();
						$validation_error = apply_filters( 'yith_wcaf_process_become_an_affiliate_errors', $validation_error, $customer_id );

						if ( $validation_error->get_error_code() ) {
							wc_add_notice( $validation_error->get_error_message(), 'error' );
						}
						else {
							$id = $this->add( array( 'user_id' => $customer_id, 'enabled' => $auto_enable, 'token'   => $this->get_default_user_token( $customer_id ) ) );

							if ( $id ) {
								wc_add_notice( __( 'Your request has been processed correctly', 'yith-woocommerce-affiliates' ) );

								// trigger new affiliate action
								do_action( 'yith_wcaf_new_affiliate', $id );
							} else {
								wc_add_notice( __( 'An error occurred while trying to create the affiliate; try later.', 'yith-woocommerce-affiliates' ), 'error' );
							}
						}
					}
					else{
						wc_add_notice( __( 'You have already affiliated with us!', 'yith-woocommerce-affiliates' ), 'error' );
					}
				}

				wp_redirect( esc_url( apply_filters( 'yith_wcaf_become_an_affiliate_redirection', remove_query_arg( 'become_an_affiliate' ) ) ) );
				die();
			}
		}

		/**
		 * Register a user as an affiliate (register form action handling)
		 *
		 * @param $customer_id int Customer ID
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function register_affiliate( $customer_id ) {
			// retrieve options
			$enabled_form = get_option( 'yith_wcaf_referral_registration_form_options' );
			$auto_enable = 'yes' == get_option( 'yith_wcaf_referral_registration_auto_enable' );

			// retrieve post data
			$first_name = ! empty( $_POST['first_name'] ) ? trim( $_POST['first_name'] ) : false;
			$last_name = ! empty( $_POST['last_name'] ) ? trim( $_POST['last_name'] ) : false;
			$payment_email = ! empty( $_POST['payment_email'] ) ? trim( $_POST['payment_email'] ) : false;

			if(
				( ! empty( $_POST['register_affiliate'] ) && isset( $_POST['register_affiliate'] ) && wp_verify_nonce( $_POST['register_affiliate'], 'yith-wcaf-register-affiliate' ) ) ||
				( ! empty( $_POST['register'] ) && isset( $_POST['woocommerce-register-nonce'] ) && wp_verify_nonce( $_POST['woocommerce-register-nonce'], 'woocommerce-register' ) && $enabled_form == 'any' )
			){
				$id = $this->add( array( 'user_id' => $customer_id, 'enabled' => $auto_enable, 'payment_email' => $payment_email, 'token' => $this->get_default_user_token( $customer_id ) ) );

				if( $first_name || $last_name ){
					wp_update_user( array_merge(
						array( 'ID' => $customer_id ),
						( $first_name ) ? array( 'first_name' => $first_name ) : array(),
						( $last_name ) ? array( 'last_name' => $last_name ) : array()
					) );
				}

				// trigger new affiliate action
				do_action( 'yith_wcaf_new_affiliate', $id );
			}
		}

		/* === PANEL HANDLING METHODS === */

		/**
		 * Print Affiliate panel
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function print_affiliate_panel() {
			// define variables to be used in template
			$affiliates_table = new YITH_WCAF_Affiliates_Table_Premium();
			$affiliates_table->prepare_items();

			include( YITH_WCAF_DIR . 'templates/admin/affiliate-panel.php' );
		}

		/**
         * Pay all unpaid commissions for an affiliate, from Affiliate panel
         *
         * @return void
         * @since 1.0.10
		 */
		public function handle_pay_commissions_panel_actions() {
			$affiliate_id = isset( $_REQUEST['affiliate_id'] ) ? $_REQUEST['affiliate_id'] : 0;
			$gateway = isset( $_REQUEST['gateway'] ) && in_array( $_REQUEST['gateway'], array_keys( YITH_WCAF_Payment_Handler_Premium()->get_available_gateways() ) ) ? $_REQUEST['gateway'] : '';

			$res = YITH_WCAF_Payment_Handler_Premium()->pay_all_affiliate_commissions( $affiliate_id, true, $gateway );

			wp_redirect( esc_url_raw( add_query_arg( array( 'page' => 'yith_wcaf_panel', 'tab' => 'affiliates', 'commissions_paid' => $res['status'] ), admin_url( 'admin.php' ) ) ) );
			die();
		}

		/* === EDIT PROFILE METHODS === */

		/**
		 * Render affiliate fields
		 *
		 * @param $user \WP_User User object
		 * @return void
		 * @since  1.0.0
		 */
		public function render_affiliate_extra_fields( $user ) {
			parent::render_affiliate_extra_fields( $user );

			$persistent_token = '';
			$selected = '';

			if( isset( $user->ID ) ) {
				$persistent_token = get_user_meta( $user->ID, '_yith_wcaf_persistent_token', true );
			}

			if( ! current_user_can( 'manage_woocommerce' ) ){
				return;
			}

			if( ! empty( $persistent_token ) ){
				$affiliate = YITH_WCAF_Affiliate_Handler()->get_affiliate_by_token( $persistent_token );

				if( $affiliate ){
					$user = get_userdata( $affiliate['user_id'] );
					if( ! is_wp_error( $user ) ){
						$username = '';

						if ( $user->first_name || $user->last_name ) {
							$username .= esc_html( ucfirst( $user->first_name ) . ' ' . ucfirst( $user->last_name ) );
						}
						else {
							$username .= esc_html( ucfirst( $user->display_name ) );
						}

						$selected = $username . ' (#' . $user->ID . ' &ndash; ' . sanitize_email( $user->user_email ) . ')';
					}
				}
			}

			?>
			<hr />
			<h3><?php _e( 'Associated Affiliate', 'yith-woocommerce-affiliates' )?></h3>
			<table class="form-table">
				<tr>
					<th><label for="persistent_token"><?php _e( 'Associated Affiliate', 'yith-woocommerce-affiliates' )?></label></th>
					<td>
                        <?php
                        yit_add_select2_fields( array(
                            'name' => 'persistent_token',
                            'id' => 'persistent_token',
                            'class' => 'wc-product-search',
                            'data-action' => 'json_search_affiliates',
                            'data-placeholder' => __( 'Select an affiliate', 'yith-woocommerce-affiliates' ),
                            'data-selected' => array( $persistent_token => $selected ),
	                        'data-allow_clear' => true,
                            'value' => esc_attr( $persistent_token ),
                            'style' => "min-width: 300px; vertical-align: middle; display: inline-block!important; margin-right: 2px;"
                        ) );
                        ?>
						<span class="description"><?php _e( 'Affiliate that will receive permanent commission from this customer\'s purchases', 'yith-woocommerce-affiliates' ) ?></span>
					</td>
				</tr>

			</table>
			<?php
		}

		/**
		 * Save affiliate fields
		 *
		 * @param $user_id int User id
		 * @return bool Whether method actually saved option or not
		 * @since  1.0.0
		 */
		public function save_affiliate_extra_fields( $user_id ) {
			if ( ! current_user_can( 'edit_user', $user_id ) ) {
				return;
			}

			parent::save_affiliate_extra_fields( $user_id );

			$persistent_affiliate_user_id = isset( $_POST['persistent_token'] ) ? trim( $_POST['persistent_token'] ) : false;

			if( $persistent_affiliate_user_id ){
				$affiliate = YITH_WCAF_Affiliate_Handler()->get_affiliate_by_id( $persistent_affiliate_user_id );

				if( $affiliate ) {
					/**
					 * Filter yith_wcaf_updated_persisten_token
					 *
					 * @param $user_id int Current user id
					 * @param $referral string Current referral token
					 * @param $order_id int Current order id (if any; null otherwise)
					 *
					 * @since 1.1.1
					 */
					do_action( 'yith_wcaf_updated_persisten_token', $user_id, $affiliate['token'], null );

					update_user_meta( $user_id, '_yith_wcaf_persistent_token', $affiliate['token'] );
				}
			}
			else{
				/**
				 * Filter yith_wcaf_updated_persisten_token
				 *
				 * @param $user_id int Current user id
				 *
				 * @since 1.1.1
				 */
				do_action( 'yith_wcaf_deleted_persisten_token', $user_id );

			    delete_user_meta( $user_id, '_yith_wcaf_persistent_token' );
            }
		}

		/**
		 * Returns single instance of the class
		 *
		 * @return \YITH_WCAF_Affiliate_Handler_Premium
		 * @since 1.0.0
		 */
		public static function get_instance(){
			if( is_null( self::$instance ) ){
				self::$instance = new self;
			}

			return self::$instance;
		}
	}
}

/**
 * Unique access to instance of YITH_WCAF_Affiliate_Handler class
 *
 * @return \YITH_WCAF_Affiliate_Handler_Premium
 * @since 1.0.0
 */
function YITH_WCAF_Affiliate_Handler_Premium(){
	return YITH_WCAF_Affiliate_Handler_Premium::get_instance();
}