<?php
/**
 * Admin class premium
 *
 * @author Your Inspiration Themes
 * @package YITH WooCommerce Affiliates
 * @version 1.0.0
 */

/*
 * This file belongs to the YIT Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 */

if ( ! defined( 'YITH_WCAF' ) ) {
	exit;
} // Exit if accessed directly

if ( ! class_exists( 'YITH_WCAF_Admin_Premium' ) ) {
	/**
	 * WooCommerce Affiliates Admin Premium
	 *
	 * @since 1.0.0
	 */
	class YITH_WCAF_Admin_Premium extends YITH_WCAF_Admin {
		/**
		 * Single instance of the class
		 *
		 * @var \YITH_WCAF_Admin_Premium
		 * @since 1.0.0
		 */
		protected static $instance;

		/**
		 * Constructor method
		 *
		 * @return \YITH_WCAF_Admin_Premium
		 * @since 1.0.0
		 */
		public function __construct() {
			add_filter( 'yith_wcaf_available_admin_tabs', array( $this, 'filter_admin_tabs' ) );
			add_filter( 'yith_wcaf_general_settings', array( $this, 'filter_general_settings' ) );

			// register plugin to licence/update system
			add_action( 'wp_loaded', array( $this, 'register_plugin_for_activation' ), 99 );
			add_action( 'admin_init', array( $this, 'register_plugin_for_updates' ) );

			parent::__construct();
		}

		/**
		 * Filters tabs for admin section
		 *
		 * @param $tabs mixed Array of available tabs
		 * @return mixed Filtered array of tabs
		 * @since 1.0.0
		 */
		public function filter_admin_tabs( $tabs ) {
			$array_chunk_1 = array_slice( $tabs, 0, 2 );
			$array_chunk_2 = array_splice( $tabs, 1, count( $tabs ) - 1 );

			$tabs = array_merge(
				$array_chunk_1,
				array(
					'rates' => __( 'Rates', 'yith-woocommerce-affiliates' ),
					'clicks' => __( 'Clicks', 'yith-woocommerce-affiliates' ),
				),
				$array_chunk_2
			);

			unset( $tabs['premium'] );
			return $tabs;
		}

		/**
		 * Filters settings for premium users
		 *
		 * @param $settings mixed Array of available settings
		 * @return mixed Filtered array of settings
		 * @since 1.0.6
		 */
		public function filter_general_settings( $settings ) {
			$settings_options = $settings['settings'];
			$before_index = 'general-options';
			$before_index_position = array_search( $before_index, array_keys( $settings_options ) );

			$settings_options_chunk_1 = array_slice( $settings_options, 0, $before_index_position + 1 );
			$settings_options_chunk_2 = array_slice( $settings_options, $before_index_position + 1, count( $settings_options ) );

			$premium_settings = array(
				'general-referral-cod' => array(
					'title' => __( 'How to get referrer id', 'yith-woocommerce-affiliates' ),
					'type' => 'select',
					'desc' => __( 'Choose how to receive referrer id during purchase', 'yith-woocommerce-affiliates' ),
					'options' => array(
						'query_string' => __( 'Via query string', 'yith-woocommerce-affiliates' ),
						'checkout' => __( 'Let users enter it in checkout page', 'yith-woocommerce-affiliates' )
					),
					'id' => 'yith_wcaf_general_referral_cod',
					'default' => 'query_string',
					'desc_tip' => true,
					'css' => 'min-width: 300px;',
				)
			);

			$settings['settings'] = array_merge(
				$settings_options_chunk_1,
				$premium_settings,
				$settings_options_chunk_2
			);

			return $settings;
		}

		/* === STATS METHODS === */

		/**
		 * Print plugin stat panel
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function print_stat_panel() {
			$page_title = __( 'Global stats', 'yith-woocommerce-affiliates' );
			$title_suffix = '';

			// init stat filters
			$from = isset( $_REQUEST['_from'] ) ? $_REQUEST['_from'] : '';
			$to = isset( $_REQUEST['_to'] ) ? $_REQUEST['_to'] : '';
			$affiliate_id = isset( $_REQUEST['_affiliate_id'] ) ? $_REQUEST['_affiliate_id'] : '';
			$need_reset = false;

			$filters = array();

			if( ! empty( $from ) ){
				$filters['interval']['start_date'] = date( 'Y-m-d 00:00:00', strtotime( $from ) );
			}

			if( ! empty( $to ) ){
				$filters['interval']['end_date'] = date( 'Y-m-d 23:59:59', strtotime( $to ) );
			}

			if( ! empty( $from ) || ! empty( $to ) ){
				$title_suffix = sprintf( ' (%s - %s)', ! empty( $from ) ? date_i18n( wc_date_format(), strtotime( $from ) ) : __( 'Ever', 'yith-woocommerce-affiliates' ), ! empty( $to ) ? date_i18n( wc_date_format(), strtotime( $to ) ) : __( 'Today', 'yith-woocommerce-affiliates' ) );
			}

			if( ! empty( $affiliate_id ) ){
				$filters['affiliate_id'] = intval( $affiliate_id );
				$affiliate = YITH_WCAF_Affiliate_Handler()->get_affiliate_by_id( $affiliate_id );

				if( $affiliate ){
					$user = get_userdata( $affiliate['user_id'] );
					if( ! is_wp_error( $user ) ){
						$username = '';

						if ( $user->first_name || $user->last_name ) {
							$username .= esc_html( ucfirst( $user->first_name ) . ' ' . ucfirst( $user->last_name ) );
						}
						else {
							$username .= esc_html( ucfirst( $user->display_name ) );
						}

						$selected = $username . ' (#' . $user->ID . ' &ndash; ' . sanitize_email( $user->user_email ) . ')';
						$page_title = sprintf( __( 'Stats for affiliate "%s"', 'yith-woocommerce-affiliates' ), $username );
					}
				}
			}

			if( ! empty( $from ) || ! empty( $to ) || ! empty( $affiliate_id ) ){
				$need_reset = true;
				$reset_link = esc_url( add_query_arg( array( 'page' => 'yith_wcaf_panel', 'tab' => 'stats' ), admin_url( 'admin.php' ) ) );
			}

			// define variables to be used in the template
			$total_amount = YITH_WCAF_Commission_Handler()->get_commission_stats( 'total_amount', array_merge( $filters, array( 'status' => array( 'pending', 'pending-payment', 'paid' ) ) ) );
			$total_paid = YITH_WCAF_Commission_Handler()->get_commission_stats( 'total_amount', array_merge( $filters, array( 'status' => array( 'pending-payment', 'paid' ) ) ) );
			$total_earned = YITH_WCAF_Commission_Handler()->get_commission_stats( 'total_earned', array_merge( $filters, array( 'status' => array( 'pending', 'pending-payment', 'paid' ) ) ) );
			$total_refunded = YITH_WCAF_Commission_Handler()->get_commission_stats( 'total_refunds', array_merge( $filters ) );

			$total_clicks = YITH_WCAF_Click_Handler()->get_hit_stats( 'total_clicks', $filters );
			$total_conversions = YITH_WCAF_Click_Handler()->get_hit_stats( 'total_conversions', $filters );
			$avg_conv_rate = ! empty( $total_clicks ) ? $total_conversions / $total_clicks * 100 : 0;
			$avg_conv_rate = ! empty( $avg_conv_rate ) ? number_format( $avg_conv_rate, 2 ) . '%' : __( 'N/A', 'yith-woocommerce-affiliates' );

			$avg_conv_time = YITH_WCAF_Click_Handler()->get_hit_stats( 'avg_conv_time', $filters );
			$readable_avg_conv_time = ! empty( $avg_conv_time ) ? human_time_diff( time(), time() + $avg_conv_time ) : __( 'N/A', 'yith-woocommerce-affiliates' );

			$page_title .= $title_suffix;

			$product_table = new YITH_WCAF_Product_Stat_Table();
			$product_table->prepare_items();

			// includes panel template
			include( YITH_WCAF_DIR . 'templates/admin/stat-panel-premium.php' );
		}

		/* === LICENCE HANDLING METHODS === */

		/**
		 * Register plugins for activation tab
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function register_plugin_for_activation() {
			if( ! class_exists( 'YIT_Plugin_Licence' ) ){
				require_once YITH_WCAF_DIR . 'plugin-fw/licence/lib/yit-licence.php';
				require_once YITH_WCAF_DIR . 'plugin-fw/licence/lib/yit-plugin-licence.php';
			}

			YIT_Plugin_Licence()->register( YITH_WCAF_INIT, YITH_WCAF_SECRET_KEY, YITH_WCAF_SLUG );
		}

		/**
		 * Register plugins for update tab
		 *
		 * @return void
		 * @since 1.0.0
		 */
		public function register_plugin_for_updates() {
			if( ! class_exists( 'YIT_Plugin_Licence' ) ){
				require_once( YITH_WCAF_DIR . 'plugin-fw/lib/yit-upgrade.php' );
			}

			YIT_Upgrade()->register( YITH_WCAF_SLUG, YITH_WCAF_INIT );
		}

		/**
		 * Returns single instance of the class
		 *
		 * @return \YITH_WCAF_Admin_Premium
		 * @since 1.0.0
		 */
		public static function get_instance(){
			if( is_null( self::$instance ) ){
				self::$instance = new self;
			}

			return self::$instance;
		}
	}
}

/**
 * Unique access to instance of YITH_WCAF_Admin_Premium class
 *
 * @return \YITH_WCAF_Admin_Premium
 * @since 1.0.0
 */
function YITH_WCAF_Admin_premium(){
	return YITH_WCAF_Admin_premium::get_instance();
}