<?php
/**
 * This file belongs to the YIT Plugin Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if ( ! class_exists( 'YLC_Mailer' ) ) {

	class YLC_Mailer {

		protected $from;
		protected $to;
		protected $subject;
		protected $message;
		protected $reply_to;
		protected $from_name;

		public function __construct( $from, $to, $subject, $message, $reply_to, $from_name = '' ) {

			$this->from      = $from;
			$this->to        = $to;
			$this->subject   = $subject;
			$this->message   = $message;
			$this->reply_to  = $reply_to;
			$this->from_name = $from_name;

		}

		public function send() {

			add_filter( 'wp_mail_from', array( $this, 'get_from_address' ) );
			add_filter( 'wp_mail_from_name', array( $this, 'get_from_name' ) );
			add_filter( 'wp_mail_content_type', array( $this, 'get_content_type' ) );

			$return = wp_mail( $this->to, $this->subject, $this->message, $this->set_headers() );

			remove_filter( 'wp_mail_from', array( $this, 'get_from_address' ) );
			remove_filter( 'wp_mail_from_name', array( $this, 'get_from_name' ) );
			remove_filter( 'wp_mail_content_type', array( $this, 'get_content_type' ) );

			return $return;
		}

		public function set_headers() {
			$headers   = array();
			$headers[] = 'content-type: text/html';
			$headers[] = 'charset=utf-8';
			//$headers[] = 'From: ' . sanitize_text_field( ( $this->from_name != '' ? $this->from_name : get_option( 'blogname' ) ) ) . ' <' . sanitize_email( apply_filters( 'ylc_sender_override', $this->from ) ) . '>';
			$headers[] = 'Reply-To: ' . $this->reply_to;

			return apply_filters( 'ylc_headers_filter', $headers );
		}

		/**
		 * Get the from name for outgoing emails.
		 *
		 * @return string
		 */
		public function get_from_name() {
			return wp_specialchars_decode( esc_html( $this->from_name ), ENT_QUOTES );
		}

		/**
		 * Get the from address for outgoing emails.
		 *
		 * @return string
		 */
		public function get_from_address() {
			return sanitize_email( $this->from );
		}

		/**
		 * Get email content type.
		 *
		 * @return string
		 */
		public function get_content_type() {

			return 'text/html';

		}
	}


}


if ( ! function_exists( 'ylc_send_email_message' ) ) {

	/**
	 * Send email message
	 *
	 * @since   1.0.0
	 *
	 * @param   $from
	 * @param   $to
	 * @param   $subject
	 * @param   $message
	 * @param   $reply_to
	 * @param   $from_name
	 *
	 * @return  bool
	 * @author  Alberto Ruggiero
	 */
	function ylc_send_email_message( $from, $to, $subject, $message, $reply_to, $from_name = '' ) {

		$email = new YLC_Mailer( $from, $to, $subject, $message, $reply_to, $from_name );

		return $email->send();

	}

}

if ( ! function_exists( 'ylc_get_mail_body' ) ) {

	/**
	 * Get mail body
	 *
	 * @since   1.0.0
	 *
	 * @param   $template
	 * @param   $args
	 *
	 * @return  string
	 * @author  Alberto Ruggiero
	 */
	function ylc_get_mail_body( $template, $args ) {

		ob_start();

		ylc_get_template( 'email/' . $template . '.php', $args );

		return ob_get_clean();

	}

}

if ( ! function_exists( 'ylc_send_chat_data' ) ) {

	/**
	 * Send chat transcripts
	 *
	 * @since   1.0.0
	 *
	 * @param   $cnv_id
	 * @param   $from
	 * @param   $to
	 * @param   $message
	 * @param   $chat_data
	 * @param   $user
	 *
	 * @return  boolean
	 * @author  Alberto Ruggiero
	 */
	function ylc_send_chat_data( $cnv_id, $from, $to, $message, $chat_data = array(), $user = '' ) {

		$subject = __( 'Chat conversation copy', 'yith-live-chat' ) . ( ( $user != '' ) ? ': ' . $user : '' );

		$args = array(
			'subject'   => $subject,
			'mail_body' => wp_strip_all_tags( $message ),
			'cnv_id'    => $cnv_id,
			'chat_data' => $chat_data
		);

		$message = ylc_get_mail_body( 'chat-copy', $args );

		return ylc_send_email_message( $from, $to, $subject, $message, $from );

	}

}

if ( ! function_exists( 'ylc_send_offline_msg' ) ) {

	/**
	 * Send offline message
	 *
	 * @since   1.0.0
	 *
	 * @param   $from
	 * @param   $to
	 * @param   $subject
	 * @param   $user
	 * @param   $ip_address
	 * @param   $form_data
	 * @param   $mail_body
	 * @param   $page
	 * @param   $user_copy
	 *
	 * @return  bool
	 * @author  Alberto Ruggiero
	 */
	function ylc_send_offline_msg( $from, $to, $subject, $user, $ip_address, $form_data, $mail_body, $page, $user_copy = false ) {

		$args = array(
			'subject'    => $subject,
			'mail_body'  => $mail_body,
			'name'       => $form_data['name'],
			'email'      => $form_data['email'],
			'message'    => $form_data['message'],
			'os'         => $user->info( 'os' ),
			'browser'    => $user->info( 'browser' ),
			'version'    => $user->info( 'version' ),
			'ip_address' => $ip_address,
			'page'       => $page,
		);

		$message    = ylc_get_mail_body( 'offline-message', $args );
		$reply_to   = ( $user_copy ) ? $from : $form_data['email'];
		$from_name  = ( $user_copy ) ? '' : $form_data['name'];
		$from_email = ( $user_copy ) ? $from : $form_data['email'];

		return ylc_send_email_message( $from_email, $to, $subject, $message, $reply_to, $from_name );

	}

}