<?php

/**
 * Class Test_Minifier
 *
 * @group minifier
 */
class Test_Minifier extends WP_Hummingbird_UnitTestCase {

	public $to_footer = array(
			'styles' => array(),
			'scripts' => array()
	);

	private static $minified_counter = 0;

	public $to_header = array(
			'styles' => array(),
			'scripts' => array()
	);

	function setUp() {
		parent::setUp();

		$this->old_wp_scripts = isset( $GLOBALS['wp_scripts'] ) ? $GLOBALS['wp_scripts'] : null;
		remove_action( 'wp_default_scripts', 'wp_default_scripts' );
		remove_action( 'wp_default_styles', 'wp_default_styles' );
		$GLOBALS['wp_scripts']                  = new WP_Scripts();
		$GLOBALS['wp_scripts']->default_version = get_bloginfo( 'version' );

	}

	function tearDown() {
		$GLOBALS['wp_scripts'] = $this->old_wp_scripts;
		add_action( 'wp_default_scripts', 'wp_default_scripts' );
		parent::tearDown();
	}

	/**
	 * @group parser
	 */
	function test_url_parser() {
		wphb_include_sources_collector();

		$content = <<<EOT
background-image:url(http://googlefonts.es);
background-image:url(http://googlefodnts.es);
background-image:url("http://googlefodnts.es");
background-image:url('http://googlefodnts.es');
background-image:url( 'http://googlefodnts.es');
background-image:url('http://googlefodnts.es'  );
background-image:url(  'http://googlefodnts.es');
background-image:url   (  'http://googlefodnts.es');
background-image:url   (  '//googlefodnts.es');
background-image:url   (  ../style.css);
background-image:url   (  "../style.css");
background-image:url   (  "../style.css");
background-image:url   (  "../styles/style.css");
background-image:url   (  "styles/style.css");
background-image:url("styles/style.css");
background-image:url(styles/style.css);
background-image:url('styles/style.css');
EOT;

		$file_url = 'http://example.org/wp-content/plugins/wp-hummingbird/js/';
		$content = WP_Hummingbird_Sources_Group::replace_relative_urls( $file_url, $content );
		var_dump($content);
return;
		$content = <<<EOT
@font-face{font-family:dashicons;src:url(../fonts/dashicons.eot)}@font-face{font-family:dashicons;src:url(data:application/font-woff;charset=utf-8;base64,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) format('woff'),url(../fonts/dashicons.ttf) format("truetype"),url(../fonts/dashicons.svg#dashicons) format("svg");font-weight:400;font-style:normal}.dashicons,.dashicons-before:before{display:inline-block;width:20px;height:20px;font-size:20px;line-height:1;font-family:dashicons;text-decoration:inherit;font-weight:400;font-style:normal;vertical-align:top;text-align:center;-webkit-transition:color .1s ease-in 0;transition:color .1s ease-in 0;-webkit-font-smoothing:antialiased;-moz-osx-font-smoothing:grayscale}.dashicons-menu:before{content:"\f333"}.dashicons-admin-site:before{content:"\f319"}.dashicons-dashboard:before{content:"\f226"}.dashicons-admin-media:before{content:"\f104"}.dashicons-admin-page:before{content:"\f105"}.dashicons-admin-comments:before{content:"\f101"}.dashicons-admin-appearance:before{content:"\f100"}.dashicons-admin-plugins:before{content:"\f106"}.dashicons-admin-users:before{content:"\f110"}.dashicons-admin-tools:before{content:"\f107"}.dashicons-admin-settings:before{content:"\f108"}.dashicons-admin-network:before{content:"\f112"}.dashicons-admin-generic:before{content:"\f111"}.dashicons-admin-home:before{content:"\f102"}.dashicons-admin-collapse:before{content:"\f148"}.dashicons-filter:before{content:"\f536"}.dashicons-admin-customizer:before{content:"\f540"}.dashicons-admin-multisite:before{content:"\f541"}.dashicons-admin-links:before,.dashicons-format-links:before{content:"\f103"}.dashicons-admin-post:before,.dashicons-format-standard:before{content:"\f109"}.dashicons-format-image:before{content:"\f128"}.dashicons-format-gallery:before{content:"\f161"}.dashicons-format-audio:before{content:"\f127"}.dashicons-format-video:before{content:"\f126"}.dashicons-format-chat:before{content:"\f125"}.dashicons-format-status:before{content:"\f130"}.dashicons-format-aside:before{content:"\f123"}.dashicons-format-quote:before{content:"\f122"}.dashicons-welcome-edit-page:before,.dashicons-welcome-write-blog:before{content:"\f119"}.dashicons-welcome-add-page:before{content:"\f133"}.dashicons-welcome-view-site:before{content:"\f115"}.dashicons-welcome-widgets-menus:before{content:"\f116"}.dashicons-welcome-comments:before{content:"\f117"}.dashicons-welcome-learn-more:before{content:"\f118"}.dashicons-image-crop:before{content:"\f165"}.dashicons-image-rotate:before{content:"\f531"}.dashicons-image-rotate-left:before{content:"\f166"}.dashicons-image-rotate-right:before{content:"\f167"}.dashicons-image-flip-vertical:before{content:"\f168"}.dashicons-image-flip-horizontal:before{content:"\f169"}.dashicons-image-filter:before{content:"\f533"}.dashicons-undo:before{content:"\f171"}.dashicons-redo:before{content:"\f172"}.dashicons-editor-bold:before{content:"\f200"}.dashicons-editor-italic:before{content:"\f201"}.dashicons-editor-ul:before{content:"\f203"}.dashicons-editor-ol:before{content:"\f204"}.dashicons-editor-quote:before{content:"\f205"}.dashicons-editor-alignleft:before{content:"\f206"}.dashicons-editor-aligncenter:before{content:"\f207"}.dashicons-editor-alignright:before{content:"\f208"}.dashicons-editor-insertmore:before{content:"\f209"}.dashicons-editor-spellcheck:before{content:"\f210"}.dashicons-editor-distractionfree:before,.dashicons-editor-expand:before{content:"\f211"}.dashicons-editor-contract:before{content:"\f506"}.dashicons-editor-kitchensink:before{content:"\f212"}.dashicons-editor-underline:before{content:"\f213"}.dashicons-editor-justify:before{content:"\f214"}.dashicons-editor-textcolor:before{content:"\f215"}.dashicons-editor-paste-word:before{content:"\f216"}.dashicons-editor-paste-text:before{content:"\f217"}.dashicons-editor-removeformatting:before{content:"\f218"}.dashicons-editor-video:before{content:"\f219"}.dashicons-editor-customchar:before{content:"\f220"}.dashicons-editor-outdent:before{content:"\f221"}.dashicons-editor-indent:before{content:"\f222"}.dashicons-editor-help:before{content:"\f223"}.dashicons-editor-strikethrough:before{content:"\f224"}.dashicons-editor-unlink:before{content:"\f225"}.dashicons-editor-rtl:before{content:"\f320"}.dashicons-editor-break:before{content:"\f474"}.dashicons-editor-code:before{content:"\f475"}.dashicons-editor-paragraph:before{content:"\f476"}.dashicons-editor-table:before{content:"\f535"}.dashicons-align-left:before{content:"\f135"}.dashicons-align-right:before{content:"\f136"}.dashicons-align-center:before{content:"\f134"}.dashicons-align-none:before{content:"\f138"}.dashicons-lock:before{content:"\f160"}.dashicons-unlock:before{content:"\f528"}.dashicons-calendar:before{content:"\f145"}.dashicons-calendar-alt:before{content:"\f508"}.dashicons-visibility:before{content:"\f177"}.dashicons-hidden:before{content:"\f530"}.dashicons-post-status:before{content:"\f173"}.dashicons-edit:before{content:"\f464"}.dashicons-post-trash:before,.dashicons-trash:before{content:"\f182"}.dashicons-sticky:before{content:"\f537"}.dashicons-external:before{content:"\f504"}.dashicons-arrow-up:before{content:"\f142"}.dashicons-arrow-down:before{content:"\f140"}.dashicons-arrow-left:before{content:"\f141"}.dashicons-arrow-right:before{content:"\f139"}.dashicons-arrow-up-alt:before{content:"\f342"}.dashicons-arrow-down-alt:before{content:"\f346"}.dashicons-arrow-left-alt:before{content:"\f340"}.dashicons-arrow-right-alt:before{content:"\f344"}.dashicons-arrow-up-alt2:before{content:"\f343"}.dashicons-arrow-down-alt2:before{content:"\f347"}.dashicons-arrow-left-alt2:before{content:"\f341"}.dashicons-arrow-right-alt2:before{content:"\f345"}.dashicons-leftright:before{content:"\f229"}.dashicons-sort:before{content:"\f156"}.dashicons-randomize:before{content:"\f503"}.dashicons-list-view:before{content:"\f163"}.dashicons-excerpt-view:before,.dashicons-exerpt-view:before{content:"\f164"}.dashicons-grid-view:before{content:"\f509"}.dashicons-hammer:before{content:"\f308"}.dashicons-art:before{content:"\f309"}.dashicons-migrate:before{content:"\f310"}.dashicons-performance:before{content:"\f311"}.dashicons-universal-access:before{content:"\f483"}.dashicons-universal-access-alt:before{content:"\f507"}.dashicons-tickets:before{content:"\f486"}.dashicons-nametag:before{content:"\f484"}.dashicons-clipboard:before{content:"\f481"}.dashicons-heart:before{content:"\f487"}.dashicons-megaphone:before{content:"\f488"}.dashicons-schedule:before{content:"\f489"}.dashicons-wordpress:before{content:"\f120"}.dashicons-wordpress-alt:before{content:"\f324"}.dashicons-pressthis:before{content:"\f157"}.dashicons-update:before{content:"\f463"}.dashicons-screenoptions:before{content:"\f180"}.dashicons-cart:before{content:"\f174"}.dashicons-feedback:before{content:"\f175"}.dashicons-cloud:before{content:"\f176"}.dashicons-translation:before{content:"\f326"}.dashicons-tag:before{content:"\f323"}.dashicons-category:before{content:"\f318"}.dashicons-archive:before{content:"\f480"}.dashicons-tagcloud:before{content:"\f479"}.dashicons-text:before{content:"\f478"}.dashicons-media-archive:before{content:"\f501"}.dashicons-media-audio:before{content:"\f500"}.dashicons-media-code:before{content:"\f499"}.dashicons-media-default:before{content:"\f498"}.dashicons-media-document:before{content:"\f497"}.dashicons-media-interactive:before{content:"\f496"}.dashicons-media-spreadsheet:before{content:"\f495"}.dashicons-media-text:before{content:"\f491"}.dashicons-media-video:before{content:"\f490"}.dashicons-playlist-audio:before{content:"\f492"}.dashicons-playlist-video:before{content:"\f493"}.dashicons-controls-play:before{content:"\f522"}.dashicons-controls-pause:before{content:"\f523"}.dashicons-controls-forward:before{content:"\f519"}.dashicons-controls-skipforward:before{content:"\f517"}.dashicons-controls-back:before{content:"\f518"}.dashicons-controls-skipback:before{content:"\f516"}.dashicons-controls-repeat:before{content:"\f515"}.dashicons-controls-volumeon:before{content:"\f521"}.dashicons-controls-volumeoff:before{content:"\f520"}.dashicons-yes:before{content:"\f147"}.dashicons-no:before{content:"\f158"}.dashicons-no-alt:before{content:"\f335"}.dashicons-plus:before{content:"\f132"}.dashicons-plus-alt:before{content:"\f502"}.dashicons-plus-alt2:before{content:"\f543"}.dashicons-minus:before{content:"\f460"}.dashicons-dismiss:before{content:"\f153"}.dashicons-marker:before{content:"\f159"}.dashicons-star-filled:before{content:"\f155"}.dashicons-star-half:before{content:"\f459"}.dashicons-star-empty:before{content:"\f154"}.dashicons-flag:before{content:"\f227"}.dashicons-info:before{content:"\f348"}.dashicons-warning:before{content:"\f534"}.dashicons-share:before{content:"\f237"}.dashicons-share1:before{content:"\f237"}.dashicons-share-alt:before{content:"\f240"}.dashicons-share-alt2:before{content:"\f242"}.dashicons-twitter:before{content:"\f301"}.dashicons-rss:before{content:"\f303"}.dashicons-email:before{content:"\f465"}.dashicons-email-alt:before{content:"\f466"}.dashicons-facebook:before{content:"\f304"}.dashicons-facebook-alt:before{content:"\f305"}.dashicons-networking:before{content:"\f325"}.dashicons-googleplus:before{content:"\f462"}.dashicons-location:before{content:"\f230"}.dashicons-location-alt:before{content:"\f231"}.dashicons-camera:before{content:"\f306"}.dashicons-images-alt:before{content:"\f232"}.dashicons-images-alt2:before{content:"\f233"}.dashicons-video-alt:before{content:"\f234"}.dashicons-video-alt2:before{content:"\f235"}.dashicons-video-alt3:before{content:"\f236"}.dashicons-vault:before{content:"\f178"}.dashicons-shield:before{content:"\f332"}.dashicons-shield-alt:before{content:"\f334"}.dashicons-sos:before{content:"\f468"}.dashicons-search:before{content:"\f179"}.dashicons-slides:before{content:"\f181"}.dashicons-analytics:before{content:"\f183"}.dashicons-chart-pie:before{content:"\f184"}.dashicons-chart-bar:before{content:"\f185"}.dashicons-chart-line:before{content:"\f238"}.dashicons-chart-area:before{content:"\f239"}.dashicons-groups:before{content:"\f307"}.dashicons-businessman:before{content:"\f338"}.dashicons-id:before{content:"\f336"}.dashicons-id-alt:before{content:"\f337"}.dashicons-products:before{content:"\f312"}.dashicons-awards:before{content:"\f313"}.dashicons-forms:before{content:"\f314"}.dashicons-testimonial:before{content:"\f473"}.dashicons-portfolio:before{content:"\f322"}.dashicons-book:before{content:"\f330"}.dashicons-book-alt:before{content:"\f331"}.dashicons-download:before{content:"\f316"}.dashicons-upload:before{content:"\f317"}.dashicons-backup:before{content:"\f321"}.dashicons-clock:before{content:"\f469"}.dashicons-lightbulb:before{content:"\f339"}.dashicons-microphone:before{content:"\f482"}.dashicons-desktop:before{content:"\f472"}.dashicons-tablet:before{content:"\f471"}.dashicons-smartphone:before{content:"\f470"}.dashicons-phone:before{content:"\f525"}.dashicons-smiley:before{content:"\f328"}.dashicons-index-card:before{content:"\f510"}.dashicons-carrot:before{content:"\f511"}.dashicons-building:before{content:"\f512"}.dashicons-store:before{content:"\f513"}.dashicons-album:before{content:"\f514"}.dashicons-palmtree:before{content:"\f527"}.dashicons-tickets-alt:before{content:"\f524"}.dashicons-money:before{content:"\f526"}.dashicons-thumbs-up:before{content:"\f529"}.dashicons-thumbs-down:before{content:"\f542"}.dashicons-layout:before{content:"\f538"}
EOT;

		//$relatives = WP_Hummingbird_Sources_Group::get_relative_urls( $file_url, $content );
		//WP_Hummingbird_Sources_Group::transform_relative_urls( $content );
	}

	function get_echo( $callable, $args = array() ) {
		ob_start();
		call_user_func_array( $callable, $args );

		return ob_get_clean();
	}

	function test_minify() {
		global $wp_styles, $wp_scripts;

		$this->wp_styles();
		$this->wp_scripts();

		// Filter the header styles
		$minifier = wphb_get_module( 'minify' );

		/** @var WP_Hummingbird_Module_Minify $minifier */
		$to_do = $minifier->filter_styles( $wp_styles->to_do );

		// First time, all styles should be returned to WP, we haven't generated the files yet
		$this->assertEquals( $to_do, $wp_styles->to_do );
		$this->assertCount( 2, $minifier->process_queue );
		$minifier->save_data();
	}



	function wp_styles() {
		global $wp_styles;

		$wp_styles = wp_styles();

		include( ABSPATH . WPINC . '/version.php' ); // include an unmodified $wp_version

		if ( ! defined( 'SCRIPT_DEBUG' ) )
			define( 'SCRIPT_DEBUG', false !== strpos( $wp_version, '-src' ) );

		if ( ! $guessurl = site_url() )
			$guessurl = wp_guess_url();

		$wp_styles->base_url = $guessurl;
		$wp_styles->content_url = defined('WP_CONTENT_URL')? WP_CONTENT_URL : '';
		$wp_styles->default_version = get_bloginfo( 'version' );
		$wp_styles->text_direction = function_exists( 'is_rtl' ) && is_rtl() ? 'rtl' : 'ltr';
		$wp_styles->default_dirs = array('/wp-admin/', '/wp-includes/css/');

		$open_sans_font_url = '';
		if ( 'off' !== _x( 'on', 'Open Sans font: on or off' ) ) {
			$subsets = 'latin,latin-ext';

			/* translators: To add an additional Open Sans character subset specific to your language,
			 * translate this to 'greek', 'cyrillic' or 'vietnamese'. Do not translate into your own language.
			 */
			$subset = _x( 'no-subset', 'Open Sans font: add new subset (greek, cyrillic, vietnamese)' );

			if ( 'cyrillic' == $subset ) {
				$subsets .= ',cyrillic,cyrillic-ext';
			} elseif ( 'greek' == $subset ) {
				$subsets .= ',greek,greek-ext';
			} elseif ( 'vietnamese' == $subset ) {
				$subsets .= ',vietnamese';
			}

			// Hotlink Open Sans, for now
			$open_sans_font_url = "https://fonts.googleapis.com/css?family=Open+Sans:300italic,400italic,600italic,300,400,600&subset=$subsets";
		}

		$registered_styles = $this->get_registered_styles();
		foreach ( $registered_styles as $slug => $registered ) {
			$wp_styles->add( $registered['handle'], $registered['src'], $registered['deps'], $registered['ver'], $registered['args'] );

			if ( ! empty( $registered['extra'] ) ) {
				foreach ( $registered['extra'] as $key => $value )
					$wp_styles->add_data( $registered['handle'], $key, $value );
			}
		}

		$queue = array (
			0 => 'admin-bar',
			1 => 'twentyfifteen-fonts',
			2 => 'genericons',
			3 => 'twentyfifteen-style',
			4 => 'twentyfifteen-ie',
			5 => 'twentyfifteen-ie7',
			6 => 'colorbox',
			7 => 'mp-select2',
			8 => 'mp-base',
			9 => 'mp-theme',
			10 => 'query-monitor',
		);

		foreach ( $queue as $item )
			$wp_styles->enqueue( $item );

		$wp_styles->to_do = array (
				0 => 'open-sans',
				1 => 'dashicons',
				2 => 'admin-bar',
				3 => 'twentyfifteen-fonts',
				4 => 'genericons',
				5 => 'twentyfifteen-style',
				6 => 'twentyfifteen-ie',
				7 => 'twentyfifteen-ie7',
				8 => 'colorbox',
				9 => 'mp-select2',
				10 => 'mp-base',
				11 => 'mp-theme',
				12 => 'query-monitor',
		);
	}

	function wp_scripts() {
		global $wp_scripts;

		$wp_scripts = wp_scripts();

		include( ABSPATH . WPINC . '/version.php' ); // include an unmodified $wp_version

		$develop_src = false !== strpos( $wp_version, '-src' );

		if ( ! defined( 'SCRIPT_DEBUG' ) ) {
			define( 'SCRIPT_DEBUG', $develop_src );
		}

		if ( ! $guessurl = site_url() ) {
			$guessed_url = true;
			$guessurl = wp_guess_url();
		}

		$wp_scripts->base_url = $guessurl;
		$wp_scripts->content_url = defined('WP_CONTENT_URL')? WP_CONTENT_URL : '';
		$wp_scripts->default_version = get_bloginfo( 'version' );
		$wp_scripts->default_dirs = array('/wp-admin/js/', '/wp-includes/js/');

		$registered_scripts = $this->get_registered_scripts();
		foreach ( $registered_scripts as $slug => $registered ) {
			$wp_scripts->add( $registered['handle'], $registered['src'], $registered['deps'], $registered['ver'], $registered['args'] );

			if ( ! empty( $registered['extra'] ) ) {
				foreach ( $registered['extra'] as $key => $value )
					if ( 'group' == $key )
						$wp_scripts->set_group( $registered['handle'], false, $value );
					else
						$wp_scripts->add_data( $registered['handle'], $key, $value );

			}
		}

		$queue = array (
			0 => 'admin-bar',
			1 => 'twentyfifteen-skip-link-focus-fix',
			2 => 'twentyfifteen-script',
			3 => 'mp-cart',
			4 => 'mp-frontend',
			5 => 'query-monitor',
		);

		foreach ( $queue as $item )
			$wp_scripts->enqueue( $item );

		$wp_scripts->to_do = array (
				0 => 'admin-bar',
				1 => 'twentyfifteen-skip-link-focus-fix',
				2 => 'jquery-core',
				3 => 'jquery-migrate',
				4 => 'jquery',
				5 => 'twentyfifteen-script',
				6 => 'ajaxq',
				7 => 'colorbox',
				8 => 'jquery-validate',
				9 => 'mp-cart',
				10 => 'jquery-ui-core',
				11 => 'jquery-ui-widget',
				12 => 'jquery-ui-position',
				13 => 'jquery-ui-tooltip',
				14 => 'hover-intent',
				15 => 'mp-select2',
				16 => 'mp-frontend',
				17 => 'query-monitor',
		);

	}

















	function get_registered_scripts() {
		return 	array (
								'utils' =>
										array(
												'handle' => 'utils',
												'src' => '/wp-includes/js/utils.min.js',
												'deps' =>
														array (
														),
												'ver' => false,
												'args' => NULL,
												'extra' =>
														array (
																'data' => 'var userSettings = {"url":"\\/","uid":"1","time":"1449748006","secure":""};',
														),
										),
								'common' =>
										array(
												'handle' => 'common',
												'src' => '/wp-admin/js/common.min.js',
												'deps' =>
														array (
																0 => 'jquery',
																1 => 'hoverIntent',
																2 => 'utils',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var commonL10n = {"warnDelete":"You are about to permanently delete these items.\\n  \'Cancel\' to stop, \'OK\' to delete.","dismiss":"Dismiss this notice."};',
														),
										),
								'wp-a11y' =>
										array(
												'handle' => 'wp-a11y',
												'src' => '/wp-includes/js/wp-a11y.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'sack' =>
										array(
												'handle' => 'sack',
												'src' => '/wp-includes/js/tw-sack.min.js',
												'deps' =>
														array (
														),
												'ver' => '1.6.1',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'quicktags' =>
										array(
												'handle' => 'quicktags',
												'src' => '/wp-includes/js/quicktags.min.js',
												'deps' =>
														array (
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var quicktagsL10n = {"closeAllOpenTags":"Close all open tags","closeTags":"close tags","enterURL":"Enter the URL","enterImageURL":"Enter the URL of the image","enterImageDescription":"Enter a description of the image","textdirection":"text direction","toggleTextdirection":"Toggle Editor Text Direction","dfw":"Distraction-free writing mode","strong":"Bold","strongClose":"Close bold tag","em":"Italic","emClose":"Close italic tag","link":"Insert link","blockquote":"Blockquote","blockquoteClose":"Close blockquote tag","del":"Deleted text (strikethrough)","delClose":"Close deleted text tag","ins":"Inserted text","insClose":"Close inserted text tag","image":"Insert image","ul":"Bulleted list","ulClose":"Close bulleted list tag","ol":"Numbered list","olClose":"Close numbered list tag","li":"List item","liClose":"Close list item tag","code":"Code","codeClose":"Close code tag","more":"Insert Read More tag"};',
														),
										),
								'colorpicker' =>
										array(
												'handle' => 'colorpicker',
												'src' => '/wp-includes/js/colorpicker.min.js',
												'deps' =>
														array (
																0 => 'prototype',
														),
												'ver' => '3517m',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'editor' =>
										array(
												'handle' => 'editor',
												'src' => '/wp-admin/js/editor.min.js',
												'deps' =>
														array (
																0 => 'utils',
																1 => 'jquery',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'wp-fullscreen-stub' =>
										array(
												'handle' => 'wp-fullscreen-stub',
												'src' => '/wp-admin/js/wp-fullscreen-stub.min.js',
												'deps' =>
														array (
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'wp-ajax-response' =>
										array(
												'handle' => 'wp-ajax-response',
												'src' => '/wp-includes/js/wp-ajax-response.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var wpAjax = {"noPerm":"You do not have permission to do that.","broken":"An unidentified error has occurred."};',
														),
										),
								'wp-pointer' =>
										array(
												'handle' => 'wp-pointer',
												'src' => '/wp-includes/js/wp-pointer.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-widget',
																1 => 'jquery-ui-position',
														),
												'ver' => '20111129a',
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var wpPointerL10n = {"dismiss":"Dismiss"};',
														),
										),
								'autosave' =>
										array(
												'handle' => 'autosave',
												'src' => '/wp-includes/js/autosave.min.js',
												'deps' =>
														array (
																0 => 'heartbeat',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var autosaveL10n = {"autosaveInterval":"60","blog_id":"61"};',
														),
										),
								'heartbeat' =>
										array(
												'handle' => 'heartbeat',
												'src' => '/wp-includes/js/heartbeat.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var heartbeatSettings = {"ajaxurl":"\\/wp-admin\\/admin-ajax.php","nonce":"47949e5958"};',
														),
										),
								'wp-auth-check' =>
										array(
												'handle' => 'wp-auth-check',
												'src' => '/wp-includes/js/wp-auth-check.min.js',
												'deps' =>
														array (
																0 => 'heartbeat',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var authcheckL10n = {"beforeunload":"Your session has expired. You can log in again from this page or go to the login page.","interval":"180"};',
														),
										),
								'wp-lists' =>
										array(
												'handle' => 'wp-lists',
												'src' => '/wp-includes/js/wp-lists.min.js',
												'deps' =>
														array (
																0 => 'wp-ajax-response',
																1 => 'jquery-color',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'prototype' =>
										array(
												'handle' => 'prototype',
												'src' => 'https://ajax.googleapis.com/ajax/libs/prototype/1.7.1.0/prototype.js',
												'deps' =>
														array (
														),
												'ver' => '1.7.1',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'scriptaculous-root' =>
										array(
												'handle' => 'scriptaculous-root',
												'src' => 'https://ajax.googleapis.com/ajax/libs/scriptaculous/1.9.0/scriptaculous.js',
												'deps' =>
														array (
																0 => 'prototype',
														),
												'ver' => '1.9.0',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'scriptaculous-builder' =>
										array(
												'handle' => 'scriptaculous-builder',
												'src' => 'https://ajax.googleapis.com/ajax/libs/scriptaculous/1.9.0/builder.js',
												'deps' =>
														array (
																0 => 'scriptaculous-root',
														),
												'ver' => '1.9.0',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'scriptaculous-dragdrop' =>
										array(
												'handle' => 'scriptaculous-dragdrop',
												'src' => 'https://ajax.googleapis.com/ajax/libs/scriptaculous/1.9.0/dragdrop.js',
												'deps' =>
														array (
																0 => 'scriptaculous-builder',
																1 => 'scriptaculous-effects',
														),
												'ver' => '1.9.0',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'scriptaculous-effects' =>
										array(
												'handle' => 'scriptaculous-effects',
												'src' => 'https://ajax.googleapis.com/ajax/libs/scriptaculous/1.9.0/effects.js',
												'deps' =>
														array (
																0 => 'scriptaculous-root',
														),
												'ver' => '1.9.0',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'scriptaculous-slider' =>
										array(
												'handle' => 'scriptaculous-slider',
												'src' => 'https://ajax.googleapis.com/ajax/libs/scriptaculous/1.9.0/slider.js',
												'deps' =>
														array (
																0 => 'scriptaculous-effects',
														),
												'ver' => '1.9.0',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'scriptaculous-sound' =>
										array(
												'handle' => 'scriptaculous-sound',
												'src' => 'https://ajax.googleapis.com/ajax/libs/scriptaculous/1.9.0/sound.js',
												'deps' =>
														array (
																0 => 'scriptaculous-root',
														),
												'ver' => '1.9.0',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'scriptaculous-controls' =>
										array(
												'handle' => 'scriptaculous-controls',
												'src' => 'https://ajax.googleapis.com/ajax/libs/scriptaculous/1.9.0/controls.js',
												'deps' =>
														array (
																0 => 'scriptaculous-root',
														),
												'ver' => '1.9.0',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'scriptaculous' =>
										array(
												'handle' => 'scriptaculous',
												'src' => false,
												'deps' =>
														array (
																0 => 'scriptaculous-dragdrop',
																1 => 'scriptaculous-slider',
																2 => 'scriptaculous-controls',
														),
												'ver' => false,
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'cropper' =>
										array(
												'handle' => 'cropper',
												'src' => '/wp-includes/js/crop/cropper.js',
												'deps' =>
														array (
																0 => 'scriptaculous-dragdrop',
														),
												'ver' => false,
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'jquery' =>
										array(
												'handle' => 'jquery',
												'src' => false,
												'deps' =>
														array (
																0 => 'jquery-core',
																1 => 'jquery-migrate',
														),
												'ver' => '1.11.3',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'jquery-core' =>
										array(
												'handle' => 'jquery-core',
												'src' => '/wp-includes/js/jquery/jquery.js',
												'deps' =>
														array (
														),
												'ver' => '1.11.3',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'jquery-migrate' =>
										array(
												'handle' => 'jquery-migrate',
												'src' => '/wp-includes/js/jquery/jquery-migrate.min.js',
												'deps' =>
														array (
														),
												'ver' => '1.2.1',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'jquery-ui-core' =>
										array(
												'handle' => 'jquery-ui-core',
												'src' => '/wp-includes/js/jquery/ui/core.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-core' =>
										array(
												'handle' => 'jquery-effects-core',
												'src' => '/wp-includes/js/jquery/ui/effect.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-blind' =>
										array(
												'handle' => 'jquery-effects-blind',
												'src' => '/wp-includes/js/jquery/ui/effect-blind.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-bounce' =>
										array(
												'handle' => 'jquery-effects-bounce',
												'src' => '/wp-includes/js/jquery/ui/effect-bounce.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-clip' =>
										array(
												'handle' => 'jquery-effects-clip',
												'src' => '/wp-includes/js/jquery/ui/effect-clip.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-drop' =>
										array(
												'handle' => 'jquery-effects-drop',
												'src' => '/wp-includes/js/jquery/ui/effect-drop.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-explode' =>
										array(
												'handle' => 'jquery-effects-explode',
												'src' => '/wp-includes/js/jquery/ui/effect-explode.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-fade' =>
										array(
												'handle' => 'jquery-effects-fade',
												'src' => '/wp-includes/js/jquery/ui/effect-fade.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-fold' =>
										array(
												'handle' => 'jquery-effects-fold',
												'src' => '/wp-includes/js/jquery/ui/effect-fold.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-highlight' =>
										array(
												'handle' => 'jquery-effects-highlight',
												'src' => '/wp-includes/js/jquery/ui/effect-highlight.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-puff' =>
										array(
												'handle' => 'jquery-effects-puff',
												'src' => '/wp-includes/js/jquery/ui/effect-puff.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
																1 => 'jquery-effects-scale',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-pulsate' =>
										array(
												'handle' => 'jquery-effects-pulsate',
												'src' => '/wp-includes/js/jquery/ui/effect-pulsate.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-scale' =>
										array(
												'handle' => 'jquery-effects-scale',
												'src' => '/wp-includes/js/jquery/ui/effect-scale.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
																1 => 'jquery-effects-size',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-shake' =>
										array(
												'handle' => 'jquery-effects-shake',
												'src' => '/wp-includes/js/jquery/ui/effect-shake.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-size' =>
										array(
												'handle' => 'jquery-effects-size',
												'src' => '/wp-includes/js/jquery/ui/effect-size.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-slide' =>
										array(
												'handle' => 'jquery-effects-slide',
												'src' => '/wp-includes/js/jquery/ui/effect-slide.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-effects-transfer' =>
										array(
												'handle' => 'jquery-effects-transfer',
												'src' => '/wp-includes/js/jquery/ui/effect-transfer.min.js',
												'deps' =>
														array (
																0 => 'jquery-effects-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-accordion' =>
										array(
												'handle' => 'jquery-ui-accordion',
												'src' => '/wp-includes/js/jquery/ui/accordion.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-core',
																1 => 'jquery-ui-widget',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-autocomplete' =>
										array(
												'handle' => 'jquery-ui-autocomplete',
												'src' => '/wp-includes/js/jquery/ui/autocomplete.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-menu',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-button' =>
										array(
												'handle' => 'jquery-ui-button',
												'src' => '/wp-includes/js/jquery/ui/button.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-core',
																1 => 'jquery-ui-widget',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-datepicker' =>
										array(
												'handle' => 'jquery-ui-datepicker',
												'src' => '/wp-includes/js/jquery/ui/datepicker.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-core',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-dialog' =>
										array(
												'handle' => 'jquery-ui-dialog',
												'src' => '/wp-includes/js/jquery/ui/dialog.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-resizable',
																1 => 'jquery-ui-draggable',
																2 => 'jquery-ui-button',
																3 => 'jquery-ui-position',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-draggable' =>
										array(
												'handle' => 'jquery-ui-draggable',
												'src' => '/wp-includes/js/jquery/ui/draggable.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-mouse',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-droppable' =>
										array(
												'handle' => 'jquery-ui-droppable',
												'src' => '/wp-includes/js/jquery/ui/droppable.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-draggable',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-menu' =>
										array(
												'handle' => 'jquery-ui-menu',
												'src' => '/wp-includes/js/jquery/ui/menu.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-core',
																1 => 'jquery-ui-widget',
																2 => 'jquery-ui-position',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-mouse' =>
										array(
												'handle' => 'jquery-ui-mouse',
												'src' => '/wp-includes/js/jquery/ui/mouse.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-core',
																1 => 'jquery-ui-widget',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-position' =>
										array(
												'handle' => 'jquery-ui-position',
												'src' => '/wp-includes/js/jquery/ui/position.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-progressbar' =>
										array(
												'handle' => 'jquery-ui-progressbar',
												'src' => '/wp-includes/js/jquery/ui/progressbar.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-core',
																1 => 'jquery-ui-widget',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-resizable' =>
										array(
												'handle' => 'jquery-ui-resizable',
												'src' => '/wp-includes/js/jquery/ui/resizable.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-mouse',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-selectable' =>
										array(
												'handle' => 'jquery-ui-selectable',
												'src' => '/wp-includes/js/jquery/ui/selectable.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-mouse',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-selectmenu' =>
										array(
												'handle' => 'jquery-ui-selectmenu',
												'src' => '/wp-includes/js/jquery/ui/selectmenu.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-menu',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-slider' =>
										array(
												'handle' => 'jquery-ui-slider',
												'src' => '/wp-includes/js/jquery/ui/slider.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-mouse',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-sortable' =>
										array(
												'handle' => 'jquery-ui-sortable',
												'src' => '/wp-includes/js/jquery/ui/sortable.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-mouse',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-spinner' =>
										array(
												'handle' => 'jquery-ui-spinner',
												'src' => '/wp-includes/js/jquery/ui/spinner.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-button',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-tabs' =>
										array(
												'handle' => 'jquery-ui-tabs',
												'src' => '/wp-includes/js/jquery/ui/tabs.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-core',
																1 => 'jquery-ui-widget',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-tooltip' =>
										array(
												'handle' => 'jquery-ui-tooltip',
												'src' => '/wp-includes/js/jquery/ui/tooltip.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-core',
																1 => 'jquery-ui-widget',
																2 => 'jquery-ui-position',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-ui-widget' =>
										array(
												'handle' => 'jquery-ui-widget',
												'src' => '/wp-includes/js/jquery/ui/widget.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '1.11.4',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-form' =>
										array(
												'handle' => 'jquery-form',
												'src' => '/wp-includes/js/jquery/jquery.form.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '3.37.0',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-color' =>
										array(
												'handle' => 'jquery-color',
												'src' => '/wp-includes/js/jquery/jquery.color.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '2.1.1',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'suggest' =>
										array(
												'handle' => 'suggest',
												'src' => '/wp-includes/js/jquery/suggest.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '1.1-20110113',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'schedule' =>
										array(
												'handle' => 'schedule',
												'src' => '/wp-includes/js/jquery/jquery.schedule.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '20m',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-query' =>
										array(
												'handle' => 'jquery-query',
												'src' => '/wp-includes/js/jquery/jquery.query.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '2.1.7',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-serialize-object' =>
										array(
												'handle' => 'jquery-serialize-object',
												'src' => '/wp-includes/js/jquery/jquery.serialize-object.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '0.2',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-hotkeys' =>
										array(
												'handle' => 'jquery-hotkeys',
												'src' => '/wp-includes/js/jquery/jquery.hotkeys.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '0.0.2m',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-table-hotkeys' =>
										array(
												'handle' => 'jquery-table-hotkeys',
												'src' => '/wp-includes/js/jquery/jquery.table-hotkeys.min.js',
												'deps' =>
														array (
																0 => 'jquery',
																1 => 'jquery-hotkeys',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-touch-punch' =>
										array(
												'handle' => 'jquery-touch-punch',
												'src' => '/wp-includes/js/jquery/jquery.ui.touch-punch.js',
												'deps' =>
														array (
																0 => 'jquery-ui-widget',
																1 => 'jquery-ui-mouse',
														),
												'ver' => '0.2.2',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'masonry' =>
										array(
												'handle' => 'masonry',
												'src' => '/wp-includes/js/masonry.min.js',
												'deps' =>
														array (
														),
												'ver' => '3.1.2',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'jquery-masonry' =>
										array(
												'handle' => 'jquery-masonry',
												'src' => '/wp-includes/js/jquery/jquery.masonry.min.js',
												'deps' =>
														array (
																0 => 'jquery',
																1 => 'masonry',
														),
												'ver' => '3.1.2',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'thickbox' =>
										array(
												'handle' => 'thickbox',
												'src' => '/wp-includes/js/thickbox/thickbox.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '3.1-20121105',
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var thickboxL10n = {"next":"Next >","prev":"< Prev","image":"Image","of":"of","close":"Close","noiframes":"This feature requires inline frames. You have iframes disabled or your browser does not support them.","loadingAnimation":"http:\\/\\/humming.local.wordpress-wpmudev.dev\\/wp-includes\\/js\\/thickbox\\/loadingAnimation.gif"};',
														),
										),
								'jcrop' =>
										array(
												'handle' => 'jcrop',
												'src' => '/wp-includes/js/jcrop/jquery.Jcrop.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '0.9.12',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'swfobject' =>
										array(
												'handle' => 'swfobject',
												'src' => '/wp-includes/js/swfobject.js',
												'deps' =>
														array (
														),
												'ver' => '2.2-20120417',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'plupload' =>
										array(
												'handle' => 'plupload',
												'src' => '/wp-includes/js/plupload/plupload.full.min.js',
												'deps' =>
														array (
														),
												'ver' => '2.1.8',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'plupload-all' =>
										array(
												'handle' => 'plupload-all',
												'src' => false,
												'deps' =>
														array (
																0 => 'plupload',
														),
												'ver' => '2.1.1',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'plupload-html5' =>
										array(
												'handle' => 'plupload-html5',
												'src' => false,
												'deps' =>
														array (
																0 => 'plupload',
														),
												'ver' => '2.1.1',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'plupload-flash' =>
										array(
												'handle' => 'plupload-flash',
												'src' => false,
												'deps' =>
														array (
																0 => 'plupload',
														),
												'ver' => '2.1.1',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'plupload-silverlight' =>
										array(
												'handle' => 'plupload-silverlight',
												'src' => false,
												'deps' =>
														array (
																0 => 'plupload',
														),
												'ver' => '2.1.1',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'plupload-html4' =>
										array(
												'handle' => 'plupload-html4',
												'src' => false,
												'deps' =>
														array (
																0 => 'plupload',
														),
												'ver' => '2.1.1',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'plupload-handlers' =>
										array(
												'handle' => 'plupload-handlers',
												'src' => '/wp-includes/js/plupload/handlers.min.js',
												'deps' =>
														array (
																0 => 'plupload',
																1 => 'jquery',
														),
												'ver' => false,
												'args' => NULL,
												'extra' =>
														array (
																'data' => 'var pluploadL10n = {"queue_limit_exceeded":"You have attempted to queue too many files.","file_exceeds_size_limit":"%s exceeds the maximum upload size for this site.","zero_byte_file":"This file is empty. Please try another.","invalid_filetype":"This file type is not allowed. Please try another.","not_an_image":"This file is not an image. Please try another.","image_memory_exceeded":"Memory exceeded. Please try another smaller file.","image_dimensions_exceeded":"This is larger than the maximum size. Please try another.","default_error":"An error occurred in the upload. Please try again later.","missing_upload_url":"There was a configuration error. Please contact the server administrator.","upload_limit_exceeded":"You may only upload 1 file.","http_error":"HTTP error.","upload_failed":"Upload failed.","big_upload_failed":"Please try uploading this file with the %1$sbrowser uploader%2$s.","big_upload_queued":"%s exceeds the maximum upload size for the multi-file uploader when used in your browser.","io_error":"IO error.","security_error":"Security error.","file_cancelled":"File canceled.","upload_stopped":"Upload stopped.","dismiss":"Dismiss","crunching":"Crunching\\u2026","deleted":"moved to the trash.","error_uploading":"\\u201c%s\\u201d has failed to upload."};',
														),
										),
								'wp-plupload' =>
										array(
												'handle' => 'wp-plupload',
												'src' => '/wp-includes/js/plupload/wp-plupload.min.js',
												'deps' =>
														array (
																0 => 'plupload',
																1 => 'jquery',
																2 => 'json2',
																3 => 'media-models',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var pluploadL10n = {"queue_limit_exceeded":"You have attempted to queue too many files.","file_exceeds_size_limit":"%s exceeds the maximum upload size for this site.","zero_byte_file":"This file is empty. Please try another.","invalid_filetype":"This file type is not allowed. Please try another.","not_an_image":"This file is not an image. Please try another.","image_memory_exceeded":"Memory exceeded. Please try another smaller file.","image_dimensions_exceeded":"This is larger than the maximum size. Please try another.","default_error":"An error occurred in the upload. Please try again later.","missing_upload_url":"There was a configuration error. Please contact the server administrator.","upload_limit_exceeded":"You may only upload 1 file.","http_error":"HTTP error.","upload_failed":"Upload failed.","big_upload_failed":"Please try uploading this file with the %1$sbrowser uploader%2$s.","big_upload_queued":"%s exceeds the maximum upload size for the multi-file uploader when used in your browser.","io_error":"IO error.","security_error":"Security error.","file_cancelled":"File canceled.","upload_stopped":"Upload stopped.","dismiss":"Dismiss","crunching":"Crunching\\u2026","deleted":"moved to the trash.","error_uploading":"\\u201c%s\\u201d has failed to upload."};',
														),
										),
								'swfupload' =>
										array(
												'handle' => 'swfupload',
												'src' => '/wp-includes/js/swfupload/swfupload.js',
												'deps' =>
														array (
														),
												'ver' => '2201-20110113',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'swfupload-swfobject' =>
										array(
												'handle' => 'swfupload-swfobject',
												'src' => '/wp-includes/js/swfupload/plugins/swfupload.swfobject.js',
												'deps' =>
														array (
																0 => 'swfupload',
																1 => 'swfobject',
														),
												'ver' => '2201a',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'swfupload-queue' =>
										array(
												'handle' => 'swfupload-queue',
												'src' => '/wp-includes/js/swfupload/plugins/swfupload.queue.js',
												'deps' =>
														array (
																0 => 'swfupload',
														),
												'ver' => '2201',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'swfupload-speed' =>
										array(
												'handle' => 'swfupload-speed',
												'src' => '/wp-includes/js/swfupload/plugins/swfupload.speed.js',
												'deps' =>
														array (
																0 => 'swfupload',
														),
												'ver' => '2201',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'swfupload-all' =>
										array(
												'handle' => 'swfupload-all',
												'src' => false,
												'deps' =>
														array (
																0 => 'swfupload',
																1 => 'swfupload-swfobject',
																2 => 'swfupload-queue',
														),
												'ver' => '2201',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'swfupload-handlers' =>
										array(
												'handle' => 'swfupload-handlers',
												'src' => '/wp-includes/js/swfupload/handlers.min.js',
												'deps' =>
														array (
																0 => 'swfupload-all',
																1 => 'jquery',
														),
												'ver' => '2201-20110524',
												'args' => NULL,
												'extra' =>
														array (
																'data' => 'var swfuploadL10n = {"queue_limit_exceeded":"You have attempted to queue too many files.","file_exceeds_size_limit":"%s exceeds the maximum upload size for this site.","zero_byte_file":"This file is empty. Please try another.","invalid_filetype":"This file type is not allowed. Please try another.","not_an_image":"This file is not an image. Please try another.","image_memory_exceeded":"Memory exceeded. Please try another smaller file.","image_dimensions_exceeded":"This is larger than the maximum size. Please try another.","default_error":"An error occurred in the upload. Please try again later.","missing_upload_url":"There was a configuration error. Please contact the server administrator.","upload_limit_exceeded":"You may only upload 1 file.","http_error":"HTTP error.","upload_failed":"Upload failed.","big_upload_failed":"Please try uploading this file with the %1$sbrowser uploader%2$s.","big_upload_queued":"%s exceeds the maximum upload size for the multi-file uploader when used in your browser.","io_error":"IO error.","security_error":"Security error.","file_cancelled":"File canceled.","upload_stopped":"Upload stopped.","dismiss":"Dismiss","crunching":"Crunching\\u2026","deleted":"moved to the trash.","error_uploading":"\\u201c%s\\u201d has failed to upload."};',
														),
										),
								'comment-reply' =>
										array(
												'handle' => 'comment-reply',
												'src' => '/wp-includes/js/comment-reply.min.js',
												'deps' =>
														array (
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'json2' =>
										array(
												'handle' => 'json2',
												'src' => '/wp-includes/js/json2.min.js',
												'deps' =>
														array (
														),
												'ver' => '2015-05-03',
												'args' => NULL,
												'extra' =>
														array (
																'conditional' => 'lt IE 8',
														),
										),
								'underscore' =>
										array(
												'handle' => 'underscore',
												'src' => '/wp-includes/js/underscore.min.js',
												'deps' =>
														array (
														),
												'ver' => '1.6.0',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'backbone' =>
										array(
												'handle' => 'backbone',
												'src' => '/wp-includes/js/backbone.min.js',
												'deps' =>
														array (
																0 => 'underscore',
																1 => 'jquery',
														),
												'ver' => '1.1.2',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'wp-util' =>
										array(
												'handle' => 'wp-util',
												'src' => '/wp-includes/js/wp-util.min.js',
												'deps' =>
														array (
																0 => 'underscore',
																1 => 'jquery',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var _wpUtilSettings = {"ajax":{"url":"\\/wp-admin\\/admin-ajax.php"}};',
														),
										),
								'wp-backbone' =>
										array(
												'handle' => 'wp-backbone',
												'src' => '/wp-includes/js/wp-backbone.min.js',
												'deps' =>
														array (
																0 => 'backbone',
																1 => 'wp-util',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'revisions' =>
										array(
												'handle' => 'revisions',
												'src' => '/wp-admin/js/revisions.min.js',
												'deps' =>
														array (
																0 => 'wp-backbone',
																1 => 'jquery-ui-slider',
																2 => 'hoverIntent',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'imgareaselect' =>
										array(
												'handle' => 'imgareaselect',
												'src' => '/wp-includes/js/imgareaselect/jquery.imgareaselect.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'mediaelement' =>
										array(
												'handle' => 'mediaelement',
												'src' => '/wp-includes/js/mediaelement/mediaelement-and-player.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '2.18.1',
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var mejsL10n = {"language":"en-US","strings":{"Close":"Close","Fullscreen":"Fullscreen","Download File":"Download File","Download Video":"Download Video","Play\\/Pause":"Play\\/Pause","Mute Toggle":"Mute Toggle","None":"None","Turn off Fullscreen":"Turn off Fullscreen","Go Fullscreen":"Go Fullscreen","Unmute":"Unmute","Mute":"Mute","Captions\\/Subtitles":"Captions\\/Subtitles"}};
var _wpmejsSettings = {"pluginPath":"\\/wp-includes\\/js\\/mediaelement\\/"};',
														),
										),
								'wp-mediaelement' =>
										array(
												'handle' => 'wp-mediaelement',
												'src' => '/wp-includes/js/mediaelement/wp-mediaelement.js',
												'deps' =>
														array (
																0 => 'mediaelement',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'froogaloop' =>
										array(
												'handle' => 'froogaloop',
												'src' => '/wp-includes/js/mediaelement/froogaloop.min.js',
												'deps' =>
														array (
														),
												'ver' => '2.0',
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'wp-playlist' =>
										array(
												'handle' => 'wp-playlist',
												'src' => '/wp-includes/js/mediaelement/wp-playlist.js',
												'deps' =>
														array (
																0 => 'wp-util',
																1 => 'backbone',
																2 => 'mediaelement',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'zxcvbn-async' =>
										array(
												'handle' => 'zxcvbn-async',
												'src' => '/wp-includes/js/zxcvbn-async.min.js',
												'deps' =>
														array (
														),
												'ver' => '1.0',
												'args' => NULL,
												'extra' =>
														array (
																'data' => 'var _zxcvbnSettings = {"src":"http:\\/\\/humming.local.wordpress-wpmudev.dev\\/wp-includes\\/js\\/zxcvbn.min.js"};',
														),
										),
								'password-strength-meter' =>
										array(
												'handle' => 'password-strength-meter',
												'src' => '/wp-admin/js/password-strength-meter.min.js',
												'deps' =>
														array (
																0 => 'jquery',
																1 => 'zxcvbn-async',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var pwsL10n = {"short":"Very weak","bad":"Weak","good":"Medium","strong":"Strong","mismatch":"Mismatch"};',
														),
										),
								'user-profile' =>
										array(
												'handle' => 'user-profile',
												'src' => '/wp-admin/js/user-profile.min.js',
												'deps' =>
														array (
																0 => 'jquery',
																1 => 'password-strength-meter',
																2 => 'wp-util',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var userProfileL10n = {"warn":"Your new password has not been saved.","show":"Show","hide":"Hide","cancel":"Cancel","ariaShow":"Show password","ariaHide":"Hide password"};',
														),
										),
								'language-chooser' =>
										array(
												'handle' => 'language-chooser',
												'src' => '/wp-admin/js/language-chooser.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'user-suggest' =>
										array(
												'handle' => 'user-suggest',
												'src' => '/wp-admin/js/user-suggest.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-autocomplete',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'admin-bar' =>
										array(
												'handle' => 'admin-bar',
												'src' => '/wp-includes/js/admin-bar.min.js',
												'deps' =>
														array (
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'wplink' =>
										array(
												'handle' => 'wplink',
												'src' => '/wp-includes/js/wplink.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var wpLinkL10n = {"title":"Insert\\/edit link","update":"Update","save":"Add Link","noTitle":"(no title)","noMatchesFound":"No results found."};',
														),
										),
								'wpdialogs' =>
										array(
												'handle' => 'wpdialogs',
												'src' => '/wp-includes/js/wpdialog.min.js',
												'deps' =>
														array (
																0 => 'jquery-ui-dialog',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'word-count' =>
										array(
												'handle' => 'word-count',
												'src' => '/wp-admin/js/word-count.min.js',
												'deps' =>
														array (
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var wordCountL10n = {"type":"words","shortcodes":["embed","wp_caption","caption","gallery","playlist","audio","video","areas","mp_tag_cloud","mp_list_categories","mp_dropdown_categories","mp_popular_products","mp_related_products","mp_list_products","mp_product","mp_product_image","mp_buy_button","mp_product_price","mp_product_meta","mp_product_sku","mp_product_stock","mp_cart","mp_cart_widget","mp_checkout","mp_order_status","mp_order_lookup_form","mp_cart_link","mp_store_link","mp_products_link","mp_orderstatus_link","mp_store_navigation"]};',
														),
										),
								'media-upload' =>
										array(
												'handle' => 'media-upload',
												'src' => '/wp-admin/js/media-upload.min.js',
												'deps' =>
														array (
																0 => 'thickbox',
																1 => 'shortcode',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'hoverIntent' =>
										array(
												'handle' => 'hoverIntent',
												'src' => '/wp-includes/js/hoverIntent.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '1.8.1',
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'customize-base' =>
										array(
												'handle' => 'customize-base',
												'src' => '/wp-includes/js/customize-base.min.js',
												'deps' =>
														array (
																0 => 'jquery',
																1 => 'json2',
																2 => 'underscore',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'customize-loader' =>
										array(
												'handle' => 'customize-loader',
												'src' => '/wp-includes/js/customize-loader.min.js',
												'deps' =>
														array (
																0 => 'customize-base',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'customize-preview' =>
										array(
												'handle' => 'customize-preview',
												'src' => '/wp-includes/js/customize-preview.min.js',
												'deps' =>
														array (
																0 => 'customize-base',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'customize-models' =>
										array(
												'handle' => 'customize-models',
												'src' => '/wp-includes/js/customize-models.js',
												'deps' =>
														array (
																0 => 'underscore',
																1 => 'backbone',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'customize-views' =>
										array(
												'handle' => 'customize-views',
												'src' => '/wp-includes/js/customize-views.js',
												'deps' =>
														array (
																0 => 'jquery',
																1 => 'underscore',
																2 => 'imgareaselect',
																3 => 'customize-models',
																4 => 'media-editor',
																5 => 'media-views',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'customize-controls' =>
										array(
												'handle' => 'customize-controls',
												'src' => '/wp-admin/js/customize-controls.min.js',
												'deps' =>
														array (
																0 => 'customize-base',
																1 => 'wp-a11y',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var _wpCustomizeControlsL10n = {"activate":"Save & Activate","save":"Save & Publish","saveAlert":"The changes you made will be lost if you navigate away from this page.","saved":"Saved","cancel":"Cancel","close":"Close","cheatin":"Cheatin\\u2019 uh?","notAllowed":"You are not allowed to customize the appearance of this site.","previewIframeTitle":"Site Preview","loginIframeTitle":"Session expired","collapseSidebar":"Collapse Sidebar","expandSidebar":"Expand Sidebar","allowedFiles":"Allowed Files"};',
														),
										),
								'customize-widgets' =>
										array(
												'handle' => 'customize-widgets',
												'src' => '/wp-admin/js/customize-widgets.min.js',
												'deps' =>
														array (
																0 => 'jquery',
																1 => 'jquery-ui-sortable',
																2 => 'jquery-ui-droppable',
																3 => 'wp-backbone',
																4 => 'customize-controls',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'customize-preview-widgets' =>
										array(
												'handle' => 'customize-preview-widgets',
												'src' => '/wp-includes/js/customize-preview-widgets.min.js',
												'deps' =>
														array (
																0 => 'jquery',
																1 => 'wp-util',
																2 => 'customize-preview',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'customize-nav-menus' =>
										array(
												'handle' => 'customize-nav-menus',
												'src' => '/wp-admin/js/customize-nav-menus.min.js',
												'deps' =>
														array (
																0 => 'jquery',
																1 => 'wp-backbone',
																2 => 'customize-controls',
																3 => 'accordion',
																4 => 'nav-menu',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'customize-preview-nav-menus' =>
										array(
												'handle' => 'customize-preview-nav-menus',
												'src' => '/wp-includes/js/customize-preview-nav-menus.min.js',
												'deps' =>
														array (
																0 => 'jquery',
																1 => 'wp-util',
																2 => 'customize-preview',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'accordion' =>
										array(
												'handle' => 'accordion',
												'src' => '/wp-admin/js/accordion.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'shortcode' =>
										array(
												'handle' => 'shortcode',
												'src' => '/wp-includes/js/shortcode.min.js',
												'deps' =>
														array (
																0 => 'underscore',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'media-models' =>
										array(
												'handle' => 'media-models',
												'src' => '/wp-includes/js/media-models.min.js',
												'deps' =>
														array (
																0 => 'wp-backbone',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
																'data' => 'var _wpMediaModelsL10n = {"settings":{"ajaxurl":"\\/wp-admin\\/admin-ajax.php","post":{"id":0}}};',
														),
										),
								'wp-embed' =>
										array(
												'handle' => 'wp-embed',
												'src' => '/wp-includes/js/wp-embed.min.js',
												'deps' =>
														array (
														),
												'ver' => false,
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'media-views' =>
										array(
												'handle' => 'media-views',
												'src' => '/wp-includes/js/media-views.min.js',
												'deps' =>
														array (
																0 => 'utils',
																1 => 'media-models',
																2 => 'wp-plupload',
																3 => 'jquery-ui-sortable',
																4 => 'wp-mediaelement',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'media-editor' =>
										array(
												'handle' => 'media-editor',
												'src' => '/wp-includes/js/media-editor.min.js',
												'deps' =>
														array (
																0 => 'shortcode',
																1 => 'media-views',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'media-audiovideo' =>
										array(
												'handle' => 'media-audiovideo',
												'src' => '/wp-includes/js/media-audiovideo.min.js',
												'deps' =>
														array (
																0 => 'media-editor',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'mce-view' =>
										array(
												'handle' => 'mce-view',
												'src' => '/wp-includes/js/mce-view.min.js',
												'deps' =>
														array (
																0 => 'shortcode',
																1 => 'jquery',
																2 => 'media-views',
																3 => 'media-audiovideo',
														),
												'ver' => false,
												'args' => 1,
												'extra' =>
														array (
														),
										),
								'debug-bar' =>
										array(
												'handle' => 'debug-bar',
												'src' => false,
												'deps' =>
														array (
																0 => 'query-monitor',
														),
												'ver' => false,
												'args' => NULL,
												'extra' =>
														array (
														),
										),
								'twentyfifteen-skip-link-focus-fix' =>
										array(
												'handle' => 'twentyfifteen-skip-link-focus-fix',
												'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/themes/twentyfifteen/js/skip-link-focus-fix.js',
												'deps' =>
														array (
														),
												'ver' => '20141010',
												'args' => NULL,
												'extra' =>
														array (
																'group' => 1,
														),
										),
								'twentyfifteen-script' =>
										array(
												'handle' => 'twentyfifteen-script',
												'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/themes/twentyfifteen/js/functions.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '20150330',
												'args' => NULL,
												'extra' =>
														array (
																'group' => 1,
																'data' => 'var screenReaderText = {"expand":"<span class=\\"screen-reader-text\\">expand child menu<\\/span>","collapse":"<span class=\\"screen-reader-text\\">collapse child menu<\\/span>"};',
														),
										),
								'jquery-validate' =>
										array(
												'handle' => 'jquery-validate',
												'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/marketpress/ui/js/jquery.validate.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '3.0.0.7',
												'args' => NULL,
												'extra' =>
														array (
																'group' => 1,
														),
										),
								'jquery-validate-methods' =>
										array(
												'handle' => 'jquery-validate-methods',
												'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/marketpress/ui/js/jquery.validate.methods.min.js',
												'deps' =>
														array (
																0 => 'jquery-validate',
														),
												'ver' => '3.0.0.7',
												'args' => NULL,
												'extra' =>
														array (
																'group' => 1,
														),
										),
								'ajaxq' =>
										array(
												'handle' => 'ajaxq',
												'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/marketpress/ui/js/ajaxq.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '3.0.0.7',
												'args' => NULL,
												'extra' =>
														array (
																'group' => 1,
														),
										),
								'colorbox' =>
										array(
												'handle' => 'colorbox',
												'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/marketpress/ui/js/jquery.colorbox-min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '3.0.0.7',
												'args' => NULL,
												'extra' =>
														array (
																'group' => 1,
														),
										),
								'mp-cart' =>
										array(
												'handle' => 'mp-cart',
												'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/marketpress/ui/js/mp-cart.js',
												'deps' =>
														array (
																0 => 'ajaxq',
																1 => 'colorbox',
																2 => 'jquery-validate',
														),
												'ver' => '3.0.0.7',
												'args' => NULL,
												'extra' =>
														array (
																'data' => 'var mp_cart_i18n = {"ajaxurl":"http:\\/\\/humming.local.wordpress-wpmudev.dev\\/wp-admin\\/admin-ajax.php","ajax_loader":"<span class=\\"mp_ajax_loader\\"><img src=\\"http:\\/\\/humming.local.wordpress-wpmudev.dev\\/wp-content\\/plugins\\/marketpress\\/ui\\/images\\/ajax-loader.gif\\" alt=\\"\\"> Adding...<\\/span>","cart_updated_error_limit":"Cart update notice: this item has a limit per order.","is_cart_page":""};',
														),
										),
								'hover-intent' =>
										array(
												'handle' => 'hover-intent',
												'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/marketpress/ui/js/hoverintent.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '3.0.0.7',
												'args' => NULL,
												'extra' =>
														array (
																'group' => 1,
														),
										),
								'mp-select2' =>
										array(
												'handle' => 'mp-select2',
												'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/marketpress/ui/select2/select2.min.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => '3.0.0.7',
												'args' => NULL,
												'extra' =>
														array (
																'group' => 1,
														),
										),
								'mp-frontend' =>
										array(
												'handle' => 'mp-frontend',
												'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/marketpress/ui/js/frontend.js',
												'deps' =>
														array (
																0 => 'jquery-ui-tooltip',
																1 => 'colorbox',
																2 => 'hover-intent',
																3 => 'mp-select2',
														),
												'ver' => '3.0.0.7',
												'args' => NULL,
												'extra' =>
														array (
																'data' => 'var mp_i18n = {"ajaxurl":"http:\\/\\/humming.local.wordpress-wpmudev.dev\\/wp-admin\\/admin-ajax.php","loadingImage":"http:\\/\\/humming.local.wordpress-wpmudev.dev\\/wp-content\\/plugins\\/marketpress\\/ui\\/images\\/loading.gif","productsURL":"","productCats":[]};',
														),
										),
								'query-monitor' =>
										array(
												'handle' => 'query-monitor',
												'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/query-monitor/assets/query-monitor.js',
												'deps' =>
														array (
																0 => 'jquery',
														),
												'ver' => 1448962880,
												'args' => NULL,
												'extra' =>
														array (
																'group' => 1,
																'data' => 'var qm_locale = {"weekday":["Sunday","Monday","Tuesday","Wednesday","Thursday","Friday","Saturday"],"weekday_initial":{"Sunday":"S","Monday":"M","Tuesday":"T","Wednesday":"W","Thursday":"T","Friday":"F","Saturday":"S"},"weekday_abbrev":{"Sunday":"Sun","Monday":"Mon","Tuesday":"Tue","Wednesday":"Wed","Thursday":"Thu","Friday":"Fri","Saturday":"Sat"},"start_of_week":null,"month":{"01":"January","02":"February","03":"March","04":"April","05":"May","06":"June","07":"July","08":"August","09":"September","10":"October","11":"November","12":"December"},"month_abbrev":{"January":"Jan","February":"Feb","March":"Mar","April":"Apr","May":"May","June":"Jun","July":"Jul","August":"Aug","September":"Sep","October":"Oct","November":"Nov","December":"Dec"},"meridiem":{"am":"am","pm":"pm","AM":"AM","PM":"PM"},"text_direction":"ltr","number_format":{"thousands_sep":",","decimal_point":"."},"month_genitive":{"01":"January","02":"February","03":"March","04":"April","05":"May","06":"June","07":"July","08":"August","09":"September","10":"October","11":"November","12":"December"}};
var qm_l10n = {"ajax_error":"PHP Error in AJAX Response","infinitescroll_paused":"Infinite Scroll has been paused by Query Monitor","ajaxurl":"http:\\/\\/humming.local.wordpress-wpmudev.dev\\/wp-admin\\/admin-ajax.php","auth_nonce":{"on":"f02259cf7b","off":"1248a7adf7"}};',
														),
										),
						);
	}

	function get_registered_styles() {
		return array (
				'colors' =>
						array(
								'handle' => 'colors',
								'src' => true,
								'deps' =>
										array (
												0 => 'wp-admin',
												1 => 'buttons',
												2 => 'open-sans',
												3 => 'dashicons',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
										),
						),
				'wp-admin' =>
						array(
								'handle' => 'wp-admin',
								'src' => '/wp-admin/css/wp-admin.min.css',
								'deps' =>
										array (
												0 => 'open-sans',
												1 => 'dashicons',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'login' =>
						array(
								'handle' => 'login',
								'src' => '/wp-admin/css/login.min.css',
								'deps' =>
										array (
												0 => 'buttons',
												1 => 'open-sans',
												2 => 'dashicons',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'install' =>
						array(
								'handle' => 'install',
								'src' => '/wp-admin/css/install.min.css',
								'deps' =>
										array (
												0 => 'buttons',
												1 => 'open-sans',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'wp-color-picker' =>
						array(
								'handle' => 'wp-color-picker',
								'src' => '/wp-admin/css/color-picker.min.css',
								'deps' =>
										array (
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'customize-controls' =>
						array(
								'handle' => 'customize-controls',
								'src' => '/wp-admin/css/customize-controls.min.css',
								'deps' =>
										array (
												0 => 'wp-admin',
												1 => 'colors',
												2 => 'ie',
												3 => 'imgareaselect',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'customize-widgets' =>
						array(
								'handle' => 'customize-widgets',
								'src' => '/wp-admin/css/customize-widgets.min.css',
								'deps' =>
										array (
												0 => 'wp-admin',
												1 => 'colors',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'customize-nav-menus' =>
						array(
								'handle' => 'customize-nav-menus',
								'src' => '/wp-admin/css/customize-nav-menus.min.css',
								'deps' =>
										array (
												0 => 'wp-admin',
												1 => 'colors',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'press-this' =>
						array(
								'handle' => 'press-this',
								'src' => '/wp-admin/css/press-this.min.css',
								'deps' =>
										array (
												0 => 'open-sans',
												1 => 'buttons',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'ie' =>
						array(
								'handle' => 'ie',
								'src' => '/wp-admin/css/ie.min.css',
								'deps' =>
										array (
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'conditional' => 'lte IE 7',
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'buttons' =>
						array(
								'handle' => 'buttons',
								'src' => '/wp-includes/css/buttons.min.css',
								'deps' =>
										array (
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'dashicons' =>
						array(
								'handle' => 'dashicons',
								'src' => '/wp-includes/css/dashicons.min.css',
								'deps' =>
										array (
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
										),
						),
				'open-sans' =>
						array(
								'handle' => 'open-sans',
								'src' => 'https://fonts.googleapis.com/css?family=Open+Sans:300italic,400italic,600italic,300,400,600&subset=latin,latin-ext',
								'deps' =>
										array (
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
										),
						),
				'admin-bar' =>
						array(
								'handle' => 'admin-bar',
								'src' => '/wp-includes/css/admin-bar.min.css',
								'deps' =>
										array (
												0 => 'open-sans',
												1 => 'dashicons',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'wp-auth-check' =>
						array(
								'handle' => 'wp-auth-check',
								'src' => '/wp-includes/css/wp-auth-check.min.css',
								'deps' =>
										array (
												0 => 'dashicons',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'editor-buttons' =>
						array(
								'handle' => 'editor-buttons',
								'src' => '/wp-includes/css/editor.min.css',
								'deps' =>
										array (
												0 => 'dashicons',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'media-views' =>
						array(
								'handle' => 'media-views',
								'src' => '/wp-includes/css/media-views.min.css',
								'deps' =>
										array (
												0 => 'buttons',
												1 => 'dashicons',
												2 => 'wp-mediaelement',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'wp-pointer' =>
						array(
								'handle' => 'wp-pointer',
								'src' => '/wp-includes/css/wp-pointer.min.css',
								'deps' =>
										array (
												0 => 'dashicons',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'customize-preview' =>
						array(
								'handle' => 'customize-preview',
								'src' => '/wp-includes/css/customize-preview.min.css',
								'deps' =>
										array (
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
										),
						),
				'wp-embed-template-ie' =>
						array(
								'handle' => 'wp-embed-template-ie',
								'src' => '/wp-includes/css/wp-embed-template-ie.min.css',
								'deps' =>
										array (
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'conditional' => 'lte IE 8',
										),
						),
				'imgareaselect' =>
						array(
								'handle' => 'imgareaselect',
								'src' => '/wp-includes/js/imgareaselect/imgareaselect.css',
								'deps' =>
										array (
										),
								'ver' => '0.9.8',
								'args' => NULL,
								'extra' =>
										array (
										),
						),
				'wp-jquery-ui-dialog' =>
						array(
								'handle' => 'wp-jquery-ui-dialog',
								'src' => '/wp-includes/css/jquery-ui-dialog.min.css',
								'deps' =>
										array (
												0 => 'dashicons',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'mediaelement' =>
						array(
								'handle' => 'mediaelement',
								'src' => '/wp-includes/js/mediaelement/mediaelementplayer.min.css',
								'deps' =>
										array (
										),
								'ver' => '2.18.1',
								'args' => NULL,
								'extra' =>
										array (
										),
						),
				'wp-mediaelement' =>
						array(
								'handle' => 'wp-mediaelement',
								'src' => '/wp-includes/js/mediaelement/wp-mediaelement.css',
								'deps' =>
										array (
												0 => 'mediaelement',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
										),
						),
				'thickbox' =>
						array(
								'handle' => 'thickbox',
								'src' => '/wp-includes/js/thickbox/thickbox.css',
								'deps' =>
										array (
												0 => 'dashicons',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
										),
						),
				'media' =>
						array(
								'handle' => 'media',
								'src' => '/wp-admin/css/deprecated-media.min.css',
								'deps' =>
										array (
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'farbtastic' =>
						array(
								'handle' => 'farbtastic',
								'src' => '/wp-admin/css/farbtastic.css',
								'deps' =>
										array (
										),
								'ver' => '1.3u1',
								'args' => NULL,
								'extra' =>
										array (
												'rtl' => 'replace',
												'suffix' => '.min',
										),
						),
				'jcrop' =>
						array(
								'handle' => 'jcrop',
								'src' => '/wp-includes/js/jcrop/jquery.Jcrop.min.css',
								'deps' =>
										array (
										),
								'ver' => '0.9.12',
								'args' => NULL,
								'extra' =>
										array (
										),
						),
				'colors-fresh' =>
						array(
								'handle' => 'colors-fresh',
								'src' => false,
								'deps' =>
										array (
												0 => 'wp-admin',
												1 => 'buttons',
										),
								'ver' => false,
								'args' => NULL,
								'extra' =>
										array (
										),
						),
				'debug-bar' =>
						array(
								'handle' => 'debug-bar',
								'src' => false,
								'deps' =>
										array (
												0 => 'query-monitor',
										),
								'ver' => false,
								'args' => 'all',
								'extra' =>
										array (
										),
						),
				'twentyfifteen-fonts' =>
						array(
								'handle' => 'twentyfifteen-fonts',
								'src' => 'https://fonts.googleapis.com/css?family=Noto+Sans%3A400italic%2C700italic%2C400%2C700%7CNoto+Serif%3A400italic%2C700italic%2C400%2C700%7CInconsolata%3A400%2C700&subset=latin%2Clatin-ext',
								'deps' =>
										array (
										),
								'ver' => NULL,
								'args' => 'all',
								'extra' =>
										array (
										),
						),
				'genericons' =>
						array(
								'handle' => 'genericons',
								'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/themes/twentyfifteen/genericons/genericons.css',
								'deps' =>
										array (
										),
								'ver' => '3.2',
								'args' => 'all',
								'extra' =>
										array (
										),
						),
				'twentyfifteen-style' =>
						array(
								'handle' => 'twentyfifteen-style',
								'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/themes/twentyfifteen/style.css',
								'deps' =>
										array (
										),
								'ver' => false,
								'args' => 'all',
								'extra' =>
										array (
												'after' =>
														array (
																0 => '	/* Color Scheme */

	/* Background Color */
	body {
		background-color: #ffe5d1;
	}

	/* Sidebar Background Color */
	body:before,
	.site-header {
		background-color: #e53b51;
	}

	/* Box Background Color */
	.post-navigation,
	.pagination,
	.secondary,
	.site-footer,
	.hentry,
	.page-header,
	.page-content,
	.comments-area,
	.widecolumn {
		background-color: #ffffff;
	}

	/* Box Background Color */
	button,
	input[type="button"],
	input[type="reset"],
	input[type="submit"],
	.pagination .prev,
	.pagination .next,
	.widget_calendar tbody a,
	.widget_calendar tbody a:hover,
	.widget_calendar tbody a:focus,
	.page-links a,
	.page-links a:hover,
	.page-links a:focus,
	.sticky-post {
		color: #ffffff;
	}

	/* Main Text Color */
	button,
	input[type="button"],
	input[type="reset"],
	input[type="submit"],
	.pagination .prev,
	.pagination .next,
	.widget_calendar tbody a,
	.page-links a,
	.sticky-post {
		background-color: #352712;
	}

	/* Main Text Color */
	body,
	blockquote cite,
	blockquote small,
	a,
	.dropdown-toggle:after,
	.image-navigation a:hover,
	.image-navigation a:focus,
	.comment-navigation a:hover,
	.comment-navigation a:focus,
	.widget-title,
	.entry-footer a:hover,
	.entry-footer a:focus,
	.comment-metadata a:hover,
	.comment-metadata a:focus,
	.pingback .edit-link a:hover,
	.pingback .edit-link a:focus,
	.comment-list .reply a:hover,
	.comment-list .reply a:focus,
	.site-info a:hover,
	.site-info a:focus {
		color: #352712;
	}

	/* Main Text Color */
	.entry-content a,
	.entry-summary a,
	.page-content a,
	.comment-content a,
	.pingback .comment-body > a,
	.author-description a,
	.taxonomy-description a,
	.textwidget a,
	.entry-footer a:hover,
	.comment-metadata a:hover,
	.pingback .edit-link a:hover,
	.comment-list .reply a:hover,
	.site-info a:hover {
		border-color: #352712;
	}

	/* Secondary Text Color */
	button:hover,
	button:focus,
	input[type="button"]:hover,
	input[type="button"]:focus,
	input[type="reset"]:hover,
	input[type="reset"]:focus,
	input[type="submit"]:hover,
	input[type="submit"]:focus,
	.pagination .prev:hover,
	.pagination .prev:focus,
	.pagination .next:hover,
	.pagination .next:focus,
	.widget_calendar tbody a:hover,
	.widget_calendar tbody a:focus,
	.page-links a:hover,
	.page-links a:focus {
		background-color: #352712; /* Fallback for IE7 and IE8 */
		background-color: rgba( 53, 39, 18, 0.7);
	}

	/* Secondary Text Color */
	blockquote,
	a:hover,
	a:focus,
	.main-navigation .menu-item-description,
	.post-navigation .meta-nav,
	.post-navigation a:hover .post-title,
	.post-navigation a:focus .post-title,
	.image-navigation,
	.image-navigation a,
	.comment-navigation,
	.comment-navigation a,
	.widget,
	.author-heading,
	.entry-footer,
	.entry-footer a,
	.taxonomy-description,
	.page-links > .page-links-title,
	.entry-caption,
	.comment-author,
	.comment-metadata,
	.comment-metadata a,
	.pingback .edit-link,
	.pingback .edit-link a,
	.post-password-form label,
	.comment-form label,
	.comment-notes,
	.comment-awaiting-moderation,
	.logged-in-as,
	.form-allowed-tags,
	.no-comments,
	.site-info,
	.site-info a,
	.wp-caption-text,
	.gallery-caption,
	.comment-list .reply a,
	.widecolumn label,
	.widecolumn .mu_register label {
		color: #352712; /* Fallback for IE7 and IE8 */
		color: rgba( 53, 39, 18, 0.7);
	}

	/* Secondary Text Color */
	blockquote,
	.logged-in-as a:hover,
	.comment-author a:hover {
		border-color: #352712; /* Fallback for IE7 and IE8 */
		border-color: rgba( 53, 39, 18, 0.7);
	}

	/* Border Color */
	hr,
	.dropdown-toggle:hover,
	.dropdown-toggle:focus {
		background-color: #352712; /* Fallback for IE7 and IE8 */
		background-color: rgba( 53, 39, 18, 0.1);
	}

	/* Border Color */
	pre,
	abbr[title],
	table,
	th,
	td,
	input,
	textarea,
	.main-navigation ul,
	.main-navigation li,
	.post-navigation,
	.post-navigation div + div,
	.pagination,
	.comment-navigation,
	.widget li,
	.widget_categories .children,
	.widget_nav_menu .sub-menu,
	.widget_pages .children,
	.site-header,
	.site-footer,
	.hentry + .hentry,
	.author-info,
	.entry-content .page-links a,
	.page-links > span,
	.page-header,
	.comments-area,
	.comment-list + .comment-respond,
	.comment-list article,
	.comment-list .pingback,
	.comment-list .trackback,
	.comment-list .reply a,
	.no-comments {
		border-color: #352712; /* Fallback for IE7 and IE8 */
		border-color: rgba( 53, 39, 18, 0.1);
	}

	/* Border Focus Color */
	a:focus,
	button:focus,
	input:focus {
		outline-color: #352712; /* Fallback for IE7 and IE8 */
		outline-color: rgba( 53, 39, 18, 0.3);
	}

	input:focus,
	textarea:focus {
		border-color: #352712; /* Fallback for IE7 and IE8 */
		border-color: rgba( 53, 39, 18, 0.3);
	}

	/* Sidebar Link Color */
	.secondary-toggle:before {
		color: #ffffff;
	}

	.site-title a,
	.site-description {
		color: #ffffff;
	}

	/* Sidebar Text Color */
	.site-title a:hover,
	.site-title a:focus {
		color: rgba( 255, 255, 255, 0.7);
	}

	/* Sidebar Border Color */
	.secondary-toggle {
		border-color: #ffffff; /* Fallback for IE7 and IE8 */
		border-color: rgba( 255, 255, 255, 0.1);
	}

	/* Sidebar Border Focus Color */
	.secondary-toggle:hover,
	.secondary-toggle:focus {
		border-color: #ffffff; /* Fallback for IE7 and IE8 */
		border-color: rgba( 255, 255, 255, 0.3);
	}

	.site-title a {
		outline-color: #ffffff; /* Fallback for IE7 and IE8 */
		outline-color: rgba( 255, 255, 255, 0.3);
	}

	/* Meta Background Color */
	.entry-footer {
		background-color: #f1f1f1;
	}

	@media screen and (min-width: 38.75em) {
		/* Main Text Color */
		.page-header {
			border-color: #352712;
		}
	}

	@media screen and (min-width: 59.6875em) {
		/* Make sure its transparent on desktop */
		.site-header,
		.secondary {
			background-color: transparent;
		}

		/* Sidebar Background Color */
		.widget button,
		.widget input[type="button"],
		.widget input[type="reset"],
		.widget input[type="submit"],
		.widget_calendar tbody a,
		.widget_calendar tbody a:hover,
		.widget_calendar tbody a:focus {
			color: #e53b51;
		}

		/* Sidebar Link Color */
		.secondary a,
		.dropdown-toggle:after,
		.widget-title,
		.widget blockquote cite,
		.widget blockquote small {
			color: #ffffff;
		}

		.widget button,
		.widget input[type="button"],
		.widget input[type="reset"],
		.widget input[type="submit"],
		.widget_calendar tbody a {
			background-color: #ffffff;
		}

		.textwidget a {
			border-color: #ffffff;
		}

		/* Sidebar Text Color */
		.secondary a:hover,
		.secondary a:focus,
		.main-navigation .menu-item-description,
		.widget,
		.widget blockquote,
		.widget .wp-caption-text,
		.widget .gallery-caption {
			color: rgba( 255, 255, 255, 0.7);
		}

		.widget button:hover,
		.widget button:focus,
		.widget input[type="button"]:hover,
		.widget input[type="button"]:focus,
		.widget input[type="reset"]:hover,
		.widget input[type="reset"]:focus,
		.widget input[type="submit"]:hover,
		.widget input[type="submit"]:focus,
		.widget_calendar tbody a:hover,
		.widget_calendar tbody a:focus {
			background-color: rgba( 255, 255, 255, 0.7);
		}

		.widget blockquote {
			border-color: rgba( 255, 255, 255, 0.7);
		}

		/* Sidebar Border Color */
		.main-navigation ul,
		.main-navigation li,
		.widget input,
		.widget textarea,
		.widget table,
		.widget th,
		.widget td,
		.widget pre,
		.widget li,
		.widget_categories .children,
		.widget_nav_menu .sub-menu,
		.widget_pages .children,
		.widget abbr[title] {
			border-color: rgba( 255, 255, 255, 0.1);
		}

		.dropdown-toggle:hover,
		.dropdown-toggle:focus,
		.widget hr {
			background-color: rgba( 255, 255, 255, 0.1);
		}

		.widget input:focus,
		.widget textarea:focus {
			border-color: rgba( 255, 255, 255, 0.3);
		}

		.sidebar a:focus,
		.dropdown-toggle:focus {
			outline-color: rgba( 255, 255, 255, 0.3);
		}
	}',
														),
										),
						),
				'twentyfifteen-ie' =>
						array(
								'handle' => 'twentyfifteen-ie',
								'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/themes/twentyfifteen/css/ie.css',
								'deps' =>
										array (
												0 => 'twentyfifteen-style',
										),
								'ver' => '20141010',
								'args' => 'all',
								'extra' =>
										array (
												'conditional' => 'lt IE 9',
										),
						),
				'twentyfifteen-ie7' =>
						array(
								'handle' => 'twentyfifteen-ie7',
								'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/themes/twentyfifteen/css/ie7.css',
								'deps' =>
										array (
												0 => 'twentyfifteen-style',
										),
								'ver' => '20141010',
								'args' => 'all',
								'extra' =>
										array (
												'conditional' => 'lt IE 8',
										),
						),
				'colorbox' =>
						array(
								'handle' => 'colorbox',
								'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/marketpress/ui/css/colorbox.css',
								'deps' =>
										array (
										),
								'ver' => '3.0.0.7',
								'args' => 'all',
								'extra' =>
										array (
										),
						),
				'jquery-ui' =>
						array(
								'handle' => 'jquery-ui',
								'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/marketpress/ui/css/jquery-ui.min.css',
								'deps' =>
										array (
										),
								'ver' => '3.0.0.7',
								'args' => 'all',
								'extra' =>
										array (
										),
						),
				'mp-select2' =>
						array(
								'handle' => 'mp-select2',
								'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/marketpress/ui/select2/select2.css',
								'deps' =>
										array (
										),
								'ver' => '3.0.0.7',
								'args' => 'all',
								'extra' =>
										array (
										),
						),
				'mp-base' =>
						array(
								'handle' => 'mp-base',
								'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/marketpress/ui/css/marketpress.css',
								'deps' =>
										array (
										),
								'ver' => '3.0.0.7',
								'args' => 'all',
								'extra' =>
										array (
										),
						),
				'mp-theme' =>
						array(
								'handle' => 'mp-theme',
								'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/marketpress/ui/themes/default.css',
								'deps' =>
										array (
										),
								'ver' => '3.0.0.7',
								'args' => 'all',
								'extra' =>
										array (
										),
						),
				'query-monitor' =>
						array(
								'handle' => 'query-monitor',
								'src' => 'http://humming.local.wordpress-wpmudev.dev/wp-content/plugins/query-monitor/assets/query-monitor.css',
								'deps' =>
										array (
										),
								'ver' => 1448962880,
								'args' => 'all',
								'extra' =>
										array (
										),
						),
		);
	}
}