<?php
/**
 * Handles adding and inserting MarketPress short codes from within the tinymce editor
 *
 * @since 3.0
 * @package MarketPress
 */
add_action( 'wp_ajax_mp_shortcode_builder_search_products', array( 'MP_Shortcode_Builder', 'search_products' ) );

class MP_Shortcode_Builder {

	/**
	 * Refers to a single instance of the class.
	 *
	 * @since 3.0
	 * @access private
	 * @var object
	 */
	private static $_instance = null;

	/**
	 * Refers to the product categories
	 *
	 * @since 3.0
	 * @access public
	 * @var array
	 */
	protected $_product_cats = array();

	/**
	 * Refers to the product tags
	 *
	 * @since 3.0
	 * @access public
	 * @var array
	 */
	protected $_product_tags = array();

	/**
	 * Gets the single instance of the class.
	 *
	 * @since 3.0
	 * @access public
	 * @return object
	 */
	public static function get_instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new MP_Shortcode_Builder();
		}
		return self::$_instance;
	}

	/**
	 * Searches posts
	 *
	 * @since 1.0
	 * @access public
	 * @action wp_ajax_mp_shortcode_builder_search_posts
	 */
	public static function search_products() {
		add_filter( 'posts_search', array( __CLASS__, 'search_by_title_only' ), 500, 2 );

		$query	 = new WP_Query( array(
			'paged'					 => isset( $_GET[ 'page' ] ) ? $_GET[ 'page' ] : false,
			'posts_per_page'		 => -1,
			's'						 => isset( $_GET[ 'search_term' ] ) ? $_GET[ 'search_term' ] : false,
			'update_post_meta_cache' => false,
			'update_post_term_cache' => false,
			'orderby'				 => 'title',
			'order'					 => 'ASC',
			'post_type'				 => MP_Product::get_post_type(),
		) );
		$data	 = array( 'posts' => array(), 'posts_per_page' => isset( $args[ 'posts_per_page' ] ) ? $args[ 'posts_per_page' ] : '-1', 'total' => $query->found_posts );

		while ( $query->have_posts() ) : $query->the_post();
			$data[ 'posts' ][] = array( 'id' => get_the_ID(), 'text' => get_the_title() );
		endwhile;

		wp_send_json( $data );
	}

	/**
	 * Search by title only
	 *
	 * @since 3.0
	 * @access public
	 * @filter posts_search
	 * @param string $search
	 * @param object $wp_query
	 * @return string
	 */
	public static function search_by_title_only( $search, &$wp_query ) {
		global $wpdb;

		if ( empty( $search ) ) {
			return $search; // skip processing - no search term in query
		}

		$q			 = $wp_query->query_vars;
		$n			 = !empty( $q[ 'exact' ] ) ? '' : '%';
		$search		 = '';
		$searchand	 = '';

		foreach ( (array) $q[ 'search_terms' ] as $term ) {
			$term		 = esc_sql( $wpdb->esc_like( $term ) );
			$search .= "{$searchand}($wpdb->posts.post_title LIKE '{$n}{$term}{$n}')";
			$searchand	 = ' AND ';
		}

		if ( !empty( $search ) ) {
			$search = " AND ({$search}) ";
			if ( !is_user_logged_in() ) {
				$search .= " AND ($wpdb->posts.post_password = '') ";
			}
		}

		return $search;
	}

	/**
	 * Constructor.
	 *
	 * @since 3.0
	 * @access private
	 */
	private function __construct() {
		$this->_product_cats = get_terms( 'product_category', 'hide_empty=0' );
		$this->_product_tags = get_terms( 'product_tag', 'hide_empty=0' );

		add_action( 'media_buttons', array( &$this, 'media_buttons' ) );
		add_action( 'admin_enqueue_scripts', array( &$this, 'enqueue_styles_scripts' ) );
		add_action( 'in_admin_footer', array( &$this, 'display_short_code_form' ) );
		add_filter( 'wp_dropdown_cats', array( &$this, 'wp_dropdown_cats' ), 10, 2 );
	}

	/**
	 * Add data-default attribute to html generated by wp_dropdown_categories() function
	 *
	 * @since 3.0
	 * @access public
	 * @filter wp_dropdown_cats
	 * @return string
	 */
	public function wp_dropdown_cats( $html, $args ) {
		return str_replace( '<select ', '<select data-default="-1" ', $html );
	}

	/**
	 * Displays the short code form
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_short_code_form() {
		if ( did_action( 'media_buttons' ) == 0 ) {
			// Only continue if a tinymce editor exists on the current page
			return;
		}

		if ( ! function_exists( 'is_plugin_active_for_network' ) ) {
			require_once( ABSPATH . '/wp-admin/includes/plugin.php' );
		}

		if ( ! function_exists( 'mp_get_plugin_slug' ) ) {
			function mp_get_plugin_slug() {
				if ( MP_LITE ) {
					return 'wordpress-ecommerce/marketpress.php';
				} else {
					return 'marketpress/marketpress.php';
				}
			}
		}

		$shortcodes = array(
			'mp_tag_cloud'			 => __( 'Display a cloud or list of your product tags.', 'mp' ),
			'mp_list_categories'	 => __( 'Display an HTML list of your product categories.', 'mp' ),
			'mp_dropdown_categories' => __( 'Display an HTML dropdown of your product categories.', 'mp' ),
			'mp_popular_products'	 => __( 'Display a list of popular products ordered by sales.', 'mp' ),
			'mp_related_products'	 => __( 'Display products related to the one being viewed.', 'mp' ),
			'mp_list_products'		 => __( 'Display a list of products according to preference.', 'mp' ),
			'mp_product'			 => __( 'Display a single product.', 'mp' ),
			'mp_product_image'		 => __( 'Display the featured image of a given product.', 'mp' ),
			'mp_buy_button'			 => __( 'Display the buy or add to cart button.', 'mp' ),
			'mp_product_price'		 => __( 'Display the product price (and sale price).', 'mp' ),
			'mp_product_sku'		 => __( 'Display the product SKU number(s).', 'mp' ),
			'mp_product_stock'		 => __( 'Display the product stock.', 'mp' ),
			'mp_product_meta'		 => __( 'Display the full product meta box with price and buy now/add to cart button.', 'mp' ),
			'mp_cart_widget'		 => __( 'Display the cart widget.', 'mp' ),
			'mp_cart_link'			 => __( 'Display a link or url to the current shopping cart page.', 'mp' ),
			'mp_store_link'			 => __( 'Display a link or url to the current store page.', 'mp' ),
			'mp_products_link'		 => __( 'Display a link or url to the current products list page.', 'mp' ),
			'mp_orderstatus_link'	 => __( 'Display a link or url to the order status page.', 'mp' ),
			'mp_store_navigation'	 => __( 'Display a list of links to your store pages.', 'mp' ),
		);

		if ( is_multisite() && is_plugin_active_for_network( mp_get_plugin_slug() ) ) {
			$settings = get_site_option( 'mp_network_settings', array() );
			if ( ( isset($settings['main_blog']) && mp_is_main_site() ) || isset($settings['main_blog']) && !$settings['main_blog'] ) {
				$mu_shortcodes = array(
					'mp_list_global_products'	  => __( 'Display a list or grid  of your global products.', 'mp' ),
					'mp_global_categories_list'   => __( 'Display a list of your global categories.', 'mp' ),
					'mp_global_tag_cloud'		  => __( 'Display a cloud or list of your global product tags.', 'mp' ),
				);

				$shortcodes = array_merge($shortcodes, $mu_shortcodes);
			}
		}

		?>
		<div id="mp-shortcode-builder" style="display:none">
			<form id="mp-shortcode-builder-form">
				<h3><?php _e( 'Add E-Commerce Short Code', 'mp' ); ?></h3>
				<strong><?php _e( 'What would you like to do?', 'mp' ); ?></strong><br />
				<select name="shortcode">
					<option value=""><?php _e( 'Select an Action', 'mp' ); ?></option>
					<?php foreach ( $shortcodes as $shortcode => $label ) : ?>
						<option value="<?php echo esc_attr( $shortcode ); ?>"><?php echo '[' . $shortcode . '] ' . $label; ?></option>
					<?php endforeach; ?>
				</select>

				<div class="shortcode-attributes">
					<h2><?php _e( 'Attributes', 'mp' ); ?></h2>
					<?php
					foreach ( $shortcodes as $shortcode => $label ) {
						$func = 'display_' . $shortcode . '_attributes';

						if ( method_exists( $this, $func ) ) {
							call_user_func( array( &$this, $func ) );
						}
					}
					?>
				</div>
				<?php $this->display_shortcode_builder_form_buttons(); ?>
			</form>
		</div>
		<?php
	}

	/**
	 * Displays the [mp_list_categories] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_list_categories_attributes() {
		?>
		<table id="mp-list-categories-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'If a value is entered, causes the display of a link to all categories if the style is set to list.', 'mp' ); ?></span></span> show_option_all</th>
				<td>
					<input type="text" name="show_option_all" data-default="" value="" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Sort categories alphabetically, by unique Category ID, or by the count of posts in that Category.', 'mp' ); ?></span></span> orderby</th>
				<td>
					<select name="orderby" data-default="name">
						<?php foreach ( array( 'ID', 'name', 'slug', 'count', 'term_group' ) as $value ) : ?>
							<option <?php selected( $value, 'name' ); ?>><?php echo $value; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Sort order for categories (either ascending or descending).', 'mp' ); ?></span></span> order</th>
				<td>
					<select name="order" data-default="ASC">
						<?php foreach ( array( 'ASC', 'DESC' ) as $value ) : ?>
							<option <?php selected( $value, 'ASC' ); ?>><?php echo $value; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Style to display the categories list in. A value of list displays the categories as list items while none generates no special display method (the list items are separated by <br> tags).', 'mp' ); ?></span></span> style</th>
				<td>
					<select name="style" data-default="list">
						<?php foreach ( array( 'list', 'none' ) as $value ) : ?>
							<option <?php selected( $value, 'list' ); ?>><?php echo $value; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Toggles the display of the current count of posts in each category.', 'mp' ); ?></span></span> show_count</th>
				<td>
					<input type="checkbox" name="show_count" data-default="" value="1" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Toggles the display of categories with no posts.', 'mp' ); ?></span></span> hide_empty</th>
				<td>
					<input type="checkbox" name="hide_empty" data-default="1" value="1" checked />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Sets whether a category\'s description is inserted into the title attribute of the links created (i.e. &lt;a title="&lt;em&gt;Category Description&lt;/em&gt;" href="...).', 'mp' ); ?></span></span> use_desc_for_title</th>
				<td>
					<input type="checkbox" name="use_desc_for_title" data-default="1" value="1" checked />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Only display categories that are children of the category identified by this parameter.', 'mp' ); ?></span></span>  child_of</th>
				<td>
					<?php
					wp_dropdown_categories( array(
						'taxonomy'			 => 'product_category',
						'hide_empty'		 => 0,
						'show_option_none'	 => __( 'Any', 'mp' ),
						'name'				 => 'child_of',
					) );
					?>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Display a link to each category\'s rss-2 feed and set the link text to display. The default is no text and no feed displayed.', 'mp' ); ?></span></span> feed</th>
				<td>
					<input type="text" name="feed" data-default="" value="" />
				</td>
			</tr>
			<tr>
				<th scope="row">feed_type</th>
				<td>
					<input type="text" name="feed_type" data-default="" value="" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Set a URI for an image (usually an rss feed icon) to act as a link to each categories\' rss-2 feed. This parameter overrides the feed parameter. There is no default for this parameter.', 'mp' ); ?></span></span> feed_image</th>
				<td>
					<input type="text" name="feed_image" data-default="" value="" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Exclude one or more categories from the results. The parameter include must be empty.', 'mp' ); ?></span></span> exclude</th>
				<td>
					<select name="exclude" class="mp-chosen-select" data-placeholder="<?php _e( 'Select Categories', 'mp' ); ?>" multiple>
						<?php
						foreach ( $this->_product_cats as $cat ) {
							if ( isset( $cat ) ) {
								?>
								<option value="<?php echo esc_attr( isset( $cat->term_id ) ? $cat->term_id : ''  ); ?>"><?php echo isset( $cat->name ) ? $cat->name : ''; ?></option>
								<?php
							}
						}
						?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Only include these categories.', 'mp' ); ?></span></span> include</th>
				<td>
					<select name="include" class="mp-chosen-select" data-placeholder="<?php _e( 'Select Categories', 'mp' ); ?>" multiple>
						<?php
						foreach ( $this->_product_cats as $cat ) :
							if ( isset( $cat ) ) {
								?>
								<option value="<?php echo esc_attr( isset( $cat->term_id ) ? $cat->term_id : ''  ); ?>"><?php echo isset( $cat->name ) ? $cat->name : ''; ?></option>
							<?php }endforeach;
						?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Exclude category-tree from the results. The parameter include must be empty. If the hierarchical parameter is true, then use exclude instead of exclude_tree.', 'mp' ); ?></span></span> exclude_tree</th>
				<td>
					<select name="exclude_tree" class="mp-chosen-select" data-placeholder="<?php _e( 'Select Categories', 'mp' ); ?>" multiple>
						<?php foreach ( $this->_product_cats as $cat ) : ?>
							<option value="<?php echo esc_attr( isset( $cat->term_id ) ? $cat->term_id : ''  ); ?>"><?php echo isset( $cat->name ) ? $cat->name : ''; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Display sub-categories as inner list items (below the parent list item) or inline.', 'mp' ); ?></span></span> hierarchical</th>
				<td>
					<input type="checkbox" name="hierarchical" data-default="1" value="1" checked />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Set the title and style of the outer list item. If present but empty, the outer list item will not be displayed.', 'mp' ); ?></span></span> title_li</th>
				<td>
					<input type="text" name="title_li" data-default="Categories" value="Categories" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Set the text to show when no categories are listed.', 'mp' ); ?></span></span> show_option_none</th>
				<td>
					<input type="text" name="show_option_none" data-default="No Categories" value="No Categories" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Sets the number of Categories to display. This causes the SQL LIMIT value to be defined.', 'mp' ); ?></span></span> number</th>
				<td>
					<input type="text" name="number" data-default="" value="" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'This parameter controls how many levels in the hierarchy of Categories are to be included in the list of Categories. NOTE: The "hierarchical" argument, which is disabled by default, will override the depth argument, unless it is true. When the argument is false, it will display all of the categories. When it is enabled it will use the value in the "depth" argument.', 'mp' ); ?></span></span> depth</th>
				<td>
					<input type="text" name="depth" data-default="0" value="0" />
					<ul>
						<li><code>0</code> - <?php _e( 'All Categories and child Categories (Default).', 'mp' ); ?></li>
						<li><code>-1</code> - <?php _e( 'All Categories displayed in flat (no indent) form (overrides hierarchical).', 'mp' ); ?></li>
						<li><code>1</code> - <?php _e( 'Show only top level Categories.', 'mp' ); ?></li>
						<li><code>n</code> - <?php _e( 'Value of n (some number) specifies the depth (or level) to descend in displaying Categories.', 'mp' ); ?></li>
					</ul>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Calculates link or post counts by including items from child categories. If show_counts and hierarchical are true this is automatically set to true.', 'mp' ); ?></span></span> pad_counts</th>
				<td>
					<input type="checkbox" name="pad_counts" data-default="0" value="1" />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_dropdown_categories] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_dropdown_categories_attributes() {
		?>
		<table id="mp-dropdown-categories-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'If a value is entered, causes the display of a link to all categories if the style is set to list.', 'mp' ); ?></span></span> show_option_all</th>
				<td>
					<input type="text" name="show_option_all" data-default="" value="" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Sort categories alphabetically, by unique Category ID, or by the count of posts in that Category.', 'mp' ); ?></span></span> orderby</th>
				<td>
					<select name="orderby" data-default="name">
						<?php foreach ( array( 'ID', 'name', 'slug', 'count', 'term_group' ) as $value ) : ?>
							<option <?php selected( $value, 'name' ); ?>><?php echo $value; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Sort order for categories (either ascending or descending).', 'mp' ); ?></span></span> order</th>
				<td>
					<select name="order" data-default="ASC">
						<?php foreach ( array( 'ASC', 'DESC' ) as $value ) : ?>
							<option <?php selected( $value, 'ASC' ); ?>><?php echo $value; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<!--<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php /*_e( 'Style to display the categories list in. A value of list displays the categories as list items while none generates no special display method (the list items are separated by <br> tags).', 'mp' ); */?></span></span> style</th>
				<td>
					<select name="style" data-default="list">
						<?php /*foreach ( array( 'list', 'none' ) as $value ) : */?>
							<option <?php /*selected( $value, 'list' ); */?>><?php /*echo $value; */?></option>
						<?php /*endforeach; */?>
					</select>
				</td>
			</tr>-->
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Toggles the display of the current count of posts in each category.', 'mp' ); ?></span></span> show_count</th>
				<td>
					<input type="checkbox" name="show_count" data-default="" value="1" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Toggles the display of categories with no posts.', 'mp' ); ?></span></span> hide_empty</th>
				<td>
					<input type="checkbox" name="hide_empty" data-default="1" value="1" checked />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Only display categories that are children of the category identified by this parameter.', 'mp' ); ?></span></span>  child_of</th>
				<td>
					<?php
					wp_dropdown_categories( array(
						'taxonomy'			 => 'product_category',
						'hide_empty'		 => 0,
						'show_option_none'	 => __( 'Any', 'mp' ),
						'name'				 => 'child_of',
					) );
					?>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Exclude one or more categories from the results. The parameter include must be empty.', 'mp' ); ?></span></span> exclude</th>
				<td>
					<select name="exclude" class="mp-chosen-select" data-placeholder="<?php _e( 'Select Categories', 'mp' ); ?>" multiple>
						<?php
						$cats = get_categories( array(
							'hide_empty' => 0,
							'taxonomy'	 => 'product_category'
						) );

						foreach ( $cats as $cat ) :
							?>
							<option value="<?php echo esc_attr( $cat->term_id ); ?>"><?php echo $cat->name; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Exclude category-tree from the results. The parameter include must be empty. If the hierarchical parameter is true, then use exclude instead of exclude_tree.', 'mp' ); ?></span></span> exclude_tree</th>
				<td>
					<select name="exclude_tree" class="mp-chosen-select" data-placeholder="<?php _e( 'Select Categories', 'mp' ); ?>" multiple>
						<?php
						$cats = get_categories( array(
							'hide_empty' => 0,
							'taxonomy'	 => 'product_category'
						) );

						foreach ( $cats as $cat ) :
							?>
							<option value="<?php echo esc_attr( $cat->term_id ); ?>"><?php echo $cat->name; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Display sub-categories as inner list items (below the parent list item) or inline.', 'mp' ); ?></span></span> hierarchical</th>
				<td>
					<input type="checkbox" name="hierarchical" data-default="1" value="1" checked />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Set the text to show when no categories are listed.', 'mp' ); ?></span></span> show_option_none</th>
				<td>
					<input type="text" name="show_option_none" data-default="<?php echo esc_attr( __( 'No Categories', 'mp' ) ); ?>" value="<?php echo esc_attr( __( 'No Categories', 'mp' ) ); ?>" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The max depth. This is ignored unless hierarchical is set to true.', 'mp' ); ?></span></span> depth</th>
				<td>
					<input type="text" name="depth" data-default="0" value="0" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Calculates link or post counts by including items from child categories. If show_counts and hierarchical are true this is automatically set to true.', 'mp' ); ?></span></span> pad_counts</th>
				<td>
					<input type="checkbox" name="pad_counts" data-default="0" value="1" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Which category ID is selected.', 'mp' ); ?></span></span> selected</th>
				<td>
					<input type="text" name="selected" data-default="" value="" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The "id" attribute value for select element. ', 'mp' ); ?></span></span> id</th>
				<td>
					<input type="text" name="id" data-default="0" value="0" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The "name" attribute value for select element. ', 'mp' ); ?></span></span> name</th>
				<td>
					<input type="text" name="name" data-default="cat" value="cat" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The "tabindex" attribute value for select element. ', 'mp' ); ?></span></span> tab_index</th>
				<td>
					<input type="text" name="tab_index" data-default="0" value="0" />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_tag_cloud] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_tag_cloud_attributes() {
		?>
		<table id="mp-tag-cloud-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The text size of the tag with the smallest count value (units given by unit parameter).', 'mp' ); ?></span></span> smallest</th>
				<td>
					<input type="text" name="smallest" data-default="8" value="8" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The text size of the tag with the largest count value (units given by unit parameter).', 'mp' ); ?></span></span> largest</th>
				<td>
					<input type="text" name="largest" data-default="22" value="22" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Unit of measure as pertains to the smallest and largest values. This can be any CSS length value, e.g. pt, px, em, %.', 'mp' ); ?></span></span> unit</th>
				<td>
					<input type="text" name="unit" data-default="pt" value="pt" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The number of actual tags to display in the cloud. (Use "0" to display all tags.)', 'mp' ); ?></span></span> number</th>
				<td>
					<input type="text" name="number" data-default="45" value="45" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Format of the cloud display.', 'mp' ); ?></span></span> format</th>
				<td>
					<label for="mp-tag-cloud-format-flat"><input type="radio" name="format" id="mp-tag-cloud-format-flat" data-default="flat" value="flat" checked /> <?php _e( 'separated by whitespace defined by "separator" parameter.', 'mp' ); ?></label><br />
					<label for="mp-tag-cloud-format-list"><input type="radio" name="format" id="mp-tag-cloud-format-list" data-default="flat" value="list" /> <?php _e( 'UL with a class of "wp-tag-cloud"', 'mp' ); ?></label>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The text/space between tags)', 'mp' ); ?></span></span> separator</th>
				<td>
					<input type="text" name="separator" data-default="\n" value="\n" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Order of the tags.', 'mp' ); ?></span></span> orderby</th>
				<td>
					<label for="mp-tag-cloud-orderby-name"><input type="radio" name="orderby" id="mp-tag-cloud-orderby-name" data-default="name" value="name" checked /> <?php _e( 'Name', 'mp' ); ?></label><br />
					<label for="mp-tag-cloud-orderby-count"><input type="radio" name="orderby" id="mp-tag-cloud-orderby-count" data-default="count" value="count" /> <?php _e( 'Count', 'mp' ); ?></label>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Sort order.', 'mp' ); ?></span></span> order</th>
				<td>
					<label for="mp-tag-cloud-order-asc"><input type="radio" name="order" id="mp-tag-cloud-order-asc" data-default="ASC" value="ASC" checked /> <?php _e( 'Ascending', 'mp' ); ?></label><br />
					<label for="mp-tag-cloud-order-desc"><input type="radio" name="order" id="mp-tag-cloud-order-desc" data-default="ASC" value="DESC" /> <?php _e( 'Descending', 'mp' ); ?></label><br />
					<label for="mp-tag-cloud-order-rand"><input type="radio" name="order" id="mp-tag-cloud-order-rand" data-default="RAND" value="RAND" /> <?php _e( 'Random', 'mp' ); ?></label>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Comma separated list of tags (term_id) to exclude. For example, exclude=5,27 means tags that have the term_id 5 or 27 will NOT be displayed. Defaults to exclude nothing.', 'mp' ); ?></span></span> exclude</th>
				<td>
					<input type="text" name="exclude" data-default="" value="" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Comma separated list of tags (term_id) to include. For example, include=5,27 means tags that have the term_id 5 or 27 will be the only tags displayed. Defaults to include everything.', 'mp' ); ?></span></span> include</th>
				<td>
					<input type="text" name="include" data-default="" value="" />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_popular_products] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_popular_products_attributes() {
		?>
		<table id="mp-popular-products-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The maximum number of products to display', 'mp' ); ?></span></span> number</th>
				<td>
					<input type="text" name="number" data-default="5" value="5" />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_related_products] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_related_products_attributes() {
		?>
		<table id="mp-related-products-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The product to show related items for.', 'mp' ); ?></span></span> product_id</th>
				<td>
					<input type="text" class="mp-select-product" name="product_id" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'How to relate the products', 'mp' ); ?></span></span> relate_by</th>
				<td>
					<select name="relate_by" data-default="<?php echo esc_attr( mp_get_setting( 'related_products->relate_by' ) ); ?>">
						<?php
						$data = array(
							'both'		 => __( 'Category &amp; Tags', 'mp' ),
							'category'	 => __( 'Category', 'mp' ),
							'tags'		 => __( 'Tags', 'mp' ),
						);

						foreach ( $data as $value => $label ) :
							?>
							<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $value, mp_get_setting( 'related_products->relate_by' ) ); ?>><?php echo $label; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'How many related items to show.', 'mp' ); ?></span></span> limit</th>
				<td>
					<input type="text" name="limit" data-default="<?php echo esc_attr( mp_get_setting( 'related_products->show_limit' ) ); ?>" value="<?php echo esc_attr( mp_get_setting( 'related_products->show_limit' ) ); ?>" />
				</td>
			</tr>
			<!--<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php /*_e( 'Whether to display the items as a simple list or based on the list/grid view setting.', 'mp' ); */?></span></span> simple_list</th>
				<td>
					<input type="checkbox" name="simple_list" data-default="<?php /*echo esc_attr( mp_get_setting( 'related_products->simple_list' ) ); */?>" <?php /*checked( 1, mp_get_setting( 'related_products->simple_list' ) ); */?> />
				</td>
			</tr>-->
		</table>
		<?php
	}

	/**
	 * Displays the [mp_list_products] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_list_products_attributes() {
		?>
		<table id="mp-list-products-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether to paginate the product list. This is useful to only show a subset.', 'mp' ); ?></span></span> paginate</th>
				<td>
					<input type="checkbox" name="paginate" data-default="<?php echo esc_attr( mp_get_setting( 'paginate' ) ); ?>" value="1" <?php checked( 1, mp_get_setting( 'paginate' ) ); ?> />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The page number to display in the product list if "paginate" is set to true.', 'mp' ); ?></span></span> page</th>
				<td>
					<input type="text" name="page" data-default="1" value="1" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'How many products to display in the product list if "paginate" is set to true.', 'mp' ); ?></span></span> per_page</th>
				<td>
					<input type="text" name="per_page" data-default="<?php echo esc_attr( mp_get_setting( 'per_page' ) ); ?>" value="<?php echo esc_attr( mp_get_setting( 'per_page' ) ); ?>" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'What field to order products by.', 'mp' ); ?></span></span> order_by</th>
				<td>
					<select name="order_by" data-default="<?php echo esc_attr( mp_get_setting( 'order_by' ) ); ?>">
						<?php
						$data = array(
							'title'	 => __( 'Product Name', 'mp' ),
							'date'	 => __( 'Publish Date', 'mp' ),
							'ID'	 => __( 'Product ID', 'mp' ),
							'author' => __( 'Product Author', 'mp' ),
							'sales'	 => __( 'Number of Sales', 'mp' ),
							'price'	 => __( 'Product Price', 'mp' ),
							'rand'	 => __( 'Random', 'mp' ),
						);

						foreach ( $data as $value => $label ) :
							?>
							<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $value, mp_get_setting( 'order_by' ) ); ?>><?php echo $label; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Direction to order products by.', 'mp' ); ?></span></span> order</th>
				<td>
					<label><input type="radio" name="order" data-default="<?php echo esc_attr( mp_get_setting( 'order' ) ); ?>" value="ASC" <?php checked( 'ASC', mp_get_setting( 'order' ) ); ?> /> <?php _e( 'Ascending', 'mp' ); ?></label> &nbsp; &nbsp;
					<label><input type="radio" name="order" data-default="<?php echo esc_attr( mp_get_setting( 'order' ) ); ?>" value="DESC" <?php checked( 'DESC', mp_get_setting( 'order' ) ); ?> /> <?php _e( 'Descending', 'mp' ); ?></label>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Limits list to a specific product category.', 'mp' ); ?></span></span> category</th>
				<td>
					<select name="category" data-default="" class="mp-chosen-select">
						<option value=""><?php _e( 'None', 'mp' ); ?></option>
						<?php foreach ( $this->_product_cats as $term ) : ?>
							<option value="<?php echo esc_attr( isset( $term->slug ) ? $term->slug : ''  ); ?>"><?php echo isset( $term->name ) ? $term->name : ''; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Limits list to a specific product tag.', 'mp' ); ?></span></span> tag</th>
				<td>
					<select name="tag" data-default="" class="mp-chosen-select">
						<option value=""><?php _e( 'None', 'mp' ); ?></option>
						<?php foreach ( $this->_product_tags as $term ) : ?>
							<option value="<?php echo esc_attr( isset( $term->slug ) ? $term->slug : ''  ); ?>"><?php echo isset( $term->name ) ? $term->name : ''; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'If checked, products will be displayed as a list - otherwise products will be displayed as a grid.', 'mp' ); ?></span></span> list_view</th>
				<td>
					<input type="checkbox" name="list_view" data-default="<?php echo esc_attr( ( mp_get_setting( 'list_view' ) == 'list' ) ? 1 : 0  ); ?>" value="1" <?php checked( 'list', mp_get_setting( 'list_view' ) ); ?> />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether or not to show the product filters.', 'mp' ); ?></span></span> filters</th>
				<td>
					<label><input type="radio" name="filters" data-default="<?php echo esc_attr( mp_get_setting( 'show_filters' ) ); ?>" value="1" <?php checked( '1', mp_get_setting( 'show_filters' ) ); ?> /> <?php _e( 'Show', 'mp' ); ?></label> &nbsp; &nbsp;
					<label><input type="radio" name="filters" data-default="<?php echo esc_attr( mp_get_setting( 'show_filters' ) ); ?>" value="0" <?php checked( '1', mp_get_setting( 'show_filters' ) ); ?> /> <?php _e( 'Hide', 'mp' ); ?></label>
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_product] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_product_attributes() {
		?>
		<table id="mp-product-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The ID of the product to display.', 'mp' ); ?></span></span> product_id</th>
				<td>
					<input type="text" name="product_id" data-default="" class="mp-select-product" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether to display the product title.', 'mp' ); ?></span></span> title</th>
				<td>
					<input type="checkbox" name="title" data-default="0" value="1" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether and what type of content to display.', 'mp' ); ?></span></span> content</th>
				<td>
					<select name="content" data-default="full">
						<option value="0"><?php _e( 'None', 'mp' ); ?></option>
						<option value="excerpt"><?php _e( 'Excerpt', 'mp' ); ?></option>
						<option value="full" selected><?php _e( 'Full', 'mp' ); ?></option>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether and what context of image size to display.', 'mp' ); ?></span></span> image</th>
				<td>
					<select name="image" data-default="single">
						<option value="0"><?php _e( 'None', 'mp' ); ?></option>
						<option value="single" selected><?php _e( 'Single', 'mp' ); ?></option>
						<option value="list"><?php _e( 'List', 'mp' ); ?></option>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether to display the product meta (e.g. price, buy button).', 'mp' ); ?></span></span> meta</th>
				<td>
					<input type="checkbox" name="meta" data-default="0" value="1" />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_product_image] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_product_image_attributes() {
		?>
		<table id="mp-product-image-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The ID of the product.', 'mp' ); ?></span></span> product_id</th>
				<td>
					<input type="text" name="product_id" data-default="" class="mp-select-product" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'What context for preset size options.', 'mp' ); ?></span></span> context</th>
				<td>
					<select name="context" data-default="single">
						<option value="single" selected><?php _e( 'Single', 'mp' ); ?></option>
						<option value="list"><?php _e( 'List', 'mp' ); ?></option>
						<option value="widget"><?php _e( 'Widget', 'mp' ); ?></option>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Set a custom pixel width/height. If omitted defaults to the size set by "context".', 'mp' ); ?></span></span> size</th>
				<td>
					<input type="text" name="size" data-default="" value="" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Set the alignment of the image.', 'mp' ); ?></span></span> align</th>
				<td>
					<label><input type="radio" name="align" data-default="<?php echo esc_attr( mp_get_setting( 'image_alignment_single' ) ); ?>" value="alignnone" <?php checked( 'alignnone', mp_get_setting( 'image_alignment_single' ) ); ?> /> <?php _e( 'None', 'mp' ); ?></label> &nbsp; &nbsp;
					<label><input type="radio" name="align" data-default="<?php echo esc_attr( mp_get_setting( 'image_alignment_single' ) ); ?>" value="alignleft" <?php checked( 'alignleft', mp_get_setting( 'image_alignment_single' ) ); ?> /> <?php _e( 'Left', 'mp' ); ?></label> &nbsp; &nbsp;
					<label><input type="radio" name="align" data-default="<?php echo esc_attr( mp_get_setting( 'image_alignment_single' ) ); ?>" value="alignright" <?php checked( 'alignright', mp_get_setting( 'image_alignment_single' ) ); ?> /> <?php _e( 'Right', 'mp' ); ?></label> &nbsp; &nbsp;
					<label><input type="radio" name="align" data-default="<?php echo esc_attr( mp_get_setting( 'image_alignment_single' ) ); ?>" value="aligncenter" <?php checked( 'aligncenter', mp_get_setting( 'image_alignment_single' ) ); ?> /> <?php _e( 'Center', 'mp' ); ?></label> &nbsp; &nbsp;
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_buy_button] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_buy_button_attributes() {
		?>
		<table id="mp-buy-button-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The ID of the product.', 'mp' ); ?></span></span> product_id</th>
				<td>
					<input type="text" name="product_id" data-default="" class="mp-select-product" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'What context for display.', 'mp' ); ?></span></span> context</th>
				<td>
					<label><input type="radio" value="single" name="context" data-default="single" value="single" checked> <?php _e( 'Single', 'mp' ); ?></label> &nbsp; &nbsp;
					<label><input type="radio" value="list" name="context" data-default="single" value="list"> <?php _e( 'List', 'mp' ); ?></label>
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_product_price] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_product_price_attributes() {
		?>
		<table id="mp-product-price-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The ID of the product.', 'mp' ); ?></span></span> product_id</th>
				<td>
					<input type="text" name="product_id" data-default="" class="mp-select-product" />
				</td>
			</tr>
			<!--<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'A label to prepend to the price.', 'mp' ); ?></span></span> label</th>
				<td>
					<input type="text" name="label" data-default="Price:" value="Price:" />
				</td>
			</tr>-->
		</table>
		<?php
	}

	/**
	 * Displays the [mp_product_sku] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_product_sku_attributes() {
		?>
		<table id="mp-product-sku-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The ID of the product.', 'mp' ); ?></span></span> product_id</th>
				<td>
					<input type="text" name="product_id" data-default="" class="mp-select-product" />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_product_stock] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_product_stock_attributes() {
		?>
		<table id="mp-product-stock-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The ID of the product.', 'mp' ); ?></span></span> product_id</th>
				<td>
					<input type="text" name="product_id" data-default="" class="mp-select-product" />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_product_meta] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_product_meta_attributes() {
		?>
		<table id="mp-product-meta-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The ID of the product.', 'mp' ); ?></span></span> product_id</th>
				<td>
					<input type="text" name="product_id" data-default="" class="mp-select-product" />
				</td>
			</tr>
			<!--<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'A label to prepend to the price.', 'mp' ); ?></span></span> label</th>
				<td>
					<input type="text" name="label" data-default="Price:" value="Price:" />
				</td>
			</tr>-->
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'What context for display.', 'mp' ); ?></span></span> context</th>
				<td>
					<label><input type="radio" value="single" name="context" data-default="single" value="single" checked> <?php _e( 'Single', 'mp' ); ?></label> &nbsp; &nbsp;
					<label><input type="radio" value="list" name="context" data-default="single" value="list"> <?php _e( 'List', 'mp' ); ?></label>
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_cart_widget] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_cart_widget_attributes() {
		?>
		<table id="mp-cart-widget-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The title of the cart.', 'mp' ); ?></span></span> title</th>
				<td>
					<input type="text" name="title" data-default="" value="" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The text displayed before the cart.', 'mp' ); ?></span></span> custom_text</th>
				<td>
					<input type="text" name="custom_text" data-default="" value="" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether to show or hide the product image.', 'mp' ); ?></span></span> show_product_image</th>
				<td>
					<input type="checkbox" name="show_product_image" data-default="1" value="1" checked />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether to show or hide the product quantity.', 'mp' ); ?></span></span> show_product_qty</th>
				<td>
					<input type="checkbox" name="show_product_qty" data-default="1" value="1" checked />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether to show or hide the product price.', 'mp' ); ?></span></span> show_product_price</th>
				<td>
					<input type="checkbox" name="show_product_price" value="1" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether to display the cart only on store pages or not.', 'mp' ); ?></span></span> only_store_pages</th>
				<td>
					<input type="checkbox" name="only_store_pages" value="1" />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_cart_link] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_cart_link_attributes() {
		?>
		<table id="mp-cart-link-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether to return a clickable link or url.', 'mp' ); ?></span></span> url</th>
				<td>
					<input type="checkbox" name="url" data-default="1" value="1" checked />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The text to show in the link.', 'mp' ); ?></span></span> link_text</th>
				<td>
					<input type="text" name="link_text" data-default="" value="" />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_store_link] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_store_link_attributes() {
		?>
		<table id="mp-store-link-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether to return a clickable link or url.', 'mp' ); ?></span></span> url</th>
				<td>
					<input type="checkbox" name="url" data-default="1" value="1" checked />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The text to show in the link.', 'mp' ); ?></span></span> link_text</th>
				<td>
					<input type="text" name="link_text" data-default="" value="" />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_products_link] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_products_link_attributes() {
		?>
		<table id="mp-products-link-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether to return a clickable link or url.', 'mp' ); ?></span></span> url</th>
				<td>
					<input type="checkbox" name="url" data-default="1" value="1" checked />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The text to show in the link.', 'mp' ); ?></span></span> link_text</th>
				<td>
					<input type="text" name="link_text" data-default="" value="" />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_list_global_products] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_list_global_products_attributes() {
		?>
		<table id="mp-list-global-products-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Whether to paginate the product list. This is useful to only show a subset.', 'mp' ); ?></span></span> paginate</th>
				<td>
					<input type="checkbox" name="paginate" data-default="<?php echo esc_attr( mp_get_setting( 'paginate' ) ); ?>" value="1" <?php checked( 1, mp_get_setting( 'paginate' ) ); ?> />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'The page number to display in the product list if "paginate" is set to true.', 'mp' ); ?></span></span> page</th>
				<td>
					<input type="text" name="page" data-default="1" value="1" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'How many products to display in the product list if "paginate" is set to true.', 'mp' ); ?></span></span> per_page</th>
				<td>
					<input type="text" name="per_page" data-default="<?php echo esc_attr( mp_get_setting( 'per_page' ) ); ?>" value="<?php echo esc_attr( mp_get_setting( 'per_page' ) ); ?>" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'How many products to display in the product list if "paginate" is set to false.', 'mp' ); ?></span></span> limit</th>
				<td>
					<input type="text" name="limit" data-default="<?php echo esc_attr( mp_get_setting( 'limit' ) ); ?>" value="<?php echo esc_attr( mp_get_setting( 'limit' ) ); ?>" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'What field to order products by.', 'mp' ); ?></span></span> order_by</th>
				<td>
					<select name="order_by" data-default="<?php echo esc_attr( mp_get_setting( 'order_by' ) ); ?>">
						<?php
						$data = array(
							'title'	 => __( 'Product Name', 'mp' ),
							'date'	 => __( 'Publish Date', 'mp' ),
							'ID'	 => __( 'Product ID', 'mp' ),
							'author' => __( 'Product Author', 'mp' ),
							'sales'	 => __( 'Number of Sales', 'mp' ),
							'price'	 => __( 'Product Price', 'mp' ),
							'rand'	 => __( 'Random', 'mp' ),
						);

						foreach ( $data as $value => $label ) :
							?>
							<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $value, mp_get_setting( 'order_by' ) ); ?>><?php echo $label; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Direction to order products by.', 'mp' ); ?></span></span> order</th>
				<td>
					<label><input type="radio" name="order" data-default="<?php echo esc_attr( mp_get_setting( 'order' ) ); ?>" value="ASC" <?php checked( 'ASC', mp_get_setting( 'order' ) ); ?> /> <?php _e( 'Ascending', 'mp' ); ?></label> &nbsp; &nbsp;
					<label><input type="radio" name="order" data-default="<?php echo esc_attr( mp_get_setting( 'order' ) ); ?>" value="DESC" <?php checked( 'DESC', mp_get_setting( 'order' ) ); ?> /> <?php _e( 'Descending', 'mp' ); ?></label>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Limits list to a specific product category.', 'mp' ); ?></span></span> category</th>
				<td>
					<select name="category" data-default="" class="mp-chosen-select">
						<option value=""><?php _e( 'None', 'mp' ); ?></option>
						<?php foreach ( $this->_product_cats as $term ) : ?>
							<option value="<?php echo esc_attr( isset( $term->slug ) ? $term->slug : ''  ); ?>"><?php echo isset( $term->name ) ? $term->name : ''; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Limits list to a specific product tag.', 'mp' ); ?></span></span> tag</th>
				<td>
					<select name="tag" data-default="" class="mp-chosen-select">
						<option value=""><?php _e( 'None', 'mp' ); ?></option>
						<?php foreach ( $this->_product_tags as $term ) : ?>
							<option value="<?php echo esc_attr( isset( $term->slug ) ? $term->slug : ''  ); ?>"><?php echo isset( $term->name ) ? $term->name : ''; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'If checked, products will be displayed as a list - otherwise products will be displayed as a grid.', 'mp' ); ?></span></span> list_view</th>
				<td>
					<input type="checkbox" name="list_view" data-default="<?php echo esc_attr( ( mp_get_setting( 'list_view' ) == 'list' ) ? 1 : 0  ); ?>" value="1" <?php checked( 'list', mp_get_setting( 'list_view' ) ); ?> />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_global_categories_list] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_global_categories_list_attributes() {
		?>
		<table id="mp-global-categories-list-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Sort categories alphabetically, by unique Category ID, or by the count of posts in that Category.', 'mp' ); ?></span></span> orderby</th>
				<td>
					<select name="orderby" data-default="name">
						<?php foreach ( array( 'ID', 'name', 'slug', 'count', 'term_group' ) as $value ) : ?>
							<option <?php selected( $value, 'name' ); ?>><?php echo $value; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Sort order for categories (either ascending or descending).', 'mp' ); ?></span></span> order</th>
				<td>
					<select name="order" data-default="ASC">
						<?php foreach ( array( 'ASC', 'DESC' ) as $value ) : ?>
							<option <?php selected( $value, 'ASC' ); ?>><?php echo $value; ?></option>
						<?php endforeach; ?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Toggles the display of the current count of posts in each category.', 'mp' ); ?></span></span> show_count</th>
				<td>
					<input type="checkbox" name="show_count" data-default="" value="1" />
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Sets the number of Categories to display. This causes the SQL LIMIT value to be defined.', 'mp' ); ?></span></span> number</th>
				<td>
					<input type="text" name="limit" data-default="" value="" />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the [mp_global_categories_list] short code attributes
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_mp_global_tag_cloud_attributes() {
		?>
		<table id="mp-global-tag-cloud-shortcode" class="form-table" style="display:none">
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Order of the tags.', 'mp' ); ?></span></span> orderby</th>
				<td>
					<label for="mp-tag-cloud-orderby-name"><input type="radio" name="orderby" id="mp-tag-cloud-orderby-name" data-default="name" value="name" checked /> <?php _e( 'Name', 'mp' ); ?></label><br />
					<label for="mp-tag-cloud-orderby-count"><input type="radio" name="orderby" id="mp-tag-cloud-orderby-count" data-default="count" value="count" /> <?php _e( 'Count', 'mp' ); ?></label>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Sort order.', 'mp' ); ?></span></span> order</th>
				<td>
					<label for="mp-tag-cloud-order-asc"><input type="radio" name="order" id="mp-tag-cloud-order-asc" data-default="ASC" value="ASC" checked /> <?php _e( 'Ascending', 'mp' ); ?></label><br />
					<label for="mp-tag-cloud-order-desc"><input type="radio" name="order" id="mp-tag-cloud-order-desc" data-default="ASC" value="DESC" /> <?php _e( 'Descending', 'mp' ); ?></label><br />
					<label for="mp-tag-cloud-order-rand"><input type="radio" name="order" id="mp-tag-cloud-order-rand" data-default="RAND" value="RAND" /> <?php _e( 'Random', 'mp' ); ?></label>
				</td>
			</tr>
			<tr>
				<th scope="row"><span class="mp-tooltip dashicons dashicons-editor-help"><span><?php _e( 'Sets the number of Tags to display. This causes the SQL LIMIT value to be defined.', 'mp' ); ?></span></span> number</th>
				<td>
					<input type="text" name="limit" data-default="" value="" />
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Displays the short code builder submit/reset buttons
	 *
	 * @since 3.0
	 * @access public
	 */
	public function display_shortcode_builder_form_buttons() {
		?>
		<p class="submit">
			<input class="button-primary" type="submit" value="<?php _e( 'Insert Short Code', 'mp' ); ?>" />
			<!--<input class="button" type="button" value="<?php _e( 'Reset to Defaults', 'mp' ); ?>" />-->
		</p>
		<?php
	}

	/**
	 * Enqueues admin javascript/css
	 *
	 * @since 3.0
	 * @access public
	 */
	public function enqueue_styles_scripts() {
		if ( isset( $_GET['page'] ) ) {
			$page = isset( $_GET['page'] );
		} else {
			$page = '';
		}

		$screenpage = get_current_screen();

		if ( $page == 'store-settings'
			|| $page == 'store-settings-presentation'
			|| $page == 'store-settings-notifications'
			|| $page == 'store-settings-shipping'
			|| $page == 'store-settings-payments'
			|| $page == 'store-settings-productattributes'
			|| $page == 'store-settings-capabilities'
			|| $page == 'store-settings-addons'
			|| ( isset( $_GET['taxonomy'] ) && ($_GET['taxonomy'] == 'product_category' || $_GET['taxonomy'] == 'product_tag') )
			|| ( isset( $_GET['post_type'] ) && ($_GET['post_type'] == 'mp_coupon' || $_GET['post_type'] == 'mp_order' || $_GET['post_type'] == 'product' || $_GET['post_type'] == 'page') )
			|| ( isset( $screenpage->post_type ) && ( $screenpage->post_type == "product" || $screenpage->post_type == "post" || $screenpage->post_type == "mp_order" || $screenpage->post_type == "mp_coupon" || $screenpage->post_type == "page") ) )  {

			wp_enqueue_style( 'colorbox', mp_plugin_url( 'includes/admin/ui/colorbox/colorbox.css' ), false, MP_VERSION );
			wp_enqueue_script( 'colorbox', mp_plugin_url( 'ui/js/jquery.colorbox-min.js' ), false, MP_VERSION );

			wp_enqueue_style( 'mp-select2', mp_plugin_url( 'ui/select2/select2.css' ), false, MP_VERSION );
			wp_enqueue_script( 'mp-select2', mp_plugin_url( 'ui/select2/select2.min.js' ), false, MP_VERSION );

			wp_enqueue_script( 'mp-shortcode-builder', mp_plugin_url( 'includes/admin/ui/js/shortcode-builder.js', array( 'colorbox', 'chosen' ), MP_VERSION ) );
			wp_localize_script( 'mp-shortcode-builder', 'MP_ShortCode_Builder', array(
				'select_product' => __( 'Select a Product', 'mp' ),
			) );

		}
	}

	/**
	 * Displays the short code builder button above the tinymce editor.
	 *
	 * @since 3.0
	 * @access public
	 */
	public function media_buttons( $editor_id ) {
		// I was trying to get the editor_id here, but it seems the editor is unique, uniqid() is used.

		$SC_Builder = true;
		if( isset( $_REQUEST['page'] ) && $_REQUEST['page'] == 'store-settings-addons' ) {
			$SC_Builder = false;
		}
		if( $SC_Builder ) {
		?>
		<a href="javascript:;" class="button mp-shortcode-builder-button" title="<?php _e( 'Add E-Commerce Short Code', 'mp' ); ?>"><span class="wp-media-buttons-icon dashicons dashicons-cart"></span> <?php _e( 'Add E-Commerce Short Code', 'mp' ); ?></a>
		<?php
		}
	}

}

MP_Shortcode_Builder::get_instance();