<?php
/**
* Plugin Name: WP Plugin Manager Pro
* Plugin URI: https://hasthemes.com/plugins/
* Description: WP Plugin Manager is a WordPress plugin that allows you to disable/enable plugins for certain pages, posts or URI conditions.
* Version: 1.4.4
* Author: HasThemes
* Author URI: https://hasthemes.com/
* Text Domain: htpmpro
*/

defined( 'ABSPATH' ) or die();

/**
 * Define path
 */
define( 'HTPMPRO_ROOT_PL', __FILE__ );
define( 'HTPMPRO_ROOT_URL', plugins_url('', HTPMPRO_ROOT_PL) );
define( 'HTPMPRO_PL_ROOT', HTPMPRO_ROOT_PL );
define( 'HTPMPRO_ROOT_DIR', dirname( HTPMPRO_ROOT_PL ) );
define( 'HTPMPRO_PLUGIN_DIR', plugin_dir_path( __DIR__ ) );
define( 'HTPMPRO_PLUGIN_BASE', plugin_basename( HTPMPRO_ROOT_PL ) );
define( 'HTPMPRO_PLUGIN_VERSION', '1.4.4' );


class HTPMPRO_Main {

    // Singleton instance
    private static $_instance = null;

    /**
     * Instance
     * Initializes a singleton instance
     * @return self
     */
    public static function instance() {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Private constructor to prevent multiple instances
     */
    private function __construct() {

        /** Load the is_plugin_active function if it doesn't exist */
        if (!function_exists('is_plugin_active')) {
            include_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        add_action('in_admin_header', [$this, 'remove_admin_notice'], 1000);

		add_action( 'init', [$this, 'i18n'] );
		add_action( 'plugins_loaded', [$this, 'init'] );

        register_activation_hook( HTPMPRO_ROOT_PL, [$this, 'activation'] );
        register_deactivation_hook( HTPMPRO_ROOT_PL, [$this, 'deactivation'] );

        if( is_plugin_active('wp-plugin-manager/plugin-main.php') ){
            add_action('update_option_active_plugins', [$this, 'deactivate_lite_version']);
        }

        add_action( 'admin_enqueue_scripts', [$this, 'admin_scripts'] );

        add_action('init', [$this, 'create_mu_file']);

        if(is_admin()) {
            add_action('wp_loaded', function() {
                include_once HTPMPRO_ROOT_DIR . '/includes/licence/WPPluginManagerPro.php';

            });
        }
    }

    /**
     * Remove all Notices on admin pages.
     * @return void
     */
    function remove_admin_notice(){
        $current_screen = get_current_screen();
        $hide_screen = ['toplevel_page_htpmpro-options', 'plugin-manager_page_wp-plugin-manager-pro'];
        if(  in_array( $current_screen->id, $hide_screen) ){
            remove_all_actions('admin_notices');
            remove_all_actions('all_admin_notices');
        }
    }

    /**
     * Load Textdomain
     */
    function i18n() {
        load_plugin_textdomain( 'htpmpro', false, dirname( plugin_basename( HTPMPRO_ROOT_PL ) ) . '/languages' );
    }

    function init() {

        /** Include files */
        if(is_admin()) {
            require_once HTPMPRO_ROOT_DIR . '/includes/helper_functions.php';
            require_once HTPMPRO_ROOT_DIR . '/includes/plugin-options-page.php';
        }
        include_once( HTPMPRO_ROOT_DIR . '/includes/api/admin-dashboard-api.php');
        include_once( HTPMPRO_ROOT_DIR . '/includes/api/admin-settings.php');
        include_once( HTPMPRO_ROOT_DIR . '/includes/api/changelog-api.php');
        include_once( HTPMPRO_ROOT_DIR . '/includes/api/recommended-plugins-api.php');

        // Initialize REST API endpoints
        add_action('rest_api_init', function() {
            $plugins_api = new \HTPM\Api\Plugins();
            $plugins_api->register_routes();
        });
    }

    /**
     * Plugin activation hook
     */
    function activation() {
        update_option('htpmpro_status', 'active');

        // replace the old file
        $mu_plugin_file_source_path = HTPMPRO_ROOT_DIR . '/mu-plugin/htpm-mu-plugin.php';

        $mu_plugins_path = ( defined ( 'WPMU_PLUGIN_DIR' ) ) ? WPMU_PLUGIN_DIR : WP_CONTENT_DIR . '/' . 'mu-plugins';

        $mu_plugin_file_path = "$mu_plugins_path/htpm-mu-plugin.php";

        // add mu file 
        if ( file_exists( $mu_plugins_path ) ){
            copy( $mu_plugin_file_source_path, $mu_plugin_file_path );
        }
    }
    
    /**
     * Plugin deactivation hook
     */
    function deactivation(){
        update_option('htpmpro_status', 'deactive');
    }

    /**
     * Plugin deactivation lite version
     */
    function deactivate_lite_version(){
        deactivate_plugins('wp-plugin-manager/plugin-main.php');
    }
    
    /**
     * Enqueue admin scripts and styles.
     */
    function admin_scripts( $hook_suffix ) {
        if( $hook_suffix ==  'toplevel_page_htpmpro-options' ){

            // Load all Style files.
            wp_enqueue_style( 'htpmpro-admin', HTPMPRO_ROOT_URL . '/assets/css/admin-style.css' );

            // Enqueue Vue app assets and styles
            wp_enqueue_style(
                'htpm-vue-settings-style',
                HTPMPRO_ROOT_URL . '/assets/dist/css/style.css',
                array(),
                HTPMPRO_PLUGIN_VERSION,
                'all'
            );

            wp_enqueue_script(
                'htpm-vue-settings',
                HTPMPRO_ROOT_URL . '/assets/dist/js/main.js',
                array(),
                HTPMPRO_PLUGIN_VERSION,
                true
            );

            add_filter('script_loader_tag', function($tag, $handle, $src) {
                if ($handle === 'htpm-vue-settings') {
                    return '<script type="module" src="' . esc_url($src) . '"></script>';
                }
                return $tag;
            }, 10, 3);
            
            $admin_settings = WP_Plugin_Manager_Settings::get_instance();
            $localize_data = [
                'ajaxurl'          => admin_url( 'admin-ajax.php' ),
                'adminURL'         => admin_url(),
                'pluginURL'        => plugin_dir_url( __FILE__ ),
                'restUrl' => rest_url(),  // This will include the wp-json prefix
                'assetsURL'        => plugin_dir_url( __FILE__ ) . 'assets/',
                'nonce' => wp_create_nonce('wp_rest'),
                'licenseNonce'  => wp_create_nonce( 'el-license' ),
                'licenseEmail'  => get_option( 'WPPluginManagerPro_lic_email', get_bloginfo( 'admin_email' ) ),
                'message'          =>[
                    'packagedesc'=> esc_html__( 'in this package', 'htpmpro' ),
                    'allload'    => esc_html__( 'All Items have been Loaded', 'htpmpro' ),
                    'notfound'   => esc_html__( 'Nothing Found', 'htpmpro' ),
                ],
                'buttontxt'      =>[
                    'tmplibrary' => esc_html__( 'Import to Library', 'htpmpro' ),
                    'tmppage'    => esc_html__( 'Import to Page', 'htpmpro' ),
                    'import'     => esc_html__( 'Import', 'htpmpro' ),
                    'buynow'     => esc_html__( 'Buy Now', 'htpmpro' ),
                    'preview'    => esc_html__( 'Preview', 'htpmpro' ),
                    'installing' => esc_html__( 'Installing..', 'htpmpro' ),
                    'activating' => esc_html__( 'Activating..', 'htpmpro' ),
                    'active'     => esc_html__( 'Active', 'htpmpro' ),
                    'pro' => __( 'Pro', 'htpmpro' ),
                    'modal' => [
                        'title' => __( 'BUY PRO', 'htpmpro' ),
                        'desc' => __( 'Our free version is great, but it doesn\'t have all our advanced features. The best way to unlock all of the features in our plugin is by purchasing the pro version.', 'htpmpro' )
                    ],
                ],
                'existingData' => get_option('htpm_options'),
                'helpSection' => [
                    'title' => esc_html__('Need Help with Plugin Manager?', 'htpmpro'),
                    'description' => esc_html__('Our comprehensive documentation provides detailed information on how to use Plugin Manager effectively to improve your websites performance.', 'htpmpro'),
                    'documentation' => esc_html__('Documentation', 'htpmpro'),
                    'videoTutorial' => esc_html__('Video Tutorial', 'htpmpro'),
                    'support' => esc_html__('Support', 'htpmpro'),
                    'docLink' => 'https://hasthemes.com/docs/wp-plugin-manager/',
                    'videoLink' => 'https://www.youtube.com/watch?v=u94hkbTzKFU',
                    'supportLink' => 'https://hasthemes.com/contact-us/',
                    'upgradeLink' => 'https://hasthemes.com/plugins/wp-plugin-manager-pro/?utm_source=admin&utm_medium=mainmenu&utm_campaign=free#pricing',
                    'licenseLink' => 'https://hasthemes.com/plugins/wp-plugin-manager-pro/?utm_source=admin&utm_medium=mainmenu&utm_campaign=free#pricing',
                    'recommendedPluginsLink' => 'https://hasthemes.com/plugins/',
                ],
                'adminSettings' => [
                    'modal_settings_fields' => $admin_settings->get_modal_settings_fields(),
                    'is_pro' => $admin_settings->is_pro(),
                    'labels_texts' => $admin_settings->get_labels_texts(),
                    'dashboard_settings' => $admin_settings->get_dashboard_settings(),
                    'menu_settings' => $admin_settings->get_menu_settings(),
                    'recommendations_plugins' => $admin_settings->get_recommendations_plugins(),
                    'backend_modal_settings' => $admin_settings->get_backend_modal_settings(),
                    'allSettings' => get_option('htpm_options') ? get_option('htpm_options') : [],
                ],
            ];
            wp_localize_script( 'htpm-vue-settings', 'HTPMM', $localize_data );
        }
    }

    /**
     * Add mu file
     */
    function create_mu_file(){
        update_option('htpm_available_post_types', array_merge(array_keys(get_post_types( ['_builtin' => false, 'public' => true], 'names')), ['post', 'page']));
        
        $mu_plugin_file_source_path = HTPMPRO_ROOT_DIR . '/mu-plugin/htpm-mu-plugin.php';

        $mu_plugins_path = ( defined ( 'WPMU_PLUGIN_DIR' ) ) ? WPMU_PLUGIN_DIR : WP_CONTENT_DIR . '/' . 'mu-plugins';

        $mu_plugin_file_path = "$mu_plugins_path/htpm-mu-plugin.php";

        $plugin_data = get_file_data(HTPMPRO_PL_ROOT, ['Version'=>'Version'], 'plugin');
        $version = $plugin_data['Version'];

        if ( !is_dir( $mu_plugins_path ) ){
            mkdir( $mu_plugins_path, 0755, true );
            copy( $mu_plugin_file_source_path, $mu_plugin_file_path );
        } else {
            // create mu-plugins folder
            if ( !file_exists($mu_plugins_path) || version_compare($version, '1.0.8', '>') ) {
                copy( $mu_plugin_file_source_path, $mu_plugin_file_path );
            }
        }
    }

}


HTPMPRO_Main::instance();