<?php
namespace WPPayFormPro\Integrations\UserRegistration;

if (!defined('ABSPATH')) {
    exit;
}

use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Models\Submission;
use WPPayForm\App\Models\Meta;

class UserRegistrationApi
{
    public function getUserRoles()
    {
        if (!function_exists('get_editable_roles')) {
            require_once ABSPATH . 'wp-admin/includes/user.php';
        }

        $roles = get_editable_roles();

        $validRoles = [];
        foreach ($roles as $roleKey => $role) {
            if (!Arr::get($role, 'capabilities.manage_options')) {
                $validRoles[$roleKey] = $role['name'];
            }
        }
        return apply_filters('wppayform_UserRegistration_creatable_roles', $validRoles);
    }

    public function validate($settings, $settingsFields)
    {
        foreach ($settingsFields['fields'] as $field) {
            if ($field['key'] != 'CustomFields') {
                continue;
            }

            $errors = [];

            foreach ($field['primary_fileds'] as $primaryField) {
                if (!empty($primaryField['required'])) {
                    if (empty($settings[$primaryField['key']])) {
                        $errors[$primaryField['key']] = $primaryField['label'] . ' is required.';
                    }
                }
            }

            if ($errors) {
                wp_send_json_error([
                    'message' => array_shift($errors),
                    'errors' => $errors
                ], 422);
            }
        }

        return $settings;
    }

    public function registerUser($feed, $formData, $entry, $formId, $integrationKey)
    {
        if (get_current_user_id()) {
            return;
        }

        $parsedValue = $feed['processedValues'];
        $updateUserRoles = $this->updateUserRoles($parsedValue, $entry->id, $formId);
        if ($updateUserRoles) {
            return;
        }

        if (!empty($parsedValue['Email']) && !is_email($parsedValue['Email'])) {
            $parsedValue['Email'] = Arr::get(
                $formData,
                $parsedValue['Email']
            );
        }

        if (!empty($parsedValue['Email']) && !is_email($parsedValue['Email'])) {
            $this->addLog(
                "User not created, Email address is required!",
                $formId,
                $entry->id,
                'failed'
            );
            return;
        }

        if (!empty($parsedValue['Email']) && email_exists($parsedValue['Email'])) {
            $this->addLog(
                "User not created Email(" . $parsedValue['Email'] . ") already exist",
                $formId,
                $entry->id,
                'failed'
            );
            return;
        }

        if (!empty($parsedValue['username'])) {
            $userName = Arr::get($formData, $parsedValue['username']);
            if (is_array($userName)) {
                return;
            }
            if ($userName && username_exists($userName)) {
                return;
            }
            if ($userName) {
                $parsedValue['username'] = $userName;
            }
        }

        if (empty($parsedValue['username'])) {
            $parsedValue['username'] = $parsedValue['Email'];
        }

        $feed['processedValues'] = $parsedValue;

        $fullName = Arr::get($parsedValue, 'full_name');
        if ($fullName) {
            $nameArray = explode(' ', $fullName);
            if (count($nameArray) > 1) {
                $feed['processedValues']['last_name'] = array_pop($nameArray);
                $feed['processedValues']['first_name'] = implode(' ', $nameArray);
            } else {
                $feed['processedValues']['first_name'] = $fullName;
            }
        }

        do_action('wppayform_user_registration_before_start', $feed, $entry, $formId);

        $this->createUser($feed, $formData, $entry, $formId, $integrationKey);
    }

    protected function createUser($feed, $formData, $entry, $formId, $integrationKey)
    {
        $feed = apply_filters('wppayform_user_registration_feed', $feed, $entry, $formId);

        $parsedData = $feed['processedValues'];

        $email = $parsedData['Email'];
        $userName = $parsedData['username'];

        if (isset($parsedData['password'])) {
            $password = $parsedData['password'];
        } else if ($userName) {
            $password = $userName;
        } else {
            $password = wp_generate_password(8);
        }

        $userId = wp_create_user($userName, $password, $email);

        if (is_wp_error($userId)) {
            return $this->addLog(
                $userId->get_error_message(),
                $formId,
                $entry->id,
                'failed'
            );
        }

        do_action('wppayform_created_user', $userId, $feed, $entry, $formId);
        (new Meta())->updateOrderMeta('formSettings', $entry->id, '__created_user_id', '', $formId);

        $this->updateUser($parsedData, $userId);

        $this->addUserRole($parsedData, $userId);

        $this->addUserMeta($parsedData, $userId, $formId);

        $this->maybeLogin($parsedData, $userId, $entry);

        $this->maybeSendEmail($parsedData, $userId);

        do_action('wppayform_user_registration_completed', $userId, $feed, $entry, $formId);

        // Autologin process start
        $do_autologin = ((Arr::get($parsedData, 'enableAutoLogin') == 'yes') || (Arr::get($parsedData, 'enableAutoLogin') == 1));

        if (true == $do_autologin) {
            //get the user_id of the current logged-in user (if any)
            $current_user_id = get_current_user_id(); //returns 0 if no user is currently logged-in

            if (0 == $current_user_id) {
                //$current_user_id is 0, so there is not any current logged-in user
                //execute autologin stuff
                wp_clear_auth_cookie();
                wp_set_current_user($userId);
                wp_set_auth_cookie($userId);

                do_action('wppayform_log_data', [
                    'form_id' => $formId,
                    'submission_id' => $entry->id,
                    'type' => 'failed',
                    'created_by' => 'Paymattic BOT',
                    'content' => "User automatically logged in after course purchase and registration"
                ]);
            }
        }
        // Autologin process end

        $this->addLog(
            'user has been successfully created. Created User ID: ' . $userId,
            $formId,
            $entry->id,
            'success'
        );

        Submission::where('id', $entry->id)
            ->update([
                'user_id' => $userId
            ]);
    }

    protected function updateUser($parsedData, $userId)
    {
        $name = trim(Arr::get($parsedData, 'first_name') . ' ' . Arr::get($parsedData, 'last_name'));

        $data = array_filter([
            'ID' => $userId,
            'user_nicename' => $name,
            'display_name' => $name,
            'user_url' => Arr::get($parsedData, 'user_url')
        ]);

        if ($name) {
            wp_update_user($data);
        }
    }

    protected function addUserRole($parsedData, $userId)
    {
        $userRoles = $this->getUserRoles();
        $assignedRole = $parsedData['userRole'];

        if (!isset($userRoles[$assignedRole])) {
            $assignedRole = 'subscriber';
        }

        $user = new \WP_User($userId);
        $user->set_role($assignedRole);
    }

    protected function addUserMeta($parsedData, $userId, $formId)
    {
        foreach ($parsedData['userMeta'] as $userMeta) {
            $userMetas[$userMeta['label']] = $userMeta['item_value'];
        }

        $userMetas = array_merge($userMetas, [
            'first_name' => Arr::get($parsedData, 'first_name'),
            'last_name' => Arr::get($parsedData, 'last_name')
        ]);

        if (!isset($userMetas['nickname'])) {
            $userMetas['nickname'] = Arr::get($parsedData, 'first_name') . ' ' . Arr::get($parsedData, 'last_name');
        }

        foreach ($userMetas as $metaKey => $metaValue) {
            if ($metaValue && trim($metaValue)) {
                update_user_meta($userId, $metaKey, trim($metaValue));
            }
        }

        update_user_meta($userId, 'wppayform_user_id', $formId);
    }

    protected function maybeLogin($parsedData, $userId, $entry = false)
    {
        // if (Arr::isTrue($parsedData, 'enableAutoLogin')) {
        //     // check if it's payment success page
        //     // or direct url
        //     if (isset($_REQUEST['wppayform_payment_api_notify']) && $entry) {
        //         // This payment IPN request so let's keep a reference for real request
        //         (new Meta())->updateOrderMeta('formSettings', $entry->id, '_make_auto_login', $value, $entry->form_id);
        //         return;
        //     }

        //     wp_clear_auth_cookie();
        //     wp_set_current_user($userId);
        //     wp_set_auth_cookie($userId);
        // }
    }

    protected function maybeSendEmail($parsedData, $userId)
    {
        if (Arr::isTrue($parsedData, 'sendEmailToNewUser')) {
            // This will send an email with password setup link
            \wp_new_user_notification($userId, null, 'user');
        }
    }

    protected function addLog($content, $formId, $entryId, $type = 'activity')
    {
        do_action('wppayform_log_data', [
            'form_id' => $formId,
            'submission_id' => $entryId,
            'type' => $type,
            'created_by' => 'Paymattic BOT',
            'content' => $content
        ]);
    }

    public function updateUserRoles($parsedData, $entryId, $formId)
    {
        $targetRole = trim(Arr::get($parsedData, 'userRole', ''));
        $email = sanitize_email(Arr::get($parsedData, 'Email', ''));
        if (!$email || !$targetRole) {
            return false;
        }

        $user = get_user_by('email', $email);
        if (!$user) {
            return false;
        }

        // Check if user needs role update
        if (user_can($user, 'manage_options') || in_array($targetRole, $user->roles, true) || in_array('paymattic_user', $user->roles, true)) {
            return false; // Skip admins or users who already have the role
        }

        if ( $targetRole !== 'paymattic_user' || !user_can($user, 'manage_options') ) {
            // Add capabilities
            $caps = [
                'read_entry'                    => true,
                'read_subscription_entry'       => false,
                'can_sync_subscription_billings'=> false,
                'cancel_subscription'           => false,
            ];
            foreach ( $caps as $cap => $grant ) {
                $user->add_cap( $cap, $grant );
            }
        }

        $user->set_role($targetRole);
        do_action('wppayform_log_data', [
            'form_id' => $formId,
            'submission_id' => $entryId,
            'type' => 'info',
            'created_by' => 'Paymattic BOT',
            'content' => "User role updated to {$targetRole} for user {$user->user_email}"
        ]);
        return true;
    }
}
