<?php

namespace WPPayFormPro\Integrations\UserRegistration;

if (!defined('ABSPATH')) {
    exit;
}

use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Services\ConditionAssesor;
use WPPayForm\App\Services\Integrations\IntegrationManager;
use WPPayForm\Framework\Foundation\App;
use WPPayForm\App\Models\Meta;

class Bootstrap extends IntegrationManager
{
    public $category = 'wp_core';
    public $disableGlobalSettings = 'yes';
    private $userApi = null;
    public function __construct()
    {
        parent::__construct(
            App::getInstance(),
            'User Registration',
            'UserRegistration',
            '_wppayform_user_registration_settings',
            'user_registration_feeds',
            1
        );

        $this->userApi = new UserRegistrationApi;

        $this->logo = WPPAYFORM_URL . 'assets/images/integrations/user_registration.png';

        $this->description = 'Create WordPress user when a form is submitted.';

        add_filter('wppayform_save_integration_value_' . $this->integrationKey, [$this, 'validate'], 10, 3);

        add_filter('wppayform_validation_user_registration_errors', [$this, 'validateSubmittedForm'], 10, 3);

        // It's should not async, so don't comment this line bellow :)
        add_filter('wppayform_notifying_async_UserRegistration', '__return_false');

        $this->registerAdminHooks();
    }

    public function pushIntegration($integrations, $formId)
    {
        $integrations[$this->integrationKey] = [
            'category'                => 'wp_core',
            'disable_global_settings' => 'yes',
            'logo'                    => $this->logo,
            'title'                   => $this->title . ' Integration',
            'is_active'               => $this->isConfigured(),
            'config_url'              => admin_url('admin.php?page=wppayform.php#/integrations/' . $this->integrationKey . ''),
            'global_configure_url'    => admin_url('admin.php?page=wppayform.php#/integrations/' . $this->integrationKey . ''),
        ];

        return $integrations;
    }

    public function getIntegrationDefaults($settings, $formId = null)
    {
        $fields = [
            'name'                 => '',
            'Email'                => '',
            'username'             => '',
            'CustomFields'         => (object)[],
            'userRole'             => 'subscriber',
            'userMeta'             => [
                [
                    'label' => '', 'item_value' => ''
                ]
            ],
            'enableAutoLogin'      => false,
            'sendEmailToNewUser'   => false,
            'validateForUserEmail' => true,
            'conditionals'         => [
                'conditions' => [],
                'status'     => false,
                'type'       => 'all'
            ],
            'enabled'              => true
        ];

        return apply_filters('wppayform_user_registration_field_defaults', $fields, $formId);
    }

    public function getSettingsFields($settings, $formId = null)
    {
        $fields = apply_filters('wppayform_user_registration_feed_fields', [
            [
                'key'         => 'name',
                'label'       => 'Name',
                'required'    => true,
                'placeholder' => 'Your Feed Name',
                'component'   => 'text'
            ],
            [
                'key'                => 'CustomFields',
                'require_list'       => false,
                'label'              => 'Map Fields',
                'tips'               => 'Associate your user registration fields to the appropriate Paymattic fields by selecting the appropriate form field from the list.',
                'component'          => 'map_fields',
                'field_label_remote' => 'User Registration Field',
                'field_label_local'  => 'Form Field',
                'primary_fileds'     => [
                    [
                        'key'           => 'Email',
                        'label'         => 'Email Address',
                        'required'      => true,
                        'input_options' => 'emails'
                    ],
                    [
                        'key'       => 'username',
                        'label'     => 'User name',
                        'required'  => false,
                        'help_text' => 'Keep empty if you want the username and user email is the same',
                    ],
                    [
                        'key'   => 'full_name',
                        'label' => 'Full Name'
                    ],
                    [
                        'key'       => 'password',
                        'label'     => 'Password',
                        'help_text' => 'Keep empty to be auto generated',
                    ],
                ]
            ],
            [
                'require_list' => false,
                'required'     => true,
                'key'          => 'userRole',
                'label'        => 'Default User Role',
                'tips'         => 'Set a default user role for new registrations and update existing roles, excluding Admin and Paymattic User.',
                'component'    => 'radio_choice',
                'options'      => $this->userApi->getUserRoles()
            ],
            [
                'require_list' => false,
                'key'          => 'userMeta',
                'label'        => 'User Meta',
                'tips'         => 'Add user meta.',
                'component'    => 'dropdown_label_repeater',
                'field_label'  => 'User Meta Key',
                'value_label'  => 'User Meta Value'
            ],
            [
                'require_list'   => false,
                'key'            => 'enableAutoLogin',
                'label'          => 'Auto Login',
                'checkbox_label' => 'Allow the user login automatically after registration',
                'component'      => 'checkbox-single',
            ],
            [
                'require_list'   => false,
                'key'            => 'sendEmailToNewUser',
                'label'          => 'Email Notification',
                'checkbox_label' => 'Send default WordPress welcome email to user after registration',
                'component'      => 'checkbox-single',
            ],
            [
                'require_list'   => false,
                'key'            => 'validateForUserEmail',
                'label'          => 'Form Validation',
                'checkbox_label' => "Don't create a user if one already exists in the database with the same email address.",
                'component'      => 'checkbox-single',
            ]
        ], $formId);

        $fields[] = [
            'require_list' => false,
            'key'          => 'conditionals',
            'label'        => 'Conditional Logics',
            'tips'         => 'Allow User Registration integration conditionally based on your submission values',
            'component'    => 'conditional_block'
        ];
        $fields[] = [
            'require_list'   => false,
            'key'            => 'enabled',
            'label'          => 'Status',
            'component'      => 'checkbox-single',
            'checkbox_label' => 'Enable This feed',
            'inline_tip'     => 'Please note that, This action will only run if the visitor is logged out state and the email is not registered yet'
        ];

        return [
            'fields'              => $fields,
            'button_require_list' => false,
            'integration_title'   => $this->title
        ];
    }

    public function validate($settings, $integrationId, $formId)
    {
        $parseSettings = $this->userApi->validate(
            $settings,
            $this->getSettingsFields($settings)
        );

        (new Meta())->updateOrderMeta('formSettings', '', '_has_user_registration', '', $formId);

        return $parseSettings;
    }

    public function validateSubmittedForm($errors, $data, $form)
    {
        $feeds = Meta::where('form_id', $form->id)
            ->where('meta_key', 'user_registration_feeds')
            ->get();

        if (!$feeds) {
            return $errors;
        }

        foreach ($feeds as $feed) {
            $parsedValue = json_decode($feed->value, true);

            if (!Arr::isTrue($parsedValue, 'validateForUserEmail')) {
                continue;
            }

            if ($parsedValue && Arr::isTrue($parsedValue, 'enabled')) {
                // Now check if conditions matched or not
                $isConditionMatched = $this->checkCondition($parsedValue, $data);
                if (!$isConditionMatched) {
                    continue;
                }
                $email = Arr::get($data, $parsedValue['Email']);
                if (!$email) {
                    continue;
                }

                if (email_exists($email)) {
                    if (!isset($errors['restricted'])) {
                        $errors['restricted'] = [];
                    }
                    $errors['restricted'][] = __('This email is already registered. Please choose another one.', 'wppayformpro');
                    return $errors;
                }

                if (!empty($parsedValue['username'])) {
                    $userName = Arr::get($data, $parsedValue['username']);
                    if ($userName) {
                        if (username_exists($userName)) {
                            if (!isset($errors['restricted'])) {
                                $errors['restricted'] = [];
                            }
                            $errors['restricted'][] = __('This username is already registered. Please choose another one.', 'wppayformpro');
                            return $errors;
                        }
                    }
                }
            }
        }

        return $errors;
    }


    public function notify($feed, $formData, $entry, $formId)
    {
        $data = array(
            'feed' => $feed,
            'formData' => $formData,
            'entryId' => $entry->id
        );
        $feedsData = wp_unslash($data);
        /**
         * Updates on post_meta to trigger by hooks check "UserCreation.php file"
         */
        return update_post_meta($formId, 'wpf_user_reg_feeds', $feedsData);
    }

    // There is no global settings, so we need
    // to return true to make this module work.
    public function isConfigured()
    {
        return true;
    }

    // This is an absttract method, so it's required.
    public function getMergeFields($list, $listId, $formId)
    {
        // ...
    }

    // This method should return global settings. It's not required for
    // this class. So we should return the default settings otherwise
    // there will be an empty global settings page for this module.
    public function addGlobalMenu($setting)
    {
        return $setting;
    }

    private function checkCondition($parsedValue, $formData)
    {
        $conditionSettings = Arr::get($parsedValue, 'conditionals');
        if (!$conditionSettings ||
            !Arr::isTrue($conditionSettings, 'status') ||
            !count(Arr::get($conditionSettings, 'conditions'))
        ) {
            return true;
        }

        return ConditionAssesor::evaluate($parsedValue, $formData);
    }
}
