<?php

namespace WPPayFormPro\Integrations\Telegram;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use WPPayForm\Framework\Foundation\App;
use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Services\Integrations\IntegrationManager;

class Bootstrap extends IntegrationManager
{
    public function __construct()
    {
        parent::__construct(
            App::getInstance(),
            'Telegram Messenger',
            'telegram',
            '_wppayform_telegram_settings',
            'telegram_feed',
            99
        );

        $this->logo = WPPAYFORM_URL . 'assets/images/integrations/telegram.png';

        $this->description = 'Send notification to Telegram channel or group when a form is submitted';

        $this->registerAdminHooks();

        // add_filter('wppayform_notifying_async_' . $this->integrationKey, '__return_false');
    }

    public function getGlobalFields($fields)
    {

        return [
            'logo'             => $this->logo,
            'menu_title'       => __($this->title . ' Settings', 'wp-payment-form-pro'),
            'menu_description' => 'Create a Bot by sending /newbot command to @BotFather in telegram. After completing the steps @BotFather will provide you the Bot Token. <br>Create a Channel/group/supergroup. Add the Bot as Administrator to your Channel/Group.<br /> <br /><a href="https://paymattic.com/docs/how-to-integrate-telegram-with-paymattic-in-wordpress/" target="_blank" style="color: #F98435">Please read the documentation for step by step tutorial</a>',
            'valid_message'    => __($this->title . ' integration is complete', 'wp-payment-form-pro'),
            'invalid_message'  => __($this->title . ' integration is not complete', 'wp-payment-form-pro'),
            'save_button_text' => __('Save Settings', 'wp-payment-form-pro'),
            'fields'           => [
                'bot_token' => [
                    'type'        => 'password',
                    'placeholder' => 'bot_token',
                    'label_tips'  => __("Enter your Telegram Bot Token", 'wp-payment-form-pro'),
                    'label'       => __('Bot Token', 'wp-payment-form-pro'),
                ],
                'chat_id'   => [
                    'type'        => 'text',
                    'placeholder' => 'Channel Username/ID',
                    'label_tips'  => __("Enter your Telegram API channel user ID, You can also use message id. Please check documentation for more details.", 'wp-payment-form-pro'),
                    'label'       => __('Default Channel/Group Chat ID', 'wp-payment-form-pro'),
                ],
                'message'   => [
                    'type'        => 'textarea',
                    'placeholder' => 'Test Message',
                    'label_tips'  => __("Enter your Test Message", 'wp-payment-form-pro'),
                    'label'       => __('Test Message (optional)', 'wp-payment-form-pro'),
                    'tips'        => 'Provide a message if you want to send a test message now'
                ],
            ],
            'hide_on_valid'    => true,
            'discard_settings' => [
                'section_description' => 'Your ' . $this->title . ' API integration is up and running',
                'button_text'         => 'Disconnect ' . $this->title,
                'data'                => [
                    'chat_id'   => '',
                    'bot_token' => '',
                    'message'   => ''
                ],
                'show_verify'         => true
            ]
        ];
    }

    public function getGlobalSettings($settings)
    {
        $globalSettings = get_option($this->optionKey);
        if (!$globalSettings) {
            $globalSettings = [
                'status' => ''
            ];
        }
        $defaults = [
            'status'    => '',
            'chat_id'   => '',
            'bot_token' => ''
        ];

        return wp_parse_args($globalSettings, $defaults);
    }

    public function saveGlobalSettings($settings)
    {
        if (empty($settings['chat_id']) || empty($settings['bot_token'])) {
            $settings['status'] = false;
            update_option($this->optionKey, $settings, 'no');
            wp_send_json_success([
                'message' => __('Your settings has been updated and discarded', 'wp-payment-form-pro'),
                'status'  => false
            ], 200);
        }

        $responseMessage = __('Your ' . $this->integrationKey . ' api key has been verified and successfully set', 'wp-payment-form-pro');
        $status = false;
        // Verify API key now
        try {
            $api = $this->getApiClient($settings['bot_token']);

            $apiStatus = $api->getMe();
            
            if (is_wp_error($apiStatus)) {
                throw new \Exception($apiStatus->get_error_message());
            }

            $apiSettings = [
                'bot_token' => sanitize_textarea_field($settings['bot_token']),
                'status'    => true,
                'chat_id'   => sanitize_textarea_field($settings['chat_id'])
            ];

            $message = Arr::get($settings, 'message', '');
            if ($message) {
                $api->setChatId($apiSettings['chat_id']);
                $result = $api->sendMessage($message);
                if (is_wp_error($result)) {
                    $apiSettings['status'] = false;
                    $responseMessage = 'Your api key is right but, the message could not be sent to the provided chat id. Error: ' . $result->get_error_message();
                }
            }

            $status = $apiSettings['status'];

            update_option($this->optionKey, $apiSettings, 'no');
        } catch (\Exception $exception) {
            wp_send_json_error([
                'message' => $exception->getMessage()
            ], 400);
        }

        $responseCode = 200;
        if (!$status) {
            $responseCode = 423;
        }

        wp_send_json_success([
            'message' => $responseMessage,
            'status'  => $status
        ], $responseCode);
    }

    public function pushIntegration($integrations, $formId)
    {
        $integrations[$this->integrationKey] = [
            'title'                 => $this->title . ' Integration',
            'logo'                  => $this->logo,
            'is_active'             => $this->isConfigured(),
            'configure_title'       => 'Configuration required!',
            'config_url'            => admin_url('admin.php?page=wppayform.php#/integrations/' . $this->integrationKey . ''),
            'global_configure_url'  => admin_url('admin.php?page=wppayform.php#/integrations/' . $this->integrationKey . ''),
            'configure_message'     => $this->title . ' is not configured yet! Please configure your  api first',
            'configure_button_text' => 'Set ' . $this->title . ' API'
        ];
        return $integrations;
    }

    public function getIntegrationDefaults($settings, $formId)
    {
        return [
            'name'           => '',
            'send_message'   => '',
            'custom_chat_id' => '',
            'conditionals'   => [
                'conditions' => [],
                'status'     => false,
                'type'       => 'all'
            ],
            'enabled'        => true,
            'trigger_on_payment' => false
        ];
    }

    public function getSettingsFields($settings, $formId)
    {

        return [
            'fields'              => [
                [
                    'key'         => 'name',
                    'label'       => 'Name',
                    'required'    => true,
                    'placeholder' => 'Your Feed Name',
                    'component'   => 'text'
                ],
                [
                    'key'         => 'send_message',
                    'label'       => 'Message to Send',
                    'required'    => true,
                    'placeholder' => 'Telegram Message',
                    'component'   => 'value_textarea'
                ],
                [
                    'key'         => 'custom_chat_id',
                    'label'       => 'Custom Chat/Channel ID',
                    'required'    => false,
                    'placeholder' => 'Custom Chat ID',
                    'component'   => 'text',
                    'inline_tip'  => 'Provide custom chat id if you want to send a different channel or chat ID. <b>Leave blank for global chat ID</b>'
                ],
                [
                    'key'       => 'conditionals',
                    'label'     => 'Conditional Logics',
                    'tips'      => 'Allow ' . $this->title . ' integration conditionally based on your submission values',
                    'component' => 'conditional_block'
                ],
                [
                    'key'            => 'enabled',
                    'label'          => 'Status',
                    'component'      => 'checkbox-single',
                    'checkbox_label' => 'Enable This feed'
                ],
                [
                    'key' => 'trigger_on_payment',
                    'require_list' => false,
                    'label' => 'Trigger On',
                    'checkbox_label' => 'Trigger notification on payment success only',
                    'component' => 'checkbox-single',
                    'tips' => 'Push data to Telegram when payment is successful, helpful for payment data tracking.'
                ]
            ],
            'button_require_list' => false,
            'integration_title'   => $this->title
        ];
    }


    public function getMergeFields($list = false, $listId = false, $formId = false)
    {

        return [];
    }


    /*
     * Form Submission Hooks Here
     */
    public function notify($feed, $formData, $entry, $formId)
    {
        // especially for asynchronous notifications
        if( null == gettype($formData) || !$formData) {
            $formData = Arr::get($entry, 'form_data_formatted');
        }
        
        $feedData = $feed['processedValues'];

        if (empty($feedData['send_message'])) {
            // for now single file  upload only otherwise skip this
            return;
        }

        $settings = $this->getGlobalSettings([]);

        if (!$settings['status']) {
            return;
        }

        if ($chatId = Arr::get($feedData, 'custom_chat_id')) {
            if (trim($chatId)) {
                $settings['chat_id'] = $chatId;
            }
        }

        $api = $this->getApiClient($settings['bot_token'], $settings['chat_id']);
        $response = $api->sendMessage($feedData['send_message']);

        if (is_wp_error($response)) {
            $this->addLog($response->get_error_message(), $formId, $entry->id, 'failed');
            return;
        }

        $this->addLog(
            "Telegram feed has been successfully initialed and pushed data. Message ID: " . Arr::get($response, 'result.message_id'),
            $formId,
            $entry->id
        );
    }


    protected function getApiClient($token, $chatId = '')
    {
        return new TelegramApi(
            $token,
            $chatId
        );
    }
}
