<?php

namespace WPPayFormPro\GateWays\Sslcommerz;

if (!defined('ABSPATH')) {
    exit;
}

use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Models\Transaction;
use WPPayForm\App\Models\Submission;
use WPPayForm\App\Models\Form;
use WPPayForm\App\Services\PlaceholderParser;
use WPPayForm\App\Services\ConfirmationHelper;
use WPPayForm\GateWays\Sslcommerz\QueryHelper\Gateway;

class SslcommerzProcessor
{
    public $method = 'sslcommerz';

    protected $form;

    public function init()
    {
        new SslcommerzElement();
        (new SslcommerzSettings())->init();

        add_filter('wppayform/choose_payment_method_for_submission', array($this, 'choosePaymentMethod'), 10, 4);
        add_action('wppayform/form_submission_make_payment_' . $this->method, array($this, 'makeFormPayment'), 10, 6);
        add_filter('wppayform/entry_transactions_' . $this->method, array($this, 'addTransactionUrl'), 10, 2);
        add_filter('wppayform/submitted_payment_items_' . $this->method, array($this, 'validateSubscription'), 10, 4);

        add_action('wpf_ipn_endpoint_' . $this->method, function () {
            $this->verifyIPN();
            exit(200);
        });
    }

    public function verifyIPN()
    {
        $payId = Arr::get($_POST, 'val_id');
        $Api = new API();
        $vendorTransaction = $Api->validation($payId, $this->getPaymentMode());
        if (empty($vendorTransaction)) {
            return;
        }

        $reference = Arr::get($vendorTransaction, 'tran_id');
        $transaction = (new Transaction())->getTransaction($reference);
        $this->handleStatus($transaction, $vendorTransaction);
    }


    public function handleStatus($transaction, $vendorTransaction)
    {
        // check if already paid return
        if (!$transaction || $transaction->payment_method != $this->method || $transaction->status == 'paid') {
            return;
        }

        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($transaction->submission_id);

        do_action('wppayform/form_submission_activity_start', $transaction->form_id);

        $status = $vendorTransaction['status'];
        if ($status == 'VALID' || $status == 'VALIDATED') {
            $status = 'paid';
        } else {
            do_action('wppayform/form_payment_failed', $submission, $submission->form_id, $transaction, 'sslcommerz');
            return;
        }

        $updateData = array(
            'charge_id'     => Arr::get($vendorTransaction, 'val_id'),
            'payment_total' => intval(Arr::get($vendorTransaction, 'currency_amount') * 100),
            'status'        => $status,
            'card_brand'    => Arr::get($vendorTransaction, 'card_brand'),
            'payment_note'  => maybe_serialize($vendorTransaction),
        );

        $cardNo = Arr::get($vendorTransaction, 'card_no', false);
        if ($cardNo) {
            $updateData['card_last_4'] = substr($cardNo, -4);
        }

        $this->markAsPaid($status, $updateData, $transaction);
    }

    public function markAsPaid($status, $updateData, $transaction)
    {
        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($transaction->submission_id);
        $submissionData = array(
            'payment_status' => $status,
            'updated_at' => current_time('Y-m-d H:i:s')
        );

        $submissionModel->where('id', $transaction->submission_id)->update($submissionData);
        $transactionModel = new Transaction();
        $updateData['updated_at'] = current_time('Y-m-d H:i:s');
        $transaction = $transactionModel->getTransaction($transaction->id);
        $transactionModel->updateTransaction($transaction->id, $updateData);

        do_action('wppayform_log_data', [
            'form_id' => $transaction->form_id,
            'submission_id' => $transaction->submission_id,
            'type' => 'info',
            'created_by' => 'PayForm Bot',
            'content' => sprintf(__('Transaction Marked as %s and SSLCOMMERZ Transaction ID: %s', 'wp-payment-form-pro'), $status, $updateData['charge_id'])
        ]);

        if ($status === 'paid') {
            do_action('wppayform/form_payment_success_sslcommerz', $submission, $transaction, $transaction->form_id, $updateData);
            do_action('wppayform/form_payment_success', $submission, $transaction, $transaction->form_id, $updateData);
        }
    }

    public function choosePaymentMethod($paymentMethod, $elements, $formId, $form_data)
    {
        if ($paymentMethod) {
            // Already someone choose that it's their payment method
            return $paymentMethod;
        }
        // Now We have to analyze the elements and return our payment method
        foreach ($elements as $element) {
            if ((isset($element['type']) && $element['type'] == 'sslcommerz_gateway_element')) {
                return 'sslcommerz';
            }
        }
        return $paymentMethod;
    }

    public function makeFormPayment($transactionId, $submissionId, $form_data, $form, $hasSubscriptions)
    {
        $paymentMode = $this->getPaymentMode();
        $transactionModel = new Transaction();
        if ($transactionId) {
            $transactionModel->updateTransaction($transactionId, array(
                'payment_mode' => $paymentMode
            ));
        }
        $transaction = $transactionModel->getTransaction($transactionId);

        $submission = (new Submission())->getSubmission($submissionId);

        $this->handleRedirect($transaction, $submission, $form_data, $form, $paymentMode);
    }

    public function handleRedirect($transaction, $submission, $form_data, $form, $methodSettings)
    {
        $globalSettings = SslcommerzSettings::getApiKeys();

        if (!isset($globalSettings['api_key']) || !isset($globalSettings['api_secret'])) {
            return;
        }

        $transaction['hash'] = $submission->submission_hash;
        $paymentIntent = $this->makePaymentData($transaction, $submission, $form_data, $form);

        if (Arr::get($paymentIntent, 'status') === 'FAILED') {
            wp_send_json_error(array(
                'message' => __(Arr::get($paymentIntent, 'failedreason'), 'wp-payment-form-pro')
            ), 423);
        }

        if (is_wp_error($paymentIntent)) {
            do_action('wppayform_log_data', [
                'form_id' => $submission->form_id,
                'submission_id' => $submission->id,
                'type' => 'activity',
                'created_by' => 'Paymattic BOT',
                'title' => 'Sslcommerz Payment Webhook Error',
                'content' => $paymentIntent->get_error_message()
            ]);

            wp_send_json_error(array(
                'message' => __($paymentIntent->get_error_message(), 'wp-payment-form-pro')
            ), 423);
        }

        do_action('wppayform_log_data', [
            'form_id' => $form->ID,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'title' => 'Sslcommerz Payment Redirect',
            'content' => 'User redirect to Sslcommerz for completing the payment'
        ]);

        $redirectUrl = wp_sanitize_redirect(Arr::get($paymentIntent, 'redirectGatewayURL'));

        wp_send_json_success([
            'message' => __('You are redirecting to sslcommerz.com to complete the purchase. Please wait while you are redirecting....', 'wp-payment-form-pro'),
            'call_next_method' => 'normalRedirect',
            'redirect_url' => $redirectUrl
        ], 200);
    }

    public function makePaymentData($transaction, $submission, $form_data, $form)
    {
        $Api = new API();
        $submissionModel = new Submission();
        $entries = $submissionModel->getParsedSubmission($submission);

        $success_url = $this->redirectUrl($form->ID, $submission);

        $webhook_url = add_query_arg([
            'wpf_payment_api_notify' => '1',
            'payment_method' => 'sslcommerz'
        ], site_url('index.php'));

        $metadata = [];
        foreach ($entries as $label => $entry) {
            if ($entry['type'] == 'customer_name') {
                unset($metadata['customer_name']);
            }
            if ($entry['type'] == 'customer_email') {
                unset($metadata['customer_email']);
            }
            $value = $entry['value'];
            if (is_string($value) && $value) {
                $metadata[$entry['type']] = $value;
            }
            if ($entry['type'] == 'address_input') {
                $metadata['customer_address'] = $label;
            }
        }

        $rawData = $submissionModel->getSubmission(intval($submission->id))->form_data_raw;

        $addressLabel = Arr::get($metadata, 'customer_address', false);
        $addresses = [];
        if ($addressLabel) {
            $addresses = Arr::get($rawData, $addressLabel);
        }

        $args = [
            'total_amount' => floatval($transaction->payment_total / 100),
            'currency' =>  $transaction->currency,
            'tran_id' => $transaction->id,
            'product_category' => 'payform',
            'product_profile' => 'general',
            'product_name' => $form->post_title,
            'cus_name' => Arr::get($metadata, 'customer_name', "Not collected"),
            'cus_email' => Arr::get($metadata, 'customer_email', "Not collected"),
            'success_url' => wp_sanitize_redirect($success_url),
            'fail_url' => wp_sanitize_redirect(Arr::get($form_data, '__wpf_current_url')),
            'cancel_url' => wp_sanitize_redirect(Arr::get($form_data, '__wpf_current_url')),

            'cus_add1' => Arr::get($addresses, 'address_line_1', 'Not collected'),
            'cus_city' => Arr::get($addresses, 'city', 'Not collected'),
            'cus_country' => Arr::get($addresses, 'country', 'Not collected'),
            'cus_phone' => "Not collected",
            'shipping_method' => 'NO',
            'ipn_url' => $webhook_url
        ];

        $formId = $transaction->form_id;
        $keys = SslcommerzSettings::getApiKeys($formId);
        $keys['api_path'] = $keys['api_path'] . '/gwprocess/v4/api.php';
        return $Api->makeApiCall($keys, $args, 'POST');
    }

    public function redirectUrl($formId, $submission)
    {
        $confirmation = ConfirmationHelper::getFormConfirmation($formId, $submission);
        if (empty($confirmation['customUrl'])) {
            $confirmation['customUrl'] = Arr::get($submission, 'form_data_raw.__wpf_current_url');
        }
        return $confirmation['customUrl'];
    }

    protected function getPaymentMode($formId = false)
    {
        $isLive = SslcommerzSettings::isLive($formId);
        if ($isLive) {
            return 'live';
        }
        return 'test';
    }

    public function addTransactionUrl($transactions, $submissionId)
    {
        $transPath = 'https://sandbox.sslcommerz.com/manage/';
        if (SslcommerzSettings::isLive()) {
            $transPath = 'https://securepay.sslcommerz.com/manage/';
        };

        foreach ($transactions as $transaction) {
            if ($transaction->charge_id) {
                $transaction->transaction_url =  $transPath;
                $payNote = safeUnserialize($transaction->payment_note);
                if (!empty($payNote)) {
                    $currency = Arr::get($payNote, 'currency', '-');
                    $converted = [
                        'Conversion Type' => 'From ' . Arr::get($payNote, 'currency_type', '-') . ' to ' . $currency,
                        'Rate' => Arr::get($payNote, 'currency_rate') . ' ' . $currency,
                        'Calculated Total' => Arr::get($payNote, 'amount') . ' ' . $currency,
                        'Store amount' => '<strong>' . Arr::get($payNote, 'store_amount') . ' ' . $currency . '</strong>',
                        'Card Issuer' => Arr::get($payNote, 'card_issuer')
                    ];
                    $transaction->converted_currency = $converted;
                }
            }
        }
        return $transactions;
    }


    public function getLastTransaction($submissionId)
    {
        $transactionModel = new Transaction();
        $transaction = $transactionModel->where('submission_id', $submissionId)
            ->first();
        return $transaction;
    }

    public function validateSubscription($paymentItems, $formattedElements, $form_data, $subscriptionItems)
    {
        wp_send_json_error(array(
            'message' => __('Sslcommerz doesn\'t support subscriptions right now', 'wp-payment-form-pro'),
            'payment_error' => true
        ), 423);
    }
}
