<?php

namespace WPPayFormPro\GateWays\Square;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Error;
use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Models\Transaction;
use WPPayForm\App\Models\Submission;
use WPPayForm\App\Models\Subscription;
use WPPayForm\App\Models\SubscriptionTransaction;
use WPPayForm\App\Models\SubmissionActivity;
use WPPayForm\App\Models\OrderItem;
use WPPayForm\App\Services\ConfirmationHelper;
use WPPayFormPro\GateWays\Square\API;

class SquareProcessor
{
    public $method = 'square';

    protected $form;

    public function init()
    {
        new SquareElement();
        (new SquareSettings())->init();

        add_filter('wppayform/choose_payment_method_for_submission', array($this, 'choosePaymentMethod'), 10, 4);
        add_action('wppayform/form_submission_make_payment_' . $this->method, array($this, 'makeFormPayment'), 10, 7);
        add_action('wppayform_payment_frameless_' . $this->method, array($this, 'handleSessionRedirectBack'));
        add_filter('wppayform/entry_transactions_' . $this->method, array($this, 'addTransactionUrl'), 10, 2);
        add_filter('wppayform/submitted_payment_items_' . $this->method, array($this, 'validateSubscription'), 10, 4);

        // fetch all subscription entry wise
        add_action('wppayform/subscription_settings_sync_square', array($this, 'makeSubscriptionSync'), 10, 2);

        // cancel subscription
        add_action('wppayform/subscription_settings_cancel_square', array($this, 'cancelSubscription'), 10, 3);

        add_action('wppayform/process_refund_square', array($this, 'processRefundTransaction'), 10, 1);
    }

    public function choosePaymentMethod($paymentMethod, $elements, $formId, $form_data)
    {
        if ($paymentMethod) {
            // Already someone choose that it's their payment method
            return $paymentMethod;
        }
        // Now We have to analyze the elements and return our payment method
        foreach ($elements as $element) {
            if ((isset($element['type']) && $element['type'] == 'square_gateway_element')) {
                return 'square';
            }
        }
        return $paymentMethod;
    }

    public function makeFormPayment($transactionId, $submissionId, $form_data, $form, $hasSubscriptions, $totalPayable = 0, $paymentItems = array())
    {
        $paymentMode = $this->getPaymentMode();
        $transactionModel = new Transaction();
        if ($transactionId) {
            $transactionModel->updateTransaction($transactionId, array(
                'payment_mode' => $paymentMode
            ));
        }
        $transaction = $transactionModel->getTransaction($transactionId);

        $submission = (new Submission())->getSubmission($submissionId);
        $this->handleRedirect($transaction, $submission, $form, $paymentMode, $hasSubscriptions, $paymentItems);
    }

    public function redirectUrl($formId, $submission)
    {
        $confirmation = ConfirmationHelper::getFormConfirmation($formId, $submission);
        return $confirmation['customUrl'];
    }


    public function handleRedirect($transaction, $submission, $form, $methodSettings, $hasSubscriptions, $paymentItems = array())
    {
        $keys = SquareSettings::getApiKeys($form->ID);

        $listener_url = wp_sanitize_redirect($this->redirectUrl($form->ID, $submission));

        // checking for subscription payment
        if ($hasSubscriptions) {
            $this->handleSubscription($submission, $form, $keys, $listener_url, $paymentItems);
        } else {
            $orderItemsModel = new OrderItem();
            $items = $orderItemsModel->getOrderItems($submission->id);
            //temporary solution, will change in future
            $itemNames = '';
            if (count($items) == 1) {
                $itemNames .= $items[0]['item_name'];
            } else {
                foreach ($items as $key => $item) {
                    if ( count($items)-1 == $key) {
                        $itemNames .= $item->item_name;
                    } else {
                        $itemNames .= $item->item_name . ', ';
                    }
                }
            }

            $paymentArgs = array(
                "idempotency_key" => $submission->submission_hash,
                "order" => array(
                    "location_id" => Arr::get($keys, "location_id"),
                    "line_items" => [
                    [
                        "quantity" => '1',
                        "item_type" => "ITEM",
                        "metadata" => [
                            'form_id'       => 'Form Id ' . strval($form->ID) ,
                            'submission_id' => 'Submission Id ' . strval($submission->id)
                        ],
                        "name" => $itemNames,
                        "base_price_money" => [
                        "amount" => intval($transaction->payment_total),
                        "currency" => $transaction->currency
                        ]
                    ]
                    ],
                ),
                "checkout_options" => [
                    "redirect_url" => $listener_url
                ],
            );

            $paymentIntent = (new API())->makeApiCall('online-checkout/payment-links', $paymentArgs, $form->ID, 'POST');
            
            $transactionModel = new Transaction();
            $transactionModel->updateTransaction($transaction->id, array(
                'status' => 'intended',
                'charge_id' => Arr::get($paymentIntent, 'payment_link.order_id'),
                'payment_mode' => $this->getPaymentMode($submission->form_id)
            ));

            $paymentArgs = apply_filters('wppayform_square_payment_args', $paymentArgs, $submission, $transaction, $form);

            if (isset($paymentIntent['errors'])) {
                wp_send_json_error(array(
                    'message' => __(Arr::get($paymentIntent['errors'][0], 'detail'), 'wp-payment-form-pro')
                ), 423);
                do_action("wppayform/form_payment_failed", $submission, $submission->form_id, $transaction, 'square');
            }

            if (is_wp_error($paymentIntent)) {
                do_action('wppayform_log_data', [
                    'form_id' => $submission->form_id,
                    'submission_id' => $submission->id,
                    'type' => 'activity',
                    'created_by' => 'Paymattic BOT',
                    'title' => 'Square Payment Webhook Error',
                    'content' => $paymentIntent->get_error_message()
                ]);
                do_action("wppayform/form_payment_failed", $submission,  $submission->form_id, $transaction, 'square');
                wp_send_json_error(array(
                    'message' => __($paymentIntent->get_error_message(), 'wp-payment-form-pro')
                ), 423);
            }

            do_action('wppayform_log_data', [
                'form_id' => $form->ID,
                'submission_id' => $submission->id,
                'type' => 'activity',
                'created_by' => 'Paymattic BOT',
                'title' => 'Square Payment Redirect',
                'content' => 'User redirect to Square for completing the payment'
            ]);

            $redirectUrl = wp_sanitize_redirect(Arr::get($paymentIntent, 'payment_link.url'));
            
            wp_send_json_success([
                'message' => __('You are redirecting to squareup.com to complete the purchase. Please wait while you are redirecting....', 'wp-payment-form-pro'),
                'call_next_method' => 'normalRedirect',
                'redirect_url' => $redirectUrl
            ], 200);
        }
    }

    public function handleSubscription($submission, $form, $keys, $listener_url, $paymentItems = array())
    {
        $subscriptionModel = new Subscription();
        $subscriptions = $subscriptionModel->getSubscriptions($submission->id);

        $validSubscriptions = [];
        foreach ($subscriptions as $subscriptionItem) {
            if ($subscriptionItem->recurring_amount) {
                $validSubscriptions[] = $subscriptionItem;
            }
        }

        // We just need the first subscription
        $subscription = $validSubscriptions[0];

        if (!$subscription) {
            wp_send_json_error(array(
                'message' => __('No valid subscription found', 'wp-payment-form-pro')
            ), 423);
        }

        // create the subscription plan
        $response = $this->createPlan($submission, $form, $subscription);

        if (!isset($response['subscriptionPlan']) || !isset($response['subscriptionPlanVariation'])) {
            wp_send_json_error(array(
                'message' => __('Failed to create subscription plan', 'wp-payment-form-pro')
            ), 423);
        }

        $plan = Arr::get($response, 'subscriptionPlan');
        $planVariation = Arr::get($response, 'subscriptionPlanVariation');
        $signUpFee = Arr::get($response, 'signUpFee');

        if (!$plan || !$planVariation) {
            wp_send_json_error(array(
                'message' => __('Failed to create subscription plan', 'wp-payment-form-pro')
            ), 423);
        }

        $planId = Arr::get($response, 'subscriptionPlan.catalog_object.id');
        $planVariationId = Arr::get($response, 'subscriptionPlanVariation.catalog_object.id');
        
        $subscriptionModel->updateSubscription($subscriptionItem->id, [
            'status' => 'intended',
            'vendor_plan_id' => $planId,
            'vendor_response' => maybe_serialize($plan),
        ]);

        //getting the first phase amount of subscription plan
        $amount = $subscription->recurring_amount;
        if ($subscription->trial_days) {
            $amount = 0;
        }

        $initial_amount_tax = 0;
        foreach ($paymentItems as $key => $paymentItem) {
            if ($key == 'tax_payment_input' && $paymentItem['signup_fee_tax'] == 'yes') {
                $initial_amount_tax = Arr::get($paymentItem, 'signup_fee', 0);
                break;
            }
        }

        if ($signUpFee > 0) {
            $amount = $signUpFee + intval($subscription->recurring_amount) + $initial_amount_tax;
        }

        $paymentArgs = array(
            "idempotency_key" => $submission->submission_hash,
            "quick_pay" => array(
                "name" => $subscription->item_name,
                "price_money" => array(
                    "amount" => intval($amount),
                    "currency" => $submission->currency,
                ),
                "location_id" => Arr::get($keys, "location_id")
                ),
            "checkout_options" => array(
                "subscription_plan_id" => $planVariationId,
                "redirect_url" => $listener_url
            )
        );

        do_action('wppayform_log_data', [
            'form_id' => $form->ID,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'title' => 'Square Payment Redirect',
            'content' => 'User redirect to Square for compile
            ting the subscription'
        ]);

        $paymentIntent = (new API())->makeApiCall('online-checkout/payment-links', $paymentArgs, $form->ID, 'POST');

        if ($subscription->trial_days) {
            $paymentTotal = 0;
        } else {
            $paymentTotal = static::recurringAmountAfterDiscount($subscription, $submission);
        }

        $currentUserId = get_current_user_id();
        $paymentData = [
            'form_id' => $submission->form_id,
            'submission_id' => $submission->id,
            'user_id' => $currentUserId,
            'subscription_id' => $subscription->id,
            'transaction_type' => 'subscription',
            'payment_method' => 'square',
            'charge_id' => Arr::get($paymentIntent, 'payment_link.order_id'),
            'payment_total' => $paymentTotal,
            'status' => 'pending',
            'currency' => $submission->currency,
            'payment_mode' => $submission->payment_mode ? $submission->payment_mode : $this->getPaymentMode($submission->form_id),
            'payment_note' => maybe_serialize($planVariation)
        ];

        $subscriptionTransactionModel = new SubscriptionTransaction();
        $subscriptionTransactionModel->maybeInsertCharge($paymentData);
        
        $redirectUrl = Arr::get($paymentIntent, 'payment_link.url');
        if (!$redirectUrl) { 
            wp_send_json_error(['message' => 'No valid redirect url found!'], 423);
        }

        wp_send_json_success([
            'message' => __('You are redirecting to squareup.com to complete the purchase. Please wait while you are redirecting....', 'wp-payment-form-pro'),
            'call_next_method' => 'normalRedirect',
            'redirect_url' => $redirectUrl
        ], 200);
    }

    //create new subscription plan
    public function createPlan($submission, $form, $subscription)
    {
        $billingInterval = $subscription->billing_interval;
        $billingTime = $subscription->bill_times;

        // convert the billing interval to square format
        if ($billingInterval == 'month') {
            $billingInterval = 'MONTHLY';
        } else if ($billingInterval == 'week') {
            $billingInterval = 'WEEKLY';
        } else if ($billingInterval == 'fortnight') {
            $billingInterval = 'EVERY_TWO_WEEKS';
        } else if ($billingInterval == 'quarter') {
            $billingInterval = 'QUARTERLY';
        } else if ($billingInterval == 'half_year') {
            $billingInterval = 'EVERY_SIX_MONTHS';
        } else if ($billingInterval == 'year') {
            $billingInterval = 'ANNUAL';
        } else if ($billingInterval == 'daily') {
            $billingInterval = 'DAILY';
        } 

        // check if the subscription has trial days or sign up fee
        $trialDays = intval(Arr::get($subscription, 'trial_days'));
        $signUpFee = intval(Arr::get($subscription, 'initial_amount'));

        /* create subscription plan start */
        // create the subscription plan according to the square catalog object for subscription plan
        $subscriptionPlanData = array(
            'idempotency_key' => $submission->submission_hash . '_plan',
            'object' => array(
                'type' => 'SUBSCRIPTION_PLAN',
                'id' => '#' . $submission->submission_hash,
                'subscription_plan_data' => array(
                    'name' => $subscription->item_name,
                    'all_items' => false
                )
            )
        );
        $subscriptionPlan = (new API())->makeApiCall('catalog/object', $subscriptionPlanData, $form->ID, 'POST');

        if (isset($subscriptionPlan['errors'])) {
            wp_send_json_error(array(
                'message' => $subscriptionPlan['errors'][0]['detail']
            ), 423);
        }
        if (is_wp_error($subscriptionPlan)) {
            wp_send_json_error(array(
                'message' => $subscriptionPlan->get_error_message()
            ), 423);
        }
        /* create subscription plan end */

        /* add plan variations to the plan start */
        // step-1.1 First create the phases for the plan with trial days and sign up fee, add as the initial phase
        $phases = [];
        $hasInitialPhase = false;
        if ($trialDays > 0) {
            $hasInitialPhase = true;
            $phases = [
                array(
                    'cadence' => 'DAILY',
                    'periods' => $trialDays,
                    'ordinal' => 0,
                    'pricing' => array(
                        'type' => 'STATIC',
                        'price' => array(
                            'amount' => 0,
                            'currency' => $submission->currency
                        )
                    )
                )
            ];
        }

        // Signup Fee is a one-time payment
        if ($signUpFee > 0) {
            $hasInitialPhase = true;
            $phases = [
                array(
                    'cadence' => $billingInterval,
                    'periods' => 1,
                    'ordinal' => 0,
                    'pricing' => array(
                        'type' => 'STATIC',
                        'price' => array(
                            'amount' => intval($signUpFee) + intval($subscription->recurring_amount),
                            'currency' => $submission->currency
                        )
                    )
                )
            ];
        }

        // step-1.2  create the main/recurring phase for the plan with the recurring amount, we basically have two phases, one for (the trial days and sign up fee) and the other for the recurring amount
        $ordinal = 0;
        if ($hasInitialPhase) {
            $ordinal = 1;
        }

        if ($billingTime == 0) {
            $newPhase = array(
                'cadence' => $billingInterval,
                'ordinal' => $ordinal,
                'pricing' => array(
                    'type' => 'STATIC',
                    'price' => array(
                        'amount' => intval($subscription->recurring_amount),
                        'currency' => $submission->currency
                    )
                )
            );
        } else {
            $newPhase = array(
                'cadence' => $billingInterval,
                'ordinal' => $ordinal,
                'periods' => intval($billingTime),
                'pricing' => array(
                    'type' => 'STATIC',
                    'price' => array(
                        'amount' => intval($subscription->recurring_amount),
                        'currency' => $submission->currency
                    )
                )
            );
        }

        $phases[] = $newPhase; // add the new phase to the phases array

        // create the subscription plan variation according to the square catalog object for subscription plan variation
        $subscriptionPlanVariationData = array(
            'idempotency_key' => $submission->submission_hash . '_plan_variation',
            'object' => array(
                'type' => 'SUBSCRIPTION_PLAN_VARIATION',
                'id' => '#' . $submission->submission_hash,
                'subscription_plan_variation_data' => array(
                    'name' => $subscription->item_name,
                    'phases' => $phases,
                    'subscription_plan_id' => Arr::get($subscriptionPlan, 'catalog_object.id')
                )
            )
        );

        $subscriptionPlanVariation = (new API())->makeApiCall('catalog/object', $subscriptionPlanVariationData, $form->ID, 'POST');

        // handle errors on creating subscription plan
        if (isset($subscriptionPlanVariation['errors'])) {
            wp_send_json_error(array(
                'message' => $subscriptionPlanVariation['errors'][0]['detail']
            ), 423);
        }
        if (is_wp_error($subscriptionPlanVariation)) {
            wp_send_json_error(array(
                'message' => $subscriptionPlanVariation->get_error_message()
            ), 423);
        }
        /* add plan variations to the plan end */

        return array(
            'subscriptionPlan' => $subscriptionPlan,
            'subscriptionPlanVariation' => $subscriptionPlanVariation,
            'signUpFee' => $signUpFee,
        );
    }

    protected function getPaymentMode($formId = false)
    {
        $isLive = SquareSettings::isLive($formId);
        if ($isLive) {
            return 'live';
        }
        return 'test';
    }

    public function handleSessionRedirectBack($data)
    {
        $submissionId = intval($data['wppayform_payment']);
        $payId = Arr::get($data, 'transactionId');

        if (!$submissionId){
            return;
        }

        if (!$payId) {
            $transactionModel = new Transaction();
            $transaction = $transactionModel->where('submission_id', $submissionId)->first();
            $payId = Arr::get($transaction, 'charge_id');
        }

        if (!$payId) {
            return;
        }

        $submission = (new Submission())->getSubmission($submissionId);
        $transaction = $this->getLastTransaction($submissionId);

        $payment = (new API())->makeApiCall('orders/'.$payId, [], $submission->form_id);
        $status = Arr::get($payment, 'order.state');

        // This hook will be usefull for the developers to do something after the payment is processed
        do_action('wppayform/form_payment_processed', $submission->form_id, $submission, $data, $status);

        if(!$payment || !$status) {
            do_action('wppayform/form_payment_failed', $submission, $submission->form_id, $transaction, 'square');
            return;
        }

        $redirectUrl = $this->redirectUrl($submission->form_id, $submission);

        if (is_wp_error($payment)) {
            do_action('wppayform_log_data', [
                'form_id' => $submission->form_id,
                'submission_id' => $submission->id,
                'type' => 'info',
                'created_by' => 'PayForm Bot',
                'content' => $payment->get_error_message()
            ]);
        }

        if ($status == 'COMPLETED' || $status == 'OPEN') {
            wp_redirect($redirectUrl);
        }

        $transaction = $this->getLastTransaction($submissionId);

        $this->handlePaid($transaction, $payment);
    }

    public function addTransactionUrl($transactions, $submissionId)
    {
        foreach ($transactions as $transaction) {
            $path = $transaction->payment_mode === 'test' ? 'https://squareupsandbox.com/' : 'https://squareup.com/';
            if ($transaction->charge_id) {
                $transaction->transaction_url =  $path . 'dashboard/sales/transactions/' . $transaction->charge_id;
            }
        }
        return $transactions;
    }

    public function getLastTransaction($submissionId)
    {
        $transactionModel = new Transaction();
        $transaction = $transactionModel->where('submission_id', $submissionId)
            ->first();
        return $transaction;
    }

    public function handlePaid($transaction, $vendorTransaction)
    {
        if (!$transaction || $transaction->payment_method != $this->method || $transaction->status === 'paid') {
            return;
        }
        
        do_action('wppayform/form_submission_activity_start', $transaction->form_id);

        $status = 'paid';

        $updateData = [
            'payment_note'     => maybe_serialize($vendorTransaction),
            'charge_id'        => $transaction->charge_id,
        ];

        $this->markAsPaid($status, $updateData, $transaction);
    }

    public function markAsPaid($status, $updateData, $transaction)
    {
        $submissionModel = new Submission();
        $submissionData = array(
            'payment_status' => $status,
            'updated_at' => current_time('Y-m-d H:i:s')
        );

        $submissionModel->where('id', $transaction->submission_id)->update($submissionData);
        $submission = $submissionModel->getSubmission($transaction->submission_id);

        $transactionModel = new Transaction();
        $updateDate = array(
            'charge_id' => $updateData['charge_id'],
            'payment_note' => $updateData['payment_note'],
            'status' => $status,
            'updated_at' => current_time('Y-m-d H:i:s')
        );
        
        $formId = Arr::get($transaction, 'form_id');
        $submissionId = Arr::get($transaction, 'submission_id');

        $transactionModel->where('id', $transaction->id)->update($updateDate);
        $transaction = $transactionModel->getTransaction($transaction->id);

        do_action('wppayform_log_data', [
            'form_id' => $formId,
            'submission_id' => $submissionId,
            'type' => 'info',
            'created_by' => 'PayForm Bot',
            'content' => sprintf(__('Transaction Marked as paid and Square Transaction ID: %s', 'wp-payment-form-pro'), $updateDate['charge_id'])
        ]);

        do_action('wppayform/form_payment_success_square', $submission, $transaction, $formId, $updateDate);
        do_action('wppayform/form_payment_success', $submission, $transaction, $formId, $updateDate);
    }

    public function validateSubscription($paymentItems, $formattedElements, $form_data, $subscriptionItems)
    {
        $singleItemTotal = 0;
        foreach ($paymentItems as $paymentItem) {
            if (isset($paymentItem['recurring_tax']) && $paymentItem['recurring_tax'] == 'yes') {
                continue;
            }
            if ($paymentItem['line_total']) {
                $singleItemTotal += $paymentItem['line_total'];
            }
            if ($paymentItem['line_total']) {
                $singleItemTotal += $paymentItem['line_total'];
            }
        }

        $validSubscriptions = [];
        foreach ($subscriptionItems as $subscriptionItem) {
            if ($subscriptionItem['recurring_amount']) {
                $validSubscriptions[] = $subscriptionItem;
            }
        }

        if ($singleItemTotal && count($validSubscriptions)) {
            wp_send_json_error(array(
                'message' => __('Square does not support subscriptions payment and Single Amount Payment at one request', 'wp-payment-form-pro'),
                'payment_error' => true
            ), 423);
        }

        if (count($validSubscriptions) > 2) {
            wp_send_json_error(array(
                'message' => __('Square does not support multiple subscriptions at one request', 'wp-payment-form-pro'),
                'payment_error' => true
            ), 423);
        }

        return $paymentItems;
    }

    public function makeSubscriptionSync($formId, $submissionId)
    {
        if (!$submissionId) {
            return;
        }

        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($submissionId);

        $subscriptionModel = new Subscription();
        $subscriptions = $subscriptionModel->getSubscriptions($submissionId);

        if (!isset($subscriptions[0])) {
            return 'No subscription Id found!';
        };

       
        /* 
         * For future query
         * 
         * Arr::get($subscriptions, '0.vendor_subscriptipn_id');
         *  don't need to call for retrieving all the invoices related to this subscriptionId from sqaure,
         *  because they already are in transaction table for the webhooks(invoice.created and invoice.payment_made) subscription
        */

         SubmissionActivity::createActivity(array(
            'form_id' => $submission->form_id,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'content' => __('Square recurring payments synced from upstream', 'wp-payment-form')
        ));
        
        wp_send_json_success(array(
            'message' => 'Successfully synced!'
        ), 200);

    }

    public function cancelSubscription($formId, $submission, $subscription)
    {

        $subscriptionId = Arr::get($subscription, 'vendor_subscriptipn_id');
        
        if(!$subscriptionId){
            wp_send_json_error(array(
                'message' => 'No subscription ID found!',
            ), 423);
        }

        $response = (new API())->makeApiCall('subscriptions/'. $subscriptionId . '/cancel', [], $formId, 'POST');

        if(is_wp_error(($response))){
            wp_send_json_error( array('message' => $response->get_error_message()), 423 );
        }

        // Send email after cancel subscription
        do_action('wppayform/send_email_after_subscription_cancel_', $formId, $subscription, $submission);
        do_action('wppayform/form_submission_activity_start', $formId);
        do_action( 'wppayform/send_email_after_subscription_cancel_for_user', $submission );
        do_action( 'wppayform/send_email_after_subscription_cancel_for_admin', $submission );
        // add logs
        SubmissionActivity::createActivity(array(
            'form_id' => $submission->form_id,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'content' => __('Square recurring subscription Cancelled', 'wp-payment-form')
        ));

        wp_send_json_success(array(
            'message' => $response
        ), 200);
    }

    public static function recurringAmountAfterDiscount($subscription, $submission) 
    {
        if (!$subscription || !$submission) {
            return 0;
        }

        $recurringAmount = intval($subscription->recurring_amount);
        $orderItemModel = new OrderItem();
        $discountItems = $orderItemModel->getDiscountItems($submission->id);
        if ($discountItems) {
            $discountTotal = 0;
            foreach ($discountItems as $discountItem) {
                $discountTotal += intval($discountItem->line_total);
            }
            
            $recurringAmount -= $discountTotal;
        }
        return $recurringAmount;
    }

    public function processRefundTransaction ($refundData)
    {
        $transaction = Arr::get($refundData, 'transaction', []);
        $amount = intval(Arr::get($refundData, 'amount', 0) * 100);
        $reason = Arr::get($refundData, 'reason', '');
        $chargeId = Arr::get($transaction, 'charge_id');

        if (!$chargeId) {
            return new \WP_Error('missing_charge', __('Missing Square charge ID for refund.', 'wp-payment-form'));
        }

        $refundRequest = [
            'amount_money' => [
                'amount' => $amount,
                'currency' => 'USD'
            ],
            'reason' => $reason
        ];

        $response = (new API())->makeApiCall('transactions/'. $chargeId . '/refund', $refundRequest, $transaction->form_id, 'POST');

        if (is_wp_error($response) || isset($response->error)) {
            $message = is_wp_error($response) 
                ? $response->get_error_message() 
                : $response->error->message;

            return new \WP_Error('square_refund_failed', $message);
        }
        
        return $response;
    }

    public function handleFailed($transaction, $order)
    {
        if (!$transaction || $transaction->payment_method != $this->method || $transaction->status === 'failed') {
            return;
        }
        
        do_action('wppayform/form_submission_activity_start', $transaction->form_id);

        $status = 'failed';

        $updateData = [
            'payment_note'     => maybe_serialize($order),
            'charge_id'        => $transaction->charge_id,
        ];

        $this->markAsPaid($status, $updateData, $transaction);
    }
}
