<?php

namespace WPPayFormPro\GateWays\Razorpay;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use WPPayForm\Framework\Support\Arr;
use WPPayFormPro\GateWays\Razorpay\API\IPN;
use WPPayForm\App\Models\Transaction;
use WPPayForm\App\Models\Submission;
use WPPayForm\App\Models\Form;
use WPPayForm\App\Services\PlaceholderParser;
use WPPayForm\App\Services\ConfirmationHelper;

class RazorpayProcessor
{
    public $method = 'razorpay';

    protected $form;

    public function init()
    {
        new RazorpayElement();
        (new RazorpaySettings())->init();

        add_filter('wppayform/choose_payment_method_for_submission', array($this, 'choosePaymentMethod'), 10, 4);
        add_action('wppayform/form_submission_make_payment_' . $this->method, array($this, 'makeFormPayment'), 10, 6);
        add_action('wppayform_payment_frameless_' . $this->method, array($this, 'handleSessionRedirectBack'));
        add_action('wppayform_load_checkout_js_' . $this->method, array($this, 'addCheckoutJs'), 10, 3);

        add_action('wp_ajax_wppayform_razorpay_confirm_payment', array($this, 'confirmModalPayment'));
        add_action('wp_ajax_nopriv_wppayform_razorpay_confirm_payment', array($this, 'confirmModalPayment'));

        add_filter('wppayform/entry_transactions_' . $this->method, array($this, 'addTransactionUrl'), 10, 2);
        add_filter('wppayform/submitted_payment_items_' . $this->method, array($this, 'validateSubscription'), 10, 4);
        add_action('wppayform/process_refund_razorpay', array($this, 'processRefundTransaction'), 10, 1);

        add_action('wppayform_ipn_razorpay_action_failed', array($this, 'handlePaymentFailed'), 10, 3);
    }

    public function choosePaymentMethod($paymentMethod, $elements, $formId, $form_data)
    {
        if ($paymentMethod) {
            // Already someone choose that it's their payment method
            return $paymentMethod;
        }
        // Now We have to analyze the elements and return our payment method
        foreach ($elements as $element) {
            if ((isset($element['type']) && $element['type'] == 'razorpay_gateway_element')) {
                return 'razorpay';
            }
        }
        return $paymentMethod;
    }

    public function makeFormPayment($transactionId, $submissionId, $form_data, $form, $hasSubscriptions)
    {
        $paymentMode = $this->getPaymentMode();
        $transactionModel = new Transaction();
        if ($transactionId) {
            $transactionModel->updateTransaction($transactionId, array(
                'payment_mode' => $paymentMode
            ));
        }
        $transaction = $transactionModel->getTransaction($transactionId);

        $submission = (new Submission())->getSubmission($submissionId);

        $this->maybeShowModal($transaction, $submission, $form, $paymentMode);
        $this->handleRedirect($transaction, $submission, $form, $paymentMode);
    }

    public function getSuccessURL($form, $submission)
    {
       // Check If the form settings have success URL
        $confirmation = Form::getConfirmationSettings($form->ID);
        $confirmation = ConfirmationHelper::parseConfirmation($confirmation, $submission);
        if (
            ($confirmation['redirectTo'] == 'customUrl' && $confirmation['customUrl']) ||
            ($confirmation['redirectTo'] == 'customPage' && $confirmation['customPage']) ||
            ($confirmation['redirectTo'] == 'customPost' && $confirmation['customPage'])
        ) {
            if ($confirmation['redirectTo'] == 'customUrl') {
                $url = $confirmation['customUrl'];
            } else {
                $url = get_permalink(intval($confirmation['customPage']));
            }
            $url = add_query_arg(array(
                'payment_method' => 'razorpay'
            ), $url);
            $url = PlaceholderParser::parse($url, $submission);
            return wp_sanitize_redirect($url);
        }
        // now we have to check for global Success Page
        $globalSettings = get_option('wppayform_confirmation_pages');
        if (isset($globalSettings['confirmation']) && $globalSettings['confirmation']) {
            $url = add_query_arg(array(
                'wpf_submission' => $submission->submission_hash,
                'payment_method' => 'razorpay'
            ), get_permalink(intval($globalSettings['confirmation'])));
            return wp_sanitize_redirect($url);
        }
        // In case we don't have global settings
        $url = add_query_arg(array(
            'wpf_submission' => $submission->submission_hash,
            'payment_method' => 'razorpay'
        ), home_url());
        return wp_sanitize_redirect($url);
    }

    public function maybeShowModal($transaction, $submission, $form, $paymentMode)
    {
        $settings = (new RazorPaySettings())->getSettings();
        if ($settings['checkout_type'] != 'modal') {
            return;
        }

        // Create an order First
        $orderArgs = [
            'amount'   => intval($transaction->payment_total),
            'currency' => strtoupper($transaction->currency),
            'receipt'  => $submission->submission_hash,
            'notes'    => [
                'form_id'       => $form->ID,
                'submission_id' => $submission->id
            ]
        ];

        $order = (new API())->makeApiCall('orders', $orderArgs, $form->ID, 'POST');

        if (is_wp_error($order)) {
            $message = $order->get_error_message();
            do_action('wppayform_log_data', [
                'form_id' => $submission->form_id,
                'submission_id' => $submission->id,
                'type' => 'activity',
                'created_by' => 'Paymattic BOT',
                'title' => 'Razorpay Payment Webhook Error',
                'content' => $order->get_error_message()
            ]);

            wp_send_json([
                'errors'      => 'RazorPay Error: ' . $message,
                'append_data' => [
                    // '__entry_intermediate_hash' => Helper::getSubmissionMeta($submission->id, '__entry_intermediate_hash')
                ]
            ], 423);
        }

        $transactionModel = new Transaction();
        $transactionModel->updateTransaction($transaction->id, array(
            'charge_id' => $order['id']
        ));

        $keys = RazorPaySettings::getApiKeys($form->ID);

        $modalData = [
            'amount'       => intval($transaction->payment_total),
            'currency'     => strtoupper($transaction->currency),
            'description'  => $form->title,
            'reference_id' => $submission->submission_hash,
            'order_id'     => $order['id'],
            // 'name'         => $paymentSettings['business_name'],
            'key'          => $keys['api_key'],
            'prefill'      => [
                'email' => Arr::get($submission, 'customer_email')
            ],
            'theme'        => [
                'color' => '#3399cc'
            ]
        ];

        do_action('wppayform_log_data', [
            'form_id' => $submission->form_id,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'title' => 'Razorpay Modal is initiated',
            'content' => 'RazorPay Modal is initiated to complete the payment'
        ]);

        $confirmation = ConfirmationHelper::getFormConfirmation($submission->form_id, $submission);
        # Tell the client to handle the action
        wp_send_json_success([
            'nextAction'       => 'razorpay',
            'actionName'       => 'initRazorPayModal',
            'submission_id'    => $submission->id,
            'modal_data'       => $modalData,
            'transaction_hash' => $submission->submission_hash,
            'message'          => __('Payment Modal is opening, Please complete the payment', 'wp-payment-form-pro'),
            'result'           => [
                'insert_id' => $submission->id
            ]
        ], 200);
    }

    public function handleRedirect($transaction, $submission, $form, $methodSettings)
    {
        $successUrl = $this->getSuccessURL($form, $submission);
        $globalSettings = RazorPaySettings::getSettings();
        $listener_url = wp_sanitize_redirect(add_query_arg(array(
            'wppayform_payment' => $submission->id,
            'payment_method' => $this->method,
            'submission_hash' => $submission->submission_hash,
        ), $successUrl));

        $paymentArgs = array(
            'amount'       => intval($transaction->payment_total),
            'currency'     => $transaction->currency,
            'description'  => $form->title,
            'reference_id' => $transaction->transaction_hash,
            'customer'     => [
                'email' => Arr::get($submission, 'customer_email')
            ],
            'callback_url'  => $listener_url,
            'notes'        => [
                'form_id'       => $form->ID,
                'submission_id' => $submission->id
            ],
            'callback_method' => 'get',
            'notify' => [
                'email' => in_array('email', $globalSettings['notification']),
                'sms' => in_array('sms', $globalSettings['notification']),
            ]
        );

        $paymentArgs = apply_filters('wppayform_razorpay_payment_args', $paymentArgs, $submission, $transaction, $form);

        $paymentIntent = (new API())->makeApiCall('payment_links', $paymentArgs, $form->ID, 'POST');

        if (is_wp_error($paymentIntent)) {
            do_action('wppayform_log_data', [
                'form_id' => $submission->form_id,
                'submission_id' => $submission->id,
                'type' => 'activity',
                'created_by' => 'Paymattic BOT',
                'title' => 'Razorpay Payment Webhook Error',
                'content' => $paymentIntent->get_error_message()
            ]);

            wp_send_json_error(array(
                'message' => $paymentIntent->get_error_message()
            ), 423);
        }

        do_action('wppayform_log_data', [
            'form_id' => $form->ID,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'title' => 'Razorpay Payment Redirect',
            'content' => 'User redirect to Razorpay for completing the payment'
        ]);

        $redirectUrl = wp_sanitize_redirect($paymentIntent['short_url']);

        wp_send_json_success([
            'message' => __('You are redirecting to razorpay.com to complete the purchase. Please wait while you are redirecting....', 'wp-payment-form-pro'),
            'call_next_method' => 'normalRedirect',
            'redirect_url' => $redirectUrl,
        ], 200);
    }

    public function getPaymentMode($formId = false)
    {
        $isLive = RazorpaySettings::isLive($formId);
        if ($isLive) {
            return 'live';
        }
        return 'test';
    }

    public function handleSessionRedirectBack($data)
    {
        $submissionId = intval($data['wppayform_payment']);
        $submission = (new Submission())->getSubmission($submissionId);
        $transaction = $this->getLastTransaction($submissionId);

        $payId = Arr::get($data, 'razorpay_payment_id');
        $payment = (new API())->makeApiCall('payments/'.$payId, [], $submission->form_id);
        $isSuccess = false;

        if (is_wp_error($payment)) {
            $returnData = [
                'insert_id' => $submission->id,
                'title'     => __('Failed to retrieve payment data'),
                'result'    => false,
                'error'     => $payment->get_error_message()
            ];
            do_action('wppayform/form_payment_failed', $submission,  $submission->form_id, $data, 'razorpay');
        } else {
            // This hook will be usefull for the developers to do something after the payment is processed
            do_action('wppayform/form_payment_processed', $submission->form_id, $submission, $data, $payment['status']);
            $isSuccess = $payment['status'] == 'captured';
            if ($isSuccess) {
                $returnData = $this->handlePaid($submission, $transaction, $payment);
            } else {
                $returnData = [
                    'insert_id' => $submission->id,
                    'title'     => __('Failed to retrieve payment data'),
                    'result'    => false,
                    'error'     => __('Looks like you have cancelled the payment. Please try again!', 'wp-payment-form-pro')
                ];
            }
        }

        $returnData['type'] = ($isSuccess) ? 'success' : 'failed';

        if (!isset($returnData['is_new'])) {
            $returnData['is_new'] = false;
        }
    }

    public function addTransactionUrl($transactions, $submissionId)
    {
        foreach ($transactions as $transaction) {
            if ($transaction->charge_id) {
                $transaction->transaction_url =  'https://dashboard.razorpay.com/app/payments/'.$transaction->charge_id;
            }
        }
        return $transactions;
    }

    public function getLastTransaction($submissionId)
    {
        $transactionModel = new Transaction();
        $transaction = $transactionModel->where('submission_id', $submissionId)
            ->first();
        return $transaction;
    }

    public function handlePaid($submission, $transaction, $vendorTransaction)
    {
        $transaction = $this->getLastTransaction($submission->id);

        if (!$transaction || $transaction->payment_method != $this->method) {
            return;
        }

        do_action('wppayform/form_submission_activity_start', $transaction->form_id);

        if ($transaction->payment_method != 'razorpay') {
            return; // this isn't a mollie standard IPN
        }

        $status = 'paid';

        $updateData = [
            'payment_note'     => maybe_serialize($vendorTransaction),
            'charge_id'        => sanitize_text_field($vendorTransaction['id']),
        ];

        // Let's make the payment as paid
        $this->markAsPaid('paid', $updateData, $transaction);
    }

    public function markAsPaid($status, $updateData, $transaction)
    {
        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($transaction->submission_id);

        $submissionData = array(
            'payment_status' => $status,
            'updated_at' => current_time('Y-m-d H:i:s')
        );

        $submissionModel->where('id', $transaction->submission_id)->update($submissionData);

        $transactionModel = new Transaction();
        $updateDate = array(
            'charge_id' => $updateData['charge_id'],
            'payment_note' => $updateData['payment_note'],
            'status' => $status,
            'updated_at' => current_time('Y-m-d H:i:s')
        );

        $transactionModel->where('id', $transaction->id)->update($updateDate);
        $transaction = $transactionModel->getTransaction($transaction->id);
        do_action('wppayform_log_data', [
            'form_id' => $transaction->form_id,
            'submission_id' => $transaction->submission_id,
            'type' => 'info',
            'created_by' => 'PayForm Bot',
            'content' => sprintf(__('Transaction Marked as paid and Razorpay Transaction ID: %s', 'wp-payment-form-pro'), $updateDate['charge_id'])
        ]);

        do_action('wppayform/form_payment_success_razorpay', $submission, $transaction, $transaction->form_id, $updateDate);
        do_action('wppayform/form_payment_success', $submission, $transaction, $transaction->form_id, $updateDate);
    }

    public function handleRefund($refundAmount, $submission, $vendorTransaction)
    {
        $transaction = $this->getLastTransaction($submission->id);
        $this->updateRefund($vendorTransaction['status'], $refundAmount, $transaction, $submission);
    }

    public function updateRefund($newStatus, $refundAmount, $transaction, $submission)
    {
        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($submission->id);
        if ($submission->payment_status == $newStatus) {
            return;
        }

        $submissionModel->updateSubmission($submission->id, array(
            'payment_status' => $newStatus
        ));

        Transaction::where('submission_id', $submission->id)->update(array(
            'status' => $newStatus,
            'updated_at' => current_time('mysql')
        ));

        do_action('wppayform/after_payment_status_change', $submission->id, $newStatus);

        $activityContent = 'Payment status changed from <b>' . $submission->payment_status . '</b> to <b>' . $newStatus . '</b>';
        $note = wp_kses_post('Status updated by Razorpay.');
        $activityContent .= '<br />Note: ' . $note;
        do_action('wppayform_log_data', [
            'form_id' => $submission->form_id,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Razorpay',
            'content' => $activityContent
        ]);
    }

    public function validateSubscription($paymentItems, $formattedElements, $form_data, $subscriptionItems)
    {
        wp_send_json_error(array(
            'message' => __('Razorpay doesn\'t support subscriptions right now', 'wp-payment-form-pro'),
            'payment_error' => true
        ), 423);
    }

    public function addCheckoutJs($settings)
    {
        wp_enqueue_script('razorpay', 'https://checkout.razorpay.com/v1/checkout.js', [], WPPAYFORM_VERSION);
        wp_enqueue_script('wppayform_razorpay_handler', WPPAYFORM_URL . 'assets/js/razorpay-handler.js', ['jquery'], WPPAYFORM_VERSION);
    }

    public function confirmModalPayment()
    {
        $data = $_REQUEST;

        $submissionHash = sanitize_text_field(Arr::get($data, 'transaction_hash'));

        $submission = (new Submission())->getSubmissionByHash($submissionHash);

        $transaction = (new Transaction())->getLatestTransaction($submission->id);

        if (!$transaction || $transaction->status != 'pending') {
            wp_send_json([
                'errors'      => 'Payment Error: Invalid Request',
            ], 423);
        }

        $paymentId = sanitize_text_field(Arr::get($data, 'razorpay_payment_id'));
        $vendorPayment = (new API())->makeApiCall('payments/' . $paymentId, [], $transaction->form_id);


        if (is_wp_error($vendorPayment)) {
            do_action('wppayform_log_data', [
                'form_id' => $transaction->form_id,
                'submission_id' => $submission->id,
                'type' => 'activity',
                'created_by' => 'Paymattic BOT',
                'title' => 'RazorPay Payment is failed to verify',
                'content' => $vendorPayment->get_error_message()
            ]);

            wp_send_json_error(array(
                'message' => $vendorPayment->get_error_message(),
                'payment_error' => true,
                'type' => 'error',
                'form_events' => [
                    'payment_failed'
                ]
            ), 423);
        }

        if ($vendorPayment['status'] == 'paid' || $vendorPayment['status'] == 'captured') {
            do_action('wppayform_log_data', [
                'form_id' => $transaction->form_id,
                'submission_id' => $submission->id,
                'type' => 'activity',
                'created_by' => 'Paymattic BOT',
                'title' => 'RazorPay Payment is failed to verify',
                'content' => 'Razorpay payment has been marked as paid'
            ]);

            $returnData = $this->handlePaid($submission, $transaction, $vendorPayment);
            $confirmation = ConfirmationHelper::getFormConfirmation($submission->form_id, $submission);
            $returnData['payment'] = $vendorPayment;
            $returnData['confirmation'] = $confirmation;
            wp_send_json_success($returnData, 200);
        }

        wp_send_json_error(array(
            'message' => 'Payment could not be verified. Please contact site admin',
            'payment_error' => true,
            'type' => 'error',
            'form_events' => [
                'payment_failed'
            ]
        ), 423);
    }

    public function processRefundTransaction($refundData)
    {
        $transaction = Arr::get($refundData, 'transaction', []);
        $submission = Arr::get($refundData, 'submission_id', []);
        $refundAmount = Arr::get($refundData, 'amount', 0);
        $vendorTransaction = (new API())->makeApiCall('payments/' . $transaction->charge_id . '/refund', [
            'amount' => $refundAmount
        ], $transaction->form_id, 'POST');

        if (is_wp_error($vendorTransaction)) {
            return $vendorTransaction;
        }
        return $vendorTransaction;
    }


    public function markasFailed($submission, $updateData, $transaction)
    {
        $transactionModel = new Transaction();
        $updateData['updated_at'] = current_time('Y-m-d H:i:s');

        $transactionModel->where('id', $transaction->id)->update($updateData);
        $transaction = $transactionModel->getTransaction($transaction->id);
        do_action('wppayform_log_data', [
            'form_id' => $transaction->form_id,
            'submission_id' => $transaction->submission_id,
            'type' => 'info',
            'created_by' => 'PayForm Bot',
            'content' => sprintf(__('Transaction Marked as failed and Razorpay Transaction ID: %s', 'wp-payment-form-pro'), $updateData['charge_id'])
        ]);
        do_action('wppayform/form_payment_failed', $submission, $transaction->form_id, $transaction, 'razorpay');
    }

    public function handlePaymentFailed($submission, $transaction, $vendorTransaction)
    {
        if (!$transaction || $transaction->payment_method != $this->method) {
            return;
        }

        $status = 'failed';
        $updateData = [
            'status' => $status,
            'payment_note' => maybe_serialize($vendorTransaction),
            'updated_at' => current_time('Y-m-d H:i:s')
        ];

        // Let's make the payment as failed
        $this->markasFailed($submission, $updateData, $transaction);
    }
}
