<?php

namespace WPPayFormPro\GateWays\Paystack;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use WP_Error;
use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Models\Transaction;
use WPPayForm\App\Models\Submission;
use WPPayForm\App\Models\Form;
use WPPayForm\App\Services\PlaceholderParser;
use WPPayForm\App\Services\ConfirmationHelper;

class PaystackProcessor
{
    public $method = 'paystack';

    protected $form;

    public function init()
    {
        new PaystackElement();
        (new PaystackSettings())->init();

        add_filter('wppayform/choose_payment_method_for_submission', array($this, 'choosePaymentMethod'), 10, 4);
        add_action('wppayform/form_submission_make_payment_' . $this->method, array($this, 'makeFormPayment'), 10, 6);
        add_action('wppayform_load_checkout_js_' . $this->method, array($this, 'addCheckoutJs'), 10, 3);

        add_action('wp_ajax_wppayform_paystack_confirm_payment', array($this, 'confirmModalPayment'));
        add_action('wp_ajax_nopriv_wppayform_paystack_confirm_payment', array($this, 'confirmModalPayment'));
        add_filter('wppayform/entry_transactions_' . $this->method, array($this, 'addTransactionUrl'), 10, 2);
        add_filter('wppayform/submitted_payment_items_' . $this->method, array($this, 'validateSubscription'), 10, 4);
        add_action('wppayform/process_refund_paystack', array($this, 'processRefundPayStack'), 10, 1);
        add_action('wppayform_ipn_paystack_action_failed', array($this, 'handlePaymentFailed'), 10, 3);
    }

    public function choosePaymentMethod($paymentMethod, $elements, $formId, $form_data)
    {
        if ($paymentMethod) {
            // Already someone choose that it's their payment method
            return $paymentMethod;
        }
        // Now We have to analyze the elements and return our payment method
        foreach ($elements as $element) {
            if ((isset($element['type']) && $element['type'] == 'paystack_gateway_element')) {
                return 'paystack';
            }
        }
        return $paymentMethod;
    }

    public function makeFormPayment($transactionId, $submissionId, $form_data, $form, $hasSubscriptions)
    {
        $paymentMode = $this->getPaymentMode();
        $transactionModel = new Transaction();
        if ($transactionId) {
            $transactionModel->updateTransaction($transactionId, array(
                'payment_mode' => $paymentMode
            ));
        }
        $transaction = $transactionModel->getTransaction($transactionId);

        $submission = (new Submission())->getSubmission($submissionId);

        $this->maybeShowModal($transaction, $submission, $form, $paymentMode);
    }

    public function addCheckoutJs($settings)
    {
        wp_enqueue_script('paystack', 'https://js.paystack.co/v1/inline.js', [], WPPAYFORM_VERSION);
        wp_enqueue_script('wppayform_paystack_handler', WPPAYFORM_URL . 'assets/js/paystack-handler.js', ['jquery'], WPPAYFORM_VERSION);
    }

   public function getSuccessURL($form, $submission)
    {
        // Check If the form settings have success URL
        $confirmation = Form::getConfirmationSettings($form->ID);
        $confirmation = ConfirmationHelper::parseConfirmation($confirmation, $submission);
        if (
            ($confirmation['redirectTo'] == 'customUrl' && $confirmation['customUrl']) ||
            ($confirmation['redirectTo'] == 'customPage' && $confirmation['customPage']) ||
            ($confirmation['redirectTo'] == 'customPost' && $confirmation['customPage'])
        ) {
            if ($confirmation['redirectTo'] == 'customUrl') {
                $url = $confirmation['customUrl'];
            } else {
                $url = get_permalink(intval($confirmation['customPage']));
            }
            $url = add_query_arg(array(
                'payment_method' => 'paystack'
            ), $url);
            $url = PlaceholderParser::parse($url, $submission);
            return wp_sanitize_redirect($url);
        }
        // now we have to check for global Success Page
        $globalSettings = get_option('wppayform_confirmation_pages');
        if (isset($globalSettings['confirmation']) && $globalSettings['confirmation']) {
            $url = add_query_arg(array(
                'wpf_submission' => $submission->submission_hash,
                'payment_method' => 'paystack'
            ), get_permalink(intval($globalSettings['confirmation'])));
            return wp_sanitize_redirect($url);
        }
        // In case we don't have global settings
        $url = add_query_arg(array(
            'wpf_submission' => $submission->submission_hash,
            'payment_method' => 'paystack'
        ), home_url());
        return wp_sanitize_redirect($url);
    }

    public function maybeShowModal($transaction, $submission, $form, $paymentMode)
    {
        $currency = strtoupper($submission->currency);
        if (!in_array($currency, ['NGN', 'GHS', 'ZAR', 'USD', 'KES'])) {
            wp_send_json([
                'errors'      => $currency . 'is not supported by Paystack payment method'
            ], 423);
        }

        $keys = PaystackSettings::getApiKeys($form->id);
        $modalData = [
            'key'      => $keys['api_key'],
            'email'    => $submission->customer_email,
            'ref'      => $submission->submission_hash,
            'amount'   => intval($transaction->payment_total),
            'currency' => $currency, //
            'label'    => $form->post_title,
            'metadata' => [
                'payment_handler' => 'WPPayForm',
                'form_id'         => $form->ID,
                'transaction_id'  => $transaction->id,
                'form'            => $form->post_title
            ]
        ];

        do_action('wppayform_log_data', [
            'form_id' => $submission->form_id,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'title' => 'Paystack Modal is initiated',
            'content' => 'Paystack Modal is initiated to complete the payment'
        ]);

        $confirmation = ConfirmationHelper::getFormConfirmation($submission->form_id, $submission);
        # Tell the client to handle the action
        wp_send_json_success([
            'nextAction'       => 'paystack',
            'actionName'       => 'initPaystackModal',
            'submission_id'    => $submission->id,
            'modal_data'       => $modalData,
            'transaction_hash' => $submission->submission_hash,
            'message'          => __('Payment Modal is opening, Please complete the payment', 'wp-payment-form-pro'),
            'result'           => [
                'insert_id' => $submission->id
            ]
        ], 200);
    }

    protected function getPaymentMode($formId = false)
    {
        $isLive = PaystackSettings::isLive($formId);
        if ($isLive) {
            return 'live';
        }
        return 'test';
    }

    public function addTransactionUrl($transactions, $submissionId)
    {
        foreach ($transactions as $transaction) {
            if ($transaction->charge_id) {
                $transaction->transaction_url =  'https://dashboard.paystack.com/#/transactions/' . $transaction->charge_id;
            }
        }
        return $transactions;
    }


    public function getLastTransaction($submissionId)
    {
        $transactionModel = new Transaction();
        $transaction = $transactionModel->where('submission_id', $submissionId)
            ->first();
        return $transaction;
    }

    public function handlePaid($submission, $transaction, $vendorTransaction)
    {
        $transaction = $this->getLastTransaction($submission->id);

        if (!$transaction || $transaction->payment_method != $this->method) {
            return;
        }

        do_action('wppayform/form_submission_activity_start', $transaction->form_id);

        if ($transaction->payment_method != 'paystack') {
            return; // this isn't a paystack standard IPN
        }

        $status = 'paid';

        $updateData = [
            'status' => $status,
            'payment_note'     => maybe_serialize($vendorTransaction),
            'charge_id'        => sanitize_text_field(Arr::get($vendorTransaction, 'data.id')),
            'payment_total' => Arr::get($vendorTransaction, 'data.amount'),
            'currency'      => strtoupper(Arr::get($vendorTransaction, 'data.currency')),
            'card_brand' => sanitize_text_field(Arr::get($vendorTransaction, 'data.authorization.card_type')),
            'card_last_4' => intval(Arr::get($vendorTransaction, 'data.authorization.last4')),
        ];
        // Let's make the payment as paid
        $this->markAsPaid('paid', $updateData, $transaction);
    }

    public function markAsPaid($status, $updateData, $transaction)
    {
        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($transaction->submission_id);

        $submissionData = array(
            'payment_status' => $status,
            'updated_at' => current_time('Y-m-d H:i:s')
        );

        $submissionModel->where('id', $transaction->submission_id)->update($submissionData);

        $transactionModel = new Transaction();
        $updateData['updated_at'] = current_time('Y-m-d H:i:s');

        $transactionModel->where('id', $transaction->id)->update($updateData);
        $transaction = $transactionModel->getTransaction($transaction->id);
        do_action('wppayform_log_data', [
            'form_id' => $transaction->form_id,
            'submission_id' => $transaction->submission_id,
            'type' => 'info',
            'created_by' => 'PayForm Bot',
            'content' => sprintf(__('Transaction Marked as paid and Paystack Transaction ID: %s', 'wp-payment-form-pro'), $updateData['charge_id'])
        ]);

        do_action('wppayform/form_payment_success_paystack', $submission, $transaction, $transaction->form_id, $updateData);
        do_action('wppayform/form_payment_success', $submission, $transaction, $transaction->form_id, $updateData);
    }

    public function validateSubscription($paymentItems, $formattedElements, $form_data, $subscriptionItems)
    {
        wp_send_json_error(array(
            'message' => __('Paystack doesn\'t support subscriptions right now', 'wp-payment-form-pro'),
            'payment_error' => true
        ), 423);
    }

    public function confirmModalPayment()
    {
        $data = $_REQUEST;
        $transactionHash = sanitize_text_field(Arr::get($data, 'trxref'));

        $submission = (new Submission())->getSubmissionByHash($transactionHash);

        $transaction = (new Transaction())->getLatestTransaction($submission->id);

        if (!$transaction || $transaction->status != 'pending') {
            wp_send_json([
                'errors'      => 'Payment Error: Invalid Request',
            ], 423);
        }

        $paymentReference = sanitize_text_field(Arr::get($data, 'reference'));
        $vendorPayment = (new API())->makeApiCall('transaction/verify/' . $paymentReference, [], $transaction->form_id);

        if (is_wp_error($vendorPayment)) {
            do_action('wppayform_log_data', [
                'form_id' => $transaction->form_id,
                'submission_id' => $submission->id,
                'type' => 'activity',
                'created_by' => 'Paymattic BOT',
                'title' => 'Paystack Payment is failed to verify',
                'content' => $vendorPayment->get_error_message()
            ]);

            wp_send_json_error(array(
                'message' => $vendorPayment->get_error_message(),
                'payment_error' => true,
                'type' => 'error',
                'form_events' => [
                    'payment_failed'
                ]
            ), 423);
        }
        if ($vendorPayment['status'] == 'success') {
            do_action('wppayform_log_data', [
                'form_id' => $transaction->form_id,
                'submission_id' => $submission->id,
                'type' => 'activity',
                'created_by' => 'Paymattic BOT',
                'title' => 'Paystack Payment is failed to verify',
                'content' => 'Paystack payment has been marked as paid'
            ]);

            $returnData = $this->handlePaid($submission, $transaction, $vendorPayment);
            $confirmation = ConfirmationHelper::getFormConfirmation($submission->form_id, $submission);
            $returnData['payment'] = $vendorPayment;
            $returnData['confirmation'] = $confirmation;
            wp_send_json_success($returnData, 200);
        }

        wp_send_json_error(array(
            'message' => 'Payment could not be verified. Please contact site admin',
            'payment_error' => true,
            'type' => 'error',
            'form_events' => [
                'payment_failed'
            ]
        ), 423);
    }

    public function processRefundPayStack($refundData)
    {
        $transaction = Arr::get($refundData, 'transaction', []);
        $chargeId = Arr::get($transaction, 'charge_id', '');
        $amount = floatval(Arr::get($refundData, 'amount', 0));

         $validationResult = $this->validateRefund( $amount, $chargeId);
         if (is_wp_error($validationResult)) {
            return $validationResult;
        }

        $refund = (new API())->makeApiCall('refund', [
            'transaction' => $chargeId,
            'amount' => $amount * 100
        ], $transaction->form_id, 'POST');

        $refundStatus = $this->getRefundStatus($refund, $chargeId);
        if (is_wp_error($refundStatus)) {
            return $refundStatus;
        }
        return $refund;
    }

    public function validateRefund($amount, $chargeId)
    {
        if (!$chargeId) {
            return new WP_Error('missing_charge', __('Missing Paystack charge ID for refund.', 'wp-payment-form-pro'));
        }

        if ($amount < 50) {
            return new WP_Error('invalid_amount', __('Invalid amount for refund. Minimum amount is ₦50.', 'wp-payment-form-pro'));
        }
        return true;
    }

    public function getRefundStatus($refund, $chargeId)
    {
        if (is_wp_error($refund) || isset($refund->error)) {
            $message = is_wp_error($refund) 
                ? $refund->get_error_message() 
                : $refund->error->message;
            return new \WP_Error('paystack_refund_failed', $message);
        }

            // If we get here, the refund was successful or queued
        if (isset($refund['message']) && strpos($refund['message'], 'queued') !== false) {
            // Create a synthetic reference if one isn't provided
            if (!isset($refund['data']['reference']) || !$refund['data']['reference']) {
                $refund['data']['reference'] = 'queued_' . time() . '_' . $chargeId;
            }
        }
    }

    public function markasFailed($submission, $updateData, $transaction)
    {
        $transactionModel = new Transaction();
        $updateData['updated_at'] = current_time('Y-m-d H:i:s');

        $transactionModel->where('id', $transaction->id)->update($updateData);
        $transaction = $transactionModel->getTransaction($transaction->id);
        do_action('wppayform_log_data', [
            'form_id' => $transaction->form_id,
            'submission_id' => $transaction->submission_id,
            'type' => 'info',
            'created_by' => 'PayForm Bot',
            'content' => sprintf(__('Transaction Marked as failed and Paystack Transaction ID: %s', 'wp-payment-form-pro'), $updateData['charge_id'])
        ]);

        do_action('wppayform/form_payment_failed', $submission, $transaction->form_id, $transaction, 'paystack');
    }


    public function handlePaymentFailed($submission, $transaction, $vendorTransaction)
    {
        if (!$transaction || $transaction->payment_method != $this->method) {
            return;
        }

        do_action('wppayform/form_submission_activity_start', $transaction->form_id);

        if ($transaction->payment_method != 'paystack') {
            return; // this isn't a paystack standard IPN
        }

        $status = 'failed';
        $updateData = [
            'status' => $status,
            'payment_note'     => maybe_serialize($vendorTransaction),
            'charge_id'        => sanitize_text_field(Arr::get($vendorTransaction, 'data.id')),
        ];
        // Let's make the payment as failed
        $this->markasFailed($submission, $updateData, $transaction);
    }
}
