<?php

namespace WPPayFormPro\GateWays\Payrexx;

if (!defined('ABSPATH')) {
    exit;
}

use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Models\Transaction as WpfTransaction;
use WPPayForm\App\Models\Submission;
use WPPayForm\App\Models\Form;
use WPPayForm\App\Services\PlaceholderParser;
use WPPayForm\App\Services\ConfirmationHelper;
use WPPayFormPro\GateWays\Payrexx\QueryHelper\Gateway;
use WPPayFormPro\GateWays\Payrexx\QueryHelper\Transaction;

class PayrexxProcessor
{
    public $method = 'payrexx';

    protected $form;

    public function init()
    {
        new PayrexxElement();
        (new PayrexxSettings())->init();

        add_filter('wppayform/choose_payment_method_for_submission', array($this, 'choosePaymentMethod'), 10, 4);
        add_action('wppayform/form_submission_make_payment_' . $this->method, array($this, 'makeFormPayment'), 10, 6);
        add_action('wppayform_load_checkout_js_' . $this->method, array($this, 'addCheckoutJs'), 10, 3);

        add_filter('wppayform/entry_transactions_' . $this->method, array($this, 'addTransactionUrl'), 10, 2);
        add_filter('wppayform/submitted_payment_items_' . $this->method, array($this, 'validateSubscription'), 10, 4);

        add_action('wpf_ipn_endpoint_' . $this->method, function () {
            $this->verifyIPN();
            exit(200);
        });
    }

    public function handleRedirect($transaction, $submission, $form)
    {
        $successUrl = $this->getSuccessURL($form, $submission);
        $transaction->hash = $submission->submission_hash;
        $globalSettings = PayrexxSettings::getApiKeys();

        if (!isset($globalSettings['instance']) || !isset($globalSettings['api_key'])) {
            return;
        }

        $transaction->listener_url = wp_sanitize_redirect(add_query_arg(array(
            'wppayform_payment' => $submission->id,
            'payment_method' => $this->method,
            'submission_hash' => $submission->submission_hash,
        ), $successUrl));

        $paymentIntent = $this->makePaymentData($transaction);

        if (is_wp_error($paymentIntent)) {
            do_action('wppayform_log_data', [
                'form_id' => $submission->form_id,
                'submission_id' => $submission->id,
                'type' => 'activity',
                'created_by' => 'Paymattic BOT',
                'title' => 'Payrexx Payment Webhook Error',
                'content' => $paymentIntent->get_error_message()
            ]);

            wp_send_json_error(array(
                'message' => __($paymentIntent->get_error_message(), 'wp-payment-form-pro')
            ), 423);
        }

        do_action('wppayform_log_data', [
            'form_id' => $form->ID,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'title' => 'Payrexx Payment Redirect',
            'content' => 'User redirect to Payrexx for completing the payment'
        ]);

        $redirectUrl = wp_sanitize_redirect(Arr::get($paymentIntent, 'data.0.link'));

        wp_send_json_success([
            'message' => __('You are redirecting to payrexx.com to complete the purchase. Please wait while you are redirecting....', 'wp-payment-form-pro'),
            'call_next_method' => 'normalRedirect',
            'redirect_url' => $redirectUrl
        ], 200);
    }

    public function verifyIPN()
    {
        if (!isset($_POST['transaction'])) {
            return;
        }
        $vendorTransaction = $_POST['transaction'];

        $payId = $vendorTransaction['id'];

        $globalSettings = PayrexxSettings::getApiKeys();
        $Api = new API(Arr::get($globalSettings, 'instance'), Arr::get($globalSettings, 'api_key'));

        $transactionIns = new Transaction();
        $transactionIns->setId($payId);

        $latestTransaction = $Api->getOne($transactionIns);
        if (Arr::get($latestTransaction, 'status') === 'success') {
            $vendorTransaction = Arr::get($latestTransaction, 'data.0');
        } else {
            return;
        }

        $reference = sanitize_text_field(Arr::get($vendorTransaction, 'referenceId'));

        $transactionModel = new WpfTransaction();
        $transaction = $transactionModel->getTransaction($reference);

        $submission = (new Submission())->getSubmission($transaction->submission_id);

        $vendorTransaction['instance'] = Arr::get($globalSettings, 'instance');

        $this->handleStatus($submission, $transaction, $vendorTransaction);
    }

    public function choosePaymentMethod($paymentMethod, $elements, $formId, $form_data)
    {
        if ($paymentMethod) {
            // Already someone choose that it's their payment method
            return $paymentMethod;
        }
        // Now We have to analyze the elements and return our payment method
        foreach ($elements as $element) {
            if ((isset($element['type']) && $element['type'] == 'payrexx_gateway_element')) {
                return 'payrexx';
            }
        }
        return $paymentMethod;
    }

    public function makeFormPayment($transactionId, $submissionId, $form_data, $form, $hasSubscriptions)
    {
        $transactionModel = new WpfTransaction();
        if ($transactionId) {
            $transactionModel->updateTransaction($transactionId, array(
                'payment_mode' => ''
            ));
        }
        $transaction = $transactionModel->getTransaction($transactionId);

        $submission = (new Submission())->getSubmission($submissionId);

        $this->maybeShowModal($transaction, $submission, $form);
        $this->handleRedirect($transaction, $submission, $form);
    }

    public function makePaymentData($transaction)
    {
        $globalSettings = PayrexxSettings::getApiKeys();

        $Api = new API(Arr::get($globalSettings, 'instance'), Arr::get($globalSettings, 'api_key'));
        $gateway = new Gateway();

        $gateway->setAmount($transaction->payment_total);
        $gateway->setCurrency($transaction->currency);
        $gateway->setSuccessRedirectUrl($transaction->listener_url);
        $gateway->setSkipResultPage(true);
        $gateway->setReferenceId($transaction->id);
        $gateway->setPreAuthorization(false);
        $gateway->setReservation(false);

        $paymentIntent = $Api->create($gateway);
        return $paymentIntent;
    }

    public function addCheckoutJs($settings)
    {
        wp_enqueue_script('payrexx', 'https://media.payrexx.com/modal/v1/modal.min.js', ['jquery'], WPPAYFORM_VERSION);
        wp_enqueue_script('wppayform_payrexx_handler', WPPAYFORM_URL . 'assets/js/payrexx-handler.js', ['jquery'], WPPAYFORM_VERSION);
    }

   public function getSuccessURL($form, $submission)
    {
       // Check If the form settings have success URL
        $confirmation = Form::getConfirmationSettings($form->ID);
        $confirmation = ConfirmationHelper::parseConfirmation($confirmation, $submission);
        if (
            ($confirmation['redirectTo'] == 'customUrl' && $confirmation['customUrl']) ||
            ($confirmation['redirectTo'] == 'customPage' && $confirmation['customPage']) ||
            ($confirmation['redirectTo'] == 'customPost' && $confirmation['customPage'])
        ) {
            if ($confirmation['redirectTo'] == 'customUrl') {
                $url = $confirmation['customUrl'];
            } else {
                $url = get_permalink(intval($confirmation['customPage']));
            }
            $url = add_query_arg(array(
                'payment_method' => 'payrexx'
            ), $url);
            $url = PlaceholderParser::parse($url, $submission);
            return wp_sanitize_redirect($url);
        }
        // now we have to check for global Success Page
        $globalSettings = get_option('wppayform_confirmation_pages');
        if (isset($globalSettings['confirmation']) && $globalSettings['confirmation']) {
            $url = add_query_arg(array(
                'wpf_submission' => $submission->submission_hash,
                'payment_method' => 'payrexx'
            ), get_permalink(intval($globalSettings['confirmation'])));
            return wp_sanitize_redirect($url);
        }
        // In case we don't have global settings
        $url = add_query_arg(array(
            'wpf_submission' => $submission->submission_hash,
            'payment_method' => 'payrexx'
        ), home_url());
        return wp_sanitize_redirect($url);
    }

    public function maybeShowModal($transaction, $submission, $form)
    {
        $globalSettings = PayrexxSettings::getSettings();
        if (Arr::get($globalSettings, 'checkout_type') !== 'modal') {
            return;
        }

        $currency = strtoupper($submission->currency);
        $paymentIntent = $this->makePaymentData($transaction);

        if (is_wp_error($paymentIntent)) {
            do_action('wppayform_log_data', [
                'form_id' => $submission->form_id,
                'submission_id' => $submission->id,
                'type' => 'activity',
                'created_by' => 'Paymattic BOT',
                'title' => 'Payrexx Payment Webhook Error',
                'content' => $paymentIntent->get_error_message()
            ]);

            wp_send_json_error(array(
                'message' => __($paymentIntent->get_error_message(), 'wp-payment-form-pro')
            ), 423);
        } 

       if (Arr::get($paymentIntent, 'status', '') == 'error') {
            wp_send_json_error(array(
                'message' => __(Arr::get($paymentIntent, 'message'), 'wp-payment-form-pro')
            ), 423);
       }

        do_action('wppayform_log_data', [
            'form_id' => $submission->form_id,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'title' => 'Payrexx Modal is initiated',
            'content' => 'Payrexx Modal is initiated to complete the payment'
        ]);

        $confirmation = ConfirmationHelper::getFormConfirmation($submission->form_id, $submission);
        # Tell the client to handle the action
        $responseData = [
            'nextAction'       => 'payrexx',
            'actionName'       => 'initPayrexxModal',
            'submission_id'    => $submission->id,
            'success_url'      => $this->getSuccessURL($form, $submission),
            'modal_data'       => Arr::get($paymentIntent, 'data.0'),
            'transaction_hash' => $submission->submission_hash,
            'message'          => __('Payment Modal is opening, Please complete the payment', 'wp-payment-form-pro'),
            'result'           => [
                'insert_id' => $submission->id
            ]
        ];

        wp_send_json_success($responseData, 200);
    }

    public function addTransactionUrl($transactions, $submissionId)
    {
        foreach ($transactions as $transaction) {
            $payNote = safeUnserialize($transaction->payment_note);
            $instance =  Arr::get($payNote, 'instance');
            if ($transaction->charge_id) {
                $transaction->transaction_url = "https://" . $instance . ".payrexx.com/cadmin/index.php?cmd=checkout&act=payments&id=" . Arr::get($payNote, 'uuid');
            }
        }

        return $transactions;
    }

    public function getLastTransaction($submissionId)
    {
        $transactionModel = new WpfTransaction();
        $transaction = $transactionModel->where('submission_id', $submissionId)
            ->first();
        return $transaction;
    }

    public function handleStatus($submission, $transaction, $vendorTransaction)
    {
        if (!$transaction || $transaction->payment_method != $this->method) {
            return;
        }

        do_action('wppayform/form_submission_activity_start', $transaction->form_id);

        $payId = $vendorTransaction['id'];
        $uuid = $vendorTransaction['uuid'];

        $invoice = Arr::get($vendorTransaction, 'invoice');
        $contact = Arr::get($vendorTransaction, 'contact');

        // $fullName = Arr::get($contact, 'firstname') . ' ' . Arr::get($contact, 'lastname');
        // $fullName = sanitize_text_field($fullName);
        // $payerEmail = sanitize_text_field(Arr::get($contact, 'email'));
        $status = $vendorTransaction['status'];

        if ($status == 'confirmed') {
            $status = 'paid';
        }

        else if ($status === 'waiting') {
            $status = 'processing';
        } 

        else {
            $status = 'failed';
            do_action('wppayform/form_payment_failed', $submission, $submission->form_id, $transaction, 'payrexx');
        }
         

        $updateData = array(
            // 'charge_id'     => $uuid ? sanitize_text_field($uuid) : sanitize_text_field($payId),
            'payment_total' => intval(Arr::get($invoice, 'totalAmount')),
            'payment_mode'  => Arr::get($vendorTransaction, 'mode'),
            'status'        => $status,
            'card_brand'    => Arr::get($vendorTransaction, 'payment.brand'),
            'payment_note'  => maybe_serialize($vendorTransaction),
            'charge_id'        => sanitize_text_field($vendorTransaction['id']),
        );
        // Let's make the payment status updated
        $this->markAsPaid($status, $updateData, $transaction);
    }

    public function markAsPaid($status, $updateData, $transaction)
    {
        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($transaction->submission_id);

        $submissionData = array(
            'payment_status' => $status,
            'updated_at' => current_time('Y-m-d H:i:s')
        );

        $submissionModel->where('id', $transaction->submission_id)->update($submissionData);

        $transactionModel = new WpfTransaction();
        $updateData['updated_at'] = current_time('Y-m-d H:i:s');

        $transactionModel->where('id', $transaction->id)->update($updateData);
        $transaction = $transactionModel->getTransaction($transaction->id);

        do_action('wppayform_log_data', [
            'form_id' => $transaction->form_id,
            'submission_id' => $transaction->submission_id,
            'type' => 'info',
            'created_by' => 'PayForm Bot',
            'content' => sprintf(__('Transaction Marked as %s and Payrexx Transaction ID: %s', 'wp-payment-form-pro'), $status, $updateData['charge_id'])
        ]);

        if ($status === 'paid') {
            do_action('wppayform/form_payment_success_payrexx', $submission, $transaction, $transaction->form_id, $updateData);
            do_action('wppayform/form_payment_success', $submission, $transaction, $transaction->form_id, $updateData);
        }
    }

    public function validateSubscription($paymentItems, $formattedElements, $form_data, $subscriptionItems)
    {
        wp_send_json_error(array(
            'message' => __('Payrexx doesn\'t support subscriptions right now', 'wp-payment-form-pro'),
            'payment_error' => true
        ), 423);
    }
}
