<?php

namespace WPPayFormPro\GateWays\PayPal;

use WPPayFormPro\GateWays\BasePaymentMethod;
use WPPayForm\App\Models\Form;
use WPPayForm\App\Models\OrderItem;
use WPPayForm\App\Models\Submission;
use WPPayForm\App\Models\SubmissionActivity;
use WPPayForm\App\Models\Subscription;
use WPPayForm\App\Models\Transaction;
use WPPayForm\App\Services\AccessControl;
use WPPayForm\App\Services\GeneralSettings;
use WPPayForm\App\Services\PlaceholderParser;
use WPPayForm\App\Services\ConfirmationHelper;
use WPPayForm\Framework\Support\Arr;
use WPPayFormPro\Classes\PaymentsHelper;

class PayPal
{
    public function init()
    {
        // Init paypal Element for Editor
        new PayPalElement();
        // Choose Payment method Here
        add_filter('wppayform/choose_payment_method_for_submission', array($this, 'choosePaymentMethod'), 10, 4);
        add_action('wppayform/form_submission_make_payment_paypal', array($this, 'makeFormPayment'), 10, 7);

        add_filter('wppayform/entry_transactions_paypal', array($this, 'addTransactionUrl'), 10, 2);
        add_action('init', array($this, 'verifyIPN'));
        add_action('wppayform/form_render_after', array($this, 'checkForCancelMessage'), 10, 1);

        add_filter('wppayform/paypal_payment_args', array($this, 'maybeHasSubscription'), 10, 6);

        // View level hooks
        add_filter('wppayform/subscription_items_paypal', array($this, 'formatSubscriptionItems'), 10, 2);

        add_filter('wppayform/submitted_payment_items_paypal', array($this, 'validateSubmittedItems'), 10, 4);

        add_filter('wppayform_payment_settings_paypal', function () {
            return $this->getPaypalSettings();
        });
    }

    public function checkForCancelMessage($form)
    {
        if (isset($_REQUEST['wpf_paypal_cancel']) && isset($_REQUEST['wpf_form_id'])) {
            $submissionId = sanitize_text_field(wp_unslash($_REQUEST['wpf_paypal_cancel']));
            $formId = intval($_REQUEST['wpf_form_id']);
            // Get submission object for proper hook call
            $submission = (new Submission())->getSubmission($submissionId);
            (new Submission())->updateSubmission($submissionId, array(
                'payment_status' => 'failed'
            ));
            $transaction = (new Transaction())->getLatestTransaction($submissionId);
            (new Transaction())->updateTransaction($transaction->id, array(
                'status' => 'failed'
            ));      
            if ($submission) {
                do_action('wppayform/form_payment_failed', $submission, $formId, $transaction, 'paypal');
            }
            if ($formId == $form->ID) {
                echo '<div class="wpf_form_notices wpf_form_errors wpf_paypal_error">' . __('Looks like you have canceled the payment from paypal', 'wp-payment-form-pro') . '</div>';
            }
        }
    }

    public function addTransactionUrl($transactions, $formId)
    {
        foreach ($transactions as $transaction) {
            if ($transaction->payment_method == 'paypal' && $transaction->charge_id) {
                $transaction->transaction_url = $this->getPayPalTransactionUrl($transaction);
            }
        }
        return $transactions;
    }

    private function getPayPalTransactionUrl($transaction)
    {
        $sandbox = 'test' == $transaction->payment_mode ? 'sandbox.' : '';
        return 'https://www.' . $sandbox . 'paypal.com/activity/payment/' . $transaction->charge_id;
    }

    public function choosePaymentMethod($paymentMethod, $elements, $formId, $form_data)
    {
        if ($paymentMethod) {
            // Already someone choose that it's their payment method
            return $paymentMethod;
        }
        // Now We have to analyze the elements and return our payment method
        foreach ($elements as $element) {
            if ((isset($element['type']) && $element['type'] == 'paypal_gateway_element')) {
                return 'paypal';
            }
        }
        return $paymentMethod;
    }

    public function makeFormPayment($transactionId, $submissionId, $form_data, $form, $hasSubscriptions, $totalPayable = 0, $paymentItems = array())
    {
        $paypalSettings = $this->getPaypalSettings($form->ID);
        $paymentMode = 'live';
        if ($paypalSettings['payment_mode'] == 'test') {
            $paymentMode = 'test';
        }

        if ($paymentMode == 'test') {
            $paypal_redirect = 'https://www.sandbox.paypal.com/cgi-bin/webscr/?';
        } else {
            $paypal_redirect = 'https://www.paypal.com/cgi-bin/webscr/?';
        }

        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($submissionId);

        $listener_url = apply_filters('wppayform/paypal_ipn_url', site_url('?wpf_paypal_ipn=1&submission_id=') . $submissionId, $submission);


        // For Dev purpose only
        // $listener_url = 'https://19be48af.ngrok.io?wpf_paypal_ipn=1';

        $paypal_args = array(
            'cmd' => '_cart',
            'upload' => '1',
            'business' => $paypalSettings['paypal_email'],
            'email' => $submission->customer_email,
            'no_shipping' => (Arr::get($form_data, '__payment_require_shipping_address') == 'yes') ? '0' : '1',
            'no_note' => '1',
            'currency_code' => $submission->currency,
            'charset' => 'UTF-8',
            'custom' => $transactionId,
            'return' => esc_url_raw($this->getSuccessURL($form, $submission)),
            'notify_url' => esc_url_raw($listener_url),
            'cancel_return' => esc_url_raw($this->getCancelURL($form_data, $submission)),
            'image_url' => Arr::get($paypalSettings, 'checkout_logo'),
        );

        $cart_summary = $this->getCartSummery($submissionId, $form->ID);

        // We have to check if it's $0 order. If it's $0 then we must have to return just from here
        if (!$cart_summary && !$hasSubscriptions) {
            return;
        }

        // Now the problem is this payment may hve $0 subscription which is a really pain
        // That we have to handle

        $paypal_args = array_merge($cart_summary, $paypal_args);
        $paypal_args = apply_filters('wppayform/paypal_payment_args', $paypal_args, $submission, $form_data, $paymentMode, $hasSubscriptions, $paymentItems);

        if ($paypal_args && is_wp_error($paypal_args)) {
            return wp_send_json_error(array(
                'message' => $paypal_args->get_error_message()
            ), 423);
        }

        if (!$cart_summary && $paypal_args['cmd'] == '_cart') {
            return;
        }

        $submissionModel->updateSubmission($submissionId, array(
            'payment_mode' => $paymentMode
        ));

        if ($transactionId) {
            $transactionModel = new Transaction();
            $transactionModel->updateTransaction($transactionId, array(
                'payment_mode' => $paymentMode
            ));
        }


        $paypal_redirect .= http_build_query($paypal_args);

        SubmissionActivity::createActivity(array(
            'form_id' => $submission->form_id,
            'submission_id' => $submission->id,
            'type' => 'info',
            'created_by' => 'Payform Bot',
            'content' => __('Redirect to paypal for payment', 'wp-payment-form-pro')
        ));

        wp_send_json_success(array(
            'message' => __('Form is successfully submitted', 'wp-payment-form-pro'),
            'submission_id' => $submissionId,
            'confirmation' => array(
                'confirmation_type' => 'custom',
                'redirectTo' => 'customUrl',
                'customUrl' => $paypal_redirect,
                'messageToShow' => __('You are redirecting to paypal now', 'wp-payment-form-pro'),
                'samePageFormBehavior' => 'reset_form',
            )
        ), 200);
        exit;
    }

    private function getCancelURL($form_data, $submission)
    {
        $formUrl = Arr::get($form_data, '__wpf_current_url');
        if (!$formUrl) {
            $formUrl = home_url();
        }
        $url = add_query_arg(array(
            'wpf_form_id' => $submission->form_id,
            'wpf_paypal_cancel' => $submission->id
        ), $formUrl);
        $url .= '&wpf_form_id_' . $submission->form_id;
        return wp_sanitize_redirect($url);
    }

    private function getSuccessURL($form, $submission)
    {
        // Check If the form settings have success URL
        $confirmation = Form::getConfirmationSettings($form->ID);
        $confirmation = ConfirmationHelper::parseConfirmation($confirmation, $submission);
        if (
            ($confirmation['redirectTo'] == 'customUrl' && $confirmation['customUrl']) ||
            ($confirmation['redirectTo'] == 'customPage' && $confirmation['customPage']) ||
            ($confirmation['redirectTo'] == 'customPost' && $confirmation['customPage'])
        ) {
            if ($confirmation['redirectTo'] == 'customUrl') {
                $url = $confirmation['customUrl'];
            } else {
                $url = get_permalink(intval($confirmation['customPage']));
            }
            $url = add_query_arg(array(
                'payment_method' => 'paypal'
            ), $url);
            $url = PlaceholderParser::parse($url, $submission);
            return wp_sanitize_redirect($url);
        }
        // now we have to check for global Success Page
        $globalSettings = get_option('wppayform_confirmation_pages');
        if (isset($globalSettings['confirmation']) && $globalSettings['confirmation']) {
            $url = add_query_arg(array(
                'wpf_submission' => $submission->submission_hash,
                'payment_method' => 'paypal'
            ), get_permalink(intval($globalSettings['confirmation'])));
            return wp_sanitize_redirect($url);
        }
        // In case we don't have global settings
        $url = add_query_arg(array(
            'wpf_submission' => $submission->submission_hash,
            'payment_method' => 'paypal'
        ), home_url());
        return wp_sanitize_redirect($url);
    }

    private function getCartSummery($submissionId, $formId)
    {
        $orderItemModel = new OrderItem();
        $items = $orderItemModel->getOrderItems($submissionId);
        $discountItems = $orderItemModel->getDiscountItems($submissionId);

        $paypal_args = array();
        if ($items) {
            $counter = 1;

            foreach ($items as $item) {
                if (!$item->item_price) {
                    continue;
                }

                $paypal_args['item_name_' . $counter] = preg_replace('/[^a-zA-Z0-9\s]/', '', strip_tags(html_entity_decode($item->item_name)));
                $paypal_args['quantity_' . $counter] = $item->quantity;
                $paypal_args['amount_' . $counter] = round($item->item_price / 100, 2);
                $counter = $counter + 1;
            }
        }

        if ($discountItems) {
            $discountTotal = 0;
            foreach ($discountItems as $discountItem) {
                $discountTotal += intval($discountItem->line_total);
            }
            $paypal_args['discount_amount_cart'] = round($discountTotal / 100, 2);
        }
        return $paypal_args;
    }

    public function verifyIPN()
    {
        if (!isset($_REQUEST['wpf_paypal_ipn'])) {
            return;
        }

        if (defined('PAYFORM_PAYPAL_IPN_DEBUG')) {
            error_log(json_encode($_REQUEST));
        }

        //Check the request method is POST
        if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] != 'POST') {
            return;
        }

        $submission_id = false;
        if (!isset($_REQUEST['submission_id'])) {
            $submission_id = $_REQUEST['submission_id'];
        }

        // Set initial post data to empty string
        $post_data = '';

        // Fallback just in case post_max_size is lower than needed
        if (ini_get('allow_url_fopen')) {
            $post_data = file_get_contents('php://input');
        } else {
            // If allow_url_fopen is not enabled, then make sure that post_max_size is large enough
            ini_set('post_max_size', '12M');
        }
        // Start the encoded data collection with notification command
        $encoded_data = 'cmd=_notify-validate';

        // Get current arg separator
        $arg_separator = ini_get('arg_separator.output');

        // Verify there is a post_data
        if ($post_data || strlen($post_data) > 0) {
            // Append the data
            $encoded_data .= $arg_separator . $post_data;
        } else {
            // Check if POST is empty
            if (empty($_POST)) {
                // Nothing to do
                return;
            } else {
                // Loop through each POST
                foreach ($_POST as $key => $value) {
                    // Encode the value and append the data
                    $encoded_data .= $arg_separator . "$key=" . urlencode($value);
                }
            }
        }

        // Convert collected post data to an array
        parse_str($encoded_data, $encoded_data_array);

        foreach ($encoded_data_array as $key => $value) {
            if (false !== strpos($key, 'amp;')) {
                $new_key = str_replace('&amp;', '&', $key);
                $new_key = str_replace('amp;', '&', $new_key);
                unset($encoded_data_array[$key]);
                $encoded_data_array[$new_key] = $value;
            }
        }

        /**
         * PayPal Web IPN Verification
         *
         * Allows filtering the IPN Verification data that PayPal passes back in via IPN with PayPal Standard
         *
         *
         * @param array $data The PayPal Web Accept Data
         */
        $encoded_data_array = apply_filters('wppayform/process_paypal_ipn_data', $encoded_data_array);

        $defaults = array(
            'txn_type'       => '',
            'payment_status' => '',
            'custom'         => ''
        );

        $encoded_data_array = wp_parse_args($encoded_data_array, $defaults);

        $paymentSettings = $this->getPaypalSettings();

        if ($paymentSettings['disable_ipn_verification'] != 'yes') {
            $validate_ipn = wp_unslash($encoded_data_array); // WPCS: CSRF ok, input var ok.
            $validate_ipn['cmd'] = '_notify-validate';
            // Validate the IPN
            $remote_post_vars = array(
                'timeout' => 60,
                'redirection' => 5,
                'httpversion' => '1.1',
                'compress'    => false,
                'decompress'  => false,
                'user-agent' => 'WPPayForm IPN Verification/' . '3.0.0' . '; ' . get_bloginfo('url'),
                'body' => $validate_ipn
            );

            // Get response
            $api_response = wp_safe_remote_post($this->getPaypalRedirect(true, true), $remote_post_vars);

            if (is_wp_error($api_response)) {
                if (defined('PAYFORM_PAYPAL_IPN_DEBUG')) {
                    error_log('PAYMATTIC: IPN Verification Failed for api response error');
                }
                do_action('wppayform/paypal_ipn_verification_failed', $remote_post_vars, $encoded_data_array);
                return; // Something went wrong
            }
            if (wp_remote_retrieve_body($api_response) !== 'VERIFIED') {
                if (defined('PAYFORM_PAYPAL_IPN_DEBUG')) {
                    error_log('PAYMATTIC: IPN Verification Failed');
                }

                do_action('wppayform/paypal_ipn_not_verified', $api_response, $remote_post_vars, $encoded_data_array);
                return; // Response not okay
            }
        }

        // Check if $post_data_array has been populated
        if (!is_array($encoded_data_array) && !empty($encoded_data_array)) {
            return;
        }

        $defaults = array(
            'txn_type' => '',
            'payment_status' => '',
            'custom' => ''
        );

        $encoded_data_array = wp_parse_args($encoded_data_array, $defaults);

        if ('subscr_cancel' === $encoded_data_array['txn_type']) {
            do_action('wppayform/paypal_action_subscr_cancel', $encoded_data_array, $encoded_data_array['subscr_id']);
        }
        $payment_id = 0;

        if (!empty($encoded_data_array['parent_txn_id'])) {
            $payment_id = $this->getPaymentIdByTransactionId($encoded_data_array['parent_txn_id']);
        } elseif (!empty($encoded_data_array['txn_id'])) {
            $payment_id = $this->getPaymentIdByTransactionId($encoded_data_array['txn_id']);
        }

        if (empty($payment_id)) {
            $payment_id = !empty($encoded_data_array['custom']) ? absint($encoded_data_array['custom']) : 0;
        }

        if (defined('PAYFORM_PAYPAL_IPN_DEBUG')) {
            error_log('IPN DATA: ');
            error_log(json_encode($encoded_data_array));
        }

        if (has_action('wppayform/paypal_action_' . $encoded_data_array['txn_type'])) {
            // Allow PayPal IPN types to be processed separately
            do_action('wppayform/paypal_action_' . $encoded_data_array['txn_type'], $encoded_data_array, $payment_id);
        } else {
            if (defined('PAYFORM_PAYPAL_IPN_DEBUG')) {
                error_log('paypal_action_web_accept IPN: ');
                error_log(json_encode($encoded_data_array));
            }
            // Fallback to web accept just in case the txn_type isn't present
            do_action('wppayform/paypal_action_web_accept', $encoded_data_array, $payment_id, $submission_id);
        }
        http_response_code(200);
        exit;
    }

    private function getPaypalRedirect($ssl_check = false, $ipn = false)
    {
        $protocol = 'http://';
        if (is_ssl() || !$ssl_check) {
            $protocol = 'https://';
        }

        // Check the current payment mode
        if ($this->isTestMode()) {
            // Test mode
            if ($ipn) {
                $paypal_uri = 'https://ipnpb.sandbox.paypal.com/cgi-bin/webscr';
            } else {
                $paypal_uri = $protocol . 'www.sandbox.paypal.com/cgi-bin/webscr';
            }
        } else {
            // Live mode
            if ($ipn) {
                $paypal_uri = 'https://ipnpb.paypal.com/cgi-bin/webscr';
            } else {
                $paypal_uri = $protocol . 'www.paypal.com/cgi-bin/webscr';
            }
        }
        return apply_filters('wppayform/paypal_url', $paypal_uri, $ssl_check, $ipn);
    }

    private function getPaymentIdByTransactionId($chargeId)
    {
        $payment = Transaction::where('charge_id', $chargeId)
            ->where('payment_method', 'payapl')
            ->first();
        if ($payment) {
            return $payment->id;
        }
        return false;
    }

    public function savePaymentSettings($request)
    {
        AccessControl::checkAndPresponseError('set_payment_settings', 'global');
        $settings = $request->settings;
        $defaults = array(
            'payment_mode' => 'live',
            'paypal_email' => '',
            'disable_ipn_verification' => 'no',
            'checkout_logo' => ''
        );
        $settings = $this->mapper($defaults, $settings, false);
        // Validate the data first
        $mode = $settings['payment_mode'];

        // We require paypal Email Adddress
        if (empty($settings['paypal_email']) || !is_email($settings['paypal_email'])) {
            wp_send_json_error(array(
                'message' => __('Please enter valid email address', 'wp-payment-form-pro')
            ), 423);
        }

        // Validation Passed now let's make the data
        $data = array(
            'payment_mode' => sanitize_text_field($mode),
            'paypal_email' => sanitize_text_field($settings['paypal_email']),
            'disable_ipn_verification' => sanitize_text_field($settings['disable_ipn_verification']),
            'checkout_logo' => sanitize_text_field($settings['checkout_logo'])
        );
        do_action('wppayform/before_save_paypal_settings', $data);
        update_option('wppayform_paypal_payment_settings', $data, false);
        do_action('wppayform/after_save_paypal_settings', $data);

        $confrimationSettings = $request->confirmation_pages;
        $confirmationPages = array(
            'confirmation' => intval($confrimationSettings['confirmation']),
            'failed' => intval($confrimationSettings['failed'])
        );
        update_option('wppayform_confirmation_pages', $confirmationPages, false);

        return array(
            'message' => __('Settings successfully updated', 'wp-payment-form-pro')
        );
    }

    public function getPaymentSettings()
    {
        $pages = Form::select(array('ID', 'post_title'))
            ->where('post_type', 'page')
            ->where('post_status', 'publish')
            ->get();
        return array(
            'settings' => $this->getDynamicPaypalSettings(),
            'confirmation_pages' => GeneralSettings::getConfirmationPageSettings(),
            'pages' => $pages
        );
    }

    public function getPaypalSettings($formId = false)
    {
        $defaults = array(
            'payment_mode' => 'live',
            'paypal_email' => '',
            'disable_ipn_verification' => 'no',
            'checkout_logo' => ''
        );

        $settings = get_option('wppayform_paypal_payment_settings');
        if (!$settings) {
            $settings = array();
        }

        if ($formId) {
            $formPaymentSettings = (new PaymentsHelper())->getPaymentSettings($formId, 'admin');
            if (Arr::get($formPaymentSettings, 'paypal_account_type') == 'custom') {
                $payPalMode = Arr::get($formPaymentSettings, 'custom_paypal_mode');
                if ($payPalMode == 'test') {
                    $payPalId =  Arr::get($formPaymentSettings, 'custom_test_paypal_id');
                }
                else {
                    $payPalId =  Arr::get($formPaymentSettings, 'custom_paypal_id');
                }

                if($payPalId && $payPalMode) {
                    $settings['paypal_email'] = $payPalId;
                    $settings['payment_mode'] = $payPalMode;
                }
            }
        }

        return wp_parse_args($settings, $defaults);
    }
    private function getDynamicPaypalSettings () {
        $settings = $this->getPaypalSettings();
        return $this->mapSettings($settings);
    }
    public function mapper($defaults, $settings = [], $get = true) 
    {
        foreach ($defaults as $key => $value) {
            if($get) {
                if (isset($settings[$key])) {
                    $defaults[$key]['value'] = $settings[$key];
                }
            } else {
                if (isset($settings[$key])) {
                    $defaults[$key] = $settings[$key]['value'];
                }
            }
        }
        return $defaults;
    }
    private function mapSettings($settings) 
    {
        $defaults = array(
            'payment_mode' => array(
                'value' => 'test',
                'label' => __('Payment Mode', 'wp-payment-form'),
                'options' => array(
                    'test' => __('Sandbox Mode', 'wp-payment-form'),
                    'live' => __('Live Mode', 'wp-payment-form')
                ),
                'type' => 'payment_mode'
            ),
            'paypal_email' => array(
                'value' => '',
                'label' => __('PayPal Email', 'wp-payment-form'),
                'type' => 'gmail',
                'placeholder' => __('PayPal Email Address', 'wp-payment-form')
            ),
            'disable_ipn_verification' => array(
                'value' => 'no',
                'label' => __('Disable IPN Verification', 'wp-payment-form'),
                'type' => 'checkbox',
                'desc' => 'If you are unable to use Payment Data Transfer and payments are not getting marked as complete, then check this box. This forces the site to use a slightly less secure method of verifying purchases.'
            ),
            'checkout_logo' => array(
                'value' => '',
                'label' => __('Checkout Logo', 'wp-payment-form'),
                'type' => 'image'
            ),
            'is_pro_item' => array(
                'value' => 'yes',
                'label' => __('PayPal', 'wp-payment-form'),
            ),
        );
        return $this->mapper($defaults, $settings);
    }

    public function maybeHasSubscription($originalArgs, $submission, $form_data, $paymentMode, $hasSubscriptions, $paymentItems)
    {
        if (!$hasSubscriptions) {
            return $originalArgs;
        }

        $subscriptionModel = new Subscription();
        $subscriptions = $subscriptionModel->getSubscriptions($submission->id);
        $validSubscriptions = [];
        foreach ($subscriptions as $subscriptionItem) {
            if ($subscriptionItem->recurring_amount) {
                $validSubscriptions[] = $subscriptionItem;
            }
        }

        if (!$validSubscriptions || count($validSubscriptions) > 1) {
            // PayPal Standard does not support more than 1 subscriptions
            // We may add paypal express later for this on.
            return $originalArgs;
        }

        // We just need the first subscriptipn
        $subscription = $validSubscriptions[0];

        if (!$subscription->recurring_amount) {
            return $originalArgs;
        }


        unset($originalArgs['item_name_1']);
        unset($originalArgs['quantity_1']);
        unset($originalArgs['amount_1']);
        unset($originalArgs['cmd']);


        // $originalArgs['notify_url'] .= '&wpf_subscription_id='.$subscription->id;
        $originalArgs['custom'] = $subscription->id;

        // Form that subscription we have to create a transaction as parent

        $customerName = $submission->customer_name;
        $names = explode(' ', $customerName, 2);

        if (count($names) == 2) {
            $firstName = $names[0];
            $lastName = $names[1];
        } else {
            $firstName = $customerName;
            $lastName = '';
        }

        $paypal_args = array(
            'first_name' => $firstName,
            'last_name' => $lastName,
            'invoice' => $subscription->id,
            'no_shipping' => '1',
            'shipping' => '0',
            'no_note' => '1',
            'rm' => '2',
            'cbt' => get_bloginfo('name'),
            'sra' => '1',
            'src' => '1',
            'cmd' => '_xclick-subscriptions'
        );

        $initial_amount = round($subscription->initial_amount / 100, 2);
        $recurring_amount = round($subscription->recurring_amount / 100, 2);

        if ($subscription->quantity) {
            $recurring_amount = $recurring_amount * intval($subscription->quantity);
        }

        $initial_amount_tax = 0;
        foreach ($paymentItems as $key => $paymentItem) {
            if ($key == 'tax_payment_input' && $paymentItem['signup_fee_tax'] == 'yes') {
                $initial_amount_tax = Arr::get($paymentItem, 'signup_fee', 0);
                if ($initial_amount_tax > 0) {
                    $initial_amount_tax = round($initial_amount_tax / 100, 2);
                }
                break;
            }
        }

        if ($initial_amount) {
            $paypal_args['a1'] = round($initial_amount + $initial_amount_tax + $recurring_amount, 2);
            $paypal_args['p1'] = 1;
        } elseif ($subscription->trial_days) {
            $paypal_args['a1'] = 0;
            $paypal_args['p1'] = $subscription->trial_days;
            $paypal_args['t1'] = 'D';
        }

        $paypal_args['a3'] = $recurring_amount;

        $paypal_args['item_name'] = $subscription->item_name . ' (' . $subscription->plan_name . ') - ' . $subscription->form_id;

        $intervalCount = Arr::get($subscription->original_plan, 'interval_count', 1);
        // frequency_interval is the number of billing periods that make up one billing cycle
        $paypal_args['p3'] = $intervalCount;

        switch ($subscription->billing_interval) {
            case 'daily':
                $paypal_args['t3'] = 'D';
                break;
            case 'week':
                $paypal_args['t3'] = 'W';
                break;
            case 'month':
                $paypal_args['t3'] = 'M';
                break;
            case 'year':
                $paypal_args['t3'] = 'Y';
                break;
            case 'fortnight':
                $paypal_args['t3'] = 'W';
                $paypal_args['p3'] = 2;
                break;
            case 'quarter':
                $paypal_args['t3'] = 'M';
                $paypal_args['p3'] = 3;
                break;
            case 'half_year':
                $paypal_args['t3'] = 'M';
                $paypal_args['p3'] = 6;
                break;
        }

        if ($initial_amount) {
            $paypal_args['t1'] = $paypal_args['t3'];
        }

        $intervalCountFromOriginalPlan = Arr::get($subscription->original_plan, 'interval_count', 1);

        $billingInterval = $subscription->billing_interval;
        if ($intervalCountFromOriginalPlan > 1 ) {
            if ($billingInterval == 'daily' && $intervalCountFromOriginalPlan > 365) {
                return new \WP_Error('interval_count', __('Daily interval count can not be more than 365.', 'wp-payment-form'));
            } 
            else if ($billingInterval == 'week' && $intervalCountFromOriginalPlan > 52) {
                 return new \WP_Error('interval_count', __('Weekly interval can not be more than 52.', 'wp-payment-form')); 
            } else if ($billingInterval == 'month' && $intervalCountFromOriginalPlan > 12) {
                return new \WP_Error('interval_count', __('Monthly interval count can not be more than 12.', 'wp-payment-form'));
            } else if ($billingInterval == 'year' && $intervalCountFromOriginalPlan > 1) {
                return new \WP_Error('interval_count', __('Yearly interval count can not be more than 1.', 'wp-payment-form'));
            } else if ($billingInterval == 'fortnight' && $intervalCountFromOriginalPlan > 26) {
               return new \WP_Error('interval_count', __('Fortnightly interval count can not be more than 52.', 'wp-payment-form'));
            } else if ($billingInterval == 'quarter' && $intervalCountFromOriginalPlan > 4) {
                 return new \WP_Error('interval_count', __('Quarterly interval count can not be more than 4.', 'wp-payment-form'));
            } else if ($billingInterval == 'half_year' && $intervalCountFromOriginalPlan > 2) {
                 return new \WP_Error('interval_count', __('Half yearly interval count can not be more than 2.', 'wp-payment-form'));
            }
         }

        if ($intervalCountFromOriginalPlan > 1) {
            $paypal_args['p3'] = $paypal_args['p3'] * $intervalCountFromOriginalPlan;
        }

        if ($subscription->bill_times > 1) {
            if ($initial_amount) {
                $subscription->bill_times = $subscription->bill_times - 1;
            }
            $billTimes = $subscription->bill_times <= 52 ? absint($subscription->bill_times) : 52;
            $paypal_args['srt'] = $billTimes;
        }

        $orderItemModel = new OrderItem();
        $discountItems = $orderItemModel->getDiscountItems($submission->id);
        if ($discountItems) {
            $discountTotal = 0;
            foreach ($discountItems as $discountItem) {
                $discountTotal += intval($discountItem->line_total);
            }
            if (isset($paypal_args['a3'])) {
                $paypal_args['a3'] -= round($discountTotal / 100, 2);
            }
        }

        return wp_parse_args($paypal_args, $originalArgs);
    }

    public function formatSubscriptionItems($items, $transaction)
    {
        $paymentResponse = $transaction->payment_note;
        $items[] = array(
            'item_name' => Arr::get($paymentResponse, 'transaction_subject'),
            'status' => $transaction->status,
            'payment_total' => $transaction->payment_total,
            'transaction_id' => $transaction->charge_id,
            'payment_method' => 'paypal',
            'view_url' => $this->getPayPalTransactionUrl($transaction)
        );
        return $items;
    }

    private function isTestMode($formId = false)
    {
        $settings = $this->getPaypalSettings($formId);
        return $settings['payment_mode'] != 'live';
    }

    public function validateSubmittedItems($paymentItems, $formattedElements, $form_data, $subscriptionItems)
    {
        $singleItemTotal = 0;
        foreach ($paymentItems as $paymentItem) {
            if (isset($paymentItem['recurring_tax']) && $paymentItem['recurring_tax'] == 'yes') {
                continue;
            }
            if ($paymentItem['line_total']) {
                $singleItemTotal += $paymentItem['line_total'];
            }
        }

        $validSubscriptions = [];
        foreach ($subscriptionItems as $subscriptionItem) {
            if ($subscriptionItem['recurring_amount']) {
                $validSubscriptions[] = $subscriptionItem;
            }
        }

        if ($singleItemTotal && count($validSubscriptions)) {
            wp_send_json_error(array(
                'message' => __('PayPal does not support subscriptions payment and Single Amount Payment at one request', 'wp-payment-form-pro'),
                'payment_error' => true
            ), 423);
        }

        if (count($validSubscriptions) > 2) {
            wp_send_json_error(array(
                'message' => __('PayPal does not support multiple subscriptions at one request', 'wp-payment-form-pro'),
                'payment_error' => true
            ), 423);
        }

        return $paymentItems;
    }
}
