<?php

namespace WPPayFormPro\GateWays\Billplz;

use WPPayFormPro\GateWays\BasePaymentMethod;
use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Services\AccessControl;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class BillplzSettings extends BasePaymentMethod
{

    public function __construct()
    {
    /**
     * Automatically create global payment settings page
     * @param  String: key, title, routes_query, 'logo')
     */
        parent::__construct(
            'billplz',
            'Billplz',
            [],
            WPPAYFORM_URL . 'assets/images/gateways/billplz.svg'
        );

    }

    /**
     * @function mapperSettings, To map key => value before store
     * @function validateSettings, To validate before save settings
     */

    public function init()
    {
        add_filter('wppayform_payment_method_settings_mapper_'.$this->key, array($this, 'mapperSettings'));
        add_filter('wppayform_payment_method_settings_validation_' . $this->key, array($this, 'validateSettings'), 10, 2);
    }

    /**
     * @return Array of global fields
     */
    public function globalFields() {
        return array(
            'is_active' => array(
                'value' => 'no',
                'label' => __('Enable/Disable', 'wp-payment-form'),
            ),
            'payment_mode' => array(
                'value' => 'test',
                'label' => __('Payment Mode', 'wp-payment-form'),
                'options' => array(
                    'test' => __('Test Mode', 'wp-payment-form'),
                    'live' => __('Live Mode', 'wp-payment-form')
                ),
                'type' => 'payment_mode'
            ),
            'checkout_type' => array(
                'value' => 'modal',
                'label' => __('Checkout Logo', 'wp-payment-form'),
                'options' => ['modal' => 'Modal checkout style', 'hosted' => 'Hosted checkout style'],
            ),
            'live_collection_id' => array(
                'value' => 'live',
                'label' => __('Live Collection Id', 'wp-payment-form'),
                'type' => 'live_pub_key',
                'placeholder' => __('Live Collection Id', 'wp-payment-form')
            ),
            'collection_id' => array(
                'value' => 'test',
                'label' => __('Test Collection Id', 'wp-payment-form'),
                'type' => 'test_pub_key',
                'placeholder' => __('Test Collection Id', 'wp-payment-form')
            ),
            'live_secret_key' => array(
                'value' => '',
                'label' => __('Live Secret Key', 'wp-payment-form'),
                'type' => 'live_secret_key',
                'placeholder' => __('Live Secret Key', 'wp-payment-form')
            ),
            'secret_key' => array(
                'value' => '',
                'label' => __('Test Secret Key', 'wp-payment-form'),
                'type' => 'test_secret_key',
                'placeholder' => __('Test Secret Key', 'wp-payment-form')
            ),
            'verify_keys' => array(
                'type' => 'verify_button',
                'label' => 'Verify and Save Settings',
                'req_type' => 'get',
                'method' => 'billplz'
            ),
            'desc' => array(
                'value' => '<div> <p style="color: #d48916;">NB: Please add Email, Name and mobile field on your form to get payment data correctly.</p> <p>See our <a href="https://paymattic.com/docs/how-to-configure-billplz-in-wordpress-with-paymattic/" target="_blank" rel="noopener">documentation</a> to get more information about Billplz setup.</p> </div>',
                'label' => __('Payrexx IPN Settings', 'wp-payment-form'),
                'type' => 'html_attr'
            ),
            'is_pro_item' => array(
                'value' => 'yes',
                'label' => __('PayPal', 'wp-payment-form'),
            ),
        );
    }

    /**
     * @return Array of default fields
     */
    public static function settingsKeys()
    {
        return array(
            'is_active' => 'no',
            'payment_mode' => 'test',
            'checkout_type' => 'hosted',
            'collection_id' => '',
            'secret_key' => '',
            'live_collection_id' => '',
            'live_secret_key' => ''
        );
    }
    
    /**
     * @return Array of global_payments settings fields
     */
    public function getPaymentSettings()
    {
        $settings = $this->mapper(
            $this->globalFields(), 
            static::getSettings()
        );

        return array(
            'settings' => $settings,
            'is_key_defined' => false
        );
    }

    public static function getSettings()
    {
        $settings = get_option('wppayform_payment_settings_billplz', array());
        return wp_parse_args($settings, static::settingsKeys());
    }

    public function mapperSettings ($settings)
    {
        return $this->mapper(
            static::settingsKeys(), 
            $settings, 
            false
        );
    }

    // should modify on save settings
    public function getCollections($settings)
    {
        $isLive = Arr::get($settings, 'payment_mode') == 'live' ? true : false;
        $keys = static::ApiRoutes($isLive, $settings);
        $collectionId = $keys['collection_id'];
        $Api = new API($keys);

        if ($collectionId == '') {
            wp_send_json_error(array(
                'message' => __('Collection Id is required', 'wp-payment-form-pro')
            ), 423);
        }

        return $Api->getCollection($collectionId);
    }

    public function validateSettings($errors, $settings)
    {
        AccessControl::checkAndPresponseError('set_payment_settings', 'global');
        $mode = Arr::get($settings, 'payment_mode');

        if ($mode == 'test') {
            if (empty(Arr::get($settings, 'collection_id')) || empty(Arr::get($settings, 'secret_key'))) {
                $errors['collection_id'] = __('Please provide Test collection Id and Secret key', 'wp-payment-form-pro');
            }
        }

        if ($mode == 'live') {
            if (empty(Arr::get($settings, 'live_collection_id')) || empty(Arr::get($settings, 'live_secret_key'))) {
                $errors['live_collection_id'] = __('Please provide Live Collection Id and Secret key', 'wp-payment-form-pro');
            }
        }

        return $errors;
    }

    public static function isLive($formId = false)
    {
        $settings = self::getSettings();
        $mode = Arr::get($settings, 'payment_mode');
        return $mode == 'live';
    }

    public static function ApiRoutes($isLive, $settings)
    {
        if ($isLive) {
            return array(
                'collection_id' => Arr::get($settings, 'live_collection_id'),
                'api_secret' => Arr::get($settings, 'live_secret_key'),
                'api_path' => 'https://www.billplz.com/api/'
            );
        }
        return array(
            'collection_id' => Arr::get($settings, 'collection_id'),
            'api_secret' => Arr::get($settings, 'secret_key'),
            'api_path' => 'https://www.billplz-sandbox.com/api/'
        );
    }

    public static function getApiKeys($formId = false)
    {
        return static::ApiRoutes(
            static::isLive($formId),
            static::getSettings()
        );
    }
}
