<?php

namespace WPPayFormPro\GateWays\Billplz;

if (!defined('ABSPATH')) {
    exit;
}

use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Models\Transaction;
use WPPayForm\App\Models\Submission;
use WPPayForm\App\Models\Form;
use WPPayForm\App\Services\PlaceholderParser;
use WPPayForm\App\Services\ConfirmationHelper;
use WPPayForm\GateWays\Billplz\QueryHelper\Gateway;

class BillplzProcessor
{
    public $method = 'billplz';

    protected $form;

    public function init()
    {
        new BillplzElement();
        (new BillplzSettings())->init();

        add_filter('wppayform/choose_payment_method_for_submission', array($this, 'choosePaymentMethod'), 10, 4);
        add_action('wppayform/form_submission_make_payment_' . $this->method, array($this, 'makeFormPayment'), 10, 6);
        add_filter('wppayform/entry_transactions_' . $this->method, array($this, 'addTransactionUrl'), 10, 2);
        add_filter('wppayform/submitted_payment_items_' . $this->method, array($this, 'validateSubscription'), 10, 4);

        add_action('wppayform_payment_frameless_billplz', array($this, 'handlePaidFromRedirectUrl'), 10, 1);

        add_action('wpf_ipn_endpoint_' . $this->method, function () {
            $this->verifyIPN();
            exit(200);
        });

        // Optional hook to verify before save
        add_filter('wppayform_verify_payment_keys_' . $this->method, array($this, 'verifyKeys'), 10, 2);
    }


    /**
     * @return Array
     * array(status, message)
     */
    public function verifyKeys($response, $keys)
    {
        $settingsInstance = (new BillplzSettings());
        
        $settings = $settingsInstance->mapperSettings(
            Arr::get($keys, 'settings', [])
        );
        $res = $settingsInstance->getCollections($settings);

        if (is_wp_error($res)) {
            return array(
                'status' => false,
                'message' => 'Validation failed/Access denied'
            );
        }

        if ( isset($res['status']) && $res['status'] === 'active' ) {
            return array(
                'status' => true,
                'message' => $res['title'],
                'data' => $res
            );
        } else {
            return array(
                'status' => false,
                'message' => 'Access denied'
            );
        }
    }

    public function verifyIPN()
    {
        $payId = Arr::get($_POST, 'id');
        $keys = BillplzSettings::getApiKeys();
        $Api = new API($keys);

        $vendorTransaction = $Api->validation($payId);

        if (!$vendorTransaction || empty($vendorTransaction)) {
            return;
        }

        // We already have the transactionId as reference_2
        $transactionId = Arr::get($vendorTransaction, 'reference_2');

        $transaction = (new Transaction())->getTransaction($transactionId);
        $this->handleStatus($transaction, $vendorTransaction);
    }

    public function handleStatus($transaction, $vendorTransaction)
    {
        // check if already paid return
        if (!$transaction || $transaction->payment_method != $this->method || 'paid' === $transaction->state) {
            return;
        }

        do_action('wppayform/form_submission_activity_start', $transaction->form_id);

        $status = $vendorTransaction['state'];

        $updateData = array(
            'charge_id'     => Arr::get($vendorTransaction, 'id'),
            'payment_total' => intval(Arr::get($vendorTransaction, 'paid_amount')),
            'status'        => $status,
            'payment_note'  => maybe_serialize($vendorTransaction),
        );

        $this->updatePaymentStatus($status, $updateData, $transaction);
    }

    public function updatePaymentStatus($status, $updateData, $transaction)
    {
        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($transaction->submission_id);
        $submissionData = array(
            'payment_status' => $status,
            'updated_at' => current_time('Y-m-d H:i:s')
        );

        $submissionModel->where('id', $transaction->submission_id)->update($submissionData);
        $transactionModel = new Transaction();
        $updateData['updated_at'] = current_time('Y-m-d H:i:s');
        $transaction = $transactionModel->getTransaction($transaction->id);
        $transactionModel->updateTransaction($transaction->id, $updateData);

        do_action('wppayform_log_data', [
            'form_id' => $transaction->form_id,
            'submission_id' => $transaction->submission_id,
            'type' => 'info',
            'created_by' => 'PayForm Bot',
            'content' => sprintf(__('Transaction Marked as %s and Billplz Transaction ID: %s', 'wp-payment-form-pro'), $status, $updateData['charge_id'])
        ]);

        if ($status === 'paid') {
            do_action('wppayform/form_payment_success_billplz', $submission, $transaction, $transaction->form_id, $updateData);
            do_action('wppayform/form_payment_success', $submission, $transaction, $transaction->form_id, $updateData);
        } else {
            do_action('wppayform/form_payment_failed_billplz', $submission->form_id, $submission, $transaction);
            do_action('wppayform/form_payment_failed', $submission, $submission->form_id, $transaction, 'billplz');
        }
    }

    public function handlePaidFromRedirectUrl($data)
    {
        $billplzData = $data['billplz'];
        $payId = Arr::get($billplzData, 'id');
        $keys = BillplzSettings::getApiKeys();

        $Api = new API($keys);
        $vendorTransaction = $Api->validation($payId);
        if (!$vendorTransaction || empty($vendorTransaction)) {
            return;
        }

        $transactionId = Arr::get($vendorTransaction, 'reference_2');
        $transaction = (new Transaction())->getTransaction($transactionId);
        $this->handleStatus($transaction, $vendorTransaction);
    } 

    public function choosePaymentMethod($paymentMethod, $elements, $formId, $form_data)
    {
        if ($paymentMethod) {
            // Already someone choose that it's their payment method
            return $paymentMethod;
        }
        // Now We have to analyze the elements and return our payment method
        foreach ($elements as $element) {
            if ((isset($element['type']) && $element['type'] == 'billplz_gateway_element')) {
                return 'billplz';
            }
        }
        return $paymentMethod;
    }

    public function makeFormPayment($transactionId, $submissionId, $form_data, $form, $hasSubscriptions)
    {
        $paymentMode = $this->getPaymentMode();
        $transactionModel = new Transaction();
        if ($transactionId) {
            $transactionModel->updateTransaction($transactionId, array(
                'payment_mode' => $paymentMode
            ));
        }
        $transaction = $transactionModel->getTransaction($transactionId);

        $submission = (new Submission())->getSubmission($submissionId);

        $this->handleRedirect($transaction, $submission, $form_data, $form, $paymentMode);
    }

    public function handleRedirect($transaction, $submission, $form_data, $form, $methodSettings)
    {
        $globalSettings = BillplzSettings::getApiKeys();

        if (!isset($globalSettings['api_secret'])) {
            return;
        }

        $transaction['hash'] = $submission->submission_hash;

        $paymentIntent = $this->makePaymentData($transaction, $submission, $form_data, $form);

        if (is_wp_error($paymentIntent)) {
            wp_send_json_error(array(
                'message' => $paymentIntent->get_error_message()
            ), 422);
        }

        do_action('wppayform_log_data', [
            'form_id' => $form->ID,
            'submission_id' => $submission->id,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'title' => 'Billplz Payment Redirect',
            'content' => 'User redirect to Billplz for completing the payment'
        ]);

        $redirectUrl = wp_sanitize_redirect(Arr::get($paymentIntent, 'url'));

        wp_send_json_success([
            'message' => __('You are redirecting to Billplz.com to complete the purchase. Please wait while you are redirecting....', 'wp-payment-form-pro'),
            'call_next_method' => 'normalRedirect',
            'redirect_url' => $redirectUrl
        ], 200);
    }

    public function callbackUrl()
    {
        return add_query_arg([
            'wpf_payment_api_notify' => '1',
            'payment_method'=> 'billplz'
        ], site_url('index.php'));
    }

    public function makePaymentData($transaction, $submission, $form_data, $form)
    {
        $submissionModel = new Submission();
        $entries = $submissionModel->getParsedSubmission($submission);

        $currency = $transaction->currency;

        if ('MYR' !== $currency) {
            return new \WP_Error('currency_not_supported', __('Billplz only support MYR currency', 'wp-payment-form-pro'));
        }

        $success_url = $this->redirectUrl($form->ID, $submission);
        $webhook_url = $this->callbackUrl();

        $metadata = [];
        foreach ($entries as $label => $entry) {
            $value = $entry['value'];
            if (is_string($value) && $value) {
                $metadata[$entry['type']] = $value;
            }
        }

        $args = [
            'description' => $form->post_title,
            'amount' => floatval($transaction->payment_total),
            'name' => Arr::get($metadata, 'customer_name', ''),
            'email' => Arr::get($metadata, 'customer_email', ''),
            'callback_url' => $webhook_url,
            'redirect_url' => $success_url,
            'reference_1_label' => 'hash',
            'reference_1' => $transaction->hash,
            'reference_2_label' => 'transaction_id',
            'reference_2' => $transaction->id,
        ];

        // dd($args, 'in BillplzProcessor');
        if ($phone = Arr::get($metadata, 'phone', false)) {
            $args['mobile'] = str_replace(' ', '', $phone);
        }

        $keys = BillplzSettings::getApiKeys($transaction->form_id);
        $Api = new API($keys);

        return $Api->createBill($args);
    }

    public function redirectUrl($formId, $submission)
    {
        // Check If the form settings have success URL
        $confirmation = Form::getConfirmationSettings($formId);
        $confirmation = ConfirmationHelper::parseConfirmation($confirmation, $submission);
        if (
            ($confirmation['redirectTo'] == 'customUrl' && $confirmation['customUrl']) ||
            ($confirmation['redirectTo'] == 'customPage' && $confirmation['customPage'])
        ) {
            if ($confirmation['redirectTo'] == 'customUrl') {
                $url = $confirmation['customUrl'];
            } else {
                $url = get_permalink(intval($confirmation['customPage']));
            }
            $url = add_query_arg(array(
                'payment_method' => 'billplz'
            ), $url);
            $url = PlaceholderParser::parse($url, $submission);
            return wp_sanitize_redirect($url);
        }
        // now we have to check for global Success Page
        $globalSettings = get_option('wppayform_confirmation_pages');
        if (isset($globalSettings['confirmation']) && $globalSettings['confirmation']) {
            $url = add_query_arg(array(
                'wpf_submission' => $submission->submission_hash,
                'wppayform_payment' => $submission->id,
                'payment_method' => 'billplz'
            ), get_permalink(intval($globalSettings['confirmation'])));
            return wp_sanitize_redirect($url);
        }
        // In case we don't have global settings
        $url = add_query_arg(array(
            'wpf_submission' => $submission->submission_hash,
            'wppayform_payment' => $submission->id,
            'payment_method' => 'billplz'
        ), home_url());
        return wp_sanitize_redirect($url);
    }

    protected function getPaymentMode($formId = false)
    {
        $isLive = BillplzSettings::isLive($formId);
        if ($isLive) {
            return 'live';
        }
        return 'test';
    }

    public function addTransactionUrl($transactions, $submissionId)
    {
        $transPath = 'https://www.billplz-sandbox.com/bills/';
        if (BillplzSettings::isLive()) {
            $transPath = 'https://billplz.com/bills/';
        };

        foreach ($transactions as $transaction) {
            if ($transaction->charge_id) {
                $transaction->transaction_url =  $transPath . $transaction->charge_id;
            }
        }
        return $transactions;
    }


    public function getLastTransaction($submissionId)
    {
        $transactionModel = new Transaction();
        $transaction = $transactionModel->where('submission_id', $submissionId)
            ->first();
        return $transaction;
    }

    public function validateSubscription($paymentItems, $formattedElements, $form_data, $subscriptionItems)
    {
        wp_send_json_error(array(
            'message' => __('Billplz doesn\'t support subscriptions right now', 'wp-payment-form-pro'),
            'payment_error' => true
        ), 423);
    }
}
