<?php
namespace WPPayFormPro\Classes\EmailNotification;

use WPPayForm\App\Models\SubmissionActivity;
use WPPayForm\App\Models\Subscription;
use WPPayFormPro\Classes\EmailNotification\ResendEmail;
use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Models\Form;

if (!defined('ABSPATH')) {
    exit;
}

class ReminderEmailScheduler
{
    const REMINDER_HOOK = 'wppayform/send_reminder_email';
    const REMINDER_SENT_META_KEY = 'wppayform/wpf_reminder_email_sent';
    public function register()
    {
        // Hook into WordPress initialization  
        add_action('wppayform/daily_reminder_task', [$this, 'initReminderEmailScheduler']);
        
        // Register hook for sending reminder emails  
        add_action(self::REMINDER_HOOK, [$this, 'processReminderEmail'], 10, 4);
        
        // Add hook for cleaning up old notification logs
        add_action('wppayform/remove_old_notification_logs', [$this, 'removeOldScheuledActionLogs']);
    }

    public function initReminderEmailScheduler()
    {
        // Schedule daily check for upcoming subscription reminders  
        $this->getUpcomingSubscriptions();
        do_action('wppayform/remove_old_notification_logs');
    }


    public function getUpcomingSubscriptions()
    {
        $allFroms = new Form();
        $allFormIds = $allFroms->getFormIds();
        if (empty($allFormIds)) {
            error_log("No form IDs found");
            return [];
        }
        foreach ($allFormIds as $form) {
            $formId = Arr::get($form, 'id', 0);  
            if (!$formId) {
                continue;
            }

            $formEmailSettings = $this->formEmailSettings($formId);

            if (!$formEmailSettings) {
                // error_log("Email notifications disabled for Form ID: {$formId}");
                continue; // Skip this form
            }

            $subscriptions = new Subscription();
            $getSubscriptions = $subscriptions->getSubscriptionsById($formId);
            
            if (!is_array($getSubscriptions) && empty($getSubscriptions)) {
                continue;
            }

            foreach ($getSubscriptions as $subscription) {
                $submissionId = Arr::get($subscription, 'submission_id', '');

                $nextBillingDate = $this->nextBillingDate($subscription);
                $today = current_time('Y-m-d');
                $notifications = $this->getEmailNotifications($formId);
                if (empty($notifications)) {
                    continue;
                }

                foreach ($notifications as $notificationId => $notification) {
                    $emailTitle = Arr::get($notification, 'title', '');
                    $status = Arr::get($notification, 'status', '');
                    $sendingAction = Arr::get($notification, 'sending_action', '');
                    $reminderDays = Arr::get($notification, 'reminder_days', 0);
                    $reminderTime = Arr::get($notification, 'reminder_time', '');
                    $calculateBillingDate = $this->calculateBillingDate($nextBillingDate, $reminderDays, $reminderTime);
                    
                    if (date('Y-m-d', strtotime($calculateBillingDate)) !== $today || $status !== 'active' || $sendingAction !== 'wppayform/send_reminder_email_to_user') {
                        continue;
                    }

                    $this->scheduleReminderAction($formId, $submissionId, $notificationId, $calculateBillingDate, $emailTitle);
                }
            }
        }
    }

    private function scheduleReminderAction($formId, $submissionId, $notificationId, $nextReminderDate, $emailTitle)
    {
        $timestamp = strtotime($nextReminderDate);
        $currentDate = date('Y-m-d');  

        // Check if an action is already scheduled for this submission on the same day  
        $existingAction = as_next_scheduled_action(  
            self::REMINDER_HOOK,  
            [  
                'form_id' => $formId,  
                'submission_id' => $submissionId,  
                'notification_id' => $notificationId,  
                'email_title' => $emailTitle  
            ],  
            'wppayform-scheduler-task'  
        );  

        // If an action exists and is scheduled for today, return early  
        if ($existingAction && date('Y-m-d', $existingAction) === $currentDate) {  
            // error_log("Reminder already scheduled today for Submission ID: {$submissionId}");  
            return false;  
        }  
        
        $actionResult = as_schedule_single_action(
            $timestamp,
            self::REMINDER_HOOK,
            [
                'form_id' => $formId,
                'submission_id' => $submissionId,
                'notification_id' => $notificationId,
                'email_title' => $emailTitle
            ],
            'wppayform-scheduler-task'
        );
        if ($actionResult) {
            error_log("Reminder email scheduled successfully for Submission ID: {$submissionId}, Notification ID: {$notificationId}");
        } else {
            error_log("Failed to schedule reminder email for Submission ID: {$submissionId}, Notification ID: {$notificationId}");
        }
    }

    public function processReminderEmail($formId, $submissionId, $notificationId, $emailTitle)
    {
        try {
            $info = [
                'type' => 'subscription_reminder',
                'notification_id' => $notificationId,
            ];

            // Send reminder email  
            $resendEmail = new ResendEmail();
            $isSent = $resendEmail->initEmailHooks($formId, $submissionId, $info);

            if ($isSent) {
                $this->logReminderEmailActivity($formId, $submissionId, $emailTitle, 'sent');
            } else {
                $this->logReminderEmailActivity($formId, $submissionId, $emailTitle, 'failed');
            }
        } catch (\Exception $e) {
            error_log("Error sending reminder email: " . $e->getMessage());
        }
    }

    private function getEmailNotifications($formId)
    {
        $notifications = get_post_meta($formId, 'wpf_email_notifications', true);
        return $notifications;
    }

    public static function nextBillingDate($subscription)
    {
        $subscriptionCreatedAt = Arr::get($subscription, 'subscription_date', '');
        $billingInterval = Arr::get($subscription, 'billing_interval', '');

        $intervalMap = [
            'daily' => '1',
            'week' => '7',
            'fortnight' => '14',
            'month' => '30',
            'quarter' => '90',
            'half_year' => '180',
            'year' => '365',
        ];

        $billingInterval = Arr::get($intervalMap, $billingInterval, '1');
        $billCount = Arr::get($subscription, 'bill_count', 0);
        $trialDays = Arr::get($subscription, 'trial_days', 0);
        $nextD = strtotime($subscriptionCreatedAt . ' + ' . $billCount * $billingInterval . ' days');

        if ($trialDays > 0) {
            $nextD = strtotime($nextD . ' + ' . $trialDays . ' days');
        }
        return Date('Y-m-d H:i:s',  $nextD);
    }

    private function getNotifications($formId)
    {
        $notifications = $this->getEmailNotifications($formId);
        
        if (!is_array($notifications) || empty($notifications)) {
            return 0;
        }
        $result = [];
        foreach ($notifications as $itemId => $item) {
            $status = Arr::get($item, 'status', '');
            $sendingAction = Arr::get($item, 'sending_action', '');
            $reminderDays = Arr::get($item, 'reminder_days', 1);
            if ($status === 'active' && $sendingAction === 'wppayform/send_reminder_email_to_user') {
                $result[] = [
                    'reminder_days' => $reminderDays,
                    'notification_id' => $itemId
                ];
            }
        }
        return $result;
    }

    private function calculateBillingDate($nextReminderDate, $reminderDays, $reminderTime)
    {
        if ($reminderTime === 'before') {
            $nextPaymentDate = strtotime($nextReminderDate . " - $reminderDays days");
        } else if ($reminderTime === 'after') {
            $nextPaymentDate = strtotime($nextReminderDate . " + $reminderDays days");
        } else {
            $nextPaymentDate = strtotime($nextReminderDate);
        }
        return date('Y-m-d H:i:s', $nextPaymentDate);
    }


    private function formEmailSettings($formId)
    {
        $notifications = $this->getEmailNotifications($formId);
        if (!is_array($notifications) || empty($notifications)) {
            return false;
        }
        foreach ($notifications as $item) {
            $status = Arr::get($item, 'status', '');
            $sendingAction = Arr::get($item, 'sending_action', '');
            if ($status === 'active' && $sendingAction === 'wppayform/send_reminder_email_to_user') {
                return true;
            }
        }
        return false;
    }

    private function logReminderEmailActivity($formId, $submissionId, $emailTitle, $status)
    {  
        SubmissionActivity::createActivity(array(
            'form_id' => $formId,
            'submission_id' => $submissionId,
            'type' => 'activity',
            'created_by' => 'Paymattic BOT',
            'content' => "Reminder email {$status} for {$emailTitle}."
        ));
    }

    public function removeOldScheuledActionLogs($group_slug = 'wppayform-scheduler-task', $days_old = 7)
    {
        global $wpdb;
        // Get the timestamp for 7 days ago
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days_old} days"));

        // Get the group ID
        $group_id = $wpdb->get_var($wpdb->prepare(
            "SELECT group_id FROM {$wpdb->prefix}actionscheduler_groups WHERE slug = %s",
            $group_slug
        ));

        if (!$group_id) {
            return false; // Group not found
        }
        // Delete old actions and their associated logs
        $deleted = $wpdb->query($wpdb->prepare("
            DELETE a, l
            FROM {$wpdb->prefix}actionscheduler_actions a
            LEFT JOIN {$wpdb->prefix}actionscheduler_logs l ON a.action_id = l.action_id
            WHERE a.group_id = %d
            AND a.status IN ('complete', 'failed')
            AND a.scheduled_date_gmt < %s", $group_id, $cutoff_date));

        // Clean up orphaned claims
        $wpdb->query("
            DELETE c
            FROM {$wpdb->prefix}actionscheduler_claims c
            LEFT JOIN {$wpdb->prefix}actionscheduler_actions a ON c.claim_id = a.claim_id
            WHERE a.action_id IS NULL");

        return $deleted;
    }

    private function errorHandler($code, $message, $data = array())
    {
        $error = new \WP_Error($code, $message, $data);
        wp_die($error);
    }
}
