<?php

namespace WPPayFormPro\Classes\Dashboard;

use EDD\Compat\Customer;
use WPPayForm\Framework\Support\Arr;
use WPPayForm\App\Modules\Builder\PaymentReceipt;
use WPPayForm\App\App;
use WPPayFormPro\Classes\ReportsController;
use WPPayForm\App\Models\Submission;
use WPPayForm\App\Models\Customers;
use WPPayForm\App\Http\Controllers\SubmissionController;
use WPPayFormPro\Classes\EmailNotification\EmailHandler;
use WPPayForm\App\Models\SubmissionActivity;
use WPPayForm\App\Services\PlaceholderParser;
use WPPayForm\App\Services\AccessControl;

use WPPayForm\App\Models\Subscription;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Ajax Handler Class
 * @since 1.0.0
 */
class Render
{
    public function render()
    {
        // Localized all js and css file
        $this->localizedScript();
        // Check the user is log in or not
        $current_user = wp_get_current_user()->get('user_email');
        $is_paymattic_user = in_array('paymattic_user', wp_get_current_user()->roles) || in_array('administrator', wp_get_current_user()->roles);

        if (!$current_user) {
            // if user not login then return with message
            return $this->getUnAuthorizedHtmlContent();
        } 
        // else if (!$is_paymattic_user) {
        //     // if user not paymattic user then return with message

        //     return '<h3 style="padding: 20px"> You are not authorized to access this page. </h3>';
        // }
        $permittedRoles = get_option('_wppayform_dashboard_permitted_roles', []);
        $currentUser = wp_get_current_user();  
        $userRoles   = (array) $currentUser->roles;  
        if (empty(array_intersect($userRoles, $permittedRoles)) && !$is_paymattic_user && !current_user_can('manage_options')) {
            return '<h3 style="padding:20px;">You are not authorized to access this page.</h3>';
        }
        // Get user permissions from options that admin provide this user
        $paymatticUserPermissions = $this->getUserPermissions();

        $submissionsItems = $this->getAllSubmissions($current_user);
        // get user total spent amount with specific currency
        $payment_total = $this->sumAmountsByCurrency($submissionsItems['entries']);

        $submissionsItems['subscriptions'] = $this->getSubscriptionItems($submissionsItems);
		$menus = static::menus($paymatticUserPermissions);

        // trigger action user dashboard loading
        do_action('wppayform/before_user_dashboard_loaded', $current_user, $submissionsItems, $payment_total, $paymatticUserPermissions);

        ob_start();
        App::make('view')->render('user.dashboard', [
            'donationItems' => $submissionsItems,
            'payment_total' => $payment_total,
            'permissions' => $paymatticUserPermissions,
	        'menus' => $menus,
        ]);

        $view = ob_get_clean();
        return $view;
    }

	public static function menus($permissions)
	{
		$read_entry = Arr::get($permissions, 'read_entry');
		$read_subscription_entry = Arr::get($permissions, 'read_subscription_entry');
		$can_sync_subscription_billings = Arr::get($permissions, 'can_sync_subscription_billings');
		$cancel_subscription = Arr::get($permissions, 'cancel_subscription');

		if ($read_entry !== 'yes' && $read_subscription_entry !== 'yes') {
			return [];
		}

		$menus = [
			[
				'name' => __('Dashboard', 'wp-payment-form'),
				'slug' => 'wpf-user-dashboard',
				'icon' => 'dashicons dashicons-admin-home'
			],
		];

		if ($read_subscription_entry === 'yes') {
			$menus[] = [
				'name' =>  __('Manage Subscription', 'wp-payment-form'),
				'slug' => 'wpf-subscription',
				'icon' => 'dashicons dashicons-list-view'
			];
		}

		return apply_filters('wppayform/customer_dashboard/menus', $menus);

	}

    public function getSubscriptionItems($submissionsItems)
    {
        $subscriptionItems = [];
        foreach (Arr::get($submissionsItems, 'subscriptions', []) as $subscriptionItem) {
            $submission = new SubmissionController();
            $form_id = Arr::get($subscriptionItem, 'form_id');
            $submission_id = Arr::get($subscriptionItem, 'submission_id');
            $submissionData = $submission->getSubmissionPrepared($form_id, $submission_id);
            $subscriptionItem['submission'] = $submissionData;
            array_push($subscriptionItems, $subscriptionItem->toArray());
        }

        return $subscriptionItems;
    }

    public function getAllSubmissions($current_user)
    {
        $submissionsItems = (new Customers)->customer($current_user);
        if (!$submissionsItems) {
            return array(
                'info' => [],
                'entries' => []
            );
        }
        foreach ($submissionsItems['entries'] as $key => $entry) {
            // dd($entry->toArray());
            $entry->discounts = (new Submission())->getDiscounts($entry->id, $entry->form_id, $entry);
            $entry = (new PaymentReceipt())->getSubmissionTotal($entry);
            $submissionsItems['entries'][$key] = $entry;
        }
        $submissionsItems['info'] = $submissionsItems['info']->toArray();
        $submissionsItems['entries'] = $submissionsItems['entries']->toArray();
        
        return $submissionsItems;
    }

    public function localizedScript()
    {
        wp_enqueue_style(
            'wppayform_user_dashboard',
            WPPAYFORM_URL . 'assets/css/wppayform_user_dashboard.css',
            array(),
            WPPAYFORM_VERSION
        );

        wp_enqueue_script('wppayform_user_dashboard', WPPAYFORM_URL . 'assets/js/payforms-user_dashboard.js', array('jquery'), WPPAYFORM_VERSION, true);
        wp_localize_script(
            'wppayform_user_dashboard',
            'wpPayFormsDashboard',
            array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'login_url' => wp_login_url(),
                'nonce' => wp_create_nonce('wppayform_user_dashboard_nonce'),
            )
        );
    }

    public function getUnAuthorizedHtmlContent()
    {
        ob_start();
        App::make('view')->render('user.login_redirect', [
            'redirect_url' => wp_login_url(),
        ]);

        $view = ob_get_clean();
        return $view;
    }

    public function getUserPermissions()
    {
        $paymatticUserPermissions = get_option('_wppayform_enable_paymattic_user_dashboard', []);
        $paymatticUserPermissions = Arr::get($paymatticUserPermissions, 'paymatticUserPermissions.paymattic_user.capabilities', []);
        foreach ($paymatticUserPermissions as $permissionKey => $permission) {
            $paymatticUserPermissions[$permissionKey] = $permission == true ? 'yes' : 'no';
        }

        return $paymatticUserPermissions;
    }

    public function sumAmountsByCurrency($inputArray)
    {
        $result = array();

        foreach ($inputArray as $entry) {
            $currency = $entry['currency'];
            $amount = (float) $entry['payment_total'];

            if (isset($result[$currency])) {
                $result[$currency] += $amount;
            } else {
                $result[$currency] = $amount;
            }
        }
        return $result ;
    }

    public function syncSubscriptionAjaxController()
    {
        $formId = sanitize_text_field(Arr::get($_REQUEST, 'form_id', ''));
        $submissionHash = sanitize_text_field(Arr::get($_REQUEST, 'submission_hash', ''));
        // dd("ds ", $formId, $submissionHash);
        $submissionId = $this->validateAjaxRequest($formId, $submissionHash);
        $submission = new SubmissionController();

        return $submission->syncSubscription($formId, $submissionId);
    }

    public function validateAjaxRequest($formId, $submissionHash)
    {
        $submissionModal = new Submission();
        $submission = $submissionModal->getSubmissionByHash($submissionHash);
        if (empty($submission) || $submission->form_id != $formId) {
            wp_send_json_error(
                array(
                    'message' => __("Invalid request!", 'wp-payment-form'),
                ),
                423
            );
        }

        return $submission->id;
    }

    public function cancelSubscription()
    {
        $subscriptionModal = new Subscription();
        $formId = sanitize_text_field(Arr::get($_REQUEST, 'form_id', ''));
        $subscription_id = sanitize_text_field(Arr::get($_REQUEST, 'subscription_id', ''));
        $submissionHash = sanitize_text_field(Arr::get($_REQUEST, 'submission_hash', ''));
        $submissionId = $this->validateAjaxRequest($formId, $submissionHash);
        $subscription = $subscriptionModal->getSubscription($subscription_id);
        $submissionModel = new Submission();
        $submission = $submissionModel->getSubmission($submissionId);
      
        if (empty($submission->payment_method)) {
            wp_send_json_error(
                array(
                    'message' => __("Can't cancel the subscription at this moment. Please try again", 'wp-payment-form'),
                ),
                423
            );
        }


        add_action('wppayform/send_email_after_subscription_cancel_', array($this, 'sendSubscriptionCancelEmail'), 10, 3);
        do_action('wppayform/subscription_settings_cancel_' . $submission->payment_method, $formId, $submission, $subscription);
    }

    public function sendSubscriptionCancelEmail($formId, $subscription, $submission)
    {

        $checkHasEmailOrNot = $this->hasFormEmailSettingsOrNot($formId);

       if (!$checkHasEmailOrNot) {
            $email_template_settings = get_option( '_wppayform_global_cancel_subsription_email_settings', null );
            $subscriptionPlan = $subscription['plan_name'];
            foreach ($email_template_settings as $key => $email_template_setting) {
                if (Arr::get($email_template_setting, 'status', null) == 'active' && Arr::get($email_template_settings, 'enable_cancel_subscription_email', null) == 'yes') {
                    if ($key == 'customer_email_settings') {
                        $email_template_setting['email_to'] = wp_get_current_user()->get('user_email');
                        $email_template_setting['email_subject'] = Arr::get($email_template_setting, 'email_subject', '') ? Arr::get($email_template_setting, 'email_subject', '') : 'Subscription plan '. $subscriptionPlan . 'is cancelled';
                    } else {
                        $email_template_setting['email_subject'] = Arr::get($email_template_setting, 'email_subject', '') ? Arr::get($email_template_setting, 'email_subject', '') : 'Subscription plan '. $subscriptionPlan .' is cancelled by '. wp_get_current_user()->get('user_email');
                    }
                    $prependString = "<h4> Subscription plan ". $subscriptionPlan ." is cancelled.</h4> </br> ";
                    $email_template_setting['email_body'] =  $prependString . Arr::get($email_template_setting, 'email_body', '');
                   
                    (new EmailHandler())->processEmailNotification($email_template_setting, $submission, 'auto');
                }
            }
       }
    }

    public function hasFormEmailSettingsOrNot($formId)
    {
        $notifications = get_post_meta($formId, 'wpf_email_notifications', true);
        $notifier = [];
        foreach ($notifications as $id => $item) {
            if ($item['status'] == 'active') {
                if ($item['sending_action'] == 'wppayform/send_email_after_subscription_cancel_for_user' || $item['sending_action'] == 'wppayform/send_email_after_subscription_cancel_for_admin') {
                    return true;
                }
            }
            array_push($notifier, $id);
        }

        return false;
    }
}


