<?php

namespace WPPayFormPro\Classes\Dashboard;

// use community class/models
use FluentCommunity\App\Models\BaseSpace;
use FluentCommunity\App\Models\SpaceUserPivot;
use FluentCommunity\App\Services\LockscreenService;
use FluentCommunity\Modules\Course\Model\Course;
use FluentCommunity\Modules\Course\Model\CourseLesson;
use FluentCommunity\Modules\Course\Model\CourseTopic;
use FluentCommunity\Modules\Course\Services\CourseHelper;
use WPPayForm\Framework\Support\Arr;


class FluentCommunity
{
    public function __construct()
    {
        // add_action('fluent_community/portal_loaded', [$this, 'register']);
    }

    public function register()
    {
        add_action('wp_ajax_get_fluent_community_spaces', array($this, 'handleEndpoints'));
        add_action('wp_ajax_nopriv_get_fluent_community_spaces', array($this, 'handleEndpoints'));
    }

    public function render()
    {
       
    }

    public function handleEndpoints()
    {
        if(! isset( $_REQUEST['nonce'] ) || !wp_verify_nonce(sanitize_text_field( wp_unslash($_REQUEST['nonce'])), 'wppayform_user_dashboard_nonce')){
            wp_send_json([
                "status" => 403,
                "success"=> false,
                "message" => "Something went wrong! Request not valid.",
            ]);
            wp_die();
        }

        $route = sanitize_text_field($_REQUEST['route']);
    
        $validRoutes = array(
            'get_recommended_spaces' => 'getRecommendedSpaces',
            'get_user_spaces' => 'getUserSpaces',
        );

        if (isset($validRoutes[$route])) {
            $this->{$validRoutes[$route]}();
        }
    }

    public function getRecommendedSpaces()
    {
        $userId = get_current_user_id();
        try{
            // get all spaces, where user is not a member
            $reommendedSpaces = BaseSpace::query()->withoutGlobalScopes()
                ->select([
                    'fcom_spaces.id', 
                    'fcom_spaces.title', 
                    'fcom_spaces.type',
                    'fcom_spaces.slug',
                    'fcom_spaces.cover_photo',
                    'fcom_spaces.description',
                    'fcom_spaces.privacy',
                    'fcom_spaces.status as space_status',
                    'fcom_space_user.status as user_status',
                    'fcom_space_user.user_id'
                ]) 
                ->whereIn('type', ['course', 'community'])
                ->whereIn('fcom_spaces.status', ['published', 'active'])
                ->whereIn('fcom_spaces.privacy', ['public', 'private'])
                ->leftJoin('fcom_space_user', function ($join) use ($userId) {
                    $join->on('fcom_spaces.id', '=', 'fcom_space_user.space_id')
                        ->where('fcom_space_user.user_id', '=', $userId);
                })
                ->whereNull('fcom_space_user.space_id') // Exclude joined rows
                ->orderBy('fcom_spaces.title', 'ASC')
                ->get()
                ->map(function ($space) {
                    $space->permalink = $space->getPermalink();
                    if ($space->type === 'course') {
                        if (!$space->cover_photo) {
                            $space->cover_photo = FLUENT_COMMUNITY_PLUGIN_URL . 'assets/images/course-placeholder.jpg';
                        }
                    } else if ($space->type === 'community') {
                        if (!$space->cover_photo) {
                            $space->cover_photo = FLUENT_COMMUNITY_PLUGIN_URL . 'assets/images/space-placeholder.jpg';
                        }
                    }
                    return $space;
                });

        } catch(\Exception $e){
            wp_send_json([
                "status" => 500,
                "success"=> false,
                "message" => $e->getMessage(),
            ]);
            wp_die();
        }

        $formattedSpaces = $this->formatSpaces($reommendedSpaces);
        
        wp_send_json([
            "status" => 200,
            "success"=> true,
            "data" => [
                'spaces' => $formattedSpaces,
            ],
        ]);
    }

    public function formatSpaces($spaces)
    {
       // format spaces based on type  community and course
        $formattedSpaces = array(
            'community' => [],
            'course' => [],
        );
       
        foreach ($spaces as $space) {
            if ($space->type === 'community') {
                $space->members_count = 0;
                $space->members_count = SpaceUserPivot::where('space_id', $space->id)->where('status', 'active')
                ->whereHas('xprofile', function ($q) {
                    $q->where('status', 'active');
                })
                ->whereHas('user')
                ->count();
                $formattedSpaces['community'][] = $space;
            } else {
                $space->isEnrolled = CourseHelper::isEnrolled($space->id);
                if ($space->isEnrolled) {
                    $space->progress = CourseHelper::getCourseProgress($space->id);
                }
    
    
                $space->sectionsCount = CourseTopic::where('space_id', $space->id)->count();
                $space->lessonsCount = CourseLesson::where('space_id', $space->id)->count();
                $space->studentsCount = SpaceUserPivot::where('space_id', $space->id)->count();
                $formattedData['spaces']['course'][] = $space;
                $formattedSpaces['course'][] = $space;
            }
        }

        return $formattedSpaces;
    }

    public function getUserSpaces()
    {
        $userId = get_current_user_id();

        try{
            // now get spaces where user is a member
            $userSpaces = BaseSpace::query()->withoutGlobalScopes()
                ->select([
                    'fcom_spaces.id', 
                    'fcom_spaces.title', 
                    'fcom_spaces.type',
                    'fcom_spaces.slug',
                    'fcom_spaces.cover_photo',
                    'fcom_spaces.description',
                    'fcom_spaces.privacy',
                    'fcom_spaces.status as space_status',
                    'fcom_space_user.status as user_status',
                    'fcom_space_user.user_id'
                ]) 
                ->whereIn('type', ['course', 'community'])
                ->leftJoin('fcom_space_user', function ($join) use ($userId) {
                    $join->on('fcom_spaces.id', '=', 'fcom_space_user.space_id')
                        ->where('fcom_space_user.user_id', '=', $userId);
                })
                ->whereNotNull('fcom_space_user.space_id') // Exclude joined rows
                // ->select(['fcom_spaces.id', 'fcom_spaces.title', 'fcom_spaces.description', 'fcom_spaces.type', 'fcom_spaces.slug'])
                ->orderBy('fcom_spaces.title', 'ASC')
                ->get()
                ->map(function ($space) {
                    $space->permalink = $space->getPermalink();
                    if ($space->type === 'course') {
                        if (!$space->cover_photo) {
                            $space->cover_photo = FLUENT_COMMUNITY_PLUGIN_URL . 'assets/images/course-placeholder.jpg';
                        }
                    } else if ($space->type === 'community') {
                        if (!$space->cover_photo) {
                            $space->cover_photo = FLUENT_COMMUNITY_PLUGIN_URL . 'assets/images/space-placeholder.jpg';
                        }
                    }
                    return $space;
                });
        }catch(\Exception $e){
            wp_send_json([
                "status" => 500,
                "success"=> false,
                "message" => $e->getMessage(),
            ]);
            wp_die();

        }

        $formattedData = $this->formatUserData($userSpaces);

        wp_send_json([
            "status" => 200,
            "success"=> true,
            "data" => $formattedData
        ]);
    
    }

    public function formatUserData($userSpaces)
    {
        $formattedData = array(
            'spaces' => [],
        );

        foreach ($userSpaces as $space) {
            // get space permalink
            $space->permalink = $space->getPermalink();
            if ($space->type === 'community') {
                $space->members_count = SpaceUserPivot::where('space_id', $space->id)->where('status', 'active')
                ->whereHas('xprofile', function ($q) {
                    $q->where('status', 'active');
                })
                ->whereHas('user')
                ->count();
                $formattedData['spaces']['community'][] = $space;
            } else if ($space->type === 'course') {
                $space->isEnrolled = CourseHelper::isEnrolled($space->id);
                if ($space->isEnrolled) {
                    $space->progress = CourseHelper::getCourseProgress($space->id);
                }
    
    
                $space->sectionsCount = CourseTopic::where('space_id', $space->id)->count();
                $space->lessonsCount = CourseLesson::where('space_id', $space->id)->count();
                $space->studentsCount = SpaceUserPivot::where('space_id', $space->id)->count();
                $formattedData['spaces']['course'][] = $space;
            }
        }

        return $formattedData;
    }
}