<?php

namespace WPPayFormPro\Classes\Components;

use WPPayForm\App\Modules\FormComponents\BaseComponent;
use WPPayForm\Framework\Support\Arr;

if (!defined('ABSPATH')) {
    exit;
}

class TaxItemComponent extends BaseComponent
{
    public function __construct()
    {
        parent::__construct('tax_payment_input', 6);
        add_filter('wppayform/submitted_payment_items', array($this, 'pushTaxItems'), 999, 4);
        add_filter('wppayform/validate_component_on_save_tax_payment_input', array($this, 'validateOnSave'), 1, 3);
    }

    public function component()
    {
        return array(
            'type' => 'tax_payment_input',
            'editor_title' => 'Tax Calculated Amount',
            'group' => 'payment',
            'postion_group' => 'payment',
            'is_pro' => 'yes',
            'editor_elements' => array(
                'label' => array(
                    'label' => 'Field Label',
                    'type' => 'text',
                    'group' => 'general'
                ),
                'tax_percent' => array(
                    'label' => 'Tax Percentage',
                    'type' => 'number',
                    'group' => 'general'
                ),
                'target_product' => array(
                    'label' => 'Target Product Item',
                    'type' => 'onetime_products_selector',
                    'group' => 'general',
                    'info' => 'Please select the product in where this tax percentage will be applied'
                ),
                'tax_hidden' => array(
                    'label' => 'Hide Tax Input',
                    'type' => 'switch',
                    'group' => 'general'
                ),
                'admin_label' => array(
                    'label' => 'Admin Label',
                    'type' => 'text',
                    'group' => 'advanced'
                ),
                'wrapper_class' => array(
                    'label' => 'Field Wrapper CSS Class',
                    'type' => 'text',
                    'group' => 'advanced'
                ),
                'element_class' => array(
                    'label' => 'Input Element CSS Class',
                    'type' => 'text',
                    'group' => 'advanced'
                ),
                'conditional_render' => array(
                    'type' => 'conditional_render',
                    'group' => 'advanced',
                    'label' => 'Conditional render',
                    'selection_type' => 'Conditional logic',
                    'conditional_logic' => array(
                        'yes' => 'Yes',
                        'no' => 'No'
                    ),
                    'conditional_type' => array(
                        'any' => 'Any',
                        'all' => 'All'
                    ),
                ),
            ),
            'field_options' => array(
                'label' => 'Tax Amount',
                'tax_percent' => '10',
                'tax_hidden' => 'no',
                'conditional_logic_option' => array(
                    'conditional_logic' => 'no',
                    'conditional_type'  => 'any',
                    'options' => array(
                        array(
                            'target_field' => '',
                            'condition' => '',
                            'value' => ''
                        )
                    ),
                ),
            )
        );
    }

    public function render($element, $form, $elements)
    {
        $inputType = Arr::get($element, 'type', false);
        $fieldOptions = Arr::get($element, 'field_options', false);
        $disable = Arr::get($fieldOptions, 'disable', false);
        $taxHidden = Arr::get($fieldOptions, 'tax_hidden', 'no');
        if (!$fieldOptions || $disable) {
            return;
        }
        add_filter('wppayform/form_css_classes', function ($classes, $reneringForm) use ($form) {
            if ($reneringForm->ID == $form->ID) {
                $classes[] = 'wpf_has_tax_item';
            }
            return $classes;
        }, 10, 2);
        $inputId = 'wpf_tax_' . $form->ID . '_' . $element['id'];

        $fieldOptions['label'] = $fieldOptions['label'] . ': <span class="wpf_calc_tax" data-target_tax="' . $inputId . '"></span>';
        $controlClass = $this->elementControlClass($element);
        $taxPercent = Arr::get($fieldOptions, 'tax_percent'); ?>
        <div data-element_type="<?php echo $this->elementName; ?>"
             class="<?php echo $controlClass; ?>" style="display: <?php echo $taxHidden == 'yes' ? 'none' : 'block'; ?>">
            <?php  $labelAttributes = [
                'data-tax_percent' => $taxPercent,
                'class' => 'wpf_tax_line_item',
                'data-is_tax_line' => 'yes',
                'id' => $inputId,
                'condition_id' => $element['id'],
                'data-target_product' => Arr::get($fieldOptions, 'target_product')
            ] ?>
            <?php
            $this->buildLabel($fieldOptions, $form, $labelAttributes);            
            ?>
        </div>
        <?php
    }

    public function validateOnSave($error, $element, $formId)
    {
        $disable = Arr::get($element, 'field_options.disable', false);

        if ($disable) {
            return;
        }

        if (!Arr::get($element, 'field_options.target_product')) {
            $error = __('Target Product is required for item:', 'wp-payment-form-pro') . ' ' . Arr::get($element, 'field_options.label');
        }
        return $error;
    }

    public function pushTaxItems($paymentItems, $formattedElements, $formData, $discountPercent = 0)
    {
        if (!$paymentItems && !$formattedElements['payment']) {
            return $paymentItems;
        }

        foreach ($formattedElements['payment'] as $element) {
            if ($element['type'] == 'recurring_payment_item') {
                $paymentItems[] = $element;
            }
        }

        if (!$paymentItems) {
            return $paymentItems;
        }
        $signupFee = [];
        // structure payment items as in object of (type, parent_holder, item_name, quantity, item_price, line_total, created_at, updated_at)
        foreach ($paymentItems as $key => $payItem) {
            if ('recurring_payment_item' != $payItem['type']) {
                continue;
            }

            // Get the selected pricing option index from form data
            $paymentId = Arr::get($payItem, 'id', '');
            $selectedIndex = isset($formData[$paymentId]) ? intval($formData[$paymentId]) : 0;
            // Get all pricing options
            $pricingOptions = Arr::get($payItem, 'options.recurring_payment_options.pricing_options', []);
            $selectedOption = isset($pricingOptions[$selectedIndex]) ? $pricingOptions[$selectedIndex] : $pricingOptions[0] ?? [];
            
            if (Arr::get($selectedOption, 'has_signup_fee') === 'yes') {
                $signupFee[$paymentId] = wpPayFormConverToCents(Arr::get($selectedOption, 'signup_fee', 0));
            }

            // $pricingOptions = Arr::get($payItem, 'options.recurring_payment_options.pricing_options.0', []);
            $paymentItems[$key] = array(
                'type' => $payItem['type'],
                'parent_holder' => $payItem['id'],
                'item_name' => $payItem['label'],
                'quantity' => 1,
                'item_price' => wpPayFormConverToCents(Arr::get($selectedOption, 'subscription_amount', 0)),
                'line_total' => wpPayFormConverToCents(Arr::get($selectedOption, 'subscription_amount', 0)),
                'recurring_tax' => 'yes',
                'signup_fee' => $signupFee[$paymentId] ?? 0
            );
        }

        $taxItems = $this->getTaxItems($paymentItems, $formattedElements['payment'], $formData, $discountPercent, $signupFee);
        $taxItems = apply_filters('wppayform/form_tax_items', $taxItems, $paymentItems, $formattedElements);

        if ($taxItems) {
            $paymentItems = array_merge($paymentItems, $taxItems);
        }

        return $paymentItems;
    }

    private function getTaxItems($paymentItems, $items, $formData, $discountPercent = 0, $signupFee = [])
    {
        if (Arr::get($items, 'tax_payment_input.options.disable', false)) {
            return;
        }
       
        if (!$paymentItems && !$items) {
            return $items;
        }

        $recurringTaxStatus = [];  
        $signupFeeStatus = [];
        foreach ($paymentItems as $payItem) {  
            $parentHolder = $payItem['parent_holder'];  
            $recurringTaxStatus[$parentHolder] = $payItem['type'] === 'recurring_payment_item' ? 'yes' : 'no';  

            $signupKeyValue = Arr::get($payItem, 'signup_fee', 0);
            $signupFeeStatus[$parentHolder] = $signupKeyValue > 0 ? 'yes' : 'no';
        }  
        
        // let's format the $paymentItems as in object
        $recurringTax = 'no';
        $itemizedTotal = [];
        foreach ($paymentItems as $payItem) {
            // if ('recurring_payment_item' == $payItem['type']) {
            //     $recurringTax = 'yes';
            // }
            $prductName = $payItem['parent_holder'];
            if (!isset($itemizedTotal[$prductName])) {
                $itemizedTotal[$prductName] = $payItem['line_total'];
            } else {
                $itemizedTotal[$prductName] += $payItem['line_total'];
            }
        }


        $taxItems = array();
        foreach ($items as $itemKey => $item) {
            if ($item['type'] != 'tax_payment_input') {
                continue;
            }

            // Check conditional logic  
            $checkTaxCondition = $this->checkTaxCondition($item, $formData);  
            if ($checkTaxCondition === false) {  
                continue;  
            }  

            $targetProduct = Arr::get($item, 'options.target_product');
            if (!isset($itemizedTotal[$targetProduct])) {
                continue;
            }

            $taxPercent = Arr::get($item, 'options.tax_percent');
            if (!$taxPercent || !$itemizedTotal[$targetProduct]) {
                continue;
            }

            // keep the original itemized total value for restoring later
            $originalItemizedTotal = $itemizedTotal[$targetProduct];
            if ($discountPercent) {
                $discounts = intval($itemizedTotal[$targetProduct] * $discountPercent) / 100;
                $itemizedTotal[$targetProduct] -= $discounts;
            }

            $signupFee_tax = $signupFeeStatus[$targetProduct] ?? 'no';
            $signupFeeTax = 0;

            $signupFeeValue = Arr::get($signupFee, $targetProduct, 0);            
            if ($signupFeeValue > 0) {
                $signupFeeTax = (int) round($signupFeeValue * ($taxPercent / 100), 0, PHP_ROUND_HALF_UP);
            }

            $recurringTax = $recurringTaxStatus[$targetProduct] ?? 'no';

            $taxItems[$itemKey] = array(
                'type' => 'tax_line',
                'parent_holder' => $targetProduct,
                'item_name' => strip_tags(Arr::get($item, 'options.label') . '(' . $taxPercent . '%)'),
                'quantity' => 1,
                'item_price' => (int) round($itemizedTotal[$targetProduct] * ($taxPercent / 100), 0, PHP_ROUND_HALF_UP),
                'line_total' => (int) round($itemizedTotal[$targetProduct] * ($taxPercent / 100), 0, PHP_ROUND_HALF_UP),
                'recurring_tax' => $recurringTax,
                'signup_fee_tax' => $signupFee_tax,
                'signup_fee' => $signupFeeTax,
                'created_at' => current_time('Y-m-d H:i:s'),
                'updated_at' => current_time('Y-m-d H:i:s'),
            );
            /*
            restore the original itemized total value for the next tax item, 
            otherwise the next tax item will be calculated based on the previously discounted value,
            if there is a discount applied to the target product,
            */ 
            $itemizedTotal[$targetProduct] = $originalItemizedTotal;
        }
        return $taxItems;
    }

    private function checkTaxCondition ($item, $formData) {
        $conditionLogic = Arr::get($item, 'options.conditional_logic_option', []);  
        if (!empty($conditionLogic['options']) && isset($conditionLogic['conditional_logic']) && $conditionLogic['conditional_logic'] === 'yes') {  
            foreach ($conditionLogic['options'] as $conditionField) {  
                $targetField = Arr::get($conditionField, 'target_field', '');  
                $condition = Arr::get($conditionField, 'condition', '');
                $value = Arr::get($conditionField, 'value', ''); 
                if (!isset($formData[$targetField])) {  
                    return false;  
                }  
                $checkCondition = $this->checkCondition($condition, $formData[$targetField], $value);
                if (!$checkCondition) {  
                    return false;  
                }  
            }  
        }  
        return true; 
    }

    private function checkCondition($condition, $formValue, $value)   
    {  
        switch ($condition) {  
            case 'equal':  
                return $formValue === $value;  
    
            case 'not_equal':  
                return $formValue !== $value;  
    
            case 'starts_with':  
                return strpos($formValue, $value) === 0;  
    
            case 'ends_with':  
                return substr($formValue, -strlen($value)) === $value;  
    
            case 'contains':  
                return strpos($formValue, $value) !== false;  
    
            case 'not_contains':  
                return strpos($formValue, $value) === false;  
    
            case 'greater_than':  
                return $formValue > $value;  
    
            case 'less_than':  
                return $formValue < $value;  
    
            case 'greater_or_equal':  
                return $formValue >= $value;  
    
            case 'less_or_equal':  
                return $formValue <= $value;  
    
            default:  
                return false;  
        }  
    }
}