<?php
/*
  Plugin Name: WP Media folder Gallery Addon
  Plugin URI: http://www.joomunited.com
  Description: WP Media Folder Gallery Addon enhances WPMF plugin by adding a full image gallery management
  Author: Joomunited
  Version: 2.6.11
  Update URI: https://www.joomunited.com/juupdater_files/wp-media-folder-gallery-addon.json
  Author URI: http://www.joomunited.com
  Text Domain: wp-media-folder-gallery-addon
  Domain Path: /languages
  Requires Plugins: wp-media-folder
  Licence : GNU General Public License version 2 or later; http://www.gnu.org/licenses/gpl-2.0.html
  Copyright : Copyright (C) 2014 JoomUnited (http://www.joomunited.com). All rights reserved.
 */
// Prohibit direct script loading
defined('ABSPATH') || die('No direct script access allowed!');

//Check plugin requirements
if (version_compare(PHP_VERSION, '5.6', '<')) {
    if (!function_exists('wpmfGalleryShowError')) {
        /**
         * Show notice
         *
         * @return void
         */
        function wpmfGalleryShowError()
        {
            echo '<div class="error"><p>';
            echo '<strong>WP Media Folder Gallery Addon</strong>';
            echo ' need at least PHP 5.6 version, please update php before installing the plugin.</p></div>';
        }
    }

    //Add actions
    add_action('admin_notices', 'wpmfGalleryShowError');
    //Do not load anything more
    return;
}
if (!defined('WPMF_GALLERY_ADDON_PLUGIN_DIR')) {
    /**
     * Path to WP Media Folder Gallery addon plugin
     */
    define('WPMF_GALLERY_ADDON_PLUGIN_DIR', plugin_dir_path(__FILE__));
}

if (!defined('WPMF_GALLERY_ADDON_PLUGIN_URL')) {
    /**
     * Url to WP Media Folder Gallery addon plugin
     */
    define('WPMF_GALLERY_ADDON_PLUGIN_URL', plugin_dir_url(__FILE__));
}

if (!defined('WPMF_GALLERY_ADDON_FILE')) {
    /**
     * Path to this file
     */
    define('WPMF_GALLERY_ADDON_FILE', __FILE__);
}

if (!defined('WPMF_GALLERY_ADDON_DOMAIN')) {
    /**
     * Text domain
     */
    define('WPMF_GALLERY_ADDON_DOMAIN', 'wp-media-folder-gallery-addon');
}

if (!defined('WPMF_GALLERY_ADDON_VERSION')) {
    /**
     * Plugin version
     */
    define('WPMF_GALLERY_ADDON_VERSION', '2.6.11');
}

if (!defined('WPMF_GALLERY_ADDON_TAXO')) {
    /**
     * Gallery taxonomy name
     */
    define('WPMF_GALLERY_ADDON_TAXO', 'wpmf-gallery-category');
}

//JUtranslation
add_filter('wpmf_get_addons', function ($addons) {
    $addon                          = new stdClass();
    $addon->main_plugin_file        = __FILE__;
    $addon->extension_name          = 'WP Media Folder Gallery Addon';
    $addon->extension_slug          = 'wpmf-gallery-addon';
    $addon->text_domain             = 'wp-media-folder-gallery-addon';
    $addon->language_file           = plugin_dir_path(__FILE__) . 'languages' . DIRECTORY_SEPARATOR . 'wp-media-folder-gallery-addon-en_US.mo';
    $addons[$addon->extension_slug] = $addon;
    return $addons;
});

/**
 * Load Jutranslation
 *
 * @return void
 */
function wpmfGalleryAddonsInit()
{
    if (!class_exists('\Joomunited\WPMFGALLERYADDON\JUCheckRequirements')) {
        require_once(trailingslashit(dirname(__FILE__)) . 'requirements.php');
    }

    if (class_exists('\Joomunited\WPMFGALLERYADDON\JUCheckRequirements')) {
        // Plugins name for translate
        $args = array(
            'plugin_name' => esc_html__('WP Media Folder Gallery Addon', 'wp-media-folder-gallery-addon'),
            'plugin_path' => 'wp-media-folder-gallery-addon/wp-media-folder-gallery-addon.php',
            'plugin_textdomain' => 'wp-media-folder-gallery-addon',
            'requirements' => array(
                'plugins'     => array(
                    array(
                        'name' => 'WP Media Folder',
                        'path' => 'wp-media-folder/wp-media-folder.php',
                        'requireVersion' => '5.5'
                    )
                ),
                'php_version' => '7.4'
            )
        );
        $wpmfCheck = call_user_func('\Joomunited\WPMFGALLERYADDON\JUCheckRequirements::init', $args);

        if (!$wpmfCheck['success']) {
            // Do not load anything more
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- No action, nonce is not required
            unset($_GET['activate']);
            return;
        }
    }
}

/**
 * Get plugin path
 *
 * @return string
 */
function wpmfGalleryAddons_getPath()
{
    return 'wp-media-folder-gallery-addon/wp-media-folder-gallery-addon.php';
}

include_once(ABSPATH . 'wp-admin/includes/plugin.php');

register_activation_hook(__FILE__, 'wpmfGalleryInstall');

/**
 * Add some options
 *
 * @return void
 */
function wpmfGalleryInstall()
{
    /* create number of items per page for image selection */
    if (!get_option('wpmf_gallery_img_per_page', false)) {
        update_option('wpmf_gallery_img_per_page', 20);
    }

    if (!get_option('wpmfgrl_relationships_media', false)) {
        add_option('wpmfgrl_relationships_media', array(), '', 'yes');
    }
}

/**
 * Sort parents before children
 * http://stackoverflow.com/questions/6377147/sort-an-array-placing-children-beneath-parents
 *
 * @param array   $objects List folder
 * @param array   $result  Result
 * @param integer $parent  Parent of folder
 * @param integer $depth   Depth of folder
 *
 * @return array           output
 */
function wpmfParentSort(array $objects, array &$result = array(), $parent = 0, $depth = 0)
{
    foreach ($objects as $key => $object) {
        $order = get_term_meta($object->term_id, 'wpmf_order', true);
        if (empty($order)) {
            $order = 0;
        }
        $object->order = $order;

        if ((int) $object->parent === (int) $parent) {
            $object->depth = $depth;
            array_push($result, $object);
            unset($objects[$key]);
            wpmfParentSort($objects, $result, $object->term_id, $depth + 1);
        }
    }
    return $result;
}

/**
 * Order attachment by order
 *
 * @param integer $a Item details
 * @param integer $b Item details
 *
 * @return mixed
 */
function wpmfSortByOrder($a, $b)
{
    return (int)$a->order - (int)$b->order;
}

/* Register WPMF_GALLERY_ADDON_TAXO taxonomy */
add_action('init', 'wpmfGalleryRegisterTaxonomy', 0);
if (!function_exists('wpmfGalleryRegisterTaxonomy')) {
    /**
     * Register gallery taxonomy
     *
     * @return void
     */
    function wpmfGalleryRegisterTaxonomy()
    {
        if (!taxonomy_exists('wpmf-category')) {
            register_taxonomy(
                'wpmf-category',
                'attachment',
                array(
                    'hierarchical' => true,
                    'show_in_nav_menus' => false,
                    'show_ui' => false,
                    'public' => false,
                    'labels' => array(
                        'name' => __('WPMF Categories', 'wp-media-folder-gallery-addon'),
                        'singular_name' => __('WPMF Category', 'wp-media-folder-gallery-addon'),
                        'menu_name' => __('WPMF Categories', 'wp-media-folder-gallery-addon'),
                        'all_items' => __('All WPMF Categories', 'wp-media-folder-gallery-addon'),
                        'edit_item' => __('Edit WPMF Category', 'wp-media-folder-gallery-addon'),
                        'view_item' => __('View WPMF Category', 'wp-media-folder-gallery-addon'),
                        'update_item' => __('Update WPMF Category', 'wp-media-folder-gallery-addon'),
                        'add_new_item' => __('Add New WPMF Category', 'wp-media-folder-gallery-addon'),
                        'new_item_name' => __('New WPMF Category Name', 'wp-media-folder-gallery-addon'),
                        'parent_item' => __('Parent WPMF Category', 'wp-media-folder-gallery-addon'),
                        'parent_item_colon' => __('Parent WPMF Category:', 'wp-media-folder-gallery-addon'),
                        'search_items' => __('Search WPMF Categories', 'wp-media-folder-gallery-addon'),
                    )
                )
            );
        }

        /* get image term selection */
        $glr_selection = get_term_by('name', 'Gallery Upload', 'wpmf-category');
        if (!$glr_selection) {
            $inserted = wp_insert_term('Gallery Upload', 'wpmf-category', array());
            if (!is_wp_error($inserted)) {
                $relationships = array($inserted['term_id']);
                update_option('wpmfgrl_relationships', $relationships);
            }
        }

        register_taxonomy(WPMF_GALLERY_ADDON_TAXO, 'attachment', array(
            'hierarchical' => true,
            'show_in_nav_menus' => false,
            'show_ui' => false,
            'public' => false,
            'labels' => array(
                'name' => __('WPMF Gallery Categories', 'wp-media-folder-gallery-addon'),
                'singular_name' => __('WPMF Gallery Category', 'wp-media-folder-gallery-addon'),
                'menu_name' => __('WPMF Gallery Categories', 'wp-media-folder-gallery-addon'),
                'all_items' => __('All WPMF Gallery Categories', 'wp-media-folder-gallery-addon'),
                'edit_item' => __('Edit WPMF Gallery Category', 'wp-media-folder-gallery-addon'),
                'view_item' => __('View WPMF Gallery Category', 'wp-media-folder-gallery-addon'),
                'update_item' => __('Update WPMF Gallery Category', 'wp-media-folder-gallery-addon'),
                'add_new_item' => __('Add New WPMF Gallery Category', 'wp-media-folder-gallery-addon'),
                'new_item_name' => __('New WPMF Gallery Category Name', 'wp-media-folder-gallery-addon'),
                'parent_item' => __('Parent WPMF Gallery Category', 'wp-media-folder-gallery-addon'),
                'parent_item_colon' => __('Parent WPMF Gallery Category:', 'wp-media-folder-gallery-addon'),
                'search_items' => __('Search WPMF Gallery Categories', 'wp-media-folder-gallery-addon'),
            ),
        ));
    }
}

require_once(WPMF_GALLERY_ADDON_PLUGIN_DIR . 'admin/class/wp-media-folder-gallery-addon.php');
new WpmfGlrAddonAdmin;

global $wpmfGalleryFront;
require_once(WPMF_GALLERY_ADDON_PLUGIN_DIR . 'frontend/class/wp-media-folder-gallery-addon.php');
$wpmfGalleryFront = new WpmfGlrAddonFrontEnd;


/**
 * Load elementor widget
 *
 * @return void
 */
function wpmfGalleryAddonLoadElementorWidget()
{
    require_once(WPMF_GALLERY_ADDON_PLUGIN_DIR . 'elementor-widgets/class-gallery-elementor-widget.php');
    if (defined('ELEMENTOR_VERSION') && version_compare(ELEMENTOR_VERSION, '3.5', '<')) {
        \Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \WpmfGalleryAddonElementorWidget());
    } else {
        \Elementor\Plugin::instance()->widgets_manager->register(new \WpmfGalleryAddonElementorWidget());
    }
}

if (defined('ELEMENTOR_VERSION') && version_compare(ELEMENTOR_VERSION, '3.5', '<')) {
    add_action('elementor/widgets/widgets_registered', 'wpmfGalleryAddonLoadElementorWidget');
} else {
    add_action('elementor/widgets/register', 'wpmfGalleryAddonLoadElementorWidget');
}

add_action('elementor/frontend/before_register_scripts', function () {
    wp_register_script(
        'wordpresscanvas-imagesloaded',
        WPMF_PLUGIN_URL . 'assets/js/display-gallery/imagesloaded.pkgd.min.js',
        array(),
        '3.1.5',
        true
    );
    wp_register_script('wpmfisotope', WPMF_GALLERY_ADDON_PLUGIN_URL . 'assets/js/isotope.pkgd.js', array('jquery'), WPMF_GALLERY_ADDON_VERSION, true);
    wp_register_script('wpmfpackery', WPMF_GALLERY_ADDON_PLUGIN_URL . 'assets/packery/packery.pkgd.min.js', array('jquery'), WPMF_GALLERY_ADDON_VERSION, true);
    wp_register_script(
        'wpmf-justified-script',
        WPMF_GALLERY_ADDON_PLUGIN_URL . 'assets/js/justified-gallery/jquery.justifiedGallery.min.js',
        array('jquery'),
        WPMF_GALLERY_ADDON_VERSION,
        true
    );
    wp_register_script(
        'wpmf-slick-script',
        WPMF_PLUGIN_URL . 'assets/js/slick/slick.min.js',
        array('jquery'),
        WPMF_VERSION,
        true
    );
    wp_register_script(
        'wpmf-flipster-js',
        WPMF_GALLERY_ADDON_PLUGIN_URL . 'assets/js/jquery.flipster.js',
        array('jquery'),
        WPMF_GALLERY_ADDON_VERSION,
        true
    );
});

add_action('elementor/editor/before_enqueue_scripts', function () {
    wp_enqueue_script(
        'wordpresscanvas-imagesloaded',
        WPMF_PLUGIN_URL . 'assets/js/display-gallery/imagesloaded.pkgd.min.js',
        array(),
        '3.1.5',
        true
    );
    wp_enqueue_script('wpmfisotope', WPMF_GALLERY_ADDON_PLUGIN_URL . 'assets/js/isotope.pkgd.js', array('jquery'), WPMF_GALLERY_ADDON_VERSION, true);
    wp_enqueue_script('wpmfpackery', WPMF_GALLERY_ADDON_PLUGIN_URL . 'assets/packery/packery.pkgd.min.js', array('jquery'), WPMF_GALLERY_ADDON_VERSION, true);
    wp_enqueue_script(
        'wpmf-justified-script',
        WPMF_GALLERY_ADDON_PLUGIN_URL . 'assets/js/justified-gallery/jquery.justifiedGallery.min.js',
        array('jquery'),
        WPMF_GALLERY_ADDON_VERSION,
        true
    );
    wp_enqueue_script(
        'wpmf-slick-script',
        WPMF_PLUGIN_URL . 'assets/js/slick/slick.min.js',
        array('jquery'),
        WPMF_VERSION,
        true
    );

    wp_enqueue_script(
        'wpmf-flipster-js',
        WPMF_GALLERY_ADDON_PLUGIN_URL . 'assets/js/jquery.flipster.js',
        array('jquery'),
        WPMF_GALLERY_ADDON_VERSION,
        true
    );
    wp_enqueue_script(
        'wpmf-gallery-js',
        WPMF_GALLERY_ADDON_PLUGIN_URL . 'assets/js/gallery.js',
        array('jquery', 'wpmfisotope', 'wpmf-justified-script'),
        WPMF_GALLERY_ADDON_VERSION,
        true
    );

    $option_current_theme = get_option('current_theme');
    $gallery_configs = get_option('wpmf_gallery_settings');

    if (isset($gallery_configs['progressive_loading']) && (int)$gallery_configs['progressive_loading'] === 0) {
        $progressive_loading = 0;
    } else {
        $progressive_loading = 1;
    }

    wp_localize_script('wpmf-gallery-js', 'wpmfgallery', array(
        'wpmf_current_theme' => $option_current_theme,
        'gallery_configs' => $gallery_configs,
        'progressive_loading' => (int)$progressive_loading,
        'wpmf_gallery_nonce' => wp_create_nonce('wpmf_gallery_nonce'),
        'ajaxurl' => admin_url('admin-ajax.php')
    ));
});

/**
 * Enqueue script in divi gallery addon module
 *
 * @return void
 */
function wpmfInitGalleryAddonDivi()
{
    require_once(WPMF_GALLERY_ADDON_PLUGIN_DIR . 'frontend/class/wp-media-folder-gallery-addon.php');
    $gallery_addon = new WpmfGlrAddonFrontEnd;
    $gallery_addon->galleryScripts();
    $gallery_addon->enqueueScript('divi');
}

add_action('wpmf_init_gallery_addon_divi', 'wpmfInitGalleryAddonDivi');

/**
 * Enqueue script in bakery gallery addon module
 *
 * @return void
 */
function wpmfVcInitGalleryAddon()
{
    require_once WPMF_GALLERY_ADDON_PLUGIN_DIR . '/bakery-widgets/Gallery.php';
}

add_action('wpmf_vc_init_gallery_addon', 'wpmfVcInitGalleryAddon');

/**
 * This action registers all styles(fonts) to be enqueue later
 *
 * @return void
 */
function wpmfAddonVcEnqueueJsCss()
{
    wp_enqueue_script('jquery-masonry');
}

add_action('vc_frontend_editor_enqueue_js_css', 'wpmfAddonVcEnqueueJsCss');

if (is_plugin_active('wp-media-folder/wp-media-folder.php')) {
    if (!function_exists('wpmfGalleryAddonTnitAvada')) {
        /**
         * Init Avada module
         *
         * @return void
         */
        function wpmfGalleryAddonTnitAvada()
        {
            if (!defined('AVADA_VERSION') || !defined('FUSION_BUILDER_VERSION')) {
                return;
            }

            require_once WPMF_GALLERY_ADDON_PLUGIN_DIR . 'avada-widgets/Gallery.php';
            if (fusion_is_builder_frame()) {
                add_action('fusion_builder_enqueue_live_scripts', 'wpmfAddonAvadaEnqueueSeparateLiveScripts');
            }
            add_action('fusion_builder_admin_scripts_hook', 'fusion_builder_admin_scripts_hook');
        }

        add_action('init', 'wpmfGalleryAddonTnitAvada');
    }
}

/**
 * Avada enqueue admin scripts
 *
 * @return void
 */
function fusion_builder_admin_scripts_hook()
{
    wp_enqueue_script('wpmf_fusion_admin_gallery_addon_element', WPMF_GALLERY_ADDON_PLUGIN_URL . 'avada-widgets/js/avada_backend.js', array(), WPMF_GALLERY_ADDON_VERSION, true);
}

/**
 * Avada enqueue live scripts
 *
 * @return void
 */
function wpmfAddonAvadaEnqueueSeparateLiveScripts()
{
    wp_enqueue_script('jquery-masonry');
    $js_folder_url = FUSION_LIBRARY_URL . '/assets' . ((true === FUSION_LIBRARY_DEV_MODE) ? '' : '/min') . '/js';
    wp_enqueue_script('isotope', $js_folder_url . '/library/isotope.js', array(), FUSION_BUILDER_VERSION, true);
    wp_enqueue_script('packery', $js_folder_url . '/library/packery.js', array(), FUSION_BUILDER_VERSION, true);
    wp_enqueue_script('images-loaded', $js_folder_url . '/library/imagesLoaded.js', array(), FUSION_BUILDER_VERSION, true);
    wp_enqueue_script(
        'wpmf-fusion-slick-script',
        WPMF_PLUGIN_URL . 'assets/js/slick/slick.min.js',
        array('jquery'),
        WPMF_VERSION,
        true
    );

    wp_enqueue_script(
        'wpmf-justified-script',
        WPMF_GALLERY_ADDON_PLUGIN_URL . 'assets/js/justified-gallery/jquery.justifiedGallery.min.js',
        array('jquery'),
        WPMF_GALLERY_ADDON_VERSION,
        true
    );

    wp_enqueue_script(
        'wpmf-fusion-flipster-js',
        WPMF_GALLERY_ADDON_PLUGIN_URL . 'assets/js/jquery.flipster.js',
        array('jquery'),
        WPMF_GALLERY_ADDON_VERSION,
        true
    );

    wp_enqueue_script('wpmf_fusion_view_gallery_addon_element', WPMF_GALLERY_ADDON_PLUGIN_URL . 'avada-widgets/js/avada.js', array(), WPMF_GALLERY_ADDON_VERSION, true);
}

/**
 * Get gallery default params
 *
 * @return array
 */
function wpmfGalleryAddonGetDefaultParams()
{
    return array(
        'layout' => 'vertical',
        'row_height' => 200,
        'aspect_ratio' => 'default',
        'columns' => 3,
        'size' => 'medium',
        'targetsize' => 'large',
        'link' => 'file',
        'wpmf_orderby' => 'date',
        'wpmf_order' => 'DESC',
        'display_tree' => 0,
        'display_tag' => 0,
        'sub_galleries_listing' => 0,
        'disable_overlay' => 0,
        'download_all' => 0,
        'animation' => 'slide',
        'duration' => 4000,
        'auto_animation' => 1,
        'number_lines' => 1,
        'show_buttons' => 1,
        'auto_from_folder' => 0,
        'tree_width' => 250,
        'folder' => 0,
        'google_photos_album' => 0,
        'include_children' => 0,
        'hover_color' => '#000',
        'hover_opacity' => '0.4',
        'hover_title_position' => 'center_center',
        'hover_title_size' => 16,
        'hover_title_color' => '#fff',
        'hover_desc_position' => 'none',
        'hover_desc_size' => 14,
        'hover_desc_color' => '#fff',
        'gutterwidth' => 5
    );
}

/**
 * Create thumbnail
 *
 * @param integer $attachment_id Attachment ID
 * @param string  $size          Size name
 *
 * @return array|boolean
 */
function wpmfGalleryCreateThumbnail($attachment_id, $size)
{
    $uploadpath = wp_upload_dir();
    $sizes = explode('x', $size);
    $filepath = get_attached_file($attachment_id);
    // get origin file path, not contain watermark
    $pathinfo = pathinfo($filepath);
    $check_name_wtm       = 'imageswatermark';
    $name_change_file_wtm = pathinfo($pathinfo['basename'], PATHINFO_FILENAME) . $check_name_wtm;
    $name_change_file_wtm .= '.' . pathinfo($pathinfo['basename'], PATHINFO_EXTENSION);
    $newfile              = $pathinfo['dirname'] . '/' . $name_change_file_wtm;
    if (file_exists($newfile)) {
        $filepath = $newfile;
    }

    if (!file_exists($filepath)) {
        return false;
    }

    $infopath = pathinfo($filepath);
    list($width, $height) = getimagesize($filepath);
    if (empty($width)) {
        return false;
    }
    $new_width = $sizes[0];
    $new_height = floor($height * ($sizes[0] / $width));

    $file_url = wp_get_attachment_image_url($attachment_id, 'full');
    $info_url = pathinfo($file_url);
    $filename = $info_url['filename'] . '-' . $new_width . 'x' . $new_height . '.' . $info_url['extension'];
    $download_url =  $uploadpath['baseurl'] . '/wpmf-gallery/' . $filename;
    $intermediate_file = $uploadpath['basedir'] . '/wpmf-gallery/' . $filename;
    if (!file_exists($intermediate_file)) {
        $thumb = imagecreatetruecolor($new_width, $new_height);

        switch ($infopath['extension']) {
            case 'jpeg':
            case 'jpg':
                $old_image = imagecreatefromjpeg($filepath);
                break;
            case 'png':
                $old_image = imagecreatefrompng($filepath);
                break;
            case 'gif':
                $old_image = imagecreatefromgif($filepath);
                break;
            case 'webp':
                $old_image = imagecreatefromwebp($filepath);
                break;
            case 'bmp':
                $old_image = imagecreatefromwbmp($filepath);
                break;
        }


        imagealphablending($thumb, false);
        imagecopyresampled($thumb, $old_image, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
        switch ($infopath['extension']) {
            case 'jpeg':
            case 'jpg':
                imagejpeg($thumb, path_join($uploadpath['basedir'], $intermediate_file), 100);
                break;

            case 'png':
                imagepng($thumb, path_join($uploadpath['basedir'], $intermediate_file), 9);
                break;

            case 'gif':
                imagegif($thumb, path_join($uploadpath['basedir'], $intermediate_file));
                break;

            case 'webp':
                imagewebp($thumb, path_join($uploadpath['basedir'], $intermediate_file), 100);
                break;
            case 'bmp':
                imagewbmp($thumb, path_join($uploadpath['basedir'], $intermediate_file));
                break;
        }
    }

    return array('intermediate_file' => $intermediate_file, 'download_url' => $download_url);
}

/**
 * Render watermark
 *
 * @param string $intermediate_file Image path
 * @param string $size              Size
 *
 * @return void
 */
function wpmfGalleryrenderWatermark($intermediate_file, $size)
{
    $option_image_watermark = get_option('wpmf_option_image_watermark');
    if (!empty($option_image_watermark)) {
        $pathinfo = pathinfo($intermediate_file);
        $imageInfo = getimagesize($intermediate_file);

        // remove file
        $check_name_wtm       = 'imageswatermark';
        $name_change_file_wtm = pathinfo($pathinfo['basename'], PATHINFO_FILENAME) . $check_name_wtm;
        $name_change_file_wtm .= '.' . pathinfo($pathinfo['basename'], PATHINFO_EXTENSION);
        $file                 = $pathinfo['dirname'] . '/' . $pathinfo['basename'];
        $newfile              = $pathinfo['dirname'] . '/' . $name_change_file_wtm;
        if (file_exists($newfile)) {
            unlink($file);
            copy($newfile, $file);
            unlink($newfile);
        }

        $photograper_image_watermark_apply = wpmfGetOption('photograper_image_watermark_apply');
        if (!empty($photograper_image_watermark_apply[$size]) || !empty($photograper_image_watermark_apply['all_size'])) {
            require_once(WP_MEDIA_FOLDER_PLUGIN_DIR . 'class/class-image-watermark.php');
            $wpmfwatermark = new WpmfWatermark();
            try {
                if (!empty($pathinfo)) {
                    $wpmfwatermark->generatePicture($pathinfo['basename'], $imageInfo, $pathinfo['dirname'], $size, 'photographer');
                }
            } catch (Exception $e) {
                echo esc_html($e->getMessage());
            }
        }
    }
}

/**
 * Convert hex to rgba
 *
 * @param string          $color   Color code
 * @param boolean|integer $opacity Opacity
 *
 * @return string
 */
function wpmfConvertHex2rgba($color, $opacity = false)
{
    $default = 'rgb(0,0,0)';
    if (empty($color)) {
        return $default;
    }

    if ($color[0] === '#') {
        $color = substr($color, 1);
    }

    if (strlen($color) === 6) {
        $hex = array($color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5]);
    } elseif (strlen($color) === 3) {
        $hex = array($color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2]);
    } else {
        return $default;
    }

    $rgb = array_map('hexdec', $hex);
    if ($opacity) {
        if (abs($opacity) > 1) {
            $opacity = 1.0;
        }

        $output = 'rgba(' . implode(',', $rgb) . ',' . $opacity . ')';
    } else {
        $output = 'rgb(' . implode(',', $rgb) . ')';
    }

    return $output;
}

/**
 * Get gallery params
 *
 * @param integer $gallery_id Gallery ID
 *
 * @return array
 */
function wpmfGalleryAddonGetParams($gallery_id)
{
    $default_params = wpmfGalleryAddonGetDefaultParams();
    $gallery_configs = get_option('wpmf_gallery_settings');
    $gallery_type = get_term_meta((int)$gallery_id, 'gallery_type', true);
    if (!empty($gallery_type) && $gallery_type === 'photographer') {
        $settings         = get_option('wpmf_settings');
        if (isset($settings) && isset($settings['photograper_dimension'])) {
            $selected_dimensions = $settings['photograper_dimension'];
        } else {
            $selected_dimensions = wpmfGetOption('photograper_dimension');
        }

        if (!empty($selected_dimensions)) {
            $gallery_configs['theme']['masonry_theme']['size'] = (isset($selected_dimensions[1])) ? $selected_dimensions[1] : $selected_dimensions[0];
            $gallery_configs['theme']['masonry_theme']['targetsize'] = (isset($selected_dimensions[2])) ? $selected_dimensions[2] : $selected_dimensions[0];
        }
    }
    $galleries = get_option('wpmf_galleries');
    if (!empty($galleries[$gallery_id])) {
        $theme = $galleries[$gallery_id]['theme'];
        if (empty($gallery_configs) || empty($gallery_configs['theme'][$theme . '_theme'])) {
            $theme_configs = array();
        } else {
            $theme_configs = $gallery_configs['theme'][$theme . '_theme'];
            if (empty($theme_configs['wpmf_orderby']) && !empty($theme_configs['orderby'])) {
                $theme_configs['wpmf_orderby'] = $theme_configs['orderby'];
            }
            if (empty($theme_configs['wpmf_order']) && !empty($theme_configs['order'])) {
                $theme_configs['wpmf_order'] = $theme_configs['order'];
            }
        }

        if (empty($galleries[$gallery_id])) {
            $current_gallery_configs = array();
        } else {
            $current_gallery_configs = $galleries[$gallery_id];
            if (empty($current_gallery_configs['wpmf_orderby']) && !empty($current_gallery_configs['orderby'])) {
                $current_gallery_configs['wpmf_orderby'] = $current_gallery_configs['orderby'];
            }
            if (empty($current_gallery_configs['wpmf_order']) && !empty($current_gallery_configs['order'])) {
                $current_gallery_configs['wpmf_order'] = $current_gallery_configs['order'];
            }
        }

        $params = array_merge($default_params, $theme_configs, $current_gallery_configs);
    } else {
        $params = array();
    }

    return $params;
}

/**
 * Get image type
 *
 * @return array
 */
function wpmfGalleryAddonGetImageType()
{
    return array('image/jpg', 'image/jpeg', 'image/gif', 'image/png', 'image/bmp', 'image/tiff', 'image/x-icon', 'image/webp', 'image/avif');
}

/**
 * Get tax query
 *
 * @param integer $gallery_id Gallery ID
 * @param array   $attrs      Attributes
 *
 * @return array
 */
function wpmfGalleryAddonGetTaxQuery($gallery_id, $attrs = array())
{
    $relationships = get_option('wpmfgrl_relationships');
    $params = wpmfGalleryAddonGetParams($gallery_id);
    $tax_query = array();
    $tax_query[] = array(
        'taxonomy'         => WPMF_GALLERY_ADDON_TAXO,
        'field'            => 'term_id',
        'terms'            => $gallery_id,
        'include_children' => (!empty($attrs['include_children'])) ? true : false
    );

    if (isset($relationships[$gallery_id])) {
        $tax_query['relation'] = 'OR';
        $tax_query[] = array(
            'taxonomy'         => WPMF_TAXO,
            'field'            => 'term_id',
            'terms'            => (int) $relationships[$gallery_id],
            'include_children' => false
        );
    }

    // get gallery from folder
    if (!empty($params) && (int)$params['auto_from_folder'] === 1) {
        $tax_query['relation'] = 'OR';
        if (!empty($params['folder'])) {
            $tax_query[] = array(
                'taxonomy' => WPMF_TAXO,
                'field' => 'term_id',
                'terms' => (int)$params['folder'],
                'include_children' => (!empty($attrs['include_children'])) ? true : false
            );
        } else {
            $terms = get_categories(array('hide_empty' => false, 'taxonomy' => WPMF_TAXO));
            $cats  = array();
            foreach ($terms as $term) {
                if (!empty($term->term_id)) {
                    $cats[] = $term->term_id;
                }
            }

            $tax_query[] = array(
                'taxonomy' => WPMF_TAXO,
                'field' => 'term_id',
                'terms' => $cats,
                'operator' => 'NOT IN'
            );
        }
    }

    return $tax_query;
}

/**
 * Get video thumbnail ID
 *
 * @param integer $attachmentID Attachment ID
 *
 * @return integer
 */
function wpmfGalleryGetVideoThumbID($attachmentID)
{
    $video_thumbnail_id = get_post_meta($attachmentID, 'wpmf_video_thumbnail_id', true);
    $thumb_id = (!empty($video_thumbnail_id)) ? (int)$video_thumbnail_id : $attachmentID;
    return $thumb_id;
}

/**
 * Resort gallery
 *
 * @param array $a Gallery list
 * @param array $b Gallery list
 *
 * @return mixed
 */
function wpmfGalleryReorder($a, $b)
{
    return $a->order - $b->order;
}

/**
 * Get attachment download link
 *
 * @param object $attachment Attachment
 *
 * @return false|string|string[]
 */
function wpmfGalleryGetDownloadLink($attachment)
{
    $drive_type = get_post_meta($attachment->ID, 'wpmf_drive_type', true);
    if (empty($drive_type)) {
        if (!empty($attachment->type) && $attachment->type === 'google_photo') {
            $download_link = $attachment->url;
            $type = 'google_photo';
        } else {
            $download_link = wp_get_attachment_image_url($attachment->ID, 'full');
            $type = 'local';
        }
    } else {
        $drive_id = get_post_meta($attachment->ID, 'wpmf_drive_id', true);
        switch ($drive_type) {
            case 'onedrive':
                $download_link = admin_url('admin-ajax.php') . '?action=wpmf_onedrive_download&id=' . urlencode($drive_id) . '&link=true&dl=1';
                break;

            case 'onedrive_business':
                $download_link = admin_url('admin-ajax.php') . '?action=wpmf_onedrive_business_download&id=' . urlencode($drive_id) . '&link=true&dl=1';
                break;

            case 'google_drive':
                $download_link = admin_url('admin-ajax.php') . '?action=wpmf-download-file&id=' . urlencode($drive_id) . '&dl=1';
                break;

            case 'dropbox':
                $download_link = admin_url('admin-ajax.php') . '?action=wpmf-dbxdownload-file&id=' . urlencode($drive_id) . '&link=true&dl=1';
                break;
            default:
                $download_link = wp_get_attachment_image_url($attachment->ID, 'full');
        }

        $download_link = str_replace('&amp;', '&', $download_link);
        $download_link = str_replace('&#038;', '&', $download_link);
        $type = 'cloud';
    }

    return array('download_link' => $download_link, 'type' => $type);
}

/**
 * Render gallery comment lists
 *
 * @param integer $post_id Attachment ID
 * @param integer $parent  Parent comment
 *
 * @return string
 */
function wpmfGalleryShowComments($post_id, $parent = 0)
{
    $html = '';
    $comments = get_comments(array('post_id' => $post_id, 'parent' => $parent));
    if (!empty($comments)) {
        $html .= '<ul>';
        foreach ($comments as $comment) {
            $sup = '';
            /*if (!empty($comment->comment_author_email)) {
                $user = get_user_by('email', $comment->comment_author_email);
                if (!empty($user)) {
                    if (in_array('administrator', $user->roles)) {
                        $sup = '<sup>'. esc_html__('Photograper', 'wp-media-folder-gallery-addon') .'</sup>';
                    }
                }
            }*/

            $author = ($comment->comment_author === '') ? esc_html__('Guest', 'wp-media-folder-gallery-addon') : $comment->comment_author;
            $html .= '<li><div class="div_p"><span class="wpmf_gallery_author"><span>'. $author . '</span>'. $sup . '</span> <div>' . esc_html($comment->comment_content) . '<a href="#" class="wpmf_gallery_reply" data-attachment="'. (int)$post_id .'" data-comment="'. (int)$comment->comment_ID .'">'. esc_html__('Reply', 'wp-media-folder-gallery-addon') .'</a> <span>'. wpmfGetCommentTime($comment->comment_ID) .'</span></div></div>';
            $html .= wpmfGalleryShowComments($post_id, $comment->comment_ID);
            $html .= '</li>';
        }
        $html .= '</ul>';
    }
    return $html;
}

/**
 * Get comment time
 *
 * @param integer $comment_id Comment ID
 *
 * @return string
 */
function wpmfGetCommentTime($comment_id = 0)
{
    return sprintf(
        _x('%s ago', 'Human-readable time', 'wp-media-folder-gallery-addon'),
        human_time_diff(
            get_comment_date('U', $comment_id),
            current_time('timestamp')
        )
    );
}

/**
 * Check is cloud file disconnect
 *
 * @param integer $attachment_id Attachment ID
 *
 * @return boolean
 */
function wpmfCheckIsCloudFileDisconnect($attachment_id)
{
    $type = get_post_meta($attachment_id, 'wpmf_drive_type', true);
    $type = (!empty($type)) ? $type : '';
    $dropbox_config = get_option('_wpmfAddon_dropbox_config');
    $google_config = get_option('_wpmfAddon_cloud_config');
    $onedrive_config = get_option('_wpmfAddon_onedrive_config');
    $onedrive_business_config = get_option('_wpmfAddon_onedrive_business_config');
    if (empty($dropbox_config['dropboxToken'])) {
        if ($type === 'dropbox') {
            return true;
        }
    }

    if (empty($google_config['googleCredentials']) || empty($google_config['googleBaseFolder'])) {
        if ($type === 'google_drive') {
            return true;
        }
    }

    if (empty($onedrive_config['connected'])) {
        if ($type === 'onedrive') {
            return true;
        }
    }

    if (empty($onedrive_business_config['connected'])) {
        if ($type === 'onedrive_business') {
            return true;
        }
    }

    return false;
}

if (is_admin()) {
    if (!defined('JU_BASE')) {
        /**
         * Joomunited site url
         */
        define('JU_BASE', 'https://www.joomunited.com/');
    }

    $remote_updateinfo = JU_BASE . 'juupdater_files/wp-media-folder-gallery-addon.json';
    //end config

    require 'juupdater/juupdater.php';
    $UpdateChecker = Jufactory::buildUpdateChecker(
        $remote_updateinfo,
        __FILE__
    );
}

if (!function_exists('wpmfGalleryAddonPluginCheckForUpdates')) {
    /**
     * Plugin check for updates
     *
     * @param object $update      Update
     * @param array  $plugin_data Plugin data
     * @param string $plugin_file Plugin file
     *
     * @return array|boolean|object
     */
    function wpmfGalleryAddonPluginCheckForUpdates($update, $plugin_data, $plugin_file)
    {
        if ($plugin_file !== 'wp-media-folder-gallery-addon/wp-media-folder-gallery-addon.php') {
            return $update;
        }

        if (empty($plugin_data['UpdateURI']) || !empty($update)) {
            return $update;
        }

        $response = wp_remote_get($plugin_data['UpdateURI']);

        if (is_wp_error($response) || empty($response['body'])) {
            return $update;
        }

        $custom_plugins_data = json_decode($response['body'], true);

        $package = null;
        $token = get_site_option('wpmf_license_token');
        if (!empty($token)) {
            $package = $custom_plugins_data['download_url'] . '&token=' . $token . '&siteurl=' . get_option('siteurl');
        }

        return array(
            'version' => $custom_plugins_data['version'],
            'package' => $package
        );
    }
    add_filter('update_plugins_www.joomunited.com', 'wpmfGalleryAddonPluginCheckForUpdates', 10, 3);
}

/**
 * Check is active wp media folder gallery addon
 *
 * @return void
 */
function wp_media_folder_gallery_addon_install_action()
{
    // Check if the plugin is activated and is the 'wp-media-folder-gallery-addon'
    if (is_plugin_active('wp-media-folder-gallery-addon/wp-media-folder-gallery-addon.php')) {
        wp_media_folder_gallery_addon_activate();
    }
}

// Hook into the plugin activation process
register_activation_hook(__FILE__, 'wp_media_folder_gallery_addon_install_action');

/**
 * The function to be called after plugin activation
 *
 * @return void
 */
function wp_media_folder_gallery_addon_activate()
{
    $update_tags = get_option('_wpmfAddon_update_tags');
    
    if (!$update_tags) {
        // Get all attachments
        $all_attachment = get_posts(array(
            'post_type' => 'attachment',
            'numberposts' => -1
        ));

        // Get attachment IDs
        $attachment_ids = wp_list_pluck($all_attachment, 'ID');
        $update_terms = array();

        // Fetch metadata for each attachment ID
        foreach ($attachment_ids as $attachment_id) {
            // Get 'wpmf_img_tags' metadata for each attachment
            $img_tags_meta = get_post_meta($attachment_id, 'wpmf_img_tags', true);
            $array_img_tags = $img_tags_meta ? array_map('trim', explode(',', $img_tags_meta)) : array();

            // Get terms from 'wpmf_tag' taxonomy for each attachment
            $terms = wp_get_object_terms($attachment_id, 'wpmf_tag');
            $old_terms = array();

            if (!is_wp_error($terms) && !empty($terms)) {
                foreach ($terms as $term) {
                    // Index terms by attachment ID to manage them more easily
                    $old_terms[] = $term->name;
                }
            }

            $new_terms = array_unique(array_merge($array_img_tags, $old_terms));

            if ($new_terms !== $old_terms) {
                $update_terms[$attachment_id] = $new_terms;
            }
        }

        // Update tags for attachments that have changed
        foreach ($update_terms as $attachment_id => $new_terms) {
            wp_set_post_terms($attachment_id, $new_terms, 'wpmf_tag');
        }

        // Mark as tags updated for this addon
        update_option('_wpmfAddon_update_tags', 1);
    }
}

// Hook to execute when the plugin is updated
add_action('upgrader_process_complete', 'wp_media_folder_gallery_addon_install_action', 10, 2);
