<?php
/* Prohibit direct script loading */
defined('ABSPATH') || die('No direct script access allowed!');

/**
 * Class WpmfBunnyStorage
 * This class that holds most of the admin functionality for Bunny Storage integration
 */
class WPMFAddonBunnyStorage
{
    /**
     * Storage Zone Name
     *
     * @var string
     */
    private $storageZone;

    /**
     * API Key
     *
     * @var string
     */
    private $apiKey;

    /**
     * Region
     *
     * @var string
     */
    private $region;

    /**
     * Base url preview
     *
     * @var string
     */
    private $baseUrl;

    /**
     * WPMFAddonBunnyStorage constructor.
     *
     * @param string $storageZone Storage zone name
     * @param string $apiKey      Access key
     * @param string $hostName    Host name
     * @param string $pullZone    Pull zone name
     */
    public function __construct($storageZone, $apiKey, $hostName, $pullZone)
    {
        $this->storageZone = $storageZone;
        $this->apiKey = $apiKey;
        $parts = explode('.', $hostName);
        $this->region = isset($parts[0]) && $parts[0] !== 'storage' ? $parts[0] : 'default';
        $this->baseUrl = 'https://' . $pullZone . '.b-cdn.net';
    }

    /**
     * Upload file to Bunny Storage
     *
     * @param string $remotePath    Remote path
     * @param string $localFilePath Local file path
     *
     * @return boolean True on success, false on failure
     */
    public function upload($remotePath, $localFilePath)
    {
        if ($this->region === 'default') {
            $url = 'https://storage.bunnycdn.com/' . $this->storageZone . '/' . $remotePath;
        } else {
            $url = 'https://'. $this->region . '.storage.bunnycdn.com/' . $this->storageZone . '/' . $remotePath;
        }

        $response = wp_remote_request($url, [
            'method'  => 'PUT',
            'headers' => [
                'AccessKey' => $this->apiKey,
                'Content-Type' => 'application/octet-stream',
            ],
            'body' => file_get_contents($localFilePath),
        ]);

        return !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 201;
    }

    /**
     * Delete file from Bunny Storage
     *
     * @param string $remotePath Remote path
     *
     * @return boolean True on success, false on failure
     */
    public function delete($remotePath)
    {
        if ($this->region === 'default') {
            $url = 'https://storage.bunnycdn.com/' . $this->storageZone . '/' . $remotePath;
        } else {
            $url = 'https://' . $this->region . '.storage.bunnycdn.com/' . $this->storageZone . '/' . $remotePath;
        }

        $response = wp_remote_request($url, [
            'method'  => 'DELETE',
            'headers' => [
                'AccessKey' => $this->apiKey,
            ],
        ]);

        return !is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200;
    }

    /**
     * Get public URL for file
     *
     * @param string $remotePath Remote path
     *
     * @return string Public URL
     */
    public function getPublicUrl($remotePath)
    {
        return trailingslashit($this->baseUrl) . ltrim($remotePath, '/');
    }

    /**
     * List files in folder (optional)
     *
     * @param string $folderPath Folder path
     *
     * @return array List of files and folders from Bunny storage
     */
    public function listFiles($folderPath = '')
    {
        if ($this->region === 'default') {
            $url = 'https://storage.bunnycdn.com/' . $this->storageZone . '/' . $folderPath;
        } else {
            $url = 'https://' . $this->region . '.storage.bunnycdn.com/' . $this->storageZone . '/' . $folderPath;
        }

        $response = wp_remote_request($url, [
            'method'  => 'GET',
            'headers' => [
                'AccessKey' => $this->apiKey,
            ],
        ]);

        if (is_wp_error($response)) {
            return [];
        }

        $body = wp_remote_retrieve_body($response);
        return json_decode($body, true);
    }

    /**
     * Check connect to storage
     *
     * @return boolean True on success, false on failure
     */
    public function isConnected()
    {
        if ($this->region === 'default') {
            $url = 'https://storage.bunnycdn.com/' . $this->storageZone . '/';
        } else {
            $url = 'https://' . $this->region . '.storage.bunnycdn.com/' . $this->storageZone . '/';
        }

        $response = wp_remote_request($url, [
            'method'  => 'GET',
            'headers' => [
                'AccessKey' => $this->apiKey,
            ],
        ]);

        $code = wp_remote_retrieve_response_code($response);
        return !is_wp_error($response) && $code === 200;
    }
}
