<?php
/*
Plugin Name: WP FSAM - File Access Manager
Description: Enables admin to upload files and grant users rights to download.
Version: 1.0
Author: پرشین اسکریپت
Author URI: http://www.persianscript.ir
*/


class WP_FSAM_File_Sharing_Access_Manager {

	public $projects_table;
	public $categories_table;
	public $files_table;
	public $plugin_permalink;
	public $blog_title;
	public $upload_path;
	public $templates;

	// all the magic
 	public function __construct() {

 		// set files table name
 		global $wpdb;
		$this->projects_table = $wpdb->prefix . 'fsam_files';
		$this->categories_table = $wpdb->prefix . 'fsam_categories';
		$this->users_table = $wpdb->prefix . 'fsam_users';
		$this->files_table = $wpdb->prefix . 'fsam_uploads';

		$this->blog_title = get_bloginfo('name');

		// set plugin permalink base
		$this->plugin_permalink = 'wpfsam-manage';

 		// install DB Table on Plugin Activation
 		register_activation_hook( __FILE__, array( $this, 'install_DB' ));

 		// add administration menu
 		add_action( 'admin_menu', array( $this, 'setup_menu' ));
 		
 		// enable output control
 		add_action( 'init', array( $this, 'output_ctrl' ));

 		// add plugin shortcode to show files to users loop
 		add_shortcode( 'wp_fsam_user_files', array( $this, 'shortcode' ));
 		add_shortcode( 'wp_fsam_user_login', array( $this, 'login' ));
 		add_shortcode( 'wp_fsam_reset_password', array( $this, 'reset_password' ));

 		// add JS	
 		add_action( 'admin_enqueue_scripts', array( $this, 'JS' ));

 		// admin ajax
 		add_action( 'wp_ajax_wpfsam_sort_files', array($this, 'ajax_sort' ));

 		// HTML Email
 		add_filter('wp_mail_content_type', array( $this, 'set_content_type') );

 		// set upload path protected by .htaccess
 		$this->upload_path = plugin_dir_path( __FILE__ ) . '/protected_uploads/';

 		// Add a filter to the attributes metabox to inject template into the cache.
        if ( version_compare( floatval( get_bloginfo( 'version' ) ), '4.7', '<' ) ) {

            // 4.6 and older
            add_filter( 'page_attributes_dropdown_pages_args', array( $this, 'register_templates' ));

        } else {

            // Add a filter to the wp 4.7 version attributes metabox
            add_filter('theme_page_templates', array( $this, 'add_new_template' ));

        }

        // Add a filter to the save post to inject out template into the page cache
        add_filter( 'wp_insert_post_data',  array( $this, 'register_templates' ) );

        // Add a filter to the template include to determine if the page has our 
        // template assigned and return it's path
        add_filter( 'template_include',  array( $this, 'download_template') );

        // add templates to the array
        $this->templates = array(plugin_dir_path( __FILE__ ) . '/views/download-template.php' => 'WpFsamDownloads');

 	}

 	// register templates
    public function register_templates( $atts ) {

        // Create the key used for the themes cache
        $cache_key = 'page_templates-' . md5( get_theme_root() . '/' . get_stylesheet() );

        // Retrieve the cache list. 
        // If it doesn't exist, or it's empty prepare an array
        $templates = wp_get_theme()->get_page_templates();
        if ( empty( $templates ) ) {
            $templates = array();
        } 

        // New cache, therefore remove the old one
        wp_cache_delete( $cache_key , 'themes');

        // Now add our template to the list of templates by merging our templates
        // with the existing templates array from the cache.
        $templates = array_merge( $templates, $this->templates );

        // Add the modified cache to allow WordPress to pick it up for listing
        // available templates
        wp_cache_add( $cache_key, $templates, 'themes', 1800 );

        return $atts;

    }

    // check if current page needs our template
    public function download_template( $template ) {
        
        // Get global post
        global $post;

        // Return template if post is empty
        if ( ! $post ) {
            return $template;
        }

        // Return default template if we don't have a custom one defined
        if ( !isset( $this->templates[get_post_meta( 
            $post->ID, '_wp_page_template', true 
        )] ) ) {
            return $template;
        } 

        $file = get_post_meta( $post->ID, '_wp_page_template', true);

        // Just to be safe, we check if the file exist first
        if ( file_exists( $file ) ) {
            return $file;
        } else {
            echo $file;
        }

        // Return template
        return $template;

    }

    // Adds our template to the page dropdown for v4.7+
    public function add_new_template( $posts_templates ) {
        $posts_templates = array_merge( $posts_templates, $this->templates );
        return $posts_templates;
    }

 	// ajax sorting order
 	public function ajax_sort() {

 		if (!current_user_can('manage_options')) {
	        wp_die('You do not have sufficient permissions to access this page.');
	    }

		global $wpdb; // this is how you get access to the database

		if(array_key_exists('order', $_POST)) {

			$image_update = update_option('wpfsam_order', $_POST['order']);
			
			if($image_update) {
				echo '<div class="updated below-h2">Order updated</div>';
			}else{
				echo '<div class="updated below-h2">Order failed to update</div>';
				#$wpdb->show_errors();
				#$wpdb->print_error();
			}

		}else{
			die("AJAX: Sorting order must be in the array");
		}
		
		die(); // this is required to return a proper result
 	}


 	// HTML email filter
 	public function set_content_type($content_type){
		return 'text/html';
	}

 	// enqueue JS required by the plugin
 	public function JS() {
 			wp_enqueue_media();

 			wp_enqueue_script('thickbox');
			wp_enqueue_style('thickbox');

			wp_enqueue_script('jquery');
			wp_enqueue_script('jquery-ui-sortable');			

			// plugin custom JS/css
			wp_register_script( 'wpfsam_admin_js', plugin_dir_url(__FILE__) . 'static/js.js', false, '1.0.0' );
    		wp_enqueue_script( 'wpfsam_admin_js' );

    		wp_register_style( 'wpfsam_admin_css', plugin_dir_url(__FILE__) . 'static/wp-fsam.css', false, '1.0.0' );
    		wp_enqueue_style( 'wpfsam_admin_css' );
 	}

 	// output control
 	public function output_ctrl(){
		ob_start();
		session_start();
	}

	// a message div 
	public function message( $message ) {
		return '<div class="updated below-h2">' . $message . '</div>';
	}

 	// create HTML wrap structure
 	public function wrap( $type = 'footer', $title = '' ) {

		if( $type == 'header' ) {
			$return = '<div class="wrap">';
			$return .= sprintf( '<h2>%s</h2>', $title );

		}elseif( $type == 'footer' ) {
			$return = '</div>';
		}

		return $return;
	}


	// install DB table on plugin activation
	public function install_DB() {
		global $wpdb;

		$wpdb->query("DROP TABLE IF EXISTS " . $this->projects_table);
		$wpdb->query("DROP TABLE IF EXISTS " . $this->categories_table);
		$wpdb->query("DROP TABLE IF EXISTS " . $this->users_table);
		$wpdb->query("DROP TABLE IF EXISTS " . $this->files_table);

		$SQL = 'CREATE TABLE IF NOT EXISTS `' . $this->projects_table . '` (
				  `fileID` int(10) unsigned NOT NULL AUTO_INCREMENT,
				  `title` varchar(255) NOT NULL,
				  `description` text NOT NULL,
				  `categories` varchar(255) NOT NULL,
				  `file` varchar(255) NOT NULL,
				  PRIMARY KEY (`fileID`)
				)';

		$SQL_Categories = 'CREATE TABLE IF NOT EXISTS `' . $this->categories_table . '` (
			  `catID` int(10) unsigned NOT NULL AUTO_INCREMENT,
			  `name` varchar(255) NOT NULL,
			  PRIMARY KEY (`catID`)
			  )';

		$SQL_Users = 'CREATE TABLE IF NOT EXISTS `' . $this->users_table . '` (
					  `userID` int(10) unsigned NOT NULL AUTO_INCREMENT,
					  `name` varchar(255) NOT NULL,
					  `email` varchar(255) NOT NULL,
					  `password` varchar(32) NOT NULL,
					  `client` varchar(255) NOT NULL,
					  `categories` varchar(255) NOT NULL,
					  PRIMARY KEY (`userID`)
				)';

		$SQL_Files = 'CREATE TABLE IF NOT EXISTS `' . $this->files_table . '` (
			  `fileID` int(10) unsigned NOT NULL AUTO_INCREMENT,
			  `name` varchar(255) NOT NULL,
			  PRIMARY KEY (`fileID`)
			  )';
	
		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

		dbDelta( $SQL );
		dbDelta( $SQL_Categories );
		dbDelta( $SQL_Users );
		dbDelta( $SQL_Files );

		// create user files page
		if( !get_option( 'wpfsam_user_files_page' ) ) {

	        // CREATE RUN A SEO REPORT PAGE
	        $page = array(
	          'post_title'    => 'WPFSAM Files Access',
	          'post_content'  => '[wp_fsam_user_files]',
	          'post_status'   => 'publish',
	          'post_author'   => 1,
	          'post_type'     => 'page',
	        );

	        $id = wp_insert_post( $page );
			update_option( 'wpfsam_user_files_page', $id );

    	}// ./create_user_files_page

        // create user login page
		if( !get_option( 'wp_fsam_user_login_page' ) ) {

	        // CREATE RUN A SEO REPORT PAGE
	        $page = array(
	          'post_title'    => 'WPFSAM Login - Files Access',
	          'post_content'  => '[wp_fsam_user_login]',
	          'post_status'   => 'publish',
	          'post_author'   => 1,
	          'post_type'     => 'page',
	        );

	        $id = wp_insert_post( $page );
			update_option( 'wp_fsam_user_login_page', $id );

    	}// ./create_user_login_page

        // create user recover password
		if( !get_option( 'wp_fsam_reset_password_page' ) ) {

	        // CREATE RUN A SEO REPORT PAGE
	        $page = array(
	          'post_title'    => 'WPFSAM Reset Login - Files Access',
	          'post_content'  => '[wp_fsam_reset_password]',
	          'post_status'   => 'publish',
	          'post_author'   => 1,
	          'post_type'     => 'page',
	        );

	        $id = wp_insert_post( $page );
			update_option( 'wp_fsam_reset_password_page', $id );

    	}// ./create_user_recover_password

    	// create download page
		if( !get_option( 'wp_fsam_download_page' ) ) {

	        // CREATE RUN A SEO REPORT PAGE
	        $page = array(
	          'post_title'    => 'WP FSAM Download Page',
	          'post_content'  => '[wp_fsam_download_page]',
	          'post_status'   => 'publish',
	          'post_author'   => 1,
	          'post_type'     => 'page',
	          'page_template' => plugin_dir_path( __FILE__ ) . '/views/download-template.php'
	        );

	        $id = wp_insert_post( $page );
			update_option( 'wp_fsam_download_page', $id );

    	}// ./create_download_page

	}


 	// projects overview table
 	public function manage_files() {

 		global $wpdb;

 		require_once plugin_dir_path( __FILE__ ) . '/views/admin-nav.php';
 		echo $this->wrap( 'header', 'Manage Files' );

 		if( isset( $_GET['remove'] )) {
 			if( $wpdb->delete( $this->projects_table, array( 'fileID' => $_GET['remove'] ))) {
 				echo $this->message( 'File successfully removed.' );
 			}
 		}

 		$file_order = get_option('wpfsam_order');

 		if($file_order) 
 			$file_order = 'FIND_IN_SET(fileID, "'.implode(',', $file_order).'")';
 		else
 			$file_order = 'fileID DESC';

 		$files = $wpdb->get_results( "SELECT fileID, title, 
 									(SELECT GROUP_CONCAT(name SEPARATOR ', ') FROM " . $this->categories_table . " WHERE find_in_set( catID, categories) )AS categories 
 									FROM " . $this->projects_table . " ORDER BY " . $file_order . "");

 		if( isset($_GET['file'] )) {

 			// process form
	 		if( isset($_POST['sb-new-project']) ) :

	 			unset( $_POST['sb-new-project'] );

	 			// upload & add the file to db
	 			if(isset($_FILES['cofm_file']) AND ($_FILES['cofm_file']['error'] == 0)) {
	 				$file = $_FILES['cofm_file'];
	 				$filename = preg_replace('/[^\w\d\.]+/', '-', $file['name']);

	 				if(move_uploaded_file($file['tmp_name'], $this->upload_path . $filename)) {

	 					// here insert the file into database and return it's id
	 					$wpdb->insert( $this->files_table, array('name' => $filename) );
	 					$_POST['file'] = $wpdb->insert_id;

	 				}else{
	 					echo $this->message( 'File upload failed' );
	 				}

	 			}

	 			if(isset($_POST['categories'])) 
	 				$_POST['categories'] = implode(',', $_POST['categories']);
	 			else
	 				$_POST['categories'] = '';

	 			if( $wpdb->update( $this->projects_table, $_POST, array( 'fileID' => $_GET['file'] ) ))
	 				echo $this->message( 'File successfully updated. Go to <a href="admin.php?page=' . $this->plugin_permalink . '">Files Overview</a>' );

	 		endif;
	 		// form processing ends

 			$f = $wpdb->get_row( $wpdb->prepare( "SELECT fileID, title, description, categories, file 
 									FROM " . $this->projects_table . " WHERE fileID = %d", $_GET['file']) );

 			if(!$f) wp_die( 'No file with this #ID ');

 			$cats = $wpdb->get_var("SELECT GROUP_CONCAT(catID SEPARATOR ', ') FROM " . $this->categories_table . "
 										WHERE find_in_set(catID, '".$f->categories."')");

 			if(!$cats) $cats = array();

 			$all_cats = $wpdb->get_results("SELECT * FROM " . $this->categories_table . " ORDER BY name ASC");

 		}

 		include_once 'views/files-overview.php';	

 		echo $this->wrap();

 	}


 	// new project form
 	public function new_file() {
 		global $wpdb;

 		require_once plugin_dir_path( __FILE__ ) . '/views/admin-nav.php';
 		echo $this->wrap( 'header', 'Add File' );



 		// process form
 		if( isset($_POST['sb-new-project']) ) :

 			unset( $_POST['sb-new-project'] );

 			// upload & add the file to db
 			if(isset($_FILES['cofm_file']) AND ($_FILES['cofm_file']['error'] == 0)) {
 				$file = $_FILES['cofm_file'];
 				$filename = preg_replace('/[^\w\d\.]+/', '-', $file['name']);

 				if(move_uploaded_file($file['tmp_name'], $this->upload_path . $filename)) {

 					// here insert the file into database and return it's id
 					$wpdb->insert( $this->files_table, array('name' => $filename) );
 					$_POST['file'] = $wpdb->insert_id;

 				}else{
 					echo $this->message( 'File upload failed' );
 				}

 			}

 			if(isset($_POST['categories'])) 
 				$_POST['categories'] = implode(',', $_POST['categories']);
 			else
 				$_POST['categories'] = '';

 			if( $wpdb->insert( $this->projects_table, $_POST ))
 				echo $this->message( 'File successfully added to the website. Go to <a href="admin.php?page=' . $this->plugin_permalink . '">Files Overview</a>' );
 			else
 				echo $this->message( 'SQL Error ' );

 		endif;
 		// form processing ends

 		$categories = $wpdb->get_results("SELECT * FROM " . $this->categories_table . " ORDER BY name ASC");
 		include_once 'views/new-file.php';

 		echo $this->wrap();

 	}


 	// shortcode
 	public function shortcode() {
 		ob_start();
 		global $wpdb;

 		if(!isset($_SESSION['cg_user'])) {
 			$url = get_post_field( 'post_name', get_option( 'wp_fsam_user_login_page' ) );
 			return '<meta http-equiv="refresh" content="0; url=/'.$url.'" />';
 		}

 		//select files depending on categories current logged in user has assigned
 		$user_categories = $wpdb->get_var($wpdb->prepare("SELECT categories FROM " . $this->users_table . " 
 										WHERE userID = %d", $_SESSION['cg_user']));

 		if(!$user_categories OR empty($user_categories)) 
 			return ' - No files are assigned to your account - ';

		$user_categories = explode(',', $user_categories);

		$Files_SQL = "SELECT * FROM " . $this->projects_table . " WHERE ";
		$SQL_IN_SET = '';
		foreach( $user_categories as $category ) $SQL_IN_SET .= 'FIND_IN_SET(' . $category . ', categories) OR ';

		$SQL_IN_SET = rtrim( $SQL_IN_SET, ' OR ');

		$user_files = $wpdb->get_results( $Files_SQL . $SQL_IN_SET . ' ORDER BY fileID DESC' );

		
 		if( count( $user_files ))
 			include_once 'views/files-user.php';
 		else
 			return ' - No files to show - ';

 		return ob_get_clean();
 	}

 	// categories management
 	public function manage_categories() {
 		global $wpdb;

 		// set active tab
 		$active = 'cats';
 		require_once plugin_dir_path( __FILE__ ) . '/views/admin-nav.php';

 		echo $this->wrap( 'header', 'File Categories' );

 		// remove category
 		if( isset( $_GET['delete'] )) {
 			if( $wpdb->delete( $this->categories_table, array( 'catID' => $_GET['delete'] ))) {
 				echo $this->message( 'Category successfully removed.' );
 			}
 		}

 		// update category
 		if( isset( $_POST['sb-save-category'] )) {
 			if( $wpdb->update( $this->categories_table, array( 'name' => $_POST['name']), array('catID' => $_POST['id']) )) {
 				echo $this->message( 'Category successfully updated.' );
 			}
 		}

 		// process form
 		if( isset($_POST['sb-new-category']) ) :

 			unset( $_POST['sb-new-category'] );

 			if( $wpdb->insert( $this->categories_table, array('name' => $_POST['name']) ))
 				echo $this->message( 'Category successfully added.' );
 			else
 				echo $this->message( 'SQL Error ' . $wpdb->last_error );

 		endif;
 		// form processing ends

 		$categories = $wpdb->get_results("SELECT * FROM " . $this->categories_table . " ORDER BY name ASC");
 		include_once 'views/file-categories.php';

 		echo $this->wrap();
 	}

 	// users management
 	public function manage_users() {
 		global $wpdb;

 		// set active tab
 		$active = 'users';
 		require_once plugin_dir_path( __FILE__ ) . '/views/admin-nav.php';

 		echo $this->wrap( 'header', 'Manage Users' );

 		if( isset( $_GET['remove'] )) {
 			if( $wpdb->delete( $this->users_table, array( 'userID' => $_GET['remove'] ))) {
 				echo $this->message( 'User successfully removed.' );
 			}
 		}

 		$users = $wpdb->get_results( "SELECT userID, name, email, client, 
 									(SELECT GROUP_CONCAT(name SEPARATOR ', ') FROM " . $this->categories_table . " WHERE find_in_set( catID, categories) ) AS categories 
 									FROM " . $this->users_table . " ORDER BY userID DESC");



 		include_once 'views/users-overview.php';	

 		echo $this->wrap();

 	}

 	public function new_user() {
 		global $wpdb;

 		// set active tab
 		$active = 'users';
 		require_once plugin_dir_path( __FILE__ ) . '/views/admin-nav.php';

 		echo $this->wrap( 'header', 'New User' );

 		// process form
 		if( isset($_POST['sb-new-user']) ) :

 			unset( $_POST['sb-new-user'] );

 			if(isset($_POST['categories'])) 
 				$_POST['categories'] = implode(',', $_POST['categories']);
 			else
 				$_POST['categories'] = '';


 			$plain_password = $_POST['password'];
 			$_POST['password'] = md5($plain_password);

 			if( $wpdb->insert( $this->users_table, $_POST )) :

 				$loginURL = get_post_field( 'post_name', get_option( 'wpfsam_user_files_page' ) );

 				$mail_body = 'WELCOME TO '.$this->blog_title.', <br /><br/>
 							 We have created the following member account for you: <br/><br/>
 							 
 							 <strong><em>Login email:</em> ' . $_POST['email'] . '</strong><br/>
 							 <strong><em>Login password:</em> ' . $plain_password . '</strong><br/><br/>

 							 Login URL: <a href="'. get_bloginfo('url') . '/'.$loginURL.'">'. get_bloginfo('url') . '/'.$loginURL.'</a>
 							 ';



 				echo $this->message( 'User successfully added. Also emailed the login details.' );

 				wp_mail( $_POST['email'], ''.$this->blog_title.' Account', $mail_body );

 			else:
 				echo $this->message( 'SQL Error ' );
 			endif;

 		endif;
 		// form processing ends
 		
 		$categories = $wpdb->get_results("SELECT * FROM " . $this->categories_table . " ORDER BY name ASC");

		include_once 'views/new-user.php';	

 		echo $this->wrap();

 	}

 	public function update_user() {
 		global $wpdb;

 		// set active tab
 		$active = 'users';
 		require_once plugin_dir_path( __FILE__ ) . '/views/admin-nav.php';

 		echo $this->wrap( 'header', 'Update User' );

 		// process form
 		if( isset($_POST['sb-new-user']) ) :

 			unset( $_POST['sb-new-user'] );

 			if(isset($_POST['categories'])) 
 				$_POST['categories'] = implode(',', $_POST['categories']);
 			else
 				$_POST['categories'] = '';

 			if(isset($_POST['password']) AND !empty($_POST['password'])) {
 				
 				$plain_password = $_POST['password'];
	 			$_POST['password'] = md5($plain_password);

	 			if( $wpdb->update( $this->users_table, 
	 								array('password' => $_POST['password']), 
	 								array('userID' => $_GET['userID']) )) :

	 				$loginURL = get_post_field( 'post_name', get_option( 'wpfsam_user_files_page' ) );

	 				$mail_body = 'Account on '.$this->blog_title.', <br /><br/>
	 							 We have updated the following member account for you: <br/><br/>
	 							 
	 							 <strong><em>Login email:</em> ' . $_POST['email'] . '</strong><br/>
	 							 <strong><em>Login password:</em> ' . $plain_password . '</strong><br/><br/>

	 							 Login URL: <a href="'. get_bloginfo('url') . '/'.$loginURL.'">'. get_bloginfo('url') . '/'.$loginURL.'</a>
	 							 ';



	 				echo $this->message( 'User successfully added. Also emailed the login details.' );

	 				wp_mail( $_POST['email'], ''.$this->blog_title.' Account', $mail_body );

	 			else:
	 				echo $this->message( 'SQL Error ' );
	 			endif;
 			}//update password

 			unset( $_POST['password'] );

 			if( $wpdb->update( $this->users_table, $_POST, array('userID' => $_GET['userID']) )) :
 				echo $this->message( 'User successfully updated.' );
 			endif;

 		endif;
 		// form processing ends
 		
 		$categories = $wpdb->get_results("SELECT * FROM " . $this->categories_table . " ORDER BY name ASC");

 		$u = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM " . $this->users_table . " 
 								WHERE userID = %d", $_GET['userID']));

 		if(!$u) wp_die( "No such user #ID" );

		include_once 'views/update-user.php';	

 		echo $this->wrap();

 	}

 	// front-end user login
 	public function login() {
		global $wpdb;

		ob_start();

		if( isset($_POST['login_email']) AND isset($_POST['login_password']) ) :

			$userID = $wpdb->get_var(
						$wpdb->prepare("SELECT userID FROM " . $this->users_table . "
										WHERE email = %s AND password = %s", $_POST['login_email'], md5($_POST['login_password'])));

			if(!$userID):
				$loginURL = get_post_field( 'post_name', get_option( 'wp_fsam_reset_password_page' ) );
				echo '<div class="error">Either the email address or password you entered is incorrect. Please try again. 
									Forgot your password? <a href="'.get_bloginfo('url').'/'.$loginURL.'">Click here to recover it.</a></div>.';
			else:
				$_SESSION['cg_user'] = $userID;
				$loginURL = get_post_field( 'post_name', get_option( 'wpfsam_user_files_page' ) );
				echo '<meta http-equiv="refresh" content="0; url=' . get_bloginfo('url') . '/'.$loginURL.'" />';
			endif;

		endif;

		$loginURL = get_post_field( 'post_name', get_option( 'wp_fsam_reset_password_page' ) );

		?>
		<form method="POST" action="" id="cg-login-form">
		<dl>
		<dt>Email</dt>
		<dd><input type="email" required="required" name="login_email" /></dd>

		<dt>Password</dt>
		<dd><input type="password" required="required" name="login_password" /></dd>
		
		<dt>
			<input type="submit" name="cg-sb-login" value="Login" /><br />
			Forgot password? <a href="<?=get_bloginfo('url')?>/<?= $loginURL ?>">Click here</a>
		</dt>
		</dl>
		</form>

		<?php 
		return ob_get_clean();
	}

	// recover password
	public function reset_password() {
		global $wpdb;
		ob_start();

		if(!isset($_GET['hash'])):

		if(isset($_POST['reg_email'])) {
			$email = $wpdb->get_row($wpdb->prepare("SELECT email, MD5(CONCAT(userID, email)) AS hash_string 
									FROM " . $this->users_table . " 
									WHERE email = %s", $_POST['reg_email']));

			if(!$email) {
				echo '<div class="error">No such mail into database</div>';
			}else{
				$to = $email->email;

				$loginURL = get_post_field( 'post_name', get_option( 'wp_fsam_reset_password_page' ) );

				$mail_body = 'The following link will take you to a page to reset your password for your '.$this->blog_title.' account.';
				$mail_body .= '<a href="' . get_bloginfo('url') . '/'.$loginURL.'?hash=' . $email->hash_string . '">Password Reset Link</a>';

				#echo $mail_body;

				wp_mail( $to, ''.$this->blog_title.' - Password Reset', $mail_body );

				echo '<div class="success">We have sent you a link to reset your password. Please check your email.</div>';

			}
		}
		?>
		<form method="POST" action="" id="cg-recover-form">
		<dl>
		<dt>Enter email you registered with:</dt>
		<dd><input type="email" required="required" name="reg_email" /></dd>
		<dt><input type="submit" name="sb-reset-email" value="Submit" /></dt>
		</dl>
		</form>
		<?php
		else:
		$hash_string = trim(strip_tags($_GET['hash']));
		if(strlen($hash_string) != 32) {
			echo '<div class="error">Invalid hash</div>';
		}else{
		
			if(isset($_POST['new_password']) AND isset($_POST['confirm_password'])):

			$userID = $wpdb->get_var(
						$wpdb->prepare("SELECT userID FROM " . $this->users_table . "
									WHERE MD5(CONCAT(userID, email)) = %s", $hash_string));

			if(!$userID) {
				echo '<div class="error">No account for this reset hash.</div>';
			}else{
				$wpdb->query(
					$wpdb->prepare("UPDATE " . $this->users_table . " SET password = %s WHERE userID = %d", md5($_POST['new_password']) ,$userID));

				$loginURL = get_post_field( 'post_name', get_option( 'wp_fsam_user_login_page' ) );

				echo '<div class="success">You have successfully reset your password. <a href="' . get_bloginfo('url') . '/'.$loginURL.'">Click here to log in.</a></div>';
			}

			endif;

		}
		?>
		<form method="POST" action="" id="cg-reset-form">
		<dl>
		<dt>Enter new password:</dt>
		<dd><input type="password" required="required" name="new_password" /></dd>

		<dt>Confirm new password:</dt>
		<dd><input type="password" required="required" name="confirm_password" /></dd>

		<dt><input type="submit" name="sb-reset-password" value="Save" /></dt>
		</dl>
		</form>
		<?php
		endif;

		return ob_get_clean();
	}

	// shortcodes and pages
	public function shortcodes_and_pages(  ) {
		global $wpdb;

 		// set active tab
 		$active = 'shortcodes';
 		require_once plugin_dir_path( __FILE__ ) . '/views/admin-nav.php';

 		echo $this->wrap( 'header', 'Shortcodes & Pages' );


 		include_once 'views/shortcodes.php';	

 		echo $this->wrap();
	}


 	public function setup_menu() {
 		add_menu_page( 'WP FSAM', 'WP FSAM', 'manage_options', $this->plugin_permalink, array( $this, 'manage_files' ), plugin_dir_url(__FILE__) . 'static/small-file-icon.png' );

		add_submenu_page( $this->plugin_permalink, 'Manage Files', 'Manage Files', 'manage_options', $this->plugin_permalink, array( $this, 'manage_files' ));
		add_submenu_page( $this->plugin_permalink, 'Manage Categories', 'Manage Categories', 'manage_options', 'fsam_categories', array( $this, 'manage_categories' ));
		add_submenu_page( $this->plugin_permalink, 'Manage Users', 'Manage Users', 'manage_options', 'fsam_users',
		 array( $this, 'manage_users' ));
		add_submenu_page( $this->plugin_permalink, 'Shortcodes & Pages', 'Shortcodes & Pages', 'manage_options', 'fsam_shortcodes', array( $this, 'shortcodes_and_pages' ));
		
		add_submenu_page(  null, 'Update File', null ,'manage_options', 'fsam-update-file', array( $this, 'update_file' ));
		add_submenu_page(  null, 'Upload File', null ,'manage_options', 'fsam-upload-file', array( $this, 'new_file' ));
		add_submenu_page(  null, 'New User', null ,'manage_options', 'fsam-new-user', array( $this, 'new_user' ));
		add_submenu_page(  null, 'Update User', null ,'manage_options', 'fsam-update-user', array( $this, 'update_user' ));
		add_submenu_page(  null, 'Migrate', null ,'manage_options', 'fsam-migrate', array( $this, 'migrate' ));

 	}
 	

}

new WP_FSAM_File_Sharing_Access_Manager();