<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Order Proposal Payment Gateway.
 *
 * Provides a Order Proposal Payment Gateway. Based on code by Mike Pepper.
 *
 * @class       WC_Gateway_Order_Proposal
 * @extends     WC_Payment_Gateway
 * @version     2.1.0
 * @package     WooCommerce/Classes/Payment
 * @author      WooThemes
 */

class WC_Gateway_Order_Proposal extends WC_Payment_Gateway {

	/**
	 * Array of locales
	 *
	 * @var array
	 */
	public $locale;

	/**
	 * Gateway instructions that will be added to the thank you page and emails.
	 *
	 * @var string
	 */
	public $instructions;

	/**
	 * Constructor for the gateway.
	 */
	public function __construct() {

		$this->id                 = 'orderproposal';
		$this->icon               = apply_filters( 'woocommerce_orderproposal_icon', '' );
		$this->has_fields         = false;
		$this->method_title       = __( 'Order Proposal', 'woocommerce-order-proposal' );
		$this->method_description = __( 'Allows payments by Order Proposal', 'woocommerce-order-proposal' );

		// Load the settings.
		$this->init_form_fields();
		$this->init_settings();

		// Define user set variables
		$this->title        = $this->get_option( 'title' );
		$this->description  = $this->get_option( 'description' );
		$this->instructions = $this->get_option( 'instructions' );

		// Order Proposal account fields shown on the thanks page and in emails
		// Actions
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
		add_action( 'woocommerce_thankyou_' . $this->id, array( $this, 'thankyou_page' ) );
		add_action( 'woocommerce_email_before_order_table', array( $this, 'email_instructions' ), 10, 3 );
	}

	/**
	 * Initialise Gateway Settings Form Fields.
	 */
	public function init_form_fields() {

		$this->form_fields = array(
			'enabled' => array(
				'title'   => __( 'Enable/Disable', 'woocommerce' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable Order Proposal', 'woocommerce-order-proposal' ),
				'default' => 'no',
			),
			'title' => array(
				'title'       => __( 'Title', 'woocommerce' ),
				'type'        => 'text',
				'description' => __( 'This controls the title which the user sees during checkout.', 'woocommerce' ),
				'default'     => __( 'Order Proposal', 'woocommerce-order-proposal' ),
				'desc_tip'    => true,
			),
			'description' => array(
				'title'       => __( 'Description', 'woocommerce' ),
				'type'        => 'textarea',
				'description' => __( 'Payment method description that the customer will see during checkout.', 'woocommerce' ),
				'default'     => __( 'This will create an Order Proposal.', 'woocommerce-order-proposal' ),
				'desc_tip'    => true,
			),
			'instructions' => array(
				'title'       => __( 'Instructions', 'woocommerce' ),
				'type'        => 'textarea',
				'description' => __( 'Instructions that will be added to the thank you page and emails.', 'woocommerce-order-proposal' ),
				'default'     => '',
				'desc_tip'    => true,
			),
			'emailproposal' => array( 
				'title'	      => __( 'Order Proposal Email', 'woocommerce-order-proposal' ), 
				'description' => __( 'Send Order Proposal Email instead of Order Email.', 'woocommerce-order-proposal' ),
				'desc_tip'    => true,
				'type'        => 'checkbox',
				'default'     => 'no',
			),
			'auto_accept_proposal_for_all' => array( 
				'title'	      => __( 'Auto-accept proposal requests for all orders', 'woocommerce-order-proposal' ), 
				'description' => __( 'Automatically accept the proposal when this payment method is used during checkout for all orders.', 'woocommerce-order-proposal' ),
				'desc_tip'    => true,
				'type'        => 'checkbox',
				'default'     => 'no',
			),
			'auto_accept_proposal_for_users' => array( 
				'title'	      => __( 'Auto-accept proposal requests for specific users', 'woocommerce-order-proposal' ), 
				'description' => __( 'Automatically accept the proposal when this payment method is used during checkout for selected users.', 'woocommerce-order-proposal' ),
				'desc_tip'    => true,
				'type'        => 'multiselect',
				'class'       => 'wc-enhanced-select',
				'custom_attributes' => array(
					'data-show_for_option_name'   => 'woocommerce_orderproposal_auto_accept_proposal_for_all',
					'data-show_for_option_values' => 'false',
					'data-initial_data'           => $this->get_auto_accept_users(),
					'data-placeholder'            => __( 'Search for users', 'woocommerce-order-proposal' ),
				),
			),
		);

	}

	/**
	 * Get auto accept users as JSON for select2 dropdown.
	 * 
	 * @return string
	 */
	public function get_auto_accept_users(): string {
		$auto_accept_users = $this->get_option( 'auto_accept_proposal_for_users', array() );
		$return_users      = array();
	
		foreach ( $auto_accept_users as $index => $user_id ) {
			$user = get_user_by( 'id', absint( $user_id ) );
			if ( ! $user ) {
				continue;
			}

			$return_users[] = array(
				'id'   => absint( $user_id ),
				'text' => sprintf(
					esc_html( '%s (#%d - %s)' ),
					$user->display_name,
					absint( $user->ID ),
					$user->user_email
				)
			);
		}
	
		return json_encode( $return_users );
	}

	/**
	 * Output for the order received page.
	 *
	 * @param int $order_id
	 */
	public function thankyou_page( $order_id ) {
		if ( $this->instructions ) {
			echo wp_kses_post( wpautop( wptexturize( $this->instructions ) ) );
		}
	}

	/**
	 * Process the payment and return the result.
	 *
	 * @param int $order_id
	 * @return array
	 */
	public function process_payment( $order_id ) {

		// Disable order proposal mail
		$emails = WC_Emails::instance()->emails;

		do_action('woocommerce_order_proposal_gateway_start', $this->get_option( 'emailproposal' ));

		add_action( 'woocommerce_order_status_pending_to_order-proposalreq_notification', array( $emails['WC_Email_Admin_Order_Proposal_Gateway'], 'trigger' ), 10, 2 );

		if ($this->get_option( 'emailproposal' ) != 'no') {
			// Triggers Order Proposal Email
			add_action( 'woocommerce_order_status_pending_to_order-proposalreq_notification', array( $emails['WC_Email_Order_Proposal_Gateway'], 'trigger' ), 10, 2 );
		} else {
			add_action( 'woocommerce_order_status_pending_to_order-proposalreq_notification', array( $emails['WC_Email_Customer_Processing_Order'], 'trigger' ), 10, 2 );
		}

		$order = wc_get_order( $order_id );

		// Set order status to order-proposal or to proposal if accepted automatically
		$auto_accept_users = $this->get_option( 'auto_accept_proposal_for_users', array() );
		$is_auto_accept    = wc_string_to_bool( $this->get_option( 'auto_accept_proposal_for_all', 'no' ) ) || 
							 ( is_array( $auto_accept_users ) && in_array( $order->get_customer_id(), $auto_accept_users ) );
		$status            = apply_filters( 'wc-order-proposal-gateway-status', $is_auto_accept ? 'wc-order-proposal' : 'wc-order-proposalreq', $order_id, $this );
		$status_text       = apply_filters( 'wc-order-proposal-gateway-status-text', $is_auto_accept ? __( 'Order Proposal accepted automatically', 'woocommerce-order-proposal' ) : __( 'Order Proposal Requested', 'woocommerce-order-proposal' ), $order_id, $this );
		$order->update_status( $status, $status_text );

		// Set Order Proposal Time first so we get the default time
		$time = wc_order_proposal_get_date($order_id);
		$order->update_meta_data( WC_Order_Proposal::ORDER_PROPOSAL_TIME, $time );

		// Mark order as order-proposal
		$order->update_meta_data( WC_Order_Proposal::ORDER_PROPOSAL_USED, true );

		// Reduce stock levels
		if ( ! wc_order_proposal_no_reduce_stock() ) {
			wc_reduce_stock_levels( $order_id );
		}

		// keep title to display
		$order->set_payment_method_title( $this->get_title() );
		$order->save_meta_data();
		$order->save();

		// Remove cart
		WC()->cart->empty_cart();

		// Return thankyou redirect
		return array(
			'result'    => 'success',
			'redirect'  => $this->get_return_url( $order ),
		);
	}

	/**
	 * Check If The Gateway Is Available For Use.
	 *
	 * @return bool
	 */
	public function is_available() {
		# Do not show on order pay page in my account
		if ( is_wc_endpoint_url( 'order-pay' ) ) {
			return false;
		}

		# Check if order proposal is allowed based on product categories in checkout
		# This currently works only with the classic checkout.
		if ( is_checkout() && WC()->cart ) {
			$order_categories        = $this->get_order_product_categories( WC()->cart->get_cart() );
			$allowed_categories      = get_option( WC_Order_Proposal::ORDER_PROPOSAL_ENABLE_FOR_PRODUCT_CATEGORIES, array() );
			$allowed_categories_type = get_option( WC_Order_Proposal::ORDER_PROPOSAL_ENABLE_FOR_PRODUCT_CATEGORIES_TYPE, 'only_selected' );

			if ( ! empty( $allowed_categories ) && ! empty( $allowed_categories_type ) ) {
				switch ( $allowed_categories_type ) {
					case 'only_selected':
						return count( $allowed_categories ) === count( $order_categories ) && empty( array_diff( $allowed_categories, $order_categories ) );
					case 'at_least_one':
						return count( array_intersect( $allowed_categories, $order_categories ) ) > 0;
				}
			}
		}

		return parent::is_available();
	}

	/**
	 * Add instructions to the WC emails.
	 *
	 * @param \WC_Abstract_Order $order Order object.
	 * @param bool               $sent_to_admin  Sent to admin.
	 * @param bool               $plain_text Email format: plain text or HTML.
	 * 
	 * @return void
	 */
	public function email_instructions( \WC_Abstract_Order $order, bool $sent_to_admin, bool $plain_text = false ): void {
		if ( $this->instructions && ! $sent_to_admin && $this->id === $order->get_payment_method() ) {
			echo wp_kses_post( wpautop( wptexturize( $this->instructions ) ) . PHP_EOL );
		}
	}

	/**
	 * Function for getting order product categories.
	 * 
	 * @param array $order_items
	 * 
	 * @return array
	 */
	public function get_order_product_categories( array $order_items ): array {
		$order_product_cats = array();
		foreach ( $order_items as $item_id => $item ) {
			$order_product_cats = array_merge( $order_product_cats, wc_get_product_cat_ids( ! is_array( $item ) ? $item->get_product_id() : $item['product_id'] ) );
		}

		return array_unique( $order_product_cats );
	}
}
