<?php
/**
 * WooCommerce Google Analytics Pro
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Google Analytics Pro to newer
 * versions in the future. If you wish to customize WooCommerce Google Analytics Pro for your
 * needs please refer to http://docs.woocommerce.com/document/woocommerce-google-analytics-pro/ for more information.
 *
 * @author      SkyVerge
 * @copyright   Copyright (c) 2015-2024, SkyVerge, Inc.
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking\Events\GA4;

use SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking;
use SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking\Adapters\Product_Item_Event_Data_Adapter;
use SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking\Events\GA4_Event;

defined( 'ABSPATH' ) or exit;

/**
 * The "view item" event.
 *
 * @since 2.0.0
 */
class View_Item_Event extends GA4_Event {


	/** @var string the event ID */
	public const ID = 'view_item';


	/**
	 * @inheritdoc
	 */
	public function get_form_field_title(): string {

		return __( 'View Item', 'woocommerce-google-analytics-pro' );
	}


	/**
	 * @inheritdoc
	 */
	public function get_form_field_description(): string {

		return __( 'Triggered when a customer views a single product.', 'woocommerce-google-analytics-pro' );
	}


	/**
	 * @inheritdoc
	 */
	public function get_default_name(): string {

		return 'view_item';
	}


	/**
	 * @inheritdoc
	 */
	public function register_hooks() : void {

		add_action( 'woocommerce_after_single_product_summary', [ $this, 'handle_single_product_summary' ], 1 );
		add_filter( 'render_block_woocommerce/single-product', [ $this, 'handle_single_product_block' ], 1, 2 );
	}


	/**
	 * Handles tracking when viewing single product page summary.
	 *
	 * @since 2.1.0
	 *
	 * @internal
	 *
	 * @return void
	 */
	public function handle_single_product_summary() {
		global $product;

		$this->track( ! $product ? wc_get_product( get_the_ID() ) : $product );
	}


	/**
	 * Handles tracking when viewing a single product block.
	 *
	 * @since 2.1.0
	 *
	 * @internal
	 *
	 * @return void
	 */
	public function handle_single_product_block( string $block_content, array $block ) : string {

		if ( Tracking::is_frontend_request() && ! empty( $product_id = $block['attrs']['productId'] ) ) {

			$this->track( wc_get_product( $product_id ) );
		}

		return $block_content;
	}


	/**
	 * @inheritdoc
	 */
	public function track( $product = null ) : void {

		if ( ! $product || Tracking::not_page_reload() ) {

			$this->record_via_js( [
				'items'    => [ ( new Product_Item_Event_Data_Adapter( $product ) )->convert_from_source() ],
				'value'    => $product->get_price(),
				'currency' => get_woocommerce_currency(),
				'category' => 'Products',
			] );
		}
	}


}
