<?php
/**
 * WooCommerce Google Analytics Pro
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Google Analytics Pro to newer
 * versions in the future. If you wish to customize WooCommerce Google Analytics Pro for your
 * needs please refer to http://docs.woocommerce.com/document/woocommerce-google-analytics-pro/ for more information.
 *
 * @author      SkyVerge
 * @copyright   Copyright (c) 2015-2024, SkyVerge, Inc.
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking\Events\GA4;

use SkyVerge\WooCommerce\Google_Analytics_Pro\Helpers\Product_Helper;
use SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking;
use SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking\Adapters\Product_Item_Event_Data_Adapter;
use SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking\Events\Contracts\Deferred_Event;
use SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking\Events\GA4_Event;
use SkyVerge\WooCommerce\Google_Analytics_Pro\Tracking\Events\Traits\Has_Deferred_AJAX_Trigger;
use SkyVerge\WooCommerce\PluginFramework\v5_15_11\SV_WC_Helper;
use WC_Product;

defined( 'ABSPATH' ) or exit;

/**
 * The "select item" event.
 *
 * This event may be registered multiple times on a single page.
 *
 * @since 2.0.0
 */
class Select_Item_Event extends GA4_Event implements Deferred_Event {

	use Has_Deferred_AJAX_Trigger;

	/** @var string the event ID */
	public const ID = 'select_item';

	/** @var bool whether we're already tracking products in grid */
	protected bool $tracking_grid_products = false;

	/** @var bool whether we're already tracking products in other blocks */
	protected bool $tracking_block_products = false;

	/** @var string the ajax action name  */
	protected string $ajax_action = 'wc_google_analytics_pro_select_item';

	/** @var string the ajax callback method name  */
	protected string $ajax_callback = 'track_via_ajax';


	/**
	 * @inheritdoc
	 */
	public function get_form_field_title(): string {

		return __( 'Select Item', 'woocommerce-google-analytics-pro' );
	}


	/**
	 * @inheritdoc
	 */
	public function get_form_field_description(): string {

		return __( 'Triggered when a customer clicks a product in listing, such as search results or related products.', 'woocommerce-google-analytics-pro' );
	}


	/**
	 * @inheritdoc
	 */
	public function get_default_name(): string {

		return 'select_item';
	}


	/**
	 * @inheritdoc
	 */
	public function register_hooks() : void {

		$this->register_ajax_hooks();

		// hooks for tracking events in shortcode-based product lists
		add_action( 'woocommerce_before_shop_loop_item', [ $this, 'track' ] );

		// hooks for tracking events on product grid blocks
		add_filter( 'render_block', [ $this, 'track_in_product_grids' ], 10, 2 );

		// hooks for tracking events on the SPA-like "all products"
		add_filter( 'render_block_woocommerce/all-products', [ $this, 'track_in_products_block' ] );
	}


	/**
	 * @inheritdoc
	 *
	 * This method will be called once per each product listed on a page.
	 */
	public function track() : void {

		global $product;

		$this->record_via_js( [
			'category'       => 'Products',
			'item_list_name' => Product_Helper::get_list_type(),
			'items'          => [ ( new Product_Item_Event_Data_Adapter( $product ) )->convert_from_source() ],
		] );
	}


	/**
	 * @inheritdoc
	 *
	 * @return string
	 */
	public function get_trigger_js(): string {

		global $product;

		$product_id = esc_js( $product->get_id() );

		return <<<JS
		document.querySelector( '.products .post-{$product_id} a' ).addEventListener('click', function(event) {
			if ( ! event.target.classList.contains( 'add_to_cart_button' ) ) {
				"__INSERT_TRACKING_CALL_HERE__";
			}
		} );
		JS;
	}


	/**
	 * Tracks click events on products rendered inside a product grid.
	 *
	 * @since 2.1.0
	 * @internal
	 *
	 * @param string $block_content
	 * @param array $block
	 * @return string
	 */
	public function track_in_product_grids( string $block_content, array $block ) : string {

		if ( ! Tracking::is_frontend_request() || Tracking::do_not_track() ) {
			return $block_content;
		}

		$block_name = $block['blockName'] ?? null;

		if ( $block_name && ! $this->tracking_grid_products && Product_Helper::is_product_grid_block( $block_name ) ) {

			$track_event = <<<JS
				document.querySelectorAll( '.wc-block-grid__products' )?.forEach((grid) => {
					grid.addEventListener('click', function(event) {
						let anchor = event.target.closest('.wc-block-grid__product-link')

						if ( anchor ) {
							let parent = anchor.parentElement;
							let button = parent.querySelector('a.add_to_cart_button');
							let product_id = button ? button.getAttribute('data-product_id') : null;
							let block_name = anchor.closest('.wc-block-grid')?.getAttribute('data-block-name') ?? null;

							if ( product_id ) {
								"__INSERT_AJAX_CALL_HERE__"({ product_id, block_name });
							}
						}
					} );
				} );
			JS;

			$this->trigger_via_ajax( $track_event );
			$this->tracking_grid_products = true;
		}

		return $block_content;
	}


	/**
	 * Tracks click events on products rendered inside the "all products" block
	 *
	 * @since 2.1.0
	 *
	 * @internal
	 */
	public function track_in_products_block( string $block_content ) : string {

		if ( ! Tracking::is_frontend_request() || Tracking::do_not_track() ) {
			return $block_content;
		}

		if ( ! $this->tracking_block_products ) {

			// Note: `listName` below is only provided in some rare occasions - it's mostly undefined
			$trigger = <<<JS
			wp.hooks.addAction( 'experimental__woocommerce_blocks-product-view-link', 'woocommerce-google-analytics-pro', function({ product, listName }) {
				"__INSERT_AJAX_CALL_HERE__"({ product_id: product.id, list_name: listName });
			});
			JS;

			$this->trigger_via_ajax( $trigger );
			$this->tracking_grid_products = true;
		}

		return $block_content;
	}


	/**
	 * Tracks the event via AJAX call - used when tracking the event in blocks.
	 *
	 * @since 2.1.0
	 * @internal
	 */
	public function track_via_ajax() : void {

		check_ajax_referer( $this->ajax_action, 'security' );

		$product_id = SV_WC_Helper::get_posted_value( 'product_id' );
		$product    = $product_id ? wc_get_product( $product_id ) : null;
		$block_name = SV_WC_Helper::get_posted_value( 'block_name' );
		$list_name  = SV_WC_Helper::get_posted_value( 'list_name' );

		if ( empty( $product ) ) {
			return;
		}

		$this->record_via_api( [
			'category'       => 'Products',
			'item_list_name' => $list_name ?: Product_Helper::get_list_type( $block_name ),
			'items'          => [ ( new Product_Item_Event_Data_Adapter( $product ) )->convert_from_source() ],
		] );
	}


}
