<?php
/**
 * WooCommerce Google Analytics Pro
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce PayPal Express to newer
 * versions in the future. If you wish to customize WooCommerce PayPal Express for your
 * needs please refer to http://docs.woocommerce.com/document/woocommerce-PayPal Express/
 *
 * @author    SkyVerge
 * @copyright Copyright (c) 2015-2024, SkyVerge, Inc.
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\Google_Analytics_Pro\API\Collect_API;

use SkyVerge\WooCommerce\PluginFramework\v5_15_11 as Framework;

defined( 'ABSPATH' ) or exit;

/**
 * The Collect API request class.
 *
 * @link https://developers.google.com/analytics/devguides/collection/protocol/ga4/reference?client_type=gtag
 * @link https://www.thyngster.com/app-web-google-analytics-measurement-protocol-version-2#building-a-request
 *
 * @since 2.1.0
 */
class Request extends Framework\SV_WC_API_JSON_Request
{

	/** @var array<string, string> Page parameters map for the collect endpoint */
	protected array $page_params_map = [
		'page_location' => 'dl',
		'page_referrer' => 'dr',
		'page_title'    => 'dt',
	];

	/** @var array<string, string> Session parameters map for the collect endpoint */
	protected array $session_params_map = [
		'session_id'     => 'sid',
		'session_number' => 'sct',
	];

	/** @var array<string, string> Item (product) parameters map for the collect endpoint */
	protected array $item_params_map = [
		'item_id'        => 'id',
		'item_name'      => 'nm',
		'item_category'  => 'ca',
		'item_category2' => 'c2',
		'item_category3' => 'c3',
		'item_category4' => 'c4',
		'item_category5' => 'c5',
		'item_variant'   => 'va',
		'quantity'       => 'qt',
		'price'          => 'pr',
		// following params are included for completeness, but are not used by GA Pro at the moment
		'affiliation'    => 'af',
		'coupon'         => 'cp',
		'discount'       => 'ds',
		'item_brand'     => 'br',
		'item_list_id'   => 'li',
		'item_list_name' => 'ln',
		'location_id'    => 'lo',
	];


	/**
	 * Constructs the class.
	 *
	 * Creates the request and maps the request parameters from Measurement Protocol API format to Collect API params.
	 * This allows us to use the Collect API interchangeably with MP API.
	 *
	 * @since 2.1.0
	 *
	 * @param string $measurement_id the Google Analytics tracking ID
	 * @param array $data the request payload - same as for Measurement Protocol API, will be mapped to Collect API params
	 */
	public function __construct( string $measurement_id, array $data = [] ) {

		$this->params = [
			'v'   => 2,
			'tid' => $measurement_id,          // Measurement ID. Required
			'cid' => $data['client_id'],       // Client (anonymous) ID. Required
			'uid' => $data['user_id'] ?? null, // User ID
		];

		// even though MP API allows multiple events in a single call, we only ever track and send a single event at once
		foreach( $data['events'] as $event ) {

			$this->params = array_merge(
				$this->params,
				$this->parse_event_params( $event ),
				$this->parse_page_params( $event ),
				$this->parse_page_params( $event ),
				$this->parse_items( $event ),
				$this->parse_user_properties( $data )
			);
		}
	}


	/**
	 * Parses and returns event params from event data, including event name.
	 *
	 * @since 2.1.0
	 *
	 * @see https://www.thyngster.com/app-web-google-analytics-measurement-protocol-version-2#ecommerce
	 *
	 * @param array $event
	 * @return array
	 */
	protected function parse_event_params( array $event ) : array {

		$event_params = array_diff_key(
			$event['params'],
			$this->session_params_map,
			$this->page_params_map,
			array_flip( ['items', 'currency'] ),
		);

		return (array) array_filter( array_merge(
			[
				'en' => $event['name'],
				'cu' => $event['params']['currency'] ?? '',
			],
			array_combine(
				// event parameters must be prefixed with `ep.` or `epn.`, depending on their value type
				array_map( function ( $key, $value ) {
					$prefix = is_numeric( $value ) ? 'epn.' : 'ep.';

					return $prefix . $key;
				}, array_keys( $event_params ), $event_params ),
				// map boolean values to strings
				array_map(function( $value ) {
					if (is_bool( $value )) {
						$value = $value ? 'true' : '';
					}

					return $value;
				}, $event_params)
			) )
		);
	}


	/**
	 * Parses and returns page params from event data.
	 *
	 * @since 2.1.0
	 *
	 * @see https://www.thyngster.com/app-web-google-analytics-measurement-protocol-version-2#ecommerce
	 *
	 * @param array $event
	 * @return array
	 */
	protected function parse_page_params( array $event ): array {

		return $this->map_params(
			array_intersect_key( $event['params'], $this->page_params_map ),
			$this->page_params_map
		);
	}


	/**
	 * Parses and returns session params from event data.
	 *
	 * @since 2.1.0
	 *
	 * @see https://www.thyngster.com/app-web-google-analytics-measurement-protocol-version-2#ecommerce
	 *
	 * @param array $event
	 * @return array
	 */
	protected function parse_session_params( array $event ): array {

		return $this->map_params(
			array_intersect_key( $event['params'], $this->session_params_map ),
			$this->session_params_map
		);
	}


	/**
	 * Parses and returns items from the event data.
	 *
	 * @since 2.1.0
	 *
	 * @see https://www.thyngster.com/app-web-google-analytics-measurement-protocol-version-2#ecommerce
	 *
	 * @param array $event
	 * @return array
	 */
	protected function parse_items( array $event ) : array {

		$items = $event['params']['items'] ?? [];

		return array_reduce( array_keys( $items ), function( $carry, $key ) use ( $items ) {
			$item = $items[$key];

			$mapped_item = $this->map_params( $item, $this->item_params_map );

			// items (products) need to be prefixed with `pr{n}.`, for example `pr1`, values must be
			// concatenated together as key-value pairs of item properties, separated by tilde
			$carry[ 'pr' . ( $key + 1 ) ] = implode(
				'~',
				array_map( fn ($value, $key ) => $key . $value, $mapped_item, array_keys( $mapped_item ) )
			);

			return $carry;
		}, []);
	}


	/**
	 * Parses and returns user properties from the given data.
	 *
	 * @since 2.1.0
	 *
	 * @see https://www.thyngster.com/app-web-google-analytics-measurement-protocol-version-2#event-parameters
	 *
	 * @param array $data
	 * @return array
	 */
	protected function parse_user_properties( array $data ): array {

		$user_properties = $data['user_properties'] ?? [];

		if ( empty(  $user_properties ) ) {
			return [];
		}

		return (array) array_filter( array_combine(
			// user properties need to be prefixed with `up.`
			array_map( fn( $key ) => 'up.' . $key, array_keys( $user_properties ) ),
			array_map( fn ($property ) => $property['value'] ?? null, $user_properties )
		) );
	}


	/**
	 * Maps params (array keys) to keys as defined by the map.
	 *
	 * @since 2.1.0
	 *
	 * @param array $params
	 * @param array $map
	 * @return array
	 */
	protected function map_params( array $params, array $map ): array {

		return array_filter( array_map( fn ( $origin ) => $params[ $origin ] ?? null, array_flip( $map ) ) );
	}


}
