<?php
/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0
 * @package    CartBounty Pro - Save and recover abandoned carts for WooCommerce
 * @subpackage CartBounty Pro - Save and recover abandoned carts for WooCommerce/includes
 * @author     Streamline.lv
 */
class CartBounty_Pro{

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0
	 * @access   protected
	 * @var      Plugin_Name_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0
	 */
	public function __construct(){

		$this->plugin_name = CARTBOUNTY_PRO_PLUGIN_NAME_SLUG;
		$this->version = CARTBOUNTY_PRO_VERSION_NUMBER;

		$this->load_dependencies();
		$this->define_admin_hooks();
		$this->define_public_hooks();
		$this->define_api_connection_hooks();
		$this->define_privacy_hooks();
		$this->define_coupon_hooks();
		$this->define_automation_hooks();
		$this->define_mailchimp_hooks();
		$this->define_activecampaign_hooks();
		$this->define_getresponse_hooks();
		$this->define_wordpress_hooks();
		$this->define_bulkgate_hooks();
		$this->define_push_notification_hooks();
		$this->define_webhook_hooks();
		$this->define_wpml_hooks();
		$this->define_gtranslate_hooks();
		$this->define_reports_hooks();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Plugin_Name_Loader. Orchestrates the hooks of the plugin.
	 * - Plugin_Name_Admin. Defines all hooks for the admin area.
	 * - Plugin_Name_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0
	 * @access   private
	 */
	private function load_dependencies(){

		/**
		 * The class responsible for orchestrating the actions and filters of the core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-loader.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-cartbounty-pro-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-cartbounty-pro-public.php';

		/**
		 * The class responsible for defining all methods for dealing with API Connections
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-api-connector.php';
		
		/**
		 * The class responsible for defining all actions that occur in the MailChimp area
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-mailchimp.php';

		/**
		 * The class responsible for defining all actions that occur in the ActiveCampaign area
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-activecampaign.php';

		/**
		 * The class responsible for defining all actions that occur in the GetResponse area
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-getresponse.php';

		/**
		 * The class responsible for defining all actions that occur in the WordPress recovery area
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-wordpress.php';

		/**
		 * The class responsible for defining all actions that occur in the BulkGate recovery area
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-bulkgate.php';

		/**
		 * The class responsible for defining all actions that occur in the Push notifications recovery area
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-push-notifications.php';

		/**
		 * The class responsible for defining all actions that occur in the BulkGate recovery area
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-webhook.php';

		/**
		 * The class responsible for defining all actions that occur in the Privacy area
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-privacy.php';

		/**
		 * The class responsible for defining all methods for getting System status
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-status.php';

		/**
		 * The class responsible for defining all methods for dealing with Coupons
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-coupons.php';

		/**
		 * The class responsible for defining all methods for dealing with automation steps (used by WordPress or BulkGate)
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-automation.php';

		/**
		 * The library required for phone number validation
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/libphonenumber-for-php/vendor/autoload.php';

		/**
		 * The library required for Web push notifications
		 */
		$automation = new CartBounty_Pro_Automation();
			
		if( $automation->automation_enabled( 'push_notification' ) ){ //Loading library only after Push notifications are enabled
			require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/web-push-notifications/vendor/autoload.php';
		}

		/**
		 * The class responsible for defining all methods for handling translations for WPML plugin
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-wpml.php';

		/**
		 * The class responsible for defining all methods for handling translations for TranslatePress plugin
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-translatepress.php';

		/**
		 * The class responsible for defining all methods for handling translations for GTranslate plugin
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-gtranslate.php';

		/**
		 * The class responsible for defining all methods for dealing with reports
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-cartbounty-pro-reports.php';

		$this->loader = new CartBounty_Pro_Loader();

	}

	/**
	 * Register all of the hooks related to the admin area functionality of the plugin.
	 *
	 * @since    1.0
	 * @access   private
	 */
	private function define_admin_hooks(){
		$admin = new CartBounty_Pro_Admin( $this->get_plugin_name(), $this->get_version() );
		$status = new CartBounty_Pro_System_Status( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'admin_enqueue_scripts', $admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $admin, 'enqueue_scripts' );
		$this->loader->add_action( 'admin_menu', $admin, 'cartbounty_pro_menu', 10 ); //Creates admin menu
		$this->loader->add_action( 'admin_head', $admin, 'menu_abandoned_count' );
		$this->loader->add_action( 'admin_head', $admin, 'register_admin_tabs' );
		$this->loader->add_filter( 'set-screen-option', $admin, 'save_page_options', 12, 3 ); //Saving Screen options
		$this->loader->add_action( 'admin_init', $admin, 'schedule_events' );
		$this->loader->add_filter( 'cron_schedules', $admin, 'add_custom_wp_cron_intervals' );
		$this->loader->add_action( 'update_option_cartbounty_pro_main_settings', $admin, 'unschedule_notification_sendout_hook' );
		$this->loader->add_action( 'admin_notices', $admin, 'display_notices' ); //Output admin notices if necessary
		$this->loader->add_action( 'cartbounty_pro_notification_sendout_hook', $admin, 'send_email' ); //Hooks into WordPress cron event to launch function for sending out emails
		$this->loader->add_action( 'cartbounty_pro_delete_anonymous_carts_hook', $admin, 'delete_anonymous_carts' ); //Hooks into WordPress cron event to launch function for deleting anonymous carts
		$this->loader->add_action( 'plugins_loaded', $admin, 'check_version' ); //Hook checks if database table of plugin is not outdated
		$this->loader->add_action( 'init', $admin, 'cartbounty_pro_text_domain' ); //Hook checks if database table of plugin is not outdated
		$this->loader->add_action( 'init', $admin, 'check_for_updates' );
		$this->loader->add_filter( 'plugin_action_links_' . CARTBOUNTY_PRO_BASENAME, $admin, 'add_plugin_action_links', 10, 2 ); //Adds additional links on Plugin page
		$this->loader->add_action( 'cartbounty_pro_after_page_title', $admin, 'output_bubble_content' ); //Hooks into hook in order to output bubbles
		$this->loader->add_action( 'admin_init', $admin, 'export_cartbounty_pro_carts' ); //Runs the export carts function once the admin panel loads
		$this->loader->add_action( 'admin_init', $admin, 'export_cartbounty_meta' ); //Runs the export products, coupons, emails function once the admin panel loads
		$this->loader->add_action( 'cartbounty_pro_remove_empty_carts_hook', $admin, 'delete_empty_carts' );
		$this->loader->add_action( 'cartbounty_pro_remove_empty_carts_hook', $admin, 'delete_expired_cartbounty_transients' );
		$this->loader->add_filter( 'woocommerce_billing_fields', $admin, 'edit_checkout_fields', 10, 1 );
		$this->loader->add_action( 'woocommerce_new_order', $admin, 'handle_order', 30 ); //Handling order once it has been created (added since some themes or plugins tend to remove some default WooCommerce actions)
		$this->loader->add_action( 'woocommerce_checkout_order_processed', $admin, 'handle_order', 30 ); //Handling order once it has been processed
		$this->loader->add_action( 'woocommerce_order_status_changed', $admin, 'handle_order', 1000, 4 ); //Handling order in case of order status changes
		$this->loader->add_action( 'wp_ajax_import_carts', $admin, 'import_carts' );
		$this->loader->add_action( 'wp_loaded', $admin, 'trigger_on_load', 15 );
		$this->loader->add_action( 'profile_update', $admin, 'reset_abandoned_cart' ); //Handles clearing of abandoned cart in case a registered user changes his account data like Name, Surname, Location etc.
		$this->loader->add_action( 'wp_ajax_force_sync', $admin, 'force_sync' ); //Handles Force Sync button Ajax action for logged in users
		$this->loader->add_action( 'wp_ajax_disconnect_cartbounty', $admin, 'deactivate_license' ); //Handles CartBounty license deactivation button Ajax action for logged in users
		$this->loader->add_filter( 'admin_body_class', $admin, 'add_cartbounty_body_class' ); //Adding CartBounty class to the body tag
		$this->loader->add_action( 'wp_ajax_get_system_status', $status, 'get_system_status' );
		$this->loader->add_action( 'wp_ajax_handle_notice', $admin, 'handle_notices' ); //Handle notice actions
		$this->loader->add_action( 'plugins_loaded', $admin, 'update_type_column', 100 ); //Temporary function since version 9.3.1. Will be removed in future releases
		$this->loader->add_action( 'cartbounty_pro_admin_email_footer_end', $admin, 'add_email_badge', 100 );
		$this->loader->add_action( 'wp_ajax_save_tcpa_gdpr', $admin, 'save_tcpa_gdpr' );
		$this->loader->add_action( 'wp_ajax_find_items', $admin, 'find_items' ); //Adds ajax Item search (products, coupons etc.)
		$this->loader->add_action( 'wp_ajax_get_consent_placeholders', $admin, 'get_consent_placeholders' );
		$this->loader->add_action( 'pre_update_option', $admin, 'validate_cartbounty_fields', 10, 3 );
	}

	/**
	 * Register all of the hooks related to the public-facing functionality of the plugin.
	 *
	 * @since    1.0
	 * @access   private
	 */
	private function define_public_hooks(){
		$public = new CartBounty_Pro_Public( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'wp_enqueue_scripts', $public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $public, 'enqueue_scripts' );
		$this->loader->add_action( 'wp_ajax_nopriv_cartbounty_pro_save', $public, 'save_cart' ); //Saving cart using Ajax after any changes made by the user on the Email or Phone fields
		$this->loader->add_action( 'wp_ajax_cartbounty_pro_save', $public, 'save_cart' ); //Saving cart using Ajax after any changes made by a logged in user in the Email pr Phone fields
		$this->loader->add_action( 'woocommerce_add_to_cart', $public, 'save_cart', 200 ); //Handles data saving if an item is added to shopping cart
		$this->loader->add_action( 'woocommerce_cart_actions', $public, 'save_cart', 200 ); //Handles data updating if a cart is updated
		$this->loader->add_action( 'woocommerce_cart_item_removed', $public, 'save_cart', 200 ); //Handles data updating if an item is removed from cart
		$this->loader->add_action( 'wp', $public, 'restore_classic_checkout_fields', 10 ); //Restoring previous user input in classic checkout form
		$this->loader->add_action( 'shutdown', $public, 'restore_block_checkout_fields', 200 );
		$this->loader->add_action( 'wp_footer', $public, 'display_exit_intent_form' ); //Outputting the exit intent form in the footer of the page
		$this->loader->add_action( 'wp_ajax_nopriv_check_empty_cart', $public, 'check_empty_cart' ); //Checking if we have an empty cart in case of Ajax action
		$this->loader->add_action( 'wp_ajax_check_empty_cart', $public, 'check_empty_cart' ); //Checking if we have an empty cart in case of Ajax action if the user is logged in
		$this->loader->add_action( 'wp_ajax_nopriv_insert_early_capture', $public, 'display_early_capture_form' ); //Outputting the Early capture form in case if Ajax Add to Cart button pressed if the user is not logged in
		$this->loader->add_action( 'wp_ajax_insert_early_capture', $public, 'display_early_capture_form' ); //Outputting the Early capture form in case if Ajax Add to Cart button pressed if the user is logged in
	}

	/**
	 * Register all of the hooks related to the API Connections
	 *
	 * @since    10.9
	 * @access   private
	 */
	private function define_api_connection_hooks(){
		$api_connector = new CartBounty_Pro_API_Connector();
	}

	/**
	 * Register all of the hooks related to the privacy area functionality of the plugin.
	 *
	 * @since    9.7
	 * @access   private
	 */
	private function define_privacy_hooks(){
		$privacy = new CartBounty_Pro_Privacy();
		$this->loader->add_action( 'cartbounty_pro_remove_empty_carts_hook', $privacy, 'anonymize_abandoned_cart_data' );
		$this->loader->add_action( 'admin_init', $privacy, 'add_privacy_message' );
		$this->loader->add_filter( 'wp_privacy_personal_data_exporters', $privacy, 'cartbounty_pro_exporter', 6 );
		$this->loader->add_filter( 'wp_privacy_personal_data_erasers', $privacy, 'cartbounty_pro_eraser', 10 );
	}

	/**
	 * Register all of the hooks related to the coupon area functionality of the plugin.
	 *
	 * @since    9.7
	 * @access   private
	 */
	private function define_coupon_hooks(){
		$coupons = new CartBounty_Pro_Coupons();
		$this->loader->add_action( 'cartbounty_pro_delete_coupons_hook', $coupons, 'delete_unused_expired_coupons' );
	}

	/**
	 * Register all of the hooks related to the automation area functionality of the plugin.
	 *
	 * @since    9.7
	 * @access   private
	 */
	private function define_automation_hooks(){
		$automation = new CartBounty_Pro_Automation();
		$this->loader->add_action( 'admin_head', $automation, 'handle_stats_reset' );
		$this->loader->add_filter( 'cartbounty_pro_automation_recovery_url', $automation, 'add_click_through_tracking', 10, 3 );
		$this->loader->add_filter( 'cartbounty_pro_automation_unsubscribe_url', $automation, 'add_click_through_tracking', 10, 3 );
		$this->loader->add_action( 'update_option_cartbounty_pro_automation_steps', $automation, 'validate_automation_steps', 50); //Validating automation steps before saving them, setting priority to 50 as we want validation to fire last before saving data
		$this->loader->add_action( 'update_option_cartbounty_pro_bulkgate_steps', $automation, 'validate_automation_steps', 50); //Validating automation steps before saving them, setting priority to 50 as we want validation to fire last before saving data
		$this->loader->add_action( 'update_option_cartbounty_pro_push_notification_steps', $automation, 'validate_automation_steps', 50); //Validating automation steps before saving them, setting priority to 50 as we want validation to fire last before saving data
	}

	/**
	 * Register all of the hooks related to the MailChimp area functionality of the plugin.
	 *
	 * @since    9.7
	 * @access   private
	 */
	private function define_mailchimp_hooks(){
		$mailchimp = new CartBounty_Pro_Mailchimp();
		$this->loader->add_action( 'cartbounty_pro_sync_hook', $mailchimp, 'auto_sync' );
		$this->loader->add_action( 'admin_init', $mailchimp, 'export_last_batch_results' ); //Runs the export MailChimp last synchronization results
		$this->loader->add_action( 'woocommerce_update_coupon', $mailchimp, 'put_coupons' ); //Updating Coupon table if a new coupon code added or updated
		$this->loader->add_action( 'pre_update_option_cartbounty_pro_mailchimp_settings', $mailchimp, 'handle_settings_update', 10, 2 );
	}

	/**
	 * Register all of the hooks related to the ActiveCampaign area functionality of the plugin.
	 *
	 * @since    9.7
	 * @access   private
	 */
	private function define_activecampaign_hooks(){
		$activecampaign = new CartBounty_Pro_ActiveCampaign();
		$this->loader->add_action( 'cartbounty_pro_sync_hook', $activecampaign, 'auto_sync' );
		$this->loader->add_action( 'pre_update_option_cartbounty_pro_activecampaign_settings', $activecampaign, 'handle_settings_update', 10, 2 );
	}

	/**
	 * Register all of the hooks related to the GetResponse area functionality of the plugin.
	 *
	 * @since    9.7
	 * @access   private
	 */
	private function define_getresponse_hooks(){
		$getresponse = new CartBounty_Pro_GetResponse();
		$this->loader->add_action( 'cartbounty_pro_sync_hook', $getresponse, 'auto_sync' );
		$this->loader->add_action( 'update_option_cartbounty_pro_getresponse_settings', $getresponse, 'reset_product_table_getresponse_ids' ); //Hook checks if GetResponse store ID gets changed
		$this->loader->add_action( 'pre_update_option_cartbounty_pro_getresponse_settings', $getresponse, 'handle_settings_update', 10, 2 );
	}

	/**
	 * Register all of the hooks related to the WordPress area functionality of the plugin.
	 *
	 * @since    9.7
	 * @access   private
	 */
	private function define_wordpress_hooks(){
		$wordpress = new CartBounty_Pro_WordPress();
		$this->loader->add_action( 'cartbounty_pro_sync_hook', $wordpress, 'auto_send' );
		$this->loader->add_action( 'update_option_cartbounty_pro_automation_settings', $wordpress, 'sanitize_from_field', 50); //Custom sanitization function for "From" email name
		$this->loader->add_action( 'wp_ajax_email_preview', $wordpress, 'email_preview' );
		$this->loader->add_action( 'wp_ajax_send_test_email', $wordpress, 'send_test' );
	}

	/**
	 * Register all of the hooks related to the WordPress area functionality of the plugin.
	 *
	 * @since    9.7
	 * @access   private
	 */
	private function define_bulkgate_hooks(){
		$bulkgate = new CartBounty_Pro_BulkGate();
		$this->loader->add_action( 'cartbounty_pro_sync_hook', $bulkgate, 'auto_send' );
		$this->loader->add_action( 'wp_ajax_send_test_sms', $bulkgate, 'send_test' );
	}

	/**
	 * Register all of the hooks related to Push notifications area functionality of the plugin.
	 *
	 * @since    9.10
	 * @access   private
	 */
	private function define_push_notification_hooks(){
		$push_notification = new CartBounty_Pro_Push_Notification();
		$this->loader->add_action( 'cartbounty_pro_sync_hook', $push_notification, 'auto_send' );
		$this->loader->add_action( 'wp_footer', $push_notification, 'display_push_notification_permission_request' );
		$this->loader->add_action( 'wp_ajax_nopriv_save_notification_subscription', $push_notification, 'save_push_notification_subscription' );
		$this->loader->add_action( 'wp_ajax_save_notification_subscription', $push_notification, 'save_push_notification_subscription' );
		$this->loader->add_action( 'wp_ajax_nopriv_remove_notification_subscription', $push_notification, 'remove_push_notification_subscription' );
		$this->loader->add_action( 'wp_ajax_remove_notification_subscription', $push_notification, 'remove_push_notification_subscription' );
		$this->loader->add_action( 'wp_ajax_send_test_push_notification', $push_notification, 'send_test_push_notification' );
	}

	/**
	 * Register all of the hooks related to the WordPress area functionality of the plugin.
	 *
	 * @since    9.9
	 * @access   private
	 */
	private function define_webhook_hooks(){
		$webhook = new CartBounty_Pro_Webhook();
		$this->loader->add_action( 'cartbounty_pro_sync_hook', $webhook, 'auto_send' );
		$this->loader->add_action( 'wp_ajax_test_webhook', $webhook, 'test_webhook' );
	}

	/**
	 * Register all of the hooks related to the GetResponse area functionality of the plugin.
	 *
	 * @since    9.7
	 * @access   private
	 */
	private function define_wpml_hooks(){
		$wpml = new CartBounty_Pro_WPML();
		$this->loader->add_action( 'admin_menu', $wpml, 'register_strings' );
		$this->loader->add_action( 'updated_option', $wpml, 'update_strings' );
		$this->loader->add_action( 'wcml_multi_currency_ajax_actions', $wpml, 'add_action_to_multi_currency_ajax', 10, 1 );
	}

	/**
	 * Register all of the hooks related to the GTranslate area functionality of the plugin.
	 *
	 * @since    10.7
	 * @access   private
	 */
	private function define_gtranslate_hooks(){
		$gtranslate = new CartBounty_Pro_GTranslate();
		$this->loader->add_action( 'cartbounty_pro_wordpress_recovery_args', $gtranslate, 'translate_email' );
		$this->loader->add_action( 'cartbounty_pro_push_notification_recovery_args', $gtranslate, 'translate_push_notification' );
		$this->loader->add_action( 'cartbounty_pro_bulkgate_recovery_args', $gtranslate, 'translate_bulkgate_sms' );
	}

	/**
	 * Register all of the hooks related to the reports area functionality of the plugin.
	 *
	 * @since    10.0
	 * @access   private
	 */
	private function define_reports_hooks(){
		$reports = new CartBounty_Pro_Reports();
		$this->loader->add_action( 'admin_init', $reports, 'handle_calendar_period_submit' );
		$this->loader->add_action( 'wp_ajax_update_quick_stats', $reports, 'handle_report_updates' );
		$this->loader->add_action( 'wp_ajax_update_charts', $reports, 'handle_report_updates' );
		$this->loader->add_action( 'wp_ajax_update_top_product_count', $reports, 'handle_top_product_updates' );
		$this->loader->add_action( 'wp_ajax_update_country_count', $reports, 'handle_map_updates' );
		$this->loader->add_action( 'wp_ajax_update_map_report', $reports, 'handle_map_updates' );
		$this->loader->add_action( 'wp_ajax_apply_report_period', $reports, 'handle_ajax_submit' );
		$this->loader->add_action( 'wp_ajax_update_selected_currency', $reports, 'handle_ajax_submit' );
		$this->loader->add_action( 'wp_ajax_update_chart_type', $reports, 'handle_chart_type_updates' );
		$this->loader->add_action( 'cartbounty_pro_remove_empty_carts_hook', $reports, 'update_currency_rates' );
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0
	 */
	public function run(){
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name(){
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0
	 * @return    Plugin_Name_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader(){
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version(){
		return $this->version;
	}

}