<?php
/**
 * The WPML class.
 *
 * Takes care of synamic content translations for WPML
 *
 *
 * @since      9.7.3
 * @package    CartBounty Pro - Save and recover abandoned carts for WooCommerce
 * @subpackage CartBounty Pro - Save and recover abandoned carts for WooCommerce/includes
 * @author     Streamline.lv
 */
class CartBounty_Pro_WPML{

	/**
	 * The admin handler that manages the plugin's settings, options, and backend functionality.
	 *
	 * @since    10.9
	 * @access   protected
	 * @var      CartBounty_Pro_Admin    $admin    Provides methods to control and extend the plugin's admin area.
	 */
	protected $admin = null;

	/**
	 * Get the admin handler (lazy-loaded).
	 * Creates the connector on first use and then reuses the same instance.
	 *
	 * @since 10.9
	 * @access protected
	 * @return CartBounty_Pro_Admin
	 */
	protected function admin(){
		
		if( $this->admin === null ){
			$this->admin = new CartBounty_Pro_Admin( CARTBOUNTY_PRO_PLUGIN_NAME_SLUG, CARTBOUNTY_PRO_VERSION_NUMBER );
		}

		return $this->admin;
	}

	/**
	 * Check if WPML plugin is active or not
	 *
	 * @since    9.7.3
	 * @return   Boolean
	 */
	public function wpml_enabled(){
		global $sitepress;
		$active = false;

		if( is_object( $sitepress ) ){
			$active = true;
		}
		return $active;
	}

	/**
	 * Check if strings have been registered with WPML
	 *
	 * @since    9.7.3
	 * @return   Boolean
	 */
	public function strings_registered(){
		$admin = $this->admin();
		$registered = false;

		if ( $admin->get_settings( 'misc_settings', 'wpml_strings_registered' ) ){
			$registered = true;
		}
		return $registered;
	}

	/**
	 * Register strings for the first time if WPML has been just activated or CartBounty installed
	 * Will be registered just once
	 *
	 * @since    9.7.3
	 */
	public function register_strings(){
		if( !$this->wpml_enabled() ){ //Exit in case WPML disabled
			return;
		}
		if( $this->strings_registered() ){ //Exit if we have already run this once
			return;
		}

		$admin = $this->admin();
		$misc_settings = $admin->get_settings( 'misc_settings' );
		$misc_settings['wpml_strings_registered'] = true;
		$this->save_wpml_fields( $option = '', $register = true );
		update_option( 'cartbounty_pro_misc_settings', $misc_settings );
	}

	/**
	 * Update WPML dynamic translation strings
	 *
	 * @since    9.7.3
	 * @param    string     $option					Option name that is updated
	 */
	public function update_strings( $option ){
		if( !$this->wpml_enabled() ){ //Exit in case WPML disabled
			return;
		}
		$this->save_wpml_fields( $option );
	}

	/**
	 * Register / Update dynamic WPML translation strings
	 *
	 * @since    9.7.3
	 * @param    string     $option					Option name that is updated
	 * @param    boolean    $register				If this is translation string registration request or not. Used for registering strings for the first time
	 */
	public function save_wpml_fields( $option = '', $register = false ){
		$admin = $this->admin();
		$automation = new CartBounty_Pro_Automation();

		//Settings fields
		if( $register || $option == 'cartbounty_pro_main_settings' ){
			$checkout_consent = $admin->get_checkout_consent();
			do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $checkout_consent['name'], $checkout_consent['value'] );
			$tools_consent = $admin->get_tools_consent();
			do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $tools_consent['name'], $tools_consent['value'] );
		}

		//Exit Intent fields
		if( $register || $option == 'cartbounty_pro_exit_intent_settings' ){
			$exit_intent_content = $admin->get_exit_intent_content();
			do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $exit_intent_content['name'], $exit_intent_content['value'] );
			$exit_intent_heading = $admin->get_exit_intent_heading();
			do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $exit_intent_heading['name'], $exit_intent_heading['value'] );
		}

		//Early capture field
		if( $register || $option == 'cartbounty_pro_early_capture_settings' ){
			$early_capture_heading = $admin->get_early_capture_heading();
			do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $early_capture_heading['name'], $early_capture_heading['value'] );
		}

		//Tab notification field
		if( $register || $option == 'cartbounty_pro_tab_notification_settings' ){
			$tab_notification_message = $admin->get_tab_notification_message();
			do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $tab_notification_message['name'], $tab_notification_message['value'] );
		}

		//WordPress recovery fields
		if( $register || $option == 'cartbounty_pro_automation_steps' ){
			$automation_steps = get_option( 'cartbounty_pro_automation_steps' );
			if( is_array( $automation_steps ) ){
				foreach ( $automation_steps as $key => $step ) {
					$wordpress_content = $automation->get_content_input_fields( $key, $step, 'content', 'wordpress' );
					do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $wordpress_content['name'], $wordpress_content['value'] );
					$wordpress_heading = $automation->get_content_input_fields( $key, $step, 'heading', 'wordpress' );
					do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $wordpress_heading['name'], $wordpress_heading['value'] );
					$wordpress_subject = $automation->get_content_input_fields( $key, $step, 'subject', 'wordpress' );
					do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $wordpress_subject['name'], $wordpress_subject['value'] );
				}
			}
		}

		//BulkGate messages
		if( $register || $option == 'cartbounty_pro_bulkgate_steps' ){
			$bulkgate_steps = get_option( 'cartbounty_pro_bulkgate_steps' );
			if( is_array( $bulkgate_steps ) ){
				foreach ( $bulkgate_steps as $key => $step ) {
					$bulkgate_message = $automation->get_content_input_fields( $key, $step, 'content', 'bulkgate' );
					do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $bulkgate_message['name'], $bulkgate_message['value'] );
				}
			}
		}

		//Push notification messages
		if( $register || $option == 'cartbounty_pro_push_notification_steps' ){
			$automation_steps = get_option( 'cartbounty_pro_push_notification_steps' );
			if( is_array( $automation_steps ) ){
				foreach ( $automation_steps as $key => $step ) {
					$notification_content = $automation->get_content_input_fields( $key, $step, 'content', 'push_notification' );
					do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $notification_content['name'], $notification_content['value'] );
					$notification_heading = $automation->get_content_input_fields( $key, $step, 'heading', 'push_notification' );
					do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $notification_heading['name'], $notification_heading['value'] );
				}
			}
		}

		//Push notification message permission request settings
		if( $register || $option == 'cartbounty_pro_push_notification_settings' ){
			$permission_request_heading = $admin->get_notification_permission_request_heading();
			do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $permission_request_heading['name'], $permission_request_heading['value'] );
			$permission_request_content = $admin->get_notification_permission_request_content();
			do_action( 'wpml_register_single_string', 'woo-save-abandoned-carts', $permission_request_content['name'], $permission_request_content['value'] );
		}
	}

	/**
	 * Adding AJAX action that requires WPML multi-currency filters
	 * This is necessary to save abandoned carts in correct currency selected by user after saving abandoned cart data
	 *
	 * @since    9.9
	 * @param    array     $ajax_actions					The array including AJAX actions
	 */
	public function add_action_to_multi_currency_ajax( $ajax_actions ){
		$ajax_actions[] = 'cartbounty_pro_save'; // Add a AJAX action to the array
		return $ajax_actions;
	}
}