<?php

namespace WpieFw\Modules;

use WpieFw\Modules\Definition\WpieModulesDefinition;
use WpieFw\Modules\Resolver\WpieModuleValuesResolverFactory;
use WpieFw\Modules\Resolver\WpieModuleValuesWpOptionsResolver;
use WpieFw\Exceptions\WpieInvalidArgumentException;

if( ! defined( 'ABSPATH' ) ) exit;

/**
 * @author WeePie Plugins
 *
 * An instance of this class retrieves the stored (saved) values through the WpieModuleValuesResolverFactory instance
 */
final class WpieModulesValues
{
	/**
	 * @var WpieModuleValuesWpOptionsResolver
	 */
	private $resolver;

	/**
	 * @param WpieModuleValuesResolverFactory $factory
	 * @param WpieModulesDefinition $definitions
	 *
	 * @uses WpieModuleValuesResolverFactory::getResolver()
	 */
	public function __construct( WpieModuleValuesResolverFactory $factory, WpieModulesDefinition $definitions )
	{
		$this->resolver = $factory->getResolver( $definitions );
	}

	/**
	 * Get the value for a definition
	 *
	 * @param string $shortcut
	 * @param array $replacements
	 * @return string
	 */
	public function getDefinitionValue( $shortcut = '', array $replacements = [] )
	{
		return $this->resolver->getValueForShortcut( $shortcut, $replacements );
	}

	/**
	 * Get stored module files
	 *
	 * @uses WpieModuleValuesWpOptionsResolver::getFiles()
	 *
	 * @return array
	 */
	public function getFiles()
	{
		return $this->resolver->getFiles();
	}

	/**
	 * Set stored module files
	 *
	 * @param array $files
	 *
	 * @uses WpieModuleValuesWpOptionsResolver::setFiles()
	 *
	 * @return WpieModulesValues
	 */
	public function persistFiles( array $files )
	{
		$this->resolver->persistFiles( $files );

		return $this;
	}

	/**
	 * Persist shortcut
	 *
	 * @param string $name
	 * @param array $value
	 * @param string $key optional
	 * @return \WpieFw\Modules\WpieModulesValues
	 */
	public function persist( string $name, array $value, string $key = '' )
	{
		switch( $name ) {
			case 'files':
				$this->persistFiles( $value );
				break;
			default:
				throw new WpieInvalidArgumentException( sprintf( 'Could not persist for modules stored values on name: %s.', $name ) );
				break;
		}

		return $this;
	}

	/**
	 * Delete stored module fields
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::deleteFiles()
	 *
	 * @return \WpieFw\Modules\WpieModulesValues
	 */
	public function deleteFiles()
	{
		$this->resolver->deleteFiles();

		return $this;
	}

	/**
	 * Delete shortcut to all delete methods
	 *
	 * @param string $name
	 * @param string $key optional
	 *
	 * @return \WpieFw\Modules\WpieModulesValues
	 */
	public function delete( $name = '', $key = '' )
	{
		switch( $name ) {
			case 'files':
				$this->deleteFiles();
				break;
			default:
				throw new WpieInvalidArgumentException( sprintf( 'Could not delete for modules stored values on name: %s.', $name ) );
				break;
		}

		return $this;
	}
}